#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
图文匹配检测服务
完整的Web服务器实现
"""

import http.server
import socketserver
import os
import json
import urllib.parse
from pathlib import Path
import time
import requests
import logging
# from obs_upload import OBSUploader

# 配置日志
logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)
import requests
import os
import mimetypes
from typing import Dict, Optional, Union, Tuple
from urllib.parse import quote

class OBSUploader:
    def __init__(self, base_url: str = "https://open.raysgo.com", auth_token: Optional[str] = None):
        """
        Initialize the OBS uploader.
        
        Args:
            base_url: The base URL for the API
            auth_token: The authorization token for API access
        """
        self.base_url = base_url.rstrip('/')
        self.auth_token = auth_token
        self.headers = {
            'Authorization': f'Bearer {auth_token}' if auth_token else None
        }
        # Initialize mimetypes
        mimetypes.init()

    def _get_content_type(self, file_path: Union[str, bytes]) -> Tuple[str, bytes]:
        """
        Get content type and file content from file path or bytes.
        
        Args:
            file_path: Path to the file or file content as bytes
            
        Returns:
            Tuple of (content_type, file_content)
        """
        if isinstance(file_path, str):
            if not os.path.exists(file_path):
                raise FileNotFoundError(f"File not found: {file_path}")
            content_type, _ = mimetypes.guess_type(file_path)
            with open(file_path, 'rb') as f:
                file_content = f.read()
        else:
            file_content = file_path
            # For bytes input, try to detect type from first few bytes
            content_type = 'application/octet-stream'  # Default content type
            
        return content_type or 'application/octet-stream', file_content

    def get_upload_url(self, biz_code: str, object_name: str, content_type: str) -> Dict:
        """
        Get a temporary upload URL for the specified object.
        
        Args:
            biz_code: Business code for the upload
            object_name: Name/path of the object to upload
            content_type: MIME type of the file
            
        Returns:
            Dict containing the upload URL and related information
        """
        endpoint = f"{self.base_url}/aimodel/v1.0/obs/getCreatePostSignature"
        params = {
            'bizCode': biz_code,
            'objectName': object_name,
            'mimeType': content_type
        }
        
        response = requests.get(endpoint, params=params, headers=self.headers)
        response.raise_for_status()
        return response.json()

    def upload_file(self, file_path: Union[str, bytes], biz_code: str, object_name: str) -> Dict:
        """
        Upload a file using temporary credentials.
        
        Args:
            file_path: Path to the file to upload or file content as bytes
            biz_code: Business code for the upload
            object_name: Name/path of the object to upload
            
        Returns:
            Dict containing the upload result and file URL
        """
        # Get content type and file content
        content_type, file_content = self._get_content_type(file_path)
        
        # Get temporary upload URL with content type
        upload_info = self.get_upload_url(biz_code, object_name, content_type)
        
        if upload_info['errCode'] != 0:
            raise Exception(f"Failed to get upload URL: {upload_info['message']}")
        
        upload_url = upload_info['data']['temporarySignatureUrl']
        
        # Upload the file with the correct content type
        headers = {
            'Content-Type': content_type,
            'Content-Length': str(len(file_content))
        }
        
        response = requests.put(upload_url, data=file_content, headers=headers)
        response.raise_for_status()
        
        return {
            'success': True,
            'file_url': upload_info['data']['domain'] + '/' + object_name,
            'object_url_map': upload_info['data']['objectUrlMap']
        }

class ImageTextMatchingHandler(http.server.SimpleHTTPRequestHandler):
    def __init__(self, *args, **kwargs):
        # 设置静态文件目录
        super().__init__(*args, directory="web", **kwargs)
        self.uploader = OBSUploader(auth_token="dcg-4c1e3a7f4fcd415e8c93151ff539d20a")
        if self.uploader:
            print('uploader_sucess!')
        else:
            print('uploader_failed!')
    def setup(self):
        super().setup()
        if not hasattr(self, 'uploader'):
            self.uploader = OBSUploader(auth_token="dcg-4c1e3a7f4fcd415e8c93151ff539d20a")
            print('uploader_initialized in setup')
    def end_headers(self):
        # 添加CORS头
        self.send_header('Access-Control-Allow-Origin', '*')
        self.send_header('Access-Control-Allow-Methods', 'GET, POST, OPTIONS')
        self.send_header('Access-Control-Allow-Headers', 'Content-Type')
        super().end_headers()
    
    def do_OPTIONS(self):
        # 处理预检请求
        self.send_response(200)
        self.end_headers()
    
    def do_POST(self):
        """处理POST请求"""
        if self.path == '/upload':
            self.handle_upload()
        elif self.path == '/api/image_text_matching':
            self.handle_matching_api()
        else:
            self.send_error(404)
    
    def do_GET(self):
        """处理GET请求"""
        if self.path == '/api/health':
            # 健康检查端点
            self.send_response(200)
            self.send_header('Content-Type', 'application/json')
            self.end_headers()
            response = {"status": "ok", "service": "image_text_matching"}
            self.wfile.write(json.dumps(response).encode())
        else:
            # 静态文件处理
            super().do_GET()
    
    def handle_upload(self):
        """处理文件上传"""
        try:
            content_length = int(self.headers['Content-Length'])
            post_data = self.rfile.read(content_length)
            
            # 解析multipart/form-data
            content_type = self.headers['Content-Type']
            if 'multipart/form-data' in content_type:
                boundary = content_type.split('boundary=')[1].encode()
                parts = post_data.split(b'--' + boundary)
                
                for part in parts:
                    if b'Content-Disposition: form-data' in part and b'filename=' in part:
                        # 提取文件名
                        lines = part.split(b'\r\n')
                        filename = "unknown.jpg"
                        for line in lines:
                            if b'Content-Disposition' in line:
                                try:
                                    filename_start = line.find(b'filename="') + 10
                                    filename_end = line.find(b'"', filename_start)
                                    filename = line[filename_start:filename_end].decode()
                                except:
                                    pass
                                break
                        
                        # 提取文件内容
                        content_start = part.find(b'\r\n\r\n') + 4
                        content_end = part.rfind(b'\r\n')
                        if content_end == -1:
                            content_end = len(part)
                        file_content = part[content_start:content_end]
                        
                        if len(file_content) > 0:
                            # 保存文件
                            upload_dir = Path("uploads")
                            upload_dir.mkdir(exist_ok=True)
                            
                            timestamp = str(int(time.time() * 1000))
                            file_extension = filename.split('.')[-1] if '.' in filename else 'jpg'
                            save_filename = f"upload_{timestamp}.{file_extension}"
                            file_path = upload_dir / save_filename
                            
                            with open(file_path, 'wb') as f:
                                f.write(file_content)
                            
                            # Upload a file
                            try:
                                logger.info(f"file_path=/home/wangtengbo/uploads/{save_filename}")
                                result = self.uploader.upload_file(
                                    file_path=f"/home/wangtengbo/uploads/{save_filename}",
                                    biz_code="image_text_matching",
                                    object_name=f"image/{save_filename}"
                                )
                                file_url=result['file_url']
                                logger.info('用户上传的图像转换为file_url={file_url}')
                            except Exception as e:
                                print(f"Upload failed: {str(e)}") 
                            # 返回文件URL #这里需要封装一个oss
                            #file_url = f"http://localhost:8080/uploads/{save_filename}"
                            response = {"success": True, "url": file_url}
                            
                            self.send_response(200)
                            self.send_header('Content-Type', 'application/json')
                            self.end_headers()
                            self.wfile.write(json.dumps(response).encode())
                            return
            
            self.send_error(400, "Bad Request - No valid file found")
            
        except Exception as e:
            logger.error(f"Upload error: {e}")
            self.send_error(500, f"Internal Server Error: {str(e)}")
    
    def handle_matching_api(self):
        """处理图文匹配API请求"""
        try:
            content_length = int(self.headers['Content-Length'])
            post_data = self.rfile.read(content_length)
            
            # 解析JSON数据
            data = json.loads(post_data.decode('utf-8'))
            
            # 调用图文匹配检测服务
            result = self.call_matching_service(data)
            
            # 返回结果
            self.send_response(200)
            self.send_header('Content-Type', 'application/json')
            self.end_headers()
            self.wfile.write(json.dumps(result, ensure_ascii=False).encode('utf-8'))
            
        except json.JSONDecodeError as e:
            logger.error(f"JSON decode error: {e}")
            self.send_error(400, "Invalid JSON data")
        except Exception as e:
            logger.error(f"Matching API error: {e}")
            error_response = {
                "code": 500,
                "message": f"服务器内部错误: {str(e)}",
                "result": []
            }
            self.send_response(500)
            self.send_header('Content-Type', 'application/json')
            self.end_headers()
            self.wfile.write(json.dumps(error_response, ensure_ascii=False).encode('utf-8'))
    
    def call_matching_service(self, data):
        """调用外部图文匹配检测服务"""
        api_url = "http://localhost:29500/v1/image_text_matching"
        
        try:
            # 准备请求数据
            request_data = {
                "illustration_url": data.get("illustration_url", ""),
                "caption_text": data.get("caption_text", ""),
                "context_info": data.get("context_info", "")
            }
            
            logger.info(f"Calling matching service with data: {request_data}")
            
            # 调用外部服务
            response = requests.post(api_url, json=request_data, timeout=30)
            
            if response.status_code == 200:
                result = response.json()
                logger.info(f"Matching service response: {result}")
                return result
            else:
                logger.error(f"Matching service error: {response.status_code} - {response.text}")
                return {
                    "code": response.status_code,
                    "message": f"外部服务错误: {response.text}",
                    "result": []
                }
                
        except requests.exceptions.RequestException as e:
            logger.error(f"Request to matching service failed: {e}")
            return {
                "code": 503,
                "message": f"无法连接到图文匹配检测服务 (localhost:29500): {str(e)}",
                "result": []
            }
        except Exception as e:
            logger.error(f"Unexpected error in matching service call: {e}")
            return {
                "code": 500,
                "message": f"服务调用失败: {str(e)}",
                "result": []
            }


def setup_directories():
    """创建必要的目录结构"""
    directories = ["web", "uploads"]
    for directory in directories:
        os.makedirs(directory, exist_ok=True)
        print(f"✓ 创建目录: {directory}")


def create_index_html():
    """创建前端HTML文件"""
    html_content = '''<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>图文匹配检测服务</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }

        .container {
            max-width: 800px;
            margin: 0 auto;
            background: rgba(255, 255, 255, 0.95);
            border-radius: 20px;
            padding: 40px;
            box-shadow: 0 20px 40px rgba(0, 0, 0, 0.1);
            backdrop-filter: blur(10px);
        }

        .header {
            text-align: center;
            margin-bottom: 40px;
        }

        .header h1 {
            color: #333;
            font-size: 2.5rem;
            margin-bottom: 10px;
            background: linear-gradient(45deg, #667eea, #764ba2);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }

        .header p {
            color: #666;
            font-size: 1.1rem;
        }

        .config-section {
            background: #f8f9fa;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 30px;
            border-left: 4px solid #667eea;
        }

        .config-section h4 {
            color: #333;
            margin-bottom: 15px;
        }

        .config-input {
            width: 100%;
            padding: 10px;
            border: 1px solid #ddd;
            border-radius: 5px;
            margin-bottom: 10px;
        }

        .form-section {
            margin-bottom: 30px;
        }

        .form-section h3 {
            color: #333;
            margin-bottom: 15px;
            font-size: 1.3rem;
            display: flex;
            align-items: center;
        }

        .form-section h3::before {
            content: '';
            width: 4px;
            height: 20px;
            background: linear-gradient(45deg, #667eea, #764ba2);
            margin-right: 10px;
            border-radius: 2px;
        }

        .upload-area {
            border: 2px dashed #ccc;
            border-radius: 15px;
            padding: 40px;
            text-align: center;
            background: #fafafa;
            transition: all 0.3s ease;
            cursor: pointer;
            position: relative;
            overflow: hidden;
        }

        .upload-area:hover {
            border-color: #667eea;
            background: #f0f4ff;
            transform: translateY(-2px);
            box-shadow: 0 10px 20px rgba(102, 126, 234, 0.1);
        }

        .upload-area.dragover {
            border-color: #667eea;
            background: #e8f0ff;
        }

        .upload-icon {
            font-size: 3rem;
            color: #ccc;
            margin-bottom: 20px;
        }

        .upload-text {
            color: #666;
            font-size: 1.1rem;
            margin-bottom: 10px;
        }

        .upload-hint {
            color: #999;
            font-size: 0.9rem;
        }

        .file-input {
            display: none;
        }

        .preview-container {
            margin-top: 20px;
            text-align: center;
        }

        .preview-image {
            max-width: 100%;
            max-height: 300px;
            border-radius: 10px;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
            margin-bottom: 15px;
        }

        .file-info {
            background: #f8f9fa;
            padding: 15px;
            border-radius: 10px;
            margin-top: 15px;
        }

        .input-group {
            margin-bottom: 25px;
        }

        .input-group label {
            display: block;
            margin-bottom: 8px;
            color: #333;
            font-weight: 600;
        }

        .input-group input,
        .input-group textarea {
            width: 100%;
            padding: 15px;
            border: 2px solid #e0e0e0;
            border-radius: 10px;
            font-size: 1rem;
            transition: all 0.3s ease;
            background: #fff;
        }

        .input-group input:focus,
        .input-group textarea:focus {
            outline: none;
            border-color: #667eea;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        }

        .input-group textarea {
            min-height: 120px;
            resize: vertical;
        }

        .submit-btn {
            width: 100%;
            padding: 18px;
            background: linear-gradient(45deg, #667eea, #764ba2);
            color: white;
            border: none;
            border-radius: 12px;
            font-size: 1.2rem;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }

        .submit-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 15px 30px rgba(102, 126, 234, 0.3);
        }

        .submit-btn:disabled {
            opacity: 0.6;
            cursor: not-allowed;
            transform: none;
        }

        .loading {
            display: none;
            margin-left: 10px;
        }

        .spinner {
            width: 20px;
            height: 20px;
            border: 2px solid #ffffff30;
            border-top: 2px solid #ffffff;
            border-radius: 50%;
            animation: spin 1s linear infinite;
        }

        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }

        .result-section {
            margin-top: 40px;
            display: none;
        }

        .result-content {
            background: #f8f9fa;
            padding: 25px;
            border-radius: 15px;
            border-left: 5px solid #667eea;
        }

        .result-success {
            border-left-color: #28a745;
            background: #d4edda;
        }

        .result-error {
            border-left-color: #dc3545;
            background: #f8d7da;
        }

        .result-item {
            background: white;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 15px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.05);
        }

        .error-type {
            color: #dc3545;
            font-weight: 600;
            margin-bottom: 10px;
        }

        .error-text {
            color: #333;
            margin-bottom: 10px;
            padding: 10px;
            background: #f8f9fa;
            border-radius: 5px;
        }

        .error-reason {
            color: #666;
            font-style: italic;
        }

        .clear-btn {
            margin-top: 20px;
            padding: 10px 20px;
            background: #6c757d;
            color: white;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .clear-btn:hover {
            background: #5a6268;
        }

        .status-indicator {
            display: inline-block;
            width: 10px;
            height: 10px;
            border-radius: 50%;
            margin-right: 8px;
        }

        .status-online {
            background: #28a745;
        }

        .status-offline {
            background: #dc3545;
        }

        @media (max-width: 768px) {
            .container {
                padding: 20px;
                margin: 10px;
            }
            
            .header h1 {
                font-size: 2rem;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>🔍 图文匹配检测服务</h1>
            <p>上传图片并输入相关文本，检测图片与文本的匹配度</p>
        </div>

        <div class="config-section">
            <h4>⚙️ 服务配置</h4>
            <label>API服务地址:</label>
            <input type="text" id="apiUrl" class="config-input" value="http://localhost:8080" placeholder="http://localhost:8080">
            <div style="margin-top: 10px;">
                <span>服务状态: </span>
                <span class="status-indicator status-offline" id="statusDot"></span>
                <span id="statusText">检测中...</span>
                <button onclick="checkApiStatus()" style="margin-left: 10px; padding: 5px 10px; border: none; background: #667eea; color: white; border-radius: 5px; cursor: pointer;">检测连接</button>
            </div>
        </div>

        <form id="matchingForm">
            <div class="form-section">
                <h3>📸 上传图片</h3>
                <div class="upload-area" id="uploadArea">
                    <div class="upload-icon">📁</div>
                    <div class="upload-text">点击或拖拽上传图片</div>
                    <div class="upload-hint">支持 JPG、PNG、JPEG 格式</div>
                    <input type="file" id="imageFile" class="file-input" accept="image/*" required>
                </div>
                <div class="preview-container" id="previewContainer" style="display: none;">
                    <img id="previewImage" class="preview-image" alt="预览图片">
                    <div class="file-info" id="fileInfo"></div>
                </div>
            </div>

            <div class="form-section">
                <h3>✏️ 文本信息</h3>
                <div class="input-group">
                    <label for="captionText">图片标题/描述文本：</label>
                    <input type="text" id="captionText" placeholder="请输入与图片相关的标题或描述文本" required>
                </div>
                <div class="input-group">
                    <label for="contextInfo">上下文信息（可选）：</label>
                    <textarea id="contextInfo" placeholder="请输入相关的上下文信息，可以为空"></textarea>
                </div>
            </div>

            <button type="submit" class="submit-btn" id="submitBtn">
                🚀 开始检测
                <div class="loading" id="loading">
                    <div class="spinner"></div>
                </div>
            </button>
        </form>

        <div class="result-section" id="resultSection">
            <div class="form-section">
                <h3>📊 检测结果</h3>
                <div class="result-content" id="resultContent"></div>
                <button type="button" class="clear-btn" onclick="clearResults()">清除结果</button>
            </div>
        </div>
    </div>

    <script>
        const uploadArea = document.getElementById('uploadArea');
        const fileInput = document.getElementById('imageFile');
        const previewContainer = document.getElementById('previewContainer');
        const previewImage = document.getElementById('previewImage');
        const fileInfo = document.getElementById('fileInfo');
        const form = document.getElementById('matchingForm');
        const submitBtn = document.getElementById('submitBtn');
        const loading = document.getElementById('loading');
        const resultSection = document.getElementById('resultSection');
        const resultContent = document.getElementById('resultContent');
        const apiUrlInput = document.getElementById('apiUrl');
        const statusDot = document.getElementById('statusDot');
        const statusText = document.getElementById('statusText');

        // 页面加载时检测API状态
        window.onload = function() {
            checkApiStatus();
        };

        // 检测API状态
        async function checkApiStatus() {
            const apiUrl = apiUrlInput.value.trim();
            statusText.textContent = '检测中...';
            statusDot.className = 'status-indicator status-offline';
            
            try {
                const response = await fetch(`${apiUrl}/api/health`, {
                    method: 'GET',
                    timeout: 5000
                });
                
                if (response.ok) {
                    statusText.textContent = '服务在线';
                    statusDot.className = 'status-indicator status-online';
                } else {
                    statusText.textContent = '服务离线';
                    statusDot.className = 'status-indicator status-offline';
                }
            } catch (error) {
                statusText.textContent = '连接失败';
                statusDot.className = 'status-indicator status-offline';
            }
        }

        // 上传区域点击事件
        uploadArea.addEventListener('click', () => {
            fileInput.click();
        });

        // 拖拽上传功能
        uploadArea.addEventListener('dragover', (e) => {
            e.preventDefault();
            uploadArea.classList.add('dragover');
        });

        uploadArea.addEventListener('dragleave', () => {
            uploadArea.classList.remove('dragover');
        });

        uploadArea.addEventListener('drop', (e) => {
            e.preventDefault();
            uploadArea.classList.remove('dragover');
            const files = e.dataTransfer.files;
            if (files.length > 0) {
                handleFileSelect(files[0]);
            }
        });

        // 文件选择事件
        fileInput.addEventListener('change', (e) => {
            if (e.target.files.length > 0) {
                handleFileSelect(e.target.files[0]);
            }
        });

        // 处理文件选择
        function handleFileSelect(file) {
            if (!file.type.startsWith('image/')) {
                alert('请选择图片文件！');
                return;
            }

            // 显示预览
            const reader = new FileReader();
            reader.onload = (e) => {
                previewImage.src = e.target.result;
                previewContainer.style.display = 'block';
                
                // 显示文件信息
                fileInfo.innerHTML = `
                    <strong>文件名：</strong>${file.name}<br>
                    <strong>文件大小：</strong>${(file.size / 1024 / 1024).toFixed(2)} MB<br>
                    <strong>文件类型：</strong>${file.type}
                `;
            };
            reader.readAsDataURL(file);

            // 将文件对象保存到input中
            const dt = new DataTransfer();
            dt.items.add(file);
            fileInput.files = dt.files;
        }

        // 上传图片
        async function uploadImage(file) {
            const formData = new FormData();
            formData.append('image', file);

            try {
                const response = await fetch('/upload', {
                    method: 'POST',
                    body: formData
                });

                if (!response.ok) {
                    throw new Error(`上传失败: ${response.status}`);
                }

                const result = await response.json();
                return result.url;
            } catch (error) {
                console.error('上传错误:', error);
                throw error;
            }
        }

        // 表单提交事件
        form.addEventListener('submit', async (e) => {
            e.preventDefault();

            const file = fileInput.files[0];
            const captionText = document.getElementById('captionText').value.trim();
            const contextInfo = document.getElementById('contextInfo').value.trim();
            const apiUrl = apiUrlInput.value.trim();

            if (!file) {
                alert('请选择图片文件！');
                return;
            }

            if (!captionText) {
                alert('请输入图片标题或描述文本！');
                return;
            }

            if (!apiUrl) {
                alert('请输入API服务地址！');
                return;
            }

            // 显示加载状态
            submitBtn.disabled = true;
            loading.style.display = 'inline-block';
            resultSection.style.display = 'none';

            try {
                // 上传图片获取URL
                console.log('开始上传图片...');
                const imageUrl = await uploadImage(file);
                console.log('图片上传成功:', imageUrl);

                // 调用图文匹配API
                console.log('开始调用图文匹配API...');
                const response = await fetch(`${apiUrl}/api/image_text_matching`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        illustration_url: imageUrl,
                        caption_text: captionText,
                        context_info: contextInfo
                    })
                });

                const result = await response.json();
                console.log('API响应结果:', result);
                displayResult(result);

            } catch (error) {
                console.error('处理失败:', error);
                displayError(`处理失败: ${error.message}`);
            } finally {
                // 隐藏加载状态
                submitBtn.disabled = false;
                loading.style.display = 'none';
            }
        });

        // 显示结果
        function displayResult(result) {
            resultSection.style.display = 'block';
            
            if (result.code === 200) {
                if (result.result && result.result.length > 0) {
                    // 有匹配问题
                    resultContent.className = 'result-content result-error';
                    let html = '<h4>❌ 检测到图文不匹配问题：</h4>';
                    
                    result.result.forEach((item, index) => {
                        html += `
                            <div class="result-item">
                                <div class="error-type">错误类型: ${item.error_type}</div>
                                <div class="error-text"><strong>文本内容:</strong> "${item.context_text}"</div>
                                <div class="error-reason"><strong>不匹配原因:</strong> ${item.error_reson}</div>
                            </div>
                        `;
                    });
                    
                    resultContent.innerHTML = html;
                } else {
                    // 匹配成功
                    resultContent.className = 'result-content result-success';
                    resultContent.innerHTML = `
                        <h4>✅ 图文匹配检测通过</h4>
                        <p>图片与文本内容匹配度良好，未发现明显的不匹配问题。</p>
                        <div class="result-item">
                            <strong>检测模型:</strong> ${result.model || 'qwen'}<br>
                            <strong>检测时间:</strong> ${new Date().toLocaleString()}
                        </div>
                    `;
                }
            } else {
                // API错误
                resultContent.className = 'result-content result-error';
                resultContent.innerHTML = `
                    <h4>❌ 检测失败</h4>
                    <p><strong>错误信息:</strong> ${result.message}</p>
                    <p><strong>错误代码:</strong> ${result.code}</p>
                `;
            }
        }

        // 显示错误信息
        function displayError(message) {
            resultSection.style.display = 'block';
            resultContent.className = 'result-content result-error';
            resultContent.innerHTML = `
                <h4>❌ 处理失败</h4>
                <p>${message}</p>
            `;
        }

        // 清除结果
        function clearResults() {
            resultSection.style.display = 'none';
            resultContent.innerHTML = '';
        }
    </script>
</body>
</html>'''
    
    with open("web/index.html", "w", encoding="utf-8") as f:
        f.write(html_content)
    print("✓ 创建前端文件: web/index.html")


def start_server(port=8080):
    """启动Web服务器"""
    print("正在初始化图文匹配检测服务...")
    
    # 设置目录结构
    setup_directories()
    
    # 创建前端文件
    create_index_html()
    
    print("\n" + "="*50)
    print("🚀 图文匹配检测服务启动中...")
    print("="*50)
    
    # 启动服务器
    with socketserver.TCPServer(("", port), ImageTextMatchingHandler) as httpd:
        server_address = ("", port)
        print(f"✓ 服务器绑定地址: http://localhost:{port}")
        print(f"✓ 静态文件目录: web/")
        print(f"✓ 上传文件目录: uploads/")
        print(f"✓ 外部API地址: http://localhost:29500")
        
        try:
            print(f"\n🌐 服务器启动成功！")
            print(f"📱 请在浏览器中访问: http://localhost:{port}")
            print(f"🔧 管理界面: http://localhost:{port}/api/health")
            print("\n" + "="*50)
            print("按 Ctrl+C 停止服务器")
            print("="*50)
            
            httpd.serve_forever()
            
        except KeyboardInterrupt:
            print("\n\n⏹️  正在停止服务器...")
            httpd.shutdown()
            print("✓ 服务器已停止")


if __name__ == "__main__":
    import sys
    
    # 解析命令行参数
    port = 8080
    if len(sys.argv) > 1:
        try:
            port = int(sys.argv[1])
        except ValueError:
            print("❌ 端口号必须是数字")
            sys.exit(1)
    
    # 启动服务器
    start_server(port)