import requests
import os
import mimetypes
from typing import Dict, Optional, Union, Tuple
from urllib.parse import quote

class OBSUploader:
    def __init__(self, base_url: str = "https://open.raysgo.com", auth_token: Optional[str] = None):
        """
        Initialize the OBS uploader.
        
        Args:
            base_url: The base URL for the API
            auth_token: The authorization token for API access
        """
        self.base_url = base_url.rstrip('/')
        self.auth_token = auth_token
        self.headers = {
            'Authorization': f'Bearer {auth_token}' if auth_token else None
        }
        # Initialize mimetypes
        mimetypes.init()

    def _get_content_type(self, file_path: Union[str, bytes]) -> Tuple[str, bytes]:
        """
        Get content type and file content from file path or bytes.
        
        Args:
            file_path: Path to the file or file content as bytes
            
        Returns:
            Tuple of (content_type, file_content)
        """
        if isinstance(file_path, str):
            if not os.path.exists(file_path):
                raise FileNotFoundError(f"File not found: {file_path}")
            content_type, _ = mimetypes.guess_type(file_path)
            with open(file_path, 'rb') as f:
                file_content = f.read()
        else:
            file_content = file_path
            # For bytes input, try to detect type from first few bytes
            content_type = 'application/octet-stream'  # Default content type
            
        return content_type or 'application/octet-stream', file_content

    def get_upload_url(self, biz_code: str, object_name: str, content_type: str) -> Dict:
        """
        Get a temporary upload URL for the specified object.
        
        Args:
            biz_code: Business code for the upload
            object_name: Name/path of the object to upload
            content_type: MIME type of the file
            
        Returns:
            Dict containing the upload URL and related information
        """
        endpoint = f"{self.base_url}/aimodel/v1.0/obs/getCreatePostSignature"
        params = {
            'bizCode': biz_code,
            'objectName': object_name,
            'mimeType': content_type
        }
        
        response = requests.get(endpoint, params=params, headers=self.headers)
        response.raise_for_status()
        return response.json()

    def upload_file(self, file_path: Union[str, bytes], biz_code: str, object_name: str) -> Dict:
        """
        Upload a file using temporary credentials.
        
        Args:
            file_path: Path to the file to upload or file content as bytes
            biz_code: Business code for the upload
            object_name: Name/path of the object to upload
            
        Returns:
            Dict containing the upload result and file URL
        """
        # Get content type and file content
        content_type, file_content = self._get_content_type(file_path)
        
        # Get temporary upload URL with content type
        upload_info = self.get_upload_url(biz_code, object_name, content_type)
        
        if upload_info['errCode'] != 0:
            raise Exception(f"Failed to get upload URL: {upload_info['message']}")
        
        upload_url = upload_info['data']['temporarySignatureUrl']
        
        # Upload the file with the correct content type
        headers = {
            'Content-Type': content_type,
            'Content-Length': str(len(file_content))
        }
        
        response = requests.put(upload_url, data=file_content, headers=headers)
        response.raise_for_status()
        
        return {
            'success': True,
            'file_url': upload_info['data']['domain'] + '/' + object_name,
            'object_url_map': upload_info['data']['objectUrlMap']
        }

# Example usage:
if __name__ == "__main__":
    # Initialize uploader
    uploader = OBSUploader(auth_token="dcg-4c1e3a7f4fcd415e8c93151ff539d20a")
    
    # Upload a file
    try:
        result = uploader.upload_file(
            file_path="/data/wangtengbo/formula_node4_测试/logs/logs/2025-03-02/images/0d307e97071846a1b144e7dfb4d44241_20250302073213192/formula_1.png",
            biz_code="formula",
            object_name="image/test.jpg"
        )
        print(f"File uploaded successfully! URL: {result['file_url']}")
    except Exception as e:
        print(f"Upload failed: {str(e)}") 