import re

def normalize_latex_string(s: str) -> str:
    """标准化 LaTeX 字符串"""
    s = s.replace(" ", "").replace("$", "")
    s = re.sub(r'\\[ ,;!]', '', s)  # 移除空白控制符
    s = re.sub(r'\^\{\s*(.*?)\s*\}', r'^\1', s)  # 处理上标
    s = re.sub(r'\_\{\s*(.*?)\s*\}', r'_\1', s)  # 处理下标
    s = s.replace("{", "").replace("}", "")  # 移除花括号
    s = s.replace(r'\cdot', '*').replace(r'\times', '*')  # 替换乘号
    s = re.sub(r'\\frac\s*\{([^{}]+)\}\s*\{([^{}]+)\}', r'(\1/\2)', s)  # 处理分数
    s = re.sub(r'\\sqrt\s*\{([^{}]+)\}', r'sqrt(\1)', s)  # 处理平方根
    s = s.replace(r'\left', '').replace(r'\right', '')  # 移除 \left 和 \right
    s = re.sub(r'\\mathrm|\\mathbf|\\text|\\displaystyle|\\normalfont|\\rm', '', s)  # 移除样式命令
    s = s.replace("\\", "")  # 去除残余的反斜杠
    s = s.replace("≥", ">=").replace("≤", "<=")  # 标准化符号
    return s.strip()

def extract_latex(text: str) -> list:
    """提取文本中的所有 LaTeX 公式"""
    matches = re.findall(r'\$(.*?)\$', text)
    return matches

def latex_match(A: str, B: str) -> bool:
    """检查 A 是否匹配 B 中的任意公式"""
    # 提取并标准化 B 中的所有公式
    B_formulas = extract_latex(B)
    B_norm_formulas = [normalize_latex_string(f) for f in B_formulas]
    
    # 标准化 A
    A_norm = normalize_latex_string(A)
    
    # 检查 A 是否匹配任意标准化的公式
    for formula in B_norm_formulas:
        if A_norm == formula:
            return True
    return False

if __name__ == "__main__":
    text_B = r"\frac { \sqrt { 8 } } { \sqrt { 2 a } } = \frac { \sqrt { 8 } \cdot \sqrt { 2 a } } { \sqrt { 2 a } \cdot \sqrt { 2 a } } = \frac { 4 \sqrt { a } } { 2 a } = \frac { 2 \sqrt { a } } { a }"
    text_A = r"'(3)$\\frac { \\sqrt { 8 } } { \\sqrt { 2 a } } = \\frac { \\sqrt { 8 } \\cdot \\sqrt { 2 a } } { \\sqrt { 2 a } \\cdot \\sqrt { 2 a } } = \\frac { 4 \\sqrt { a } } { 2 a } = \\frac { 2 \\sqrt { a } } { a } .$"
    print(latex_match(text_B, text_A))  # 应该返回 True

