import base64
from PIL import Image
from io import BytesIO
import cv2
import numpy as np
import matplotlib.pyplot as plt

def get_base64_from_image(image):
    """
    将输入的图像转换为Base64编码的字符串。
    
    :param image: 输入图像（OpenCV格式）
    :return: Base64编码的字符串
    """
    # 将图像转换为PIL格式
    pil_image = Image.fromarray(cv2.cvtColor(image, cv2.COLOR_BGR2RGB))
    
    # 使用BytesIO将图像保存到内存中的字节流
    buffered = BytesIO()
    pil_image.save(buffered, format="PNG")
    
    # 获取字节流并进行Base64编码
    img_str = base64.b64encode(buffered.getvalue()).decode('utf-8')
    
    return img_str

def get_subimage_base64(image_path, coordinates, scale_factor=1):
    """
    提取子图并返回其Base64编码的字符串。
    
    :param image_path: 输入图像路径（可以是图像文件或PDF文件）
    :param coordinates: 以8个数字表示的四个角坐标，顺序为左上，右上，右下，左下
    :param scale_factor: 如果输入为PDF，scale_factor 用于将坐标从72dpi转换为目标图像的像素坐标（默认为1，无需转换）
    :return: Base64编码的子图
    """

    image = cv2.imread(image_path)
    if image is None:
        raise ValueError(f"Failed to read the image file '{image_path}'.")

    # 获取图像的尺寸
    height, width = image.shape[:2]
    
    # 解析输入坐标
    x1, y1, x2, y2, x3, y3, x4, y4 = map(int, coordinates)

    # 计算裁剪区域的矩形边界
    x_min = min(x1, x2, x3, x4)
    y_min = min(y1, y2, y3, y4)
    x_max = max(x1, x2, x3, x4)
    y_max = max(y1, y2, y3, y4)

    # 确保裁剪区域不超出图像边界
    x_min = max(0, x_min)
    y_min = max(0, y_min)
    x_max = min(width, x_max)
    y_max = min(height, y_max)

    # 提取ROI
    roi = image[y_min:y_max, x_min:x_max]
    
    # 如果子图为空，抛出异常
    if roi.size == 0:
        raise ValueError(f"The extracted region is empty for the given coordinates: {coordinates}")

    # 将提取的子图转换为Base64编码并返回
    return get_base64_from_image(roi)


def visualize_base64_image(base64_str, output_path=None):
    """
    可视化Base64编码的图像，并使用cv2存储图像。
    
    :param base64_str: Base64编码的图像字符串
    :param output_path: 可选的输出路径，如果提供则使用cv2保存图像到文件
    """
    # 解码Base64编码为图像字节流
    img_data = base64.b64decode(base64_str)
    
    # 使用BytesIO将字节流读取为PIL图像
    img = Image.open(BytesIO(img_data))
    
    # 将PIL图像转换为OpenCV格式（BGR）
    open_cv_image = np.array(img)
    open_cv_image = open_cv_image[:, :, ::-1]  # 从RGB转为BGR
    
    # 使用cv2保存图像，如果指定了输出路径
    if output_path:
        cv2.imwrite(output_path, open_cv_image)
        print(f"Image saved to: {output_path}")


if __name__ == '__main__':
    # 示例：图像坐标为 (left, upper, right, lower)
    image_path = "/data/wangtengbo/Deployments_Formula_Checker_v5_复现并微调_v7.0-合和OCR-没有版面/app/logs_data/sub_images/20241210142852866/formula_1.png"
    coordinates = [74, 143, 1240, 143, 1240, 241, 74, 241]
    
    # 获取子图的Base64编码
    base64_subimage = get_subimage_base64(image_path, coordinates)
   
    
    # 可视化Base64编码的子图
    visualize_base64_image(base64_subimage,output_path='/data/wangtengbo/Deployments_Formula_Checker_v5_复现并微调_v7.0-合和OCR-没有版面/app/utils/base64.jpg')
