import re

class FormulaProcessor:
    def __init__(self):
        # 预编译正则表达式，提高性能
        self.chinese_characters_pattern = re.compile(r'[\u4e00-\u9fff]')
        self.english_characters_pattern = re.compile(r'[a-zA-Z]')
        self.digits_pattern = re.compile(r'\d')
        self.expression_pattern = re.compile(r'(\b[a-zA-Z]+\b\s*=\s*\d+)')
        self.latex_formula_pattern = re.compile(r'\\[a-zA-Z]+\{.*?\}|\$.*?\$')
        self.exclusion_keywords = ['例题',  '题目','答案','讲解','练习','新课教授','化简','例','概念辨析']

    def contains_latex_formula(self, string):
        return bool(self.latex_formula_pattern.search(string))

    def is_formula(self, string):
        # 规则1：包含等号或者包含LaTeX公式
        if '=' not in string and not self.contains_latex_formula(string):
            return False
        
        # 规则3：包含“例题”、“题”、“题目”的都不是公式
        if any(keyword in string for keyword in self.exclusion_keywords):
            return False
        
        # 规则2：只有中文、英文，或者同时包含中文以及英文和数字的肯定不是公式，除非包含LaTeX公式
        if self.chinese_characters_pattern.search(string) and (
            self.english_characters_pattern.search(string) or self.digits_pattern.search(string)):
            if not self.contains_latex_formula(string):
                return False

        # 规则4：排除简单的表达式
        if re.match(r'[a-zA-Z]\s*=\s*[a-zA-Z0-9+\-*/^]+', string):
            return True

        # 通过所有规则检查，返回True
        return True

# 测试示例
if __name__ == "__main__":
    checker = FormulaProcessor()
    test_strings = [
        "1．了解二次根式、最简二次根式的概念,理解二次根式的性质 2.了解二次根式(根号下仅限于数)的加、减、乘、除运算法则,会用它们进 行有关的简单四则运算. 一 课时分配 本章教学约需8 课时,具体安排如下: 16.1二次根式2 课时 16.2二次根式的乘除3 课时 16.3二次根式的加减2 课时 小结1课时。",
        "最简二次根式的概念,理解二次根式的性质",
        "x^2 + y^2 = z^2",
        "E = mc^2",
        "F = ma",
        'a*b=10',
        "理解并掌握(\sqrt{a})^{2}=a (a \geq 0)",
        "本章内容主要有两个部分，它们分别是二次根式的有关概念、性质和二次根 式的四则运算. 本章的第一部分是二次根式的有关概念和性质.教材从两个电视塔的传播 半径之比出发，引入人二次根式的概念.接着根据有理数的算术平方根的意义，顺 理成章地推导出二次根式的两个性质： $\left( \sqrt{a} \right)^{3}=a \left( a \geq 0 \right) \rightarrow \sqrt{a^{3}}=a \left( a \geq 0 \right)$  本章的第二部分是二次根式的四则运算.教材遵循由特殊到一般的规律，由 学生通过分析、概括、交流、归纳等过程,探究得到二次根式的乘除运算法则:  $\sqrt{a}+ \sqrt{b}= \sqrt{ab} \left( a \geq 0,b \geq 0 \right) \text{和} \frac{ \sqrt{a}}{ \sqrt{b}}= \sqrt{ \frac{a}{b}} \left( a \geq 0,b>0 \right)$ .在此基础上，又通过进一步 类比,引出二次根式的加减运算.教材注重知识之间的联系，将乘法公式运用到 二次根式的四则运算中，以简化二次根式的运算"
        ,
        '1.直线与抛物线的交点问题 要解决直线与抛物线的位置关系问题，可把直线方程与抛物线方程联 立，消去y（或消去x)得出关于x（或关于y)的一个方程 $ax^{2}+bx+c=0$   $y$ ,其中二 次项系 $a$ 有可能为0，此时直线与抛物线有一个交点. 当二次项系数 $a \neq 0$ 时, $\Delta=b^{2}-4ac$  若△=0,则直线与抛物线没有公共点； 若 $\Delta>0$ ,则直线与抛物线有且只有一个公共点； 若 $\Delta<0$ 则直线与抛物线有两个不同的公共点. 2.弦长问题 设弦的端点为 $A \left( x_{1},y_{1} \right),B \left( x_{2},y_{2} \right)$  (1)一般弦长： $\left| AB \right|= \sqrt{1+k^{2}} \left| x_{1}-x_{2} \right.$ 域| $AB \left| \right.= \sqrt{1+ \frac{1}{k^{2}}} \left| y_{1}-y_{2} \right|$ (其中 k为弦所在直线的斜率） (2)焦点弦长: $\left| AB \right|=x_{1}+x_{2}-p.$  3.中点弦问题 若 $M \left( x_{0},y_{0} \right)$ 是抛物线 $y^{2}=2px \left( p>0 \right)$ 的弦 $AB$ 的中点，则直线 $AB$ 的斜率 为 $k_{AB}= \frac{p}{y_{0}}$'
        ,
        '库伦定律表达式：F=k1y2 r2' #:True
    ]

    results = [checker.is_formula(s) for s in test_strings]
    print(results)  # [False, False, True, True, True, True, True, True, True]
