import cv2
import base64
import numpy as np

def get_base64_from_image(image):
    """
    将图像转换为Base64编码。

    :param image: 输入图像的NumPy数组
    :return: Base64编码的字符串
    """
    _, buffer = cv2.imencode('.jpg', image)
    base64_str = base64.b64encode(buffer).decode('utf-8')
    return base64_str

def draw_bounding_boxes(image_path, coordinates_list):
    """
    在图像上绘制多个红色边框并返回带框图像的Base64编码。

    :param image_path: 图像路径
    :param coordinates_list: 每个元素为以8个数字表示的四个角坐标，顺序为左上，右上，右下，左下
    :return: 图像Base64编码的字符串
    """
    image = cv2.imread(image_path)
    if image is None:
        raise ValueError(f"Failed to read the image file '{image_path}'.")

    for coordinates in coordinates_list:
        # 解析输入坐标
        x1, y1, x2, y2, x3, y3, x4, y4 = map(int, coordinates)

        # 计算矩形边界
        x_min = min(x1, x2, x3, x4)
        y_min = min(y1, y2, y3, y4)
        x_max = max(x1, x2, x3, x4)
        y_max = max(y1, y2, y3, y4)

        # 确保边界不超出图像范围
        x_min = max(0, x_min)
        y_min = max(0, y_min)
        x_max = min(image.shape[1], x_max)
        y_max = min(image.shape[0], y_max)

        # 绘制红色矩形框 (B, G, R) -> (0, 0, 255)
        cv2.rectangle(image, (x_min, y_min), (x_max, y_max), (0, 0, 255), 2)

    # 转换为Base64编码
    return get_base64_from_image(image)

def get_formula_boundingbox_base64_list(return_sub_img_path_results):
    """
    根据输入路径和坐标信息，返回每张图像包含红色边框的Base64编码列表。

    :param return_sub_img_path_results: 字典，键为图像路径，值为包含边界框的坐标列表
    :return: Base64编码的列表
    """
    base64_list = []

    for image_path, coordinates_list in return_sub_img_path_results.items():
        # 在图像上绘制红色框并转换为Base64编码
        base64_image = draw_bounding_boxes(image_path, coordinates_list)
        base64_list.append(base64_image)

    return base64_list

# 示例调用
if __name__ == "__main__":
    return_sub_img_path_results = {
        "example_image.jpg": [
            [100, 50, 200, 50, 200, 150, 100, 150],
            [300, 200, 400, 200, 400, 300, 300, 300]
        ]
    }

    base64_lists = get_formula_boundingbox_base64_list(return_sub_img_path_results)
    for base64_str in base64_lists:
        print(base64_str)