import base64
from PIL import Image
from io import BytesIO
import cv2
import numpy as np
import cv2
import numpy as np
import os
from loguru import logger

def draw_box_and_save(image_path, coordinates, draw_box_sub_img_save_dir,save_img_name,scale_factor=1):
    """
    在图像上绘制多边形框并保存回原路径，返回图像路径。

    :param image_path: 输入图像路径
    :param coordinates: 以8个数字表示的四个角坐标，顺序为左上、右上、右下、左下
    :param scale_factor: 可选的缩放因子（若不需要则默认为1）
    :return: 保存后的图像路径（即 image_path）
    """
    # 读取图像

    image = cv2.imread(image_path)
    if image is None:
        raise ValueError(f"无法读取图像文件: {image_path}")

    # 解析坐标并应用缩放
    x1, y1, x2, y2, x3, y3, x4, y4 = map(float, coordinates)
    x1, y1, x2, y2, x3, y3, x4, y4 = [int(c * scale_factor) for c in (x1, y1, x2, y2, x3, y3, x4, y4)]

    # 构造多边形顶点数组
    pts = np.array([[x1, y1],
                    [x2, y2],
                    [x3, y3],
                    [x4, y4]], dtype=np.int32).reshape((-1, 1, 2))

    # 在图像上绘制红色多边形框
    cv2.polylines(image, [pts], isClosed=True, color=(0, 0, 255), thickness=8)
    save_image_path=os.path.join(draw_box_sub_img_save_dir,save_img_name+'.png')
    # 保存覆盖原图
    success = cv2.imwrite(save_image_path, image)
    if not success:
        logger.info(f'保存图像错误！\n\nsave_image_path={save_image_path}')
        raise IOError(f"无法将图像保存到: {save_image_path}")

    return save_image_path
def get_base64_from_image(image):
    """
    将输入的图像转换为Base64编码的字符串。

    :param image: 输入图像（OpenCV格式）
    :return: Base64编码的字符串
    """
    # 将图像转换为PIL格式
    pil_image = Image.fromarray(cv2.cvtColor(image, cv2.COLOR_BGR2RGB))

    # 使用BytesIO将图像保存到内存中的字节流
    buffered = BytesIO()
    pil_image.save(buffered, format="PNG")

    # 获取字节流并进行Base64编码
    img_str = base64.b64encode(buffered.getvalue()).decode('utf-8')

    return img_str

def get_subimage_base64_boxes(image_path, coordinates, scale_factor=1):
    """
    提取子图并返回其Base64编码的字符串。

    :param image_path: 输入图像路径（可以是图像文件或PDF文件）
    :param coordinates: 以8个数字表示的四个角坐标，顺序为左上，右上，右下，左下
    :param scale_factor: 如果输入为PDF，scale_factor 用于将坐标从72dpi转换为目标图像的像素坐标（默认为1，无需转换）
    :return: Base64编码的子图
    """

    image = cv2.imread(image_path)
    if image is None:
        raise ValueError(f"Failed to read the image file '{image_path}'.")

    # 获取图像的尺寸
    height, width = image.shape[:2]

    # 解析输入坐标
    x1, y1, x2, y2, x3, y3, x4, y4 = map(int, coordinates)

    # 计算矩形边界的顶点
    pts = np.array([[x1, y1], [x2, y2], [x3, y3], [x4, y4]], np.int32)
    pts = pts.reshape((-1, 1, 2))

    # 在图像上绘制红色边框
    image_with_box = image.copy()
    cv2.polylines(image_with_box, [pts], isClosed=True, color=(0, 0, 255), thickness=8)

    # 将修改后的图像转换为Base64编码并返回
    return get_base64_from_image(image_with_box)

def visualize_base64_image(base64_str, output_path=None):
    """
    可视化Base64编码的图像，并使用cv2存储图像。

    :param base64_str: Base64编码的图像字符串
    :param output_path: 可选的输出路径，如果提供则使用cv2保存图像到文件
    """
    # 解码Base64编码为图像字节流
    img_data = base64.b64decode(base64_str)

    # 使用BytesIO将字节流读取为PIL图像
    img = Image.open(BytesIO(img_data))

    # 将PIL图像转换为OpenCV格式（BGR）
    open_cv_image = np.array(img)
    open_cv_image = open_cv_image[:, :, ::-1]  # 从RGB转为BGR

    # 使用cv2保存图像，如果指定了输出路径
    if output_path:
        cv2.imwrite(output_path, open_cv_image)
        print(f"Image saved to: {output_path}")

if __name__ == '__main__':
    # 示例：图像坐标为 (left, upper, right, lower)
    image_path = "/data/wangtengbo/Deployments_Formula_Checker_v5_复现并微调_v7.0-合和OCR-没有版面/app/logs_data/sub_images/20241214164802726/formula_1.png"
    coordinates = [55, 1476, 3659, 1510, 3645, 3070, 41, 3042]

    # 获取添加红色边界框的图像的Base64编码
    base64_image_with_box = get_subimage_base64_boxes(image_path, coordinates)

    # 可视化Base64编码的图像
    visualize_base64_image(base64_image_with_box, output_path='/data/wangtengbo/Deployments_Formula_Checker_v5_复现并微调_v7.0-合和OCR-没有版面/app/utils/image_with_box.jpg')
