package com.pcloud.book.riddle.facade;

import com.pcloud.book.base.exception.BookBizException;
import com.pcloud.book.riddle.biz.RiddleRecordBiz;
import com.pcloud.book.riddle.dto.CouponClickDTO;
import com.pcloud.book.riddle.dto.GroupRiddleDTO;
import com.pcloud.book.riddle.dto.RiddleCouponDTO;
import com.pcloud.book.riddle.dto.RiddleOpenDTO;
import com.pcloud.book.riddle.entity.RiddleClickRecord;
import com.pcloud.book.riddle.entity.RiddleCoupon;
import com.pcloud.common.dto.ResponseDto;
import com.pcloud.common.page.PageBeanNew;
import com.pcloud.common.permission.PermissionException;
import com.pcloud.common.utils.SessionUtil;
import com.pcloud.common.utils.cookie.Cookie;
import com.pcloud.common.utils.string.StringUtil;
import com.pcloud.wechatgroup.message.dto.SendTextDTO;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.CookieValue;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;

/**
 * @描述：猜谜语外部接口
 * @作者：zhuyajie
 * @创建时间：17:53 2019/7/26
 * @版本：1.0
 */
@Api(description = "猜谜语外部接口")
@RequestMapping(value = "/riddle")
@RestController("riddleFacade")
public class RiddleFacade {

    @Autowired
    private RiddleRecordBiz riddleRecordBiz;

    @ApiOperation(value = "发送信息", httpMethod = "POST")
    @PostMapping("sendText")
    public ResponseDto<?> sendText(@RequestBody @ApiParam SendTextDTO sendTextDTO){
        riddleRecordBiz.riddleProcess(sendTextDTO);
        return new ResponseDto<>();
    }

    @ApiOperation(value = "猜谜语详情列表",httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "name", value = "name", required = false, dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "currentPage", value = "当前页", required = true, dataType = "int", paramType = "query"),
            @ApiImplicitParam(name = "numPerPage", value = "每页条数", required = true, dataType = "int", paramType = "query"),
    })
    @GetMapping("listPage")
    public ResponseDto<PageBeanNew<GroupRiddleDTO>> listPage(
            @RequestHeader("token") String token, @RequestParam(value = "name", required = false) String name,
            @RequestParam("currentPage") Integer currentPage, @RequestParam("numPerPage") Integer numPerPage) throws PermissionException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (null == numPerPage || null == currentPage) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "分页参数缺失");
        }
        PageBeanNew<GroupRiddleDTO> pageBean = riddleRecordBiz.listPage(partyId, currentPage, numPerPage, name);
        return new ResponseDto<>(pageBean);
    }

    @ApiOperation(value = "更新猜谜语状态",httpMethod = "POST")
    @PostMapping("updateRiddleOpenStatus")
    public ResponseDto<?> updateRiddleOpenStatus(@RequestHeader("token") String token, @RequestBody @ApiParam RiddleOpenDTO riddleOpenDTO) throws PermissionException {
        SessionUtil.getToken4Redis(token);
        if (null == riddleOpenDTO) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数缺失");
        }
        Long qrcodeId = riddleOpenDTO.getGroupQrcodeId();
        Boolean status = riddleOpenDTO.getStatus();
        if (null == qrcodeId || null == status) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数缺失");
        }
        riddleRecordBiz.updateRiddleOpenStatus(qrcodeId, status);
        return new ResponseDto<>();
    }

    @ApiOperation(value = "添加点击记录",httpMethod = "POST")
    @PostMapping("addClickTrack")
    public ResponseDto<?> addClickTrack(@CookieValue("userInfo") String userInfo, @RequestBody @ApiParam RiddleClickRecord riddleClickRecord){
        if (null == riddleClickRecord){
            throw new BookBizException(BookBizException.PARAM_IS_NULL,"参数为空");
        }
        if (StringUtil.isEmpty(riddleClickRecord.getWxGroupId()) || null == riddleClickRecord.getClickType()){
            throw new BookBizException(BookBizException.PARAM_IS_NULL,"参数缺失");
        }
        Long wechatUserId = Cookie.getId(userInfo,Cookie._WECHAT_USER_ID);
        riddleClickRecord.setWechatUserId(wechatUserId);
        riddleRecordBiz.addClickTrack(riddleClickRecord);
        return new ResponseDto<>();
    }

    @ApiOperation(value = "群猜谜语优惠券许愿次数", httpMethod = "GET")
    @ApiImplicitParam(name = "wxGroupId", value = "wxGroupId", required = false, dataType = "string", paramType = "query")
    @GetMapping("getGroupWishCount4Wechat")
    public ResponseDto<CouponClickDTO> getGroupWishCount4Wechat(
            @CookieValue("userInfo") String userInfo, @RequestParam(value = "wxGroupId", required = false) String wxGroupId) {
        if (StringUtil.isEmpty(wxGroupId)) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL,"缺少群id");
        }
        Long wechatUserId = Cookie.getId(userInfo, Cookie._WECHAT_USER_ID);
        CouponClickDTO dto = riddleRecordBiz.getGroupWishCount(wxGroupId,wechatUserId);
        return new ResponseDto<>(dto);
    }

    @ApiOperation(value = "新增/修改自定义优惠券", httpMethod = "POST")
    @PostMapping("addCoupon")
    public ResponseDto<?> addCoupon(@RequestHeader("token") String token, @RequestBody @ApiParam RiddleCoupon riddleCoupon) throws PermissionException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (null == riddleCoupon) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数缺失");
        }
        if (null == riddleCoupon.getStatus()) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "缺少是否开启自定义优惠券设置");
        }
        if (riddleCoupon.getStatus() && (StringUtil.isEmpty(riddleCoupon.getTitle()) || StringUtil.isEmpty(riddleCoupon.getPictUrl())
                || StringUtil.isEmpty(riddleCoupon.getCouponClickUrl()) || null == riddleCoupon.getZkFinalPrice()
                || null == riddleCoupon.getCouponPrice() || null == riddleCoupon.getType())) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数缺失");
        }
        riddleRecordBiz.addCoupon(riddleCoupon, adviserId);
        return new ResponseDto<>();
    }

    @ApiOperation(value = "编辑获取猜谜语自定义优惠券", httpMethod = "GET")
    @GetMapping("getByAdviser")
    public ResponseDto<RiddleCouponDTO> getByAdviser(@RequestHeader("token") String token) throws PermissionException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        RiddleCouponDTO dto = riddleRecordBiz.getByAdviser(adviserId);
        return new ResponseDto<>(dto);
    }

    @ApiOperation(value = "用户获取猜谜语自定义优惠券", httpMethod = "GET")
    @ApiImplicitParam(name = "wxGroupId", value = "wxGroupId", required = false, dataType = "string", paramType = "query")
    @GetMapping("getCouponByGroup4Wechat")
    public ResponseDto<RiddleCouponDTO> getCouponByGroup4Wechat(
            @CookieValue("userInfo") String userInfo, @RequestParam(value = "wxGroupId", required = true) String wxGroupId) {
        if (StringUtil.isEmpty(wxGroupId)) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "缺少群id");
        }
        Cookie.getUserInfo(userInfo);
        RiddleCouponDTO dto = riddleRecordBiz.getCouponByGroup(wxGroupId);
        return new ResponseDto<>(dto);
    }
}
