package com.pcloud.book.group.facade.impl;

import com.pcloud.book.base.exception.BookBizException;
import com.pcloud.book.group.biz.BookGroupBiz;
import com.pcloud.book.group.biz.BookGroupClassifyBiz;
import com.pcloud.book.group.dto.GroupStoreMyPayDto;
import com.pcloud.book.group.entity.BookGroup;
import com.pcloud.book.group.entity.BookGroupClassify;
import com.pcloud.book.group.facade.BookGroupFacade;
import com.pcloud.book.group.vo.*;
import com.pcloud.common.dto.ResponseDto;
import com.pcloud.common.exceptions.BizException;
import com.pcloud.common.page.PageBean;
import com.pcloud.common.page.PageBeanNew;
import com.pcloud.common.page.PageParam;
import com.pcloud.common.permission.PermissionException;
import com.pcloud.common.utils.SessionUtil;
import com.pcloud.common.utils.cookie.Cookie;
import com.pcloud.common.utils.string.StringUtil;
import org.codehaus.jackson.JsonParseException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.bind.annotation.CookieValue;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import io.swagger.annotations.ApiOperation;

/**
 * Description 社群书群二维码接口实现类
 * Created by PENG on 2019/4/17.
 */
@RestController("bookGroupFacade")
@RequestMapping("bookGroup")
public class BookGroupFacadeImpl implements BookGroupFacade {

    @Autowired
    private BookGroupBiz bookGroupBiz;
    @Autowired
    private BookGroupClassifyBiz bookGroupClassifyBiz;

    /**
     * 获取社群书群二维码信息
     */
    @Override
    @RequestMapping(value = "getBookGroupInfo", method = RequestMethod.GET)
    public ResponseDto<?> getBookGroupInfo(@RequestHeader("token") String token, @RequestParam(value = "bookGroupId", required = false) Long bookGroupId)
            throws BizException, PermissionException, JsonParseException {
        if (null == bookGroupId) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        SessionUtil.getToken4Redis(token);
        return new ResponseDto<>(bookGroupBiz.getBookGroupInfo(bookGroupId));
    }

    /**
     * 获取社群书群二维码信息
     */
    @Override
    @RequestMapping(value = "getBookGroupInfo4Wechat", method = RequestMethod.GET)
    public ResponseDto<?> getBookGroupInfo4Wechat(@RequestParam(value = "bookGroupId", required = false) Long bookGroupId)
            throws BizException, PermissionException, JsonParseException {
        if (null == bookGroupId) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        return new ResponseDto<>(bookGroupBiz.getBookGroupInfo4Wechat(bookGroupId));
    }

    /**
     * 获取社群书群二维码信息
     */
    @Override
    @RequestMapping(value = "getBookGroupInfoByBookId", method = RequestMethod.GET)
    public ResponseDto<?> getBookGroupInfoByBookId(@RequestHeader("token") String token, @RequestParam(value = "bookId", required = false) Long bookId,
            @RequestParam(value = "channelId", required = false) Long channelId,
            @RequestParam(value = "addType", required = false) Integer addType,
            @RequestParam(value = "sceneId", required = false) Long sceneId) throws BizException, PermissionException, JsonParseException {
        if (null == bookId || null == channelId) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        return new ResponseDto<>(bookGroupBiz.getBookGroupInfoByBookId(bookId, channelId, adviserId, addType, sceneId));
    }

    /**
     * 是否有社群码
     */
    @Override
    @RequestMapping(value = "isHaveQrcode", method = RequestMethod.GET)
    public ResponseDto<Boolean> isHaveQrcode(@RequestHeader("token") String token, @RequestParam(value = "bookId", required = false) Long bookId,
                                             @RequestParam(value = "channelId", required = false) Long channelId) throws BizException, PermissionException, JsonParseException {
        if (null == bookId || null == channelId) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        return new ResponseDto<Boolean>(bookGroupBiz.isHaveQrcode(bookId, channelId, adviserId));
    }


    /**
     * 更新群二维码信息
     */
    @Override
    @RequestMapping(value = "updateBookGroup", method = RequestMethod.POST)
    public ResponseDto<?> updateBookGroup(@RequestHeader("token") String token, @RequestBody BookGroup bookGroup)
            throws BizException, PermissionException, JsonParseException {
        if (null == bookGroup) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        bookGroupBiz.updateBookGroup(bookGroup);
        return new ResponseDto<>();
    }

    /**
     * 根据社群码ID获取书名
     */
    @Override
    @RequestMapping(value = "getBookNameByBookGroupId", method = RequestMethod.GET)
    public ResponseDto<?> getBookNameByBookGroupId(@RequestParam(value = "bookGroupId", required = false) Long bookGroupId)
            throws BizException, PermissionException, JsonParseException {
        if (null == bookGroupId) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        return new ResponseDto<>(bookGroupBiz.getBookNameByBookGroupId(bookGroupId));
    }

    /**
     * 获取社群书列表(编辑)
     */
    @Override
    @RequestMapping(value = "listBookGroup4Adviser", method = RequestMethod.GET)
    public ResponseDto<PageBean> listBookGroup4Adviser(@RequestHeader("token") String token,
            @RequestParam(value = "name", required = false) String name,
            @RequestParam(value = "bookName", required = false) String bookName,
            @RequestParam(value = "isbn", required = false) String isbn,
            @RequestParam(value = "channelId", required = false) Long channelId,
            @RequestParam(value = "typeCode", required = false) String typeCode,
            @RequestParam(value = "currentPage", required = false) Integer currentPage,
            @RequestParam(value = "numPerPage", required = false) Integer numPerPage,
            @RequestParam(value = "isMainEditor", required = false) Integer isMainEditor,
            @RequestParam(value = "isFundSupport", required = false) Integer isFundSupport,
            @RequestParam(value = "bookId", required = false) Integer bookId,
            @RequestParam(value = "startDate", required = false) String startDate,
            @RequestParam(value = "endDate", required = false) String endDate)
            throws BizException, PermissionException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (currentPage == null || numPerPage == null || currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("adviserId", adviserId);
        paramMap.put("isMainEditor", isMainEditor);
        paramMap.put("isFundSupport", isFundSupport);
        paramMap.put("channelId", channelId);
        paramMap.put("name", name != null && "".equals(name.trim()) ? null : name);
        paramMap.put("isbn", isbn != null && "".equals(isbn.trim()) ? null : isbn);
        paramMap.put("bookName", bookName != null && "".equals(bookName.trim()) ? null : bookName);
        paramMap.put("typeCode", typeCode != null && "".equals(typeCode.trim()) ? null : typeCode);
        paramMap.put("bookId", bookId);
        if (!StringUtil.isEmpty(startDate) && !StringUtil.isEmpty(endDate)) {
            paramMap.put("startDate", startDate + " 00:00:00");
            paramMap.put("endDate", endDate + " 23:59:59");
        }
        PageBean pageBean = bookGroupBiz.listBookGroup4Adviser(paramMap, new PageParam(currentPage, numPerPage), adviserId);
        return new ResponseDto<>(pageBean);
    }

    @Override
    @RequestMapping(value = "listBookGroup4Channel", method = RequestMethod.POST)
    public ResponseDto<PageBean> listBookGroup4Channel(@RequestHeader("token") String token,
                                                       @RequestBody ListBookGroup4ChannelParamVO listBookGroup4ChannelParamVO) throws BizException, PermissionException {
        Long channelId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        PageBean pageBean = bookGroupBiz.listBookGroup4Channel(channelId, listBookGroup4ChannelParamVO);
        return new ResponseDto<>(pageBean);
    }

    @Override
    @RequestMapping(value = "getUserBrowseGroup4KnowLedgeMall", method = RequestMethod.GET)
    public ResponseDto<?> getUserBrowseGroup4KnowLedgeMall(@CookieValue("userInfo")String userInfo,
                                                                  @RequestParam(value = "currentPage", required = false) Integer currentPage,
                                                                  @RequestParam(value = "numPerPage", required = false) Integer numPerPage) throws BizException, PermissionException {
        Long channelId = Cookie.getId(userInfo, Cookie._CHANNEL_ID);
        Long wechatUserId = Cookie.getId(userInfo, Cookie._WECHAT_USER_ID);
        PageBeanNew<GroupStoreMyPayDto> resultInfos= bookGroupBiz.getUserBrowseGroup4KnowLedgeMall(channelId, wechatUserId, currentPage, numPerPage);
        return new ResponseDto<>(resultInfos);
    }

    /**
     * 获取社群书列表(编辑)
     */
    @Override
    @RequestMapping(value = "listSimpleBookGroup4Adviser", method = RequestMethod.GET)
    public ResponseDto<?> listSimpleBookGroup4Adviser(@RequestHeader("token") String token,
            @RequestParam(value = "name", required = false) String name,
            @RequestParam(value = "currentPage", required = false) Integer currentPage,
            @RequestParam(value = "numPerPage", required = false) Integer numPerPage)
            throws BizException, PermissionException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("adviserId", adviserId);
        paramMap.put("name", name != null && "".equals(name.trim()) ? null : name);
        if ((null == currentPage && null != numPerPage) || (null != currentPage && null == numPerPage)) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        if (null != currentPage) {
            return new ResponseDto<>(bookGroupBiz.listSimpleBookGroup4Adviser(paramMap, new PageParam(currentPage, numPerPage)));
        } else {
            return new ResponseDto<>(bookGroupBiz.listSimpleBookGroup4Adviser(paramMap));
        }
    }

    /**
     * 获取未创建社群码的书刊列表
     */
    @Override
    @RequestMapping(value = "listBook4CreateBookGroup", method = RequestMethod.GET)
    public ResponseDto<?> listBook4CreateBookGroup(@RequestHeader("token") String token,
            @RequestParam(value = "name", required = false) String name,
            @RequestParam(value = "currentPage", required = false) Integer currentPage,
            @RequestParam(value = "numPerPage", required = false) Integer numPerPage)
            throws BizException, PermissionException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (currentPage == null || numPerPage == null || currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("adviserId", adviserId);
        paramMap.put("name", name != null && "".equals(name.trim()) ? null : name);
        PageBean pageBean = bookGroupBiz.listBook4CreateBookGroup(paramMap, new PageParam(currentPage, numPerPage), adviserId);
        return new ResponseDto<>(pageBean);
    }

    @Override
    @PostMapping("updateRankType")
    public ResponseDto<?> updateRankType(@RequestHeader("token") String token, @RequestBody UpdateRankTypeVO updateRankTypeVO) throws BizException {
        bookGroupBiz.updateRankType(updateRankTypeVO.getBookGroupId(), updateRankTypeVO.getRankType());
        return new ResponseDto<>();
    }

    @Override
    @GetMapping("getRankType")
    public ResponseDto<Integer> getRankType(@RequestHeader("token") String token, @RequestParam("bookGroupId") Long bookGroupId) throws BizException {
        Integer rankType = bookGroupBiz.getRankType(bookGroupId);
        return new ResponseDto<>(rankType == null ? 0 : rankType);
    }

    /**
     * 编辑获取社群码总数量
     */
    @Override
    @RequestMapping(value = "getTotalBookGroupCount", method = RequestMethod.GET)
    public ResponseDto<?> getTotalBookGroupCount(@RequestHeader("token") String token) throws BizException, PermissionException, JsonParseException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        return new ResponseDto<>(bookGroupBiz.getTotalBookGroupCount(adviserId));
    }

    /**
     * 获取社群书分类列表(编辑)
     */
    @Override
    @RequestMapping(value = "listSimpleBookGroupClassify4Adviser", method = RequestMethod.POST)
    public ResponseDto<?> listSimpleBookGroupClassify4Adviser(@RequestHeader("token") String token,
            @RequestBody BookGroupClassify bookGroupClassify)
            throws BizException, PermissionException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (null == bookGroupClassify) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数有误");
        }
        if (null == bookGroupClassify.getCurrentPage() || null == bookGroupClassify.getNumPerPage()) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "分页参数有误");
        }
        PageParam pageParam = new PageParam(bookGroupClassify.getCurrentPage(), bookGroupClassify.getNumPerPage());
        return new ResponseDto<>(bookGroupClassifyBiz.listSimpleBookGroupClassify4Adviser(adviserId, bookGroupClassify, pageParam));
    }

    @Override
    @RequestMapping(value = "isSpecialAgent", method = RequestMethod.GET)
    public ResponseDto<Boolean> isSpecialAgent(@RequestHeader("token") String token) throws BizException, PermissionException {
        Long agentId = (Long) SessionUtil.getVlaue(token, SessionUtil.TENANT_ID);
        return new ResponseDto<Boolean>(bookGroupBiz.isSpecialAgent(agentId));
    }

    @ApiOperation("获取应用作品统计")
    @GetMapping("getAppStatistics")
    @Override
    public ResponseDto<?> getAppStatistics(
            @RequestHeader("token") String token,
            @RequestParam("currentPage") Integer currentPage,
            @RequestParam("numPerPage") Integer numPerPage,
            @RequestParam("bookGroupId") Long bookGroupId,
            @RequestParam(value = "qrcodeId",required = false) Long qrcodeId
    ) throws BizException, PermissionException {
        Long partyId=(Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (currentPage==null||currentPage<0||numPerPage==null||numPerPage<=0){
            throw new BookBizException(BookBizException.PARAM_IS_ERROR,"分页参数错误！");
        }
        if (bookGroupId==null){
            throw new BookBizException(BookBizException.PARAM_IS_ERROR,"社群码id不能为空！");
        }
        return new ResponseDto<>(bookGroupBiz.getAppStatistics(currentPage,numPerPage,bookGroupId,qrcodeId,partyId));
    }

    @Override
    @ApiOperation(value = "导出社群码(编辑)", httpMethod = "GET")
    @RequestMapping(value = "exportGroupQrcode4Adviser", method = RequestMethod.GET)
    public ResponseDto<?> exportGroupQrcode4Adviser(@RequestHeader("token") String token,
                                                    @RequestParam(value = "name", required = false) String name,
                                                    @RequestParam(value = "bookName", required = false) String bookName,
                                                    @RequestParam(value = "isbn", required = false) String isbn,
                                                    @RequestParam(value = "channelId", required = false) Long channelId,
                                                    @RequestParam(value = "typeCode", required = false) String typeCode,
                                                    @RequestParam(value = "isMainEditor", required = false) Integer isMainEditor,
                                                    @RequestParam(value = "isFundSupport", required = false) Integer isFundSupport,
                                                    @RequestParam(value = "bookId", required = false) Integer bookId,
                                                    @RequestParam(value = "startDate", required = false) String startDate,
                                                    @RequestParam(value = "endDate", required = false) String endDate)
            throws BizException, PermissionException{
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("adviserId", adviserId);
        paramMap.put("isMainEditor", isMainEditor);
        paramMap.put("isFundSupport", isFundSupport);
        paramMap.put("channelId", channelId);
        paramMap.put("name", name != null && "".equals(name.trim()) ? null : name);
        paramMap.put("isbn", isbn != null && "".equals(isbn.trim()) ? null : isbn);
        paramMap.put("bookName", bookName != null && "".equals(bookName.trim()) ? null : bookName);
        paramMap.put("typeCode", typeCode != null && "".equals(typeCode.trim()) ? null : typeCode);
        paramMap.put("bookId", bookId);
        if (!StringUtil.isEmpty(startDate) && !StringUtil.isEmpty(endDate)) {
            paramMap.put("startDate", startDate + " 00:00:00");
            paramMap.put("endDate", endDate + " 23:59:59");
        }
        bookGroupBiz.exportGroupQrcode4Adviser(paramMap, adviserId);
        return new ResponseDto<>();
    }

    @ApiOperation("根据bookId获取社群书分类等统计")
    @GetMapping("getBookGroupStatistics")
    @Override
    public ResponseDto<?> getBookGroupStatistics(
            @RequestHeader("token") String token,
            @RequestParam("bookId") Long bookId
    ) throws BizException, PermissionException, JsonParseException {
        SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if(bookId==null){
            throw new BookBizException(BookBizException.PARAM_IS_ERROR,"bookId不能为空！");
        }
        return new ResponseDto<>(bookGroupBiz.getBookGroupStatistics(bookId));
    }

    @ApiOperation("根据bookId获取社群书分类和关键词等统计信息")
    @GetMapping("getBookGroupKeywordStatistics")
    @Override
    public ResponseDto<?> getBookGroupKeywordStatistics(
            @RequestHeader("token") String token,
            @RequestParam("bookId") Long bookId
    ) throws BizException, PermissionException, JsonParseException {
        SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if(bookId==null){
            throw new BookBizException(BookBizException.PARAM_IS_ERROR,"bookId不能为空！");
        }
        return new ResponseDto<>(bookGroupBiz.getBookGroupKeywordStatistics(bookId));
    }

    @ApiOperation("更新是否显示书名")
    @GetMapping("updateIsShowBookName")
    @Override
    public ResponseDto<?> updateIsShowBookName(
            @RequestHeader("token") String token,
            @RequestParam("isShowBookName") Boolean isShowBookName
    ) throws BizException, PermissionException, JsonParseException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (isShowBookName == null) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "isShowBookName不能为空！");
        }
        bookGroupBiz.updateIsShowBookName(isShowBookName, partyId);
        return new ResponseDto<>();
    }

    @ApiOperation("获取是否显示书名")
    @GetMapping("getIsShowBookName")
    @Override
    public ResponseDto<?> getIsShowBookName(
            @RequestHeader("token") String token
    ) throws BizException, PermissionException, JsonParseException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        return new ResponseDto<>(bookGroupBiz.getIsShowBookName(partyId));
    }


    @ApiOperation("删除社群码")
    @GetMapping("deleteBookGroup")
    @Override
    public ResponseDto<?> deleteBookGroup(
            @RequestHeader("token") String token,
            @RequestParam("bookGroupId") Long bookGroupId
    ) throws BizException, PermissionException, JsonParseException {
        SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        bookGroupBiz.deleteBookGroup(bookGroupId);
        return new ResponseDto<>();
    }

    @ApiOperation("根据社群书id获取个人二维码")
    @GetMapping("getOwnAltQrcodeInfoDTOByBookGroupId")
    @Override
    public ResponseDto<?> getOwnAltQrcodeInfoDTOByBookGroupId(
            @CookieValue("userInfo") String userInfo,
            @RequestParam("bookGroupId") Long bookGroupId
    ) throws BizException, PermissionException, JsonParseException {
        Long wechatUserId = Cookie.getId(userInfo, Cookie._WECHAT_USER_ID);
        if (bookGroupId == null) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "社群码id不能为空！");
        }
        return new ResponseDto<>(bookGroupBiz.getOwnAltQrcodeInfoDTOByBookGroupId(wechatUserId, bookGroupId));
    }

    @ApiOperation("根据分类id获取个人二维码")
    @GetMapping("getOwnAltQrcodeInfoDTOByClassifyId")
    @Override
    public ResponseDto<?> getOwnAltQrcodeInfoDTOByClassifyId(
            @CookieValue("userInfo") String userInfo,
            @RequestParam("classifyId") Long classifyId
    ) throws BizException, PermissionException, JsonParseException {
        Long wechatUserId = Cookie.getId(userInfo, Cookie._WECHAT_USER_ID);
        if (classifyId == null) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "分类不能为空！");
        }
        return new ResponseDto<>(bookGroupBiz.getOwnAltQrcodeInfoDTOByClassifyId(wechatUserId, classifyId));
    }

    @ApiOperation("获取所有的大类")
    @GetMapping("getAllLargeTemplet")
    @Override
    public ResponseDto<?> getAllLargeTemplet(
            @RequestHeader("token") String token
    ) throws BizException, PermissionException, JsonParseException {
        SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        return new ResponseDto<>(bookGroupBiz.getAllLargTemplet());
    }

    @Override
    @RequestMapping(value = "getBookGroupStatisByBookGroupId",method = RequestMethod.GET)
    public ResponseDto<?> getBookGroupStatisByBookGroupId(@RequestParam("bookGroupId")Long bookGroupId) throws BizException {
        return new ResponseDto<>(bookGroupBiz.getBookGroupStatisByBookGroupId(bookGroupId));
    }

    @Override
    @RequestMapping(value ="getGroupIncomeStatic",method = RequestMethod.POST)
    public ResponseDto<PageBeanNew<ResourcesStatisticVO>> getGroupIncomeStatic(@RequestHeader("token") String token,
                                                                               @RequestBody GroupIncomeStaticParamVO groupIncomeStaticParamVO) throws BizException, PermissionException {
        Map<String, Object> map = SessionUtil.getToken4Redis(token);
        Long adviserId = (Long) map.get(SessionUtil.PARTY_ID);
        PageBeanNew<ResourcesStatisticVO> groupIncomeStatic = bookGroupBiz.getGroupIncomeStatic(groupIncomeStaticParamVO, adviserId);
        return new ResponseDto<>(groupIncomeStatic);
    }

    @Override
    @RequestMapping(value ="getGroupScanTrend",method = RequestMethod.POST)
    public ResponseDto<List<GroupScanTrendVO>> getGroupScanTrend(@RequestHeader("token") String token,
                                                                 @RequestBody GroupScanTrendParamVO groupScanTrendParamVO) throws BizException, PermissionException {
        Map<String, Object> map = SessionUtil.getToken4Redis(token);
        Long adviserId = (Long) map.get(SessionUtil.PARTY_ID);
        List<GroupScanTrendVO> groupScanTrend = bookGroupBiz.getGroupScanTrend(groupScanTrendParamVO, adviserId);
        return new ResponseDto<>(groupScanTrend);
    }

    @Override
    @RequestMapping(value ="getClassifyStatistic",method = RequestMethod.GET)
    public ResponseDto<?> getClassifyStatistic(@RequestHeader("token") String token,
                                               @RequestParam(value = "bookGroupId",required = true) Long bookGroupId,
                                               @RequestParam(value = "currentPage",required = true) Integer currentPage,
                                               @RequestParam(value = "numPerPage",required = true) Integer numPerPage) throws BizException, PermissionException {
        Map<String, Object> map = SessionUtil.getToken4Redis(token);
        Long adviserId = (Long) map.get(SessionUtil.PARTY_ID);
        PageBeanNew<GroupStatisticVO> groupQrcodeStatistic = bookGroupBiz.getClassifyStatistic(bookGroupId, adviserId,currentPage,numPerPage);
        return new ResponseDto<>(groupQrcodeStatistic);
    }

    @Override
    @RequestMapping(value ="getQrcodeStatistic",method = RequestMethod.GET)
    public ResponseDto<?> getQrcodeStatistic(@RequestHeader("token") String token,
                                                                    @RequestParam(value = "bookGroupId",required = true) Long bookGroupId,
                                                                    @RequestParam("classifyId") Long classifyId) throws BizException, PermissionException {
        Map<String, Object> map = SessionUtil.getToken4Redis(token);
        Long adviserId = (Long) map.get(SessionUtil.PARTY_ID);
        List<WxGroupStatisticVO> qrcodeStatistic = bookGroupBiz.getQrcodeStatistic(bookGroupId, classifyId, adviserId);
        return new ResponseDto<>(qrcodeStatistic);
    }

    @Override
    @RequestMapping(value ="getTotalRescourceData",method = RequestMethod.GET)
    public ResponseDto<?> getTotalRescourceData(@RequestHeader("token") String token, @RequestParam(value = "bookGroupId",required = true) Long bookGroupId,
                                                @RequestParam(value = "classifyId",required = false) Long classifyId, @RequestParam(value = "qrcodeId",required = false) Long qrcodeId,
                                                @RequestParam("type") String type) throws BizException, PermissionException {
        Map<String, Object> map = SessionUtil.getToken4Redis(token);
        Long adviserId = (Long) map.get(SessionUtil.PARTY_ID);
        TotalRescourceDataVO totalRescourceData = bookGroupBiz.getTotalRescourceData(bookGroupId, adviserId, classifyId, qrcodeId, type);
        return new ResponseDto<>(totalRescourceData);
    }

    @Override
    @RequestMapping(value ="exportRescourceIncomeData",method = RequestMethod.GET)
    public ResponseDto<?> exportRescourceIncomeData(@RequestHeader("token") String token,
                                                    @RequestParam(value = "bookGroupId",required = true) Long bookGroupId) throws BizException, PermissionException {
        Map<String, Object> map = SessionUtil.getToken4Redis(token);
        Long adviserId = (Long) map.get(SessionUtil.PARTY_ID);
        Map<String, String> resultMap = bookGroupBiz.exportRescourceIncomeData(bookGroupId, adviserId);
        return new ResponseDto<>(resultMap);

    }

}
