package com.pcloud.book.consumer.settlement;

import java.math.BigDecimal;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.pcloud.common.core.aspect.ParamLog;
import com.pcloud.common.dto.ResponseDto;
import com.pcloud.settlementcenter.record.dto.GetGroupClassifyIncomeDTO;
import com.pcloud.settlementcenter.record.dto.GetGroupQrcodeIncomeDTO;
import com.pcloud.settlementcenter.record.dto.GroupRescourceIncomeParamDTO;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;

import com.pcloud.book.base.exception.BookBizException;
import com.pcloud.common.exceptions.BizException;
import com.pcloud.common.utils.ListUtils;
import com.pcloud.common.utils.ResponseHandleUtil;
import com.pcloud.settlementcenter.record.dto.BookIncomeDto;
import com.pcloud.settlementcenter.record.service.BookService;

/**
 * @描述：书刊收益中间件
 * @作者：lihao
 * @创建时间：2017年7月19日,下午5:36:04 @版本：1.0
 */
@Component("bookConsr")
public class BookConsr {

	/**
	 * logger
	 */
	private static final Logger LOGGER = LoggerFactory.getLogger(BookConsr.class);

	@Autowired
	private BookService bookService;

	/**
	 * 获取书籍收益信息
	 */
	public Map<String, BookIncomeDto> getBookIncomeInfo(List<Long> bookIds, List<Long> channelIds, Long adviserId)
			throws BizException {
		LOGGER.info("【书刊-结算(消)】书籍收益信息,<START>.[bookIds]=" + bookIds + ",[channelIds]=" + channelIds + ",[adviserId]="
				+ adviserId);
		Map<String, BookIncomeDto> bookIncomeMap = new HashMap<>();
		if (ListUtils.isEmpty(bookIds) || ListUtils.isEmpty(channelIds) || adviserId == null) {
			LOGGER.warn("参数缺失~！");
			return bookIncomeMap;
		}
		Map<String, Object> paramMap = new HashMap<>();
		paramMap.put("bookIdList", bookIds);
		paramMap.put("channelIdList", channelIds);
		try {
			bookIncomeMap = ResponseHandleUtil.parseMapResponse(bookService.getBookIncomeInfo(paramMap, adviserId),
					String.class, BookIncomeDto.class);
		} catch (BizException e) {
			LOGGER.warn("【书刊-结算(消)】书籍收益信息<ERROR>:" + e.getMessage(), e);
			throw new BizException(e.getCode(), e.getMessage());
		} catch (Exception e) {
			LOGGER.error("【书刊-结算(消)】书籍收益信息,<ERROR>.[getBookIncomeInfo]:" + e.getMessage(), e);
			throw new BookBizException(BookBizException.INVOKE_USER_ERROR, "书籍收益信息失败~!");
		}
		LOGGER.info("【书刊-结算(消)】书籍收益信息,<END>");
		return bookIncomeMap;
	}

	/**
	 * 获取图书收益
	 */
	public BigDecimal getBookIncomeInfo(Long bookId, Long channelId, Long adviserId) throws BizException {
		LOGGER.info("【书刊-结算(消)】获取图书收益,<START>.[bookId]=" + bookId + ",[channelId]=" + channelId + ",[adviserId]="
				+ adviserId);
		if (bookId == null || channelId == null || adviserId == null) {
			return null;
		}
		BigDecimal bookIncome = null;
		try {
			bookIncome = ResponseHandleUtil.parseResponse(bookService.getBookIncome(bookId, channelId, adviserId),
					BigDecimal.class);
		} catch (BizException e) {
			LOGGER.warn("【书刊-结算(消)】获取图书收益<ERROR>:" + e.getMessage(), e);
			throw new BizException(e.getCode(), e.getMessage());
		}catch (Exception e) {
			LOGGER.warn("【书刊-结算(消)】获取图书收益,<ERROR>.[getBookIncome]:" + e.getMessage(), e);
			throw new BookBizException(BookBizException.INVOKE_USER_ERROR, "获取图书收益失败~!");
		}
		LOGGER.info("【书刊-结算(消)】获取图书收益,<END>");
		return bookIncome;
	}
	
	/**
	 * 批量获取书刊总收益
	 */
	public Map<String, BigDecimal> getBookIncome(List<Long> bookIdList, List<Long> channelIdList,
			List<Long> adviserIdList) {
		LOGGER.info("【书刊-结算(消)】批量获取书刊总收益,<START>.[bookIdList]=" + bookIdList + ",[channelIdList]=" + channelIdList
				+ ",[adviserIdList]=" + adviserIdList);
		Map<String, Object> paramMap = new HashMap<>();
		paramMap.put("bookIdList", bookIdList);
		paramMap.put("channelIdList", channelIdList);
		paramMap.put("adviserIdList", adviserIdList);
		Map<String, BigDecimal> resultMap = ResponseHandleUtil.parseMap(bookService.getBookIncome(paramMap),
				String.class, BigDecimal.class);
		LOGGER.info("【书刊-结算(消)】批量获取书刊总收益,<END>");
		return resultMap;
	}

	@ParamLog(value = "批量获取社群码下资源的收益", isBefore = false, isAfterReturn = false)
	public Map<Long, BigDecimal> getRescourceIncome(GroupRescourceIncomeParamDTO groupRescourceIncomeParamDTO) throws BizException {
		Map<Long, BigDecimal> rescourceIncome = new HashMap<>();
		try {
			ResponseEntity<ResponseDto<Map<Long, BigDecimal>>> responseEntityP = bookService.getGroupRescourceIncome(groupRescourceIncomeParamDTO);
			rescourceIncome = ResponseHandleUtil.parseMap(responseEntityP, Long.class, BigDecimal.class);
		} catch (Exception e) {
			LOGGER.error("批量获取社群码下资源的收益.[getRescourceIncome]:" + e.getMessage(), e);
		}
		return rescourceIncome;
	}

	@ParamLog(value = "批量获取社群码下资源的总收益", isBefore = false, isAfterReturn = false)
	public BigDecimal getRescourceTotalIncome(GroupRescourceIncomeParamDTO groupRescourceIncomeParamDTO) throws BizException {
		BigDecimal rescourceTotalIncome = new BigDecimal(0);
		try {
			ResponseEntity<ResponseDto<BigDecimal>> responseEntityP = bookService.getGroupRescourceTotalIncome(groupRescourceIncomeParamDTO);
			rescourceTotalIncome = ResponseHandleUtil.parseResponse(responseEntityP, BigDecimal.class);
		} catch (Exception e) {
			LOGGER.error("批量获取社群码下资源的总收益.[getRescourceTotalIncome]:" + e.getMessage(), e);
		}
		return rescourceTotalIncome == null?BigDecimal.ZERO:rescourceTotalIncome;
	}

	@ParamLog(value = "批量获取社群码分类的收益", isBefore = false, isAfterReturn = false)
	public Map<Long,BigDecimal> getClassifyIncome(GetGroupClassifyIncomeDTO getGroupClassifyIncomeDTO) throws BizException {
		Map<Long,BigDecimal> classifyIncome = new HashMap<>();
		try {
			ResponseEntity<ResponseDto<Map<Long,BigDecimal>>> responseEntityP = bookService.getGroupClassifyIncome(getGroupClassifyIncomeDTO);
			classifyIncome = ResponseHandleUtil.parseMap(responseEntityP, Long.class, BigDecimal.class);
		} catch (Exception e) {
			LOGGER.error("批量获取社群码分类的收益.[getClassifyIncome]:" + e.getMessage(), e);
		}
		return classifyIncome;
	}

	@ParamLog(value = "批量获取社群码分类下的微信群收益", isBefore = false, isAfterReturn = false)
	public Map<Long,BigDecimal> getQrcodeIncome(GetGroupQrcodeIncomeDTO getGroupQrcodeIncomeDTO) throws BizException {
		Map<Long,BigDecimal> qrcodeIncome = new HashMap<>();
		try {
			ResponseEntity<ResponseDto<Map<Long,BigDecimal>>> responseEntityP = bookService.getGroupQrcodeIncome(getGroupQrcodeIncomeDTO);
			qrcodeIncome = ResponseHandleUtil.parseMap(responseEntityP, Long.class, BigDecimal.class);
		} catch (Exception e) {
			LOGGER.error("批量获取社群码分类下的微信群收益.[getQrcodeIncome]:" + e.getMessage(), e);
		}
		return qrcodeIncome;
	}
}
