package com.pcloud.book.clock.check.impl;

import com.pcloud.appcenter.app.dto.AppDto;
import com.pcloud.book.base.exception.BookBizException;
import com.pcloud.book.clock.check.BookClockCheck;
import com.pcloud.book.clock.dao.BookClockInfoDao;
import com.pcloud.book.clock.dao.BookClockKeywordDao;
import com.pcloud.book.clock.dao.BookGroupClockDao;
import com.pcloud.book.clock.dto.BookClockInfoDTO;
import com.pcloud.book.clock.dto.BookClockKeywordDTO;
import com.pcloud.book.clock.dto.BookGroupClockDTO;
import com.pcloud.book.clock.entity.BookClockInfo;
import com.pcloud.book.clock.entity.BookClockKeyword;
import com.pcloud.book.clock.enums.OriginTypeEnum;
import com.pcloud.book.consumer.app.AppConsr;
import com.pcloud.book.consumer.resource.ProductConsr;
import com.pcloud.book.group.biz.BookGroupClassifyBiz;
import com.pcloud.book.group.dao.BookGroupClassifyDao;
import com.pcloud.book.group.dto.GroupClassifyQrcodeDTO;
import com.pcloud.common.core.aspect.ParamLog;
import com.pcloud.common.utils.ListUtils;
import com.pcloud.common.utils.string.StringUtil;
import com.pcloud.resourcecenter.product.dto.ProDto;
import com.pcloud.resourcecenter.product.dto.ProductDto;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @Author: hzk
 * @Date: 2019/5/27 9:28
 * @Version 1.0
 */
@Component("bookClockCheck")
public class BookClockCheckImpl implements BookClockCheck {

    @Autowired
    private AppConsr appConsr;
    @Autowired
    private ProductConsr productConsr;
    @Autowired
    private BookClockKeywordDao bookClockKeywordDao;
    @Autowired
    private BookGroupClassifyBiz bookGroupClassifyBiz;
    @Autowired
    private BookClockInfoDao bookClockInfoDao;
    @Autowired
    private BookGroupClockDao booKGroupClockDao;
    @Autowired
    private BookGroupClassifyDao bookGroupClassifyDao;

    /**
     * 关联作品/应用与社群码时参数校验
     * @param bookClockInfo
     */
    @Override
    public void checkParam4CreateBookGroupClock(BookClockInfo bookClockInfo) {
        BookClockInfo newBookClockInfo = checkOriginIdAndOriginType(bookClockInfo);
        if(null == newBookClockInfo.getChannelId()){
            throw new BookBizException(BookBizException.ERROR,"运营平台为空！");
        }
        List<Long> bookGroupIdList = newBookClockInfo.getBookGroupIdList();
        if(!ListUtils.isEmpty(bookGroupIdList) && bookGroupIdList.size() > 200){
            throw new BookBizException(BookBizException.ERROR,"关联社群码不可超过200个！");
        }
        List<Long> classifyIdList = newBookClockInfo.getClassifyIdList();
        if(!ListUtils.isEmpty(classifyIdList) && classifyIdList.size() > 200){
            throw new BookBizException(BookBizException.ERROR,"关联分类不可超过200个！");
        }
        if(null != newBookClockInfo.getIsNotice() && newBookClockInfo.getIsNotice()){
            if(StringUtil.isEmpty(newBookClockInfo.getNoticeTime()) || StringUtil.isEmpty(newBookClockInfo.getNoticeContent())){
                throw new BookBizException(BookBizException.ERROR,"提醒设置参数有误！");
            }
        }
    }

    /**
     * 校验作品及应用
     * @param bookClockInfo
     * @return
     */
    @Override
    @ParamLog("校验作品及应用")
    public BookClockInfo checkOriginIdAndOriginType(BookClockInfo bookClockInfo) {
        Long originId = bookClockInfo.getOriginId();
        String originType = bookClockInfo.getOriginType();
        if(OriginTypeEnum.APP.value.equalsIgnoreCase(originType)){
            AppDto appDto = appConsr.getBaseById(originId);
            if(null == appDto){
                throw new BookBizException(BookBizException.ERROR,"该应用不存在！");
            }
            bookClockInfo.setOriginTitle(appDto.getTitle());
            bookClockInfo.setOriginImg(appDto.getSquareImg());
            bookClockInfo.setUniqueNumber(appDto.getUniqueNumber());
        }
        if(OriginTypeEnum.PRODUCT.value.equalsIgnoreCase(originType)){
            ProDto productDto = productConsr.getProBasesInfoById(originId);
            if(null == productDto){
                throw new BookBizException(BookBizException.ERROR,"该作品不存在！");
            }
            bookClockInfo.setOriginTitle(productDto.getProductName());
            bookClockInfo.setOriginImg(productDto.getCoverImg());
            bookClockInfo.setUniqueNumber(productDto.getUniqueNumber());
        }
        return bookClockInfo;
    }

    /**
     * 校验关键词
     * @param bookClockKeywordList
     * @param adviserId
     */
    @Override
    @ParamLog("校验关键词")
    public void checkBookClockKeyword(List<BookClockKeyword> bookClockKeywordList, Long adviserId, List<Long> bookGroupIdList,List<Long> classifyIdList) {
        //获取选择的分类集合
        List<Long> selectedClassifyIds = new ArrayList<>();
        if(!ListUtils.isEmpty(bookGroupIdList)){
            List<Long> classifyIdsUnderBookGroup = bookGroupClassifyDao.getIdByBookGroupIds(bookGroupIdList);
            if(!ListUtils.isEmpty(classifyIdsUnderBookGroup)){
                selectedClassifyIds.addAll(classifyIdsUnderBookGroup);
            }
        }
        if(!ListUtils.isEmpty(classifyIdList)){
            selectedClassifyIds.addAll(classifyIdList);
        }
        //获取关键词对应的分类集合
        List<Long> existClassifyIds = new ArrayList<>();
        List<String> keyWords = bookClockKeywordList.stream().map(BookClockKeyword::getKeyword).collect(Collectors.toList());
        if(ListUtils.isEmpty(keyWords)){
            throw new BookBizException(BookBizException.ERROR,"关键词为空！");
        }
        Map<String,Object> keywordMap = new HashMap<>();
        keywordMap.put("adviserId",adviserId);
        keywordMap.put("keyWords",keyWords);
        List<Long> bookClockInfoIds = bookClockKeywordDao.checkKeywordHasExist(keywordMap);
        if(!ListUtils.isEmpty(bookClockInfoIds)){
            List<BookGroupClockDTO> bookGroupClockDTOS = booKGroupClockDao.getInfoByInfoIds(bookClockInfoIds);
            if(!ListUtils.isEmpty(bookGroupClockDTOS)){
                //获取所有的分类集合
                List<Long> bookGroupIds = bookGroupClockDTOS.stream().filter(bookGroupClockDTO -> 0 == bookGroupClockDTO.getClassifyId()).map(BookGroupClockDTO::getBookGroupId).collect(Collectors.toList());
                if(!ListUtils.isEmpty(bookGroupIds)){
                    List<Long> classifyIdsUnderBookGroup = bookGroupClassifyDao.getIdByBookGroupIds(bookGroupIds);
                    if(!ListUtils.isEmpty(classifyIdsUnderBookGroup)){
                        existClassifyIds.addAll(classifyIdsUnderBookGroup);
                    }
                }
                List<Long> classifyIds = bookGroupClockDTOS.stream().filter(bookGroupClockDTO -> 0 == bookGroupClockDTO.getBookGroupId()).map(BookGroupClockDTO::getClassifyId).collect(Collectors.toList());
                if(!ListUtils.isEmpty(classifyIds)){
                    existClassifyIds.addAll(classifyIds);
                }

            }
        }
        if(!ListUtils.isEmpty(selectedClassifyIds) && !ListUtils.isEmpty(existClassifyIds)){
            if(!Collections.disjoint(selectedClassifyIds,existClassifyIds)){
                throw new BookBizException(BookBizException.ERROR,"关键词重复！");
            }
        }
    }

    /**
     * 校验关键词是否为打卡关键词
     * @param content
     * @param weixinGroupId
     * @return
     */
    @Override
    @ParamLog("校验关键词是否为打卡关键词")
    public BookClockInfoDTO checkKeywordIsClock(String content, String weixinGroupId) {
        //简单过滤非关键词的词
        if (StringUtil.isEmpty(content) || content.length() > 10) {
            return null;
        }
        //通过群id获取对应基本信息
        GroupClassifyQrcodeDTO classifyQrcodeInfo = bookGroupClassifyBiz.getClassifyQrcodeInfo(weixinGroupId);
        if(null == classifyQrcodeInfo){
            return null;
        }
        //获取该关键词对应的基本信息
        Map<String,Object> paramMap = new HashMap<>();
        paramMap.put("bookGroupId",classifyQrcodeInfo.getBookGroupId());
        paramMap.put("classifyId",classifyQrcodeInfo.getClassifyId());
        paramMap.put("adviserId",classifyQrcodeInfo.getAdviserId());
        paramMap.put("keyword",content);
        BookClockKeywordDTO bookClockKeywordDTO = bookClockKeywordDao.getInfoIdByKeywordAndweixinGroupId(paramMap);
        if(null == bookClockKeywordDTO){
            return null;
        }
        BookClockInfoDTO bookClockInfoDTO = bookClockInfoDao.getBaseInfoById(bookClockKeywordDTO.getBookClockInfoId());
        bookClockInfoDTO.setBookGroupId(classifyQrcodeInfo.getBookGroupId());
        bookClockInfoDTO.setGroupQrcodeId(classifyQrcodeInfo.getGroupQrcodeId());
        bookClockInfoDTO.setClassifyId(classifyQrcodeInfo.getClassifyId());
        bookClockInfoDTO.setChannelId(bookClockKeywordDTO.getChannelId());
        bookClockInfoDTO.setOperation(bookClockKeywordDTO.getOperation());
        return bookClockInfoDTO;
    }


    /**
     * 校验群是否为微信群打卡的群
     * @param classifyQrcodeInfo
     * @return
     */
    @Override
    @ParamLog("校验群是否为微信群打卡的群")
    public List<Long> checkGroupIsClock(GroupClassifyQrcodeDTO classifyQrcodeInfo) {
        if(null == classifyQrcodeInfo){
            return null;
        }
        //获取该关键词对应的基本信息
        Map<String,Object> paramMap = new HashMap<>();
        paramMap.put("bookGroupId",classifyQrcodeInfo.getBookGroupId());
        paramMap.put("classifyId",classifyQrcodeInfo.getClassifyId());
        paramMap.put("adviserId",classifyQrcodeInfo.getAdviserId());
        return booKGroupClockDao.getInfoIdByWeixinGroupId(paramMap);
    }

}
