package com.pcloud.book.advertising.biz;

import java.util.List;
import java.util.Map;
import com.pcloud.book.advertising.dto.AdvertisingMasterDTO;
import com.pcloud.book.advertising.dto.AdvertisingSpaceDTO;
import com.pcloud.book.advertising.dto.QrcodeAdvertisingSpaceCountDTO;
import com.pcloud.book.advertising.entity.*;
import com.pcloud.book.group.dto.AdGroupQrcodeDTO;
import com.pcloud.common.exceptions.BizException;
import com.pcloud.common.page.PageBean;
import com.pcloud.common.page.PageBeanNew;
import com.pcloud.common.page.PageParam;

/**
 * Description 广告位业务逻辑层接口
 * @author PENG
 * @date 2019/2/28
 */
public interface AdvertisingSpaceBiz {

    /**
     * 创建广告位
     * @param advertisingSpace 广告位实体
     * @return
     * @throws BizException
     */
    Long create(AdvertisingSpace advertisingSpace) throws BizException;

    /**
     * 修改广告位
     * @param advertisingSpace 广告位实体
     * @return
     * @throws BizException
     */
    void update(AdvertisingSpace advertisingSpace) throws BizException;

    /**
     * 停用广告位
     * @param advertisingSpace 广告位实体
     * @throws BizException
     */
    void stop(AdvertisingSpace advertisingSpace) throws BizException;

    /**
     * 获取广告位信息
     * @param adId 广告位ID
     * @return
     * @throws BizException
     */
    AdvertisingSpaceDTO get(Long adId) throws BizException;

    /**
     * 投放书刊
     * @param book 广告位书刊实体
     * @return
     * @throws BizException
     */
    void distributeBook(AdvertisingDistributionBook book) throws BizException;

    /**
     * 投放微信群
     * @param book 广告位书刊实体
     * @param isPlan 是否广告位计划
     * @throws BizException
     */
    void distributeWechatGroup(AdvertisingDistributionBook book, Boolean isPlan) throws BizException;

    /**
     * 获取出版社书刊权限
     * @param agentId 出版社ID
     * @return
     * @throws BizException
     */
    Boolean getAgentPermission(Long agentId) throws BizException;

    /**
     * 设置出版社书刊权限
     * @param agentPermission 广告位出版权限实体
     * @throws BizException
     */
    void setAgentPermission(AdvertisingAgentPermission agentPermission) throws BizException;

    /**
     * 设置编辑书刊权限
     * @param adviserPermission 广告位编辑权限实体
     * @throws BizException
     */
    void setAdviserPermission(AdvertisingAdviserPermission adviserPermission) throws BizException;

    /**
     * 编辑端获取书刊列表
     * @param paramMap  参数Map
     * @param pageParam 分页参数
     * @param adviserId 编辑ID
     * @return
     * @throws BizException
     */
    PageBean listBook4Adviser(Map<String, Object> paramMap, PageParam pageParam, Long adviserId) throws BizException;

    /**
     * 平台端获取某编辑书刊列表
     * @param paramMap  参数Map
     * @param pageParam 分页参数
     * @param adviserId 编辑ID
     * @return
     * @throws BizException
     */
    PageBean listBook4Platform(Map<String, Object> paramMap, PageParam pageParam, Long adviserId) throws BizException;

    /**
     * 平台端获取社群书微信群列表
     * @param paramMap  参数Map
     * @param pageParam 分页参数
     * @return
     * @throws BizException
     */
    PageBean listGroupQrcode4Platform(Map<String, Object> paramMap, PageParam pageParam) throws BizException;

    /**
     * 获取广告位投放书刊列表
     * @param adId      广告位ID
     * @param name      书刊名称
     * @param pageParam 分页参数
     * @return
     * @throws BizException
     */
    PageBean distributionBookList(Long adId, String name, PageParam pageParam) throws BizException;

    /**
     * 获取广告位投放微信群列表
     * @param adId      广告位ID
     * @param name      书刊名称
     * @param pageParam 分页参数
     * @return
     * @throws BizException
     */
    PageBean distributionWechatGroupList(Long adId, String name, PageParam pageParam) throws BizException;

    /**
     * 获取广告位投放书刊列表
     * @param name      书刊名称
     * @param pageParam 分页参数
     * @return
     * @throws BizException
     */
    PageBean advertisingBookList(String name, PageParam pageParam) throws BizException;

    /**
     * 获取广告位投放微信群列表
     * @param name      书刊名称
     * @param pageParam 分页参数
     * @return
     * @throws BizException
     */
    PageBean advertisingWechatGroupList(String name, PageParam pageParam) throws BizException;

    /**
     * 获取广告位投放微信群曝光量前几位
     * @return
     * @throws BizException
     */
    List<AdGroupQrcodeDTO> advertisingWechatGroupRankTop() throws BizException;

    /**
     * 获取广告位列表
     * @param name        名称
     * @param pageParam   分页参数
     * @param filterClose 过滤已关闭
     * @return
     * @throws BizException
     */
    PageBean list(String name, PageParam pageParam, Boolean filterClose, Boolean filterTest) throws BizException;

    /**
     * 书刊广告位明细
     * @param bookId      书刊ID
     * @param channelId   运营ID
     * @param adviserId   编辑ID
     * @param statisMonth 统计月份
     * @param pageParam   分页参数
     * @return
     * @throws BizException
     */
    PageBean advertisingDetail4Book(Long bookId, Long channelId, Long adviserId, String statisMonth, PageParam pageParam) throws BizException;

    /**
     * 导出书刊广告位明细
     * @param bookId      书刊ID
     * @param channelId   运营ID
     * @param adviserId   编辑ID
     * @param statisMonth 统计月份
     * @return
     * @throws BizException
     */
    Map<String, Object> exportAdvertisingDetail4Book(Long bookId, Long channelId, Long adviserId, String statisMonth) throws BizException;

    /**
     * 微信群广告位明细
     * @param qrcodeId    微信群ID
     * @param statisMonth 统计月份
     * @param pageParam   分页参数
     * @return
     * @throws BizException
     */
    PageBean advertisingDetail4WechatGroup(Long qrcodeId, String statisMonth, PageParam pageParam) throws BizException;

    /**
     * 导出微信群广告位明细
     * @param qrcodeId    微信群ID
     * @param statisMonth 统计月份
     * @return
     * @throws BizException
     */
    Map<String, Object> exportAdvertisingDetail4WechatGroup(Long qrcodeId, String statisMonth) throws BizException;

    /**
     * 微信群广告位点击读者列表
     * @param qrcodeId    微信群ID
     * @param adId        广告位ID
     * @param statisMonth 统计月份
     * @param pageParam   分页参数
     * @return
     * @throws BizException
     */
    PageBean clickUserList4AdvertisingWechatGroup(Long qrcodeId, Long adId, String statisMonth, PageParam pageParam) throws BizException;

    /**
     * 导出微信群广告位点击读者列表
     * @param qrcodeId    微信群ID
     * @param adId        广告位ID
     * @param statisMonth 统计月份
     * @return
     * @throws BizException
     */
    Map<String, Object> exportClickUserList4AdvertisingWechatGroup(Long qrcodeId, Long adId, String statisMonth) throws BizException;

    /**
     * 广告主广告位明细
     * @param qrcodeId    微信群ID
     * @param statisMonth 统计月份
     * @param pageParam   分页参数
     * @return
     * @throws BizException
     */
    PageBean advertisingDetail4Master(Long masterId, String statisMonth, PageParam pageParam) throws BizException;

    /**
     * 导出广告主广告位明细
     * @param masterId    广告主ID
     * @param statisMonth 统计月份
     * @return
     * @throws BizException
     */
    Map<String, Object> exportAdvertisingDetail4Master(Long masterId, String statisMonth) throws BizException;

    /**
     * 客户端获取书刊配置的广告位信息
     * @param sceneId          二维码ID
     * @param adviserId        编辑ID
     * @param channelId        运营ID
     * @param adPositionDetail 广告具体位置
     * @param bookGroupId      社群码ID
     * @return
     * @throws BizException
     */
    List<AdvertisingSpaceDTO> get4Wechat(Long sceneId, Long adviserId, Long channelId, String adPositionDetail, Long bookGroupId) throws BizException;

    /**
     * 获取社群书配置的广告位信息
     * @param bookGroupId      社群码ID
     * @param adPositionDetail 广告具体位置
     * @return
     * @throws BizException
     */
    AdvertisingSpaceDTO getBookGroupAd4Wechat(Long bookGroupId, String adPositionDetail) throws BizException;

    /**
     * 曝光量埋点
     * @param advertisingExposureRecord 广告位曝光实体
     * @throws BizException
     */
    void addExposureTrack(AdvertisingExposureRecord advertisingExposureRecord) throws BizException;

    /**
     * 点击量埋点
     * @param advertisingClickRecord 广告位点击实体
     * @throws BizException
     */
    void addClickTrack(AdvertisingClickRecord advertisingClickRecord) throws BizException;

    /**
     * 添加广告主
     * @param advertisingMaster 广告主实体
     * @throws BizException
     */
    void createMaster(AdvertisingMaster advertisingMaster) throws BizException;

    /**
     * 修改广告主
     * @param advertisingMaster 广告主实体
     * @throws BizException
     */
    void updateMaster(AdvertisingMaster advertisingMaster) throws BizException;

    /**
     * 获取广告主列表
     * @param pageParam  分页参数
     * @param masterName 广告主名称
     * @return
     * @throws BizException
     */
    PageBean getMasterList(PageParam pageParam, String masterName) throws BizException;

    /**
     * 获取所有广告主
     * @return
     * @throws BizException
     */
    List<AdvertisingMasterDTO> getAllMaster() throws BizException;

    /**
     * 获取广告主信息
     * @param masterId 广告主ID
     * @return
     * @throws BizException
     */
    AdvertisingMasterDTO getMasterInfo(Long masterId) throws BizException;

    /**
     * 每日凌晨计算昨日广告位收益
     * @throws BizException
     */
    void calculateAdvertisingIncomeYesterday() throws BizException;

    /**
     * 获取概览数据
     * @return
     * @throws BizException
     */
    Map<String, Object> overviewStatis() throws BizException;

    /**
     * 获取流量趋势
     * @param day       天数
     * @param startDate 开始时间
     * @param endDate   结束时间
     * @return
     * @throws BizException
     */
    List<Object> getFlowStatisTrend(Integer day, String startDate, String endDate) throws BizException;

    /**
     * 获取收益趋势
     * @param day       天数
     * @param startDate 开始时间
     * @param endDate   结束时间
     * @return
     * @throws BizException
     */
    List<Object> getIncomeStatisTrend(Integer day, String startDate, String endDate) throws BizException;

    /**
     * 获取微信群信息
     * @param qrcodeId 微信群ID
     * @return
     * @throws BizException
     */
    AdGroupQrcodeDTO getGroupQrcodeInfo(Long qrcodeId) throws BizException;

    /**
     *获得剪切的音视频
     * @param fileUrl
     * @param fileText
     * @param startTime
     * @param endTime
     * @param duration
     * @return
     */
    public String getCutUrl(String fileUrl, String fileText, Double startTime, Double endTime, Double duration);
    /**
     * 未转码的文件
     * @return
     */
    public List<AdvertisingSpace> getUnConvertFile(String adDetailFileId);

    /**
     * 更新文件路径
     * @param filelist
     * @param fileUrl
     */
    void updateFile(List<AdvertisingSpace> filelist, String fileUrl);
    /**
     * 获得关闭广告位的出版id
     * @return
     */
    List<Long> getCloseAgentId();

    /**
     * 获取广告位数量
     * @param qrcodeIds
     * @return
     */
    Map<Long,QrcodeAdvertisingSpaceCountDTO> mapAdvertisingSpaceNum(List<Long> qrcodeIds);

    /**
     * 更新测试状态
     * @param isTest
     * @param id
     */
    public void updateTestStatus(Boolean isTest, Long id);

    /**
     * 创建广告位计划
     * @param advertisingPlan
     * @return
     */
    Long createAdvertisingPlan(AdvertisingPlan advertisingPlan);

    /**
     * 修改广告位计划
     * @param advertisingPlan
     */
    void updateAdvertisingPlan(AdvertisingPlan advertisingPlan);

    /**
     * 发送广告位计划
     */
    void sendAdvertisingPlan(Long advertisingPutId);

    /**
     * 删除广告位计划
     */
    void deleteAdvertisingPlan(Long advertisingPlanId, Long partyId);

    /**
     * 获取广告位计划
     */
    AdvertisingPlan getAdvertisingPlan(Long advertisingPlanId);

    /**
     * 获取广告位计划列表
     */
    PageBeanNew<AdvertisingPlan> getAdvertisingPlanList(String keyword, Long partyId, Integer currentPage, Integer numPerPage);

    /**
     * 完课量埋点
     * @param advertisingPilotRecord
     */
    void addPilotTrack(AdvertisingPilotRecord advertisingPilotRecord);

    /**
     * 浏览时长埋点
     * @param advertisingBrowseRecord
     */
    String addBrowseTrack(AdvertisingBrowseRecord advertisingBrowseRecord);

    /**
     * 微信群广告投放详情
     * @param qrcodeId
     * @param statisMonth
     * @param adId
     * @param pageParam
     * @return
     */
    PageBean advertisingDistributionDetail4WechatGroup(Long qrcodeId, String statisMonth, Long adId, PageParam pageParam);

    /**
     * 更新曝光量
     * @param qrcodeId
     * @param peopleCounts
     */
    void updateExposureByQrcode(Long qrcodeId, Integer peopleCounts);
}
