package com.pcloud.book.adnews.facade.impl;

import com.pcloud.book.adnews.biz.AdNewsBiz;
import com.pcloud.book.adnews.entity.AdNewsClickRecord;
import com.pcloud.book.adnews.entity.AdNewsSet;
import com.pcloud.book.adnews.entity.AdNewsWechat;
import com.pcloud.book.adnews.facade.AdNewsFacade;
import com.pcloud.book.adnews.dto.AdNewsListParam;
import com.pcloud.book.base.exception.BookBizException;
import com.pcloud.common.dto.ResponseDto;
import com.pcloud.common.exceptions.BizException;
import com.pcloud.common.permission.PermissionException;
import com.pcloud.common.utils.ListUtils;
import com.pcloud.common.utils.NumberUtil;
import com.pcloud.common.utils.SessionUtil;
import com.pcloud.common.utils.cookie.Cookie;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Map;

/**
 * @Description
 * @Author ruansiyuan
 * @Date 2019/7/17 15:00
 **/
@RestController("adNewsFacade")
@RequestMapping("adNews")
public class AdNewsFacadeImpl implements AdNewsFacade {

    private static final Logger LOGGER = LoggerFactory.getLogger(AdNewsFacadeImpl.class);

    @Autowired
    private AdNewsBiz adNewsBiz;

    @ApiOperation("新增编辑端早晚报设置")
    @PostMapping("/createAdNewsSet")
    @Override
    public ResponseDto<?> createAdNewsSet(
            @RequestHeader("token") @ApiParam("token信息") String token,
            @RequestBody @ApiParam("编辑端早晚报设置") AdNewsSet adNewsSet
    ) throws BizException, PermissionException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (adNewsSet == null) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数错误");
        }
        adNewsSet.setCreateUser(partyId);
        adNewsSet.setUpdateUser(partyId);
        return new ResponseDto<>(adNewsBiz.createAdNewsSet(adNewsSet));
    }

    @ApiOperation("修改编辑端早晚报设置")
    @PostMapping("/updateAdNewsSet")
    @Override
    public ResponseDto<?> updateAdNewsSet(
            @RequestHeader("token") @ApiParam("token信息") String token,
            @RequestBody @ApiParam("编辑端早晚报设置") AdNewsSet adNewsSet
    ) throws BizException, PermissionException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (adNewsSet == null) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数错误");
        }
        if (adNewsSet.getId() == null) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "早晚报设置id不能为空");
        }

        adNewsSet.setUpdateUser(partyId);
        adNewsBiz.updateAdNewsSet(adNewsSet);
        return new ResponseDto<>();
    }

    @ApiOperation("获取编辑早晚报设置")
    @GetMapping("/getAdNewsSet")
    @Override
    public ResponseDto<?> getAdNewsSet(
            @RequestHeader("token") @ApiParam("token信息") String token
    ) throws BizException, PermissionException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        return new ResponseDto<>(adNewsBiz.getAdNewsSet(partyId));
    }

    @ApiOperation("编辑添加选择的早晚报素材")
    @PostMapping("/createAdNewsChooseBatch")
    @Override
    public ResponseDto<?> createAdNewsChooseBatch(
            @RequestHeader("token") @ApiParam("token信息") String token,
            @RequestBody @ApiParam("新闻id集合") List<Long> adNewsIds
    ) throws BizException, PermissionException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (ListUtils.isEmpty(adNewsIds)) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数错误");
        }
        adNewsBiz.createAdNewsChooseBatch(adNewsIds, partyId);
        return new ResponseDto<>();
    }

    @ApiOperation("编辑移除选择的早晚报素材")
    @GetMapping("/deleteAdNewsChoose")
    @Override
    public ResponseDto<?> deleteAdNewsChoose(
            @RequestHeader("token") @ApiParam("token信息") String token,
            @RequestParam @ApiParam("adNewsId") Long adNewsId
    ) throws BizException, PermissionException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (adNewsId == null) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数错误");
        }
        adNewsBiz.deleteAdNewsChoose(adNewsId, partyId);
        return new ResponseDto<>();
    }

    @ApiOperation("获取编辑选择的早晚报素材库")
    @GetMapping("/getAdNewsChooseList")
    @Override
    public ResponseDto<?> getAdNewsChooseList(
            @RequestHeader("token") @ApiParam("token信息") String token,
            @RequestParam(value = "title", required = false) @ApiParam("标题") String title,
            @RequestParam(value = "hasUsed", required = false) @ApiParam("使用状态") Boolean hasUsed,
            @RequestParam("currentPage") @ApiParam("当前页") Integer currentPage,
            @RequestParam("numPerPage") @ApiParam("每页条数") Integer numPerPage
    ) throws BizException, PermissionException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (currentPage == null || currentPage < 0) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "当前页不能为空且不能小于0！");
        }
        if (numPerPage == null || numPerPage <= 0) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "每页条数不能为空且必须大于0！");
        }
        return new ResponseDto<>(adNewsBiz.getAdNewsChooseList(partyId, title, hasUsed, currentPage, numPerPage));
    }

    @ApiOperation("早晚报素材库")
    @PostMapping("/getAdNewsList")
    @Override
    public ResponseDto<?> getAdNewsList(
            @RequestHeader("token") @ApiParam("token信息") String token,
            @RequestBody @ApiParam("查询参数") AdNewsListParam adNewsListParam
    ) throws BizException, PermissionException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (adNewsListParam.getCurrentPage() == null || adNewsListParam.getCurrentPage() < 0) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "当前页不能为空且不能小于0！");
        }
        if (adNewsListParam.getNumPerPage() == null || adNewsListParam.getNumPerPage() <= 0) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "每页条数不能为空且必须大于0！");
        }
        adNewsListParam.setPartyId(partyId);
        return new ResponseDto<>(adNewsBiz.getAdNewsList(adNewsListParam));
    }

    @Override
    @ApiOperation("禁用/解除禁用")
    @GetMapping("/saveAdNewsBan")
    public ResponseDto<?> saveAdNewsBan(
            @RequestHeader("token") @ApiParam("token信息") String token,
            @RequestParam("adNewsId") @ApiParam("新闻id") Long adNewsId,
            @RequestParam("hasBan") @ApiParam("禁止推送") Integer hasBan
    ) throws BizException, PermissionException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (!NumberUtil.isNumber(adNewsId)) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "adNewsId 不能小于0！");
        }
        if (!NumberUtil.isNumber(hasBan)) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "hasBan 必须大于0！");
        }
        adNewsBiz.saveAdNewsBan(partyId, adNewsId, hasBan);
        return new ResponseDto<>();
    }

    @Override
    @ApiOperation("编辑反馈公众号")
    @PostMapping("/addAdNewsWechat")
    public ResponseDto<?> addAdNewsWechat(
            @RequestHeader("token") @ApiParam("token信息") String token,
            @RequestBody @ApiParam("AdNewsWechat") AdNewsWechat adNewsWechat
    ) throws BizException, PermissionException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        adNewsWechat.setAdviserId(partyId);
        return new ResponseDto<>(adNewsBiz.addAdNewsWechat(adNewsWechat));
    }

    @Override
    @ApiOperation("获取编辑自有公众号")
    @GetMapping("/getAdNewsWechatListForAdviser")
    public ResponseDto<?> getAdNewsWechatListForAdviser(
            @RequestHeader("token") @ApiParam("token信息") String token,
            @RequestParam("currentPage") @ApiParam("当前页") Integer currentPage,
            @RequestParam("numPerPage") @ApiParam("每页条数") Integer numPerPage
    ) throws BizException, PermissionException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (currentPage == null || currentPage < 0) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "当前页不能为空且不能小于0！");
        }
        if (numPerPage == null || numPerPage <= 0) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "每页条数不能为空且必须大于0！");
        }
        return new ResponseDto<>(adNewsBiz.getAdNewsWechatListForAdviser(partyId, currentPage, numPerPage));
    }

    @ApiOperation("添加编辑选择的公众号")
    @PostMapping("/createAdNewsWechatChooseBatch")
    @Override
    public ResponseDto<?> createAdNewsWechatChooseBatch(
            @RequestHeader("token") @ApiParam("token信息") String token,
            @RequestBody @ApiParam("选择的公众号主键") List<Long> adNewsWechatIds
    ) throws BizException, PermissionException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        adNewsBiz.createAdNewsWechatChooseBatch(adNewsWechatIds, partyId);
        return new ResponseDto<>();
    }

    @Override
    @ApiOperation("移除编辑已选择的公众号")
    @GetMapping("/deleteAdNewsWechatChoose")
    public ResponseDto<?> deleteAdNewsWechatChoose(
            @RequestHeader("token") @ApiParam("token信息") String token,
            @RequestParam(value = "adNewsWechatId") @ApiParam("公众号主键") Long adNewsWechatId
    ) throws BizException, PermissionException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        adNewsBiz.deleteAdNewsWechatChoose(adNewsWechatId, partyId);
        return new ResponseDto<>();
    }

    @Override
    @ApiOperation("获取编辑已选择的公众号")
    @GetMapping("/getAdNewsWechatChooseList")
    public ResponseDto<?> getAdNewsWechatChooseList(
            @RequestHeader("token") @ApiParam("token信息") String token,
            @RequestParam("currentPage") @ApiParam("当前页") Integer currentPage,
            @RequestParam("numPerPage") @ApiParam("每页条数") Integer numPerPage
    ) throws BizException, PermissionException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (currentPage == null || currentPage < 0) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "当前页不能为空且不能小于0！");
        }
        if (numPerPage == null || numPerPage <= 0) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "每页条数不能为空且必须大于0！");
        }
        return new ResponseDto<>(adNewsBiz.getAdNewsWechatChooseList(partyId, currentPage, numPerPage));
    }

    @Override
    @ApiOperation("获取所有的公众号列表")
    @GetMapping("/getAdNewsWechatList")
    public ResponseDto<?> getAdNewsWechatList(
            @RequestHeader("token") @ApiParam("token信息") String token,
            @RequestParam("currentPage") @ApiParam("当前页") Integer currentPage,
            @RequestParam("numPerPage") @ApiParam("每页条数") Integer numPerPage
    ) throws BizException, PermissionException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (currentPage == null || currentPage < 0) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "当前页不能为空且不能小于0！");
        }
        if (numPerPage == null || numPerPage <= 0) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "每页条数不能为空且必须大于0！");
        }
        return new ResponseDto<>(adNewsBiz.getAdNewsWechatList(currentPage, numPerPage));
    }

    @Override
    @ApiOperation("获取早晚报发送记录")
    @GetMapping("/getAdNewsMessageStatistic")
    public ResponseDto<?> getAdNewsMessageStatistic(
            @RequestHeader("token") @ApiParam("token信息") String token,
            @RequestParam(value = "title", required = false) @ApiParam("群名称、群分类、社群码、社群书") String title,
            @RequestParam(value = "beginTime", required = false) @ApiParam("开始日期") String beginTime,
            @RequestParam(value = "endTime", required = false) @ApiParam("结束日期") String endTime,
            @RequestParam(value = "proLabelId", required = false) @ApiParam("专业") Long proLabelId,
            @RequestParam(value = "depLabelId", required = false) @ApiParam("深度") Long depLabelId,
            @RequestParam(value = "purLabelId", required = false) @ApiParam("目的") Long purLabelId,
            @RequestParam(value = "currentPage", required = false, defaultValue = "0") @ApiParam("当前页") Integer currentPage,
            @RequestParam(value = "numPerPage", required = false, defaultValue = "10") @ApiParam("每页条数") Integer numPerPage
    ) throws BizException, PermissionException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (currentPage == null || currentPage < 0) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "当前页不能为空且不能小于0！");
        }
        if (numPerPage == null || numPerPage <= 0) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "每页条数不能为空且必须大于0！");
        }
        return new ResponseDto<>(adNewsBiz.getAdNewsMessageStatistic(partyId, title, beginTime, endTime, proLabelId, depLabelId, purLabelId, currentPage, numPerPage));
    }

    @Override
    @ApiOperation("查询早晚报发送详情")
    @GetMapping("/getAdNewsGroupRecord")
    public ResponseDto<?> getAdNewsGroupRecord(
            @RequestHeader("token") @ApiParam("token信息") String token,
            @RequestParam("messageStatisticId") @ApiParam("统计表id") Long messageStatisticId
    ) throws BizException, PermissionException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        return new ResponseDto<>(adNewsBiz.getAdNewsGroupRecord(partyId, messageStatisticId));
    }

    @Override
    @ApiOperation("记录用户点击量")
    @PostMapping("/addAdNewsClickRecord")
    public ResponseDto<?> addAdNewsClickRecord(
            @CookieValue("userInfo") @ApiParam("用户信息") String userInfo,
            @RequestBody @ApiParam("用户点击量对象") AdNewsClickRecord adNewsClickRecord
    ) throws BizException, PermissionException {
        Map<String, Object> map = Cookie.getUserInfo(userInfo);
        Long channelId = (Long) map.get(Cookie._CHANNEL_ID);
        Long wechatUserId = (Long) map.get(Cookie._WECHAT_USER_ID);

        LOGGER.info("addAdNewsClickRecord Map => " + map.toString());

        adNewsClickRecord.setWechatUserId(wechatUserId);
        return new ResponseDto<>(adNewsBiz.addAdNewsClickRecord(adNewsClickRecord));
    }
}
