package com.pcloud.book.book.facade.impl;

import com.pcloud.book.book.biz.SearchBiz;
import com.pcloud.book.book.entity.HotBook;
import com.pcloud.book.book.entity.SearchRecord;
import com.pcloud.book.book.facade.SearchFacade;
import com.pcloud.common.core.constant.SystemCode;
import com.pcloud.common.dto.ResponseDto;
import com.pcloud.common.exceptions.BizException;
import com.pcloud.common.permission.PermissionException;
import com.pcloud.common.utils.SessionUtil;
import com.pcloud.common.utils.cookie.Cookie;
import com.pcloud.wechatgroup.message.enums.IsSystem;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.CookieValue;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.Map;


@Slf4j
@RestController("searchFacade")
@RequestMapping("search")
public class SearchFacadeImpl implements SearchFacade {

    @Autowired
    private SearchBiz searchBiz;

    @ApiOperation("创建热门书")
    @PostMapping("createHotBook")
    @Override
    public ResponseDto<?> createHotBook(
            @RequestHeader("token") String token,
            @RequestBody HotBook hotBook
    ) throws PermissionException {
        SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        searchBiz.createHotBook(hotBook);
        return new ResponseDto<>();
    }

    @ApiOperation("删除热门书")
    @GetMapping("deleteHotBook")
    @Override
    public ResponseDto<?> deleteHotBook(
            @RequestHeader("token") String token,
            @RequestParam("id") Long id
    ) throws PermissionException {
        SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        searchBiz.deleteHotBook(id);
        return new ResponseDto<>();
    }

    @ApiOperation("获取热门书PC")
    @GetMapping("getHotBookList4PC")
    @Override
    public ResponseDto<?> getHotBookList4PC(
            @RequestHeader("token") @ApiParam("token信息") String token,
            @RequestParam("currentPage") @ApiParam("当前页") Integer currentPage,
            @RequestParam("numPerPage") @ApiParam("每页条数") Integer numPerPage
    ) throws BizException, PermissionException {
        SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        return new ResponseDto<>(searchBiz.getHotBookList(currentPage,numPerPage));
    }

    @ApiOperation("获取热门书H5")
    @GetMapping("getHotBookList4H5")
    @Override
    public ResponseDto<?> getHotBookList4H5(
            @RequestParam("currentPage") @ApiParam("当前页") Integer currentPage,
            @RequestParam("numPerPage") @ApiParam("每页条数") Integer numPerPage
    ) throws BizException, PermissionException {
        return new ResponseDto<>(searchBiz.getHotBookList(currentPage,numPerPage));
    }

    @ApiOperation("修改热门书排序值")
    @PostMapping("updateHotBookSeqNum")
    @Override
    public ResponseDto<?> updateHotBookSeqNum(
            @RequestHeader("token") String token,
            @RequestBody HotBook hotBook
    ) throws PermissionException {
        SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        searchBiz.updateHotBookSeqNum(hotBook);
        return new ResponseDto<>();
    }

    @ApiOperation("创建搜索记录H5")
    @PostMapping("createSearchRecord4H5")
    @Override
    public ResponseDto<?> createSearchRecord4H5(
            @CookieValue(value = "userInfo") String  userInfo,
            @RequestBody SearchRecord searchRecord
    ) throws PermissionException {
        Cookie.getId(userInfo, Cookie._WECHAT_USER_ID);
        Long officialAccountId = Cookie.getId(userInfo, Cookie._OFFICIAL_ACCOUNTS_ID);
        searchBiz.createSearchRecord4H5(searchRecord, officialAccountId);
        return new ResponseDto<>();
    }

    @ApiOperation("获取热门搜索列表")
    @GetMapping("getSearchList")
    @Override
    public ResponseDto<?> getSearchList(
            @RequestHeader("token") @ApiParam("token信息") String token,
            @RequestParam(value = "content" ,required = false) @ApiParam("内容") String content,
            @RequestParam("currentPage") @ApiParam("当前页") Integer currentPage,
            @RequestParam("numPerPage") @ApiParam("每页条数") Integer numPerPage
    ) throws BizException, PermissionException {
        SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        return new ResponseDto<>(searchBiz.getSearchList(content,currentPage,numPerPage));
    }

    @ApiOperation("导出热门搜索列表")
    @GetMapping("exportSearchList")
    @Override
    public ResponseDto<?> exportSearchList(
            @RequestHeader("token") @ApiParam("token信息") String token,
            @RequestParam(value = "content" ,required = false) @ApiParam("内容") String content
    ) throws BizException, PermissionException {
        String systemCode = (String) SessionUtil.getVlaue(token, SessionUtil.SYSTEM_CODE);
        if (!SystemCode.pcloud.code.equalsIgnoreCase(systemCode)) {
            throw new PermissionException(PermissionException.PERMISSION_NOT_FOUND);
        }
        Map<String,Object> map =  SessionUtil.getToken4Redis(token);
        String isSystem = (String)map.get(SessionUtil.IS_SYSTEM);
        Long partyId = (Long) map.get(SessionUtil.PARTY_ID);
        if (IsSystem.NOT_SYSTEM.code.equals(isSystem)){
            partyId = (Long) map.get(SessionUtil.MEMBER_ID);
        }
        searchBiz.exportSearchList(systemCode,partyId,content);
        return new ResponseDto<>();
    }

    @ApiOperation("获取某个热门搜索列表")
    @GetMapping("getSearchRecordList")
    @Override
    public ResponseDto<?> getSearchRecordList(
            @RequestHeader("token") @ApiParam("token信息") String token,
            @RequestParam("content" ) @ApiParam("内容") String content,
            @RequestParam(value = "bookId", required = false) @ApiParam("书id") Long bookId,
            @RequestParam(value = "agentId", required = false) @ApiParam("出版社id") Long agentId,
            @RequestParam("currentPage") @ApiParam("当前页") Integer currentPage,
            @RequestParam("numPerPage") @ApiParam("每页条数") Integer numPerPage
    ) throws BizException, PermissionException {
        SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        return new ResponseDto<>(searchBiz.getSearchRecordList(content,bookId,agentId,currentPage,numPerPage));
    }

    @ApiOperation("导出某个热门搜索列表")
    @GetMapping("exportSearchRecordList")
    @Override
    public ResponseDto<?> exportSearchRecordList(
            @RequestHeader("token") @ApiParam("token信息") String token,
            @RequestParam("content" ) @ApiParam("内容") String content,
            @RequestParam(value = "bookId", required = false) @ApiParam("书id") Long bookId,
            @RequestParam(value = "agentId", required = false) @ApiParam("出版社id") Long agentId
    ) throws BizException, PermissionException {
        String systemCode = (String) SessionUtil.getVlaue(token, SessionUtil.SYSTEM_CODE);
        if (!SystemCode.pcloud.code.equalsIgnoreCase(systemCode)) {
            throw new PermissionException(PermissionException.PERMISSION_NOT_FOUND);
        }
        Map<String,Object> map =  SessionUtil.getToken4Redis(token);
        String isSystem = (String)map.get(SessionUtil.IS_SYSTEM);
        Long partyId = (Long) map.get(SessionUtil.PARTY_ID);
        if (IsSystem.NOT_SYSTEM.code.equals(isSystem)){
            partyId = (Long) map.get(SessionUtil.MEMBER_ID);
        }
        searchBiz.exportSearchRecordList(systemCode,partyId,content,bookId,agentId);
        return new ResponseDto<>();
    }

    @ApiOperation("获取用户最近搜索列表H5")
    @GetMapping("getUserSearchList4H5")
    @Override
    public ResponseDto<?> getUserSearchList4H5(
         @CookieValue(value = "userInfo") String  userInfo,
            @RequestParam(value = "wxId", required = false) @ApiParam("用户id") String wxId,
            @RequestParam(value = "wechatUserId", required = false) @ApiParam("微信id") Long wechatUserId,
            @RequestParam("currentPage") @ApiParam("当前页") Integer currentPage,
            @RequestParam("numPerPage") @ApiParam("每页条数") Integer numPerPage
    ) throws BizException, PermissionException {
        return new ResponseDto<>(searchBiz.getUserSearchList4H5(wxId,  Cookie.getId(userInfo, Cookie._WECHAT_USER_ID), currentPage,numPerPage));
    }

    @ApiOperation("删除用户搜索记录H5")
    @GetMapping("deleteUserSearchList4H5")
    @Override
    public ResponseDto<?> deleteUserSearchList4H5(
            @RequestParam(value = "wxId", required = false) @ApiParam("用户id") String wxId,
            @RequestParam(value = "wechatUserId", required = false) @ApiParam("微信id") Long wechatUserId
    ) throws BizException, PermissionException {
        searchBiz.deleteUserSearchList4H5(wxId,wechatUserId);
        return new ResponseDto<>();
    }

    @ApiOperation("获取搜索书籍的用户头像H5")
    @GetMapping("getBookSearchUserList4H5")
    @Override
    public ResponseDto<?> getBookSearchUserList4H5(
            @RequestParam("bookId" ) @ApiParam("书id") Long bookId
    ) throws BizException, PermissionException {
        return new ResponseDto<>(searchBiz.getBookSearchUserList4H5(bookId));
    }

    @ApiOperation("获取用户搜索书籍前几")
    @GetMapping("getBookSearchTop")
    @Override
    public ResponseDto<?> getBookSearchTop(
            @RequestParam("top" ) @ApiParam("前几") Integer top
    ) throws BizException, PermissionException {
        return new ResponseDto<>(searchBiz.getBookSearchTop(top));
    }

    @ApiOperation("丢书新增搜索记录")
    @PostMapping("createSearchRecord4Sale")
    public ResponseDto<?> createSearchRecord4Sale(
            @CookieValue(value = "userInfo") String  userInfo,
            @RequestBody SearchRecord searchRecord
    ) throws PermissionException {
        Cookie.getId(userInfo, Cookie._WECHAT_USER_ID);
        Long officialAccountId = Cookie.getId(userInfo, Cookie._OFFICIAL_ACCOUNTS_ID);
        searchBiz.createSearchRecord4Sale(searchRecord, officialAccountId);
        return new ResponseDto<>();
    }

    @ApiOperation("获取用户最近丢书搜索列表")
    @GetMapping("getUserSearchList4Sale")
    public ResponseDto<?> getUserSearchList4Sale(
            @CookieValue(value = "userInfo") String  userInfo,
            @RequestParam(value = "wxId", required = false) @ApiParam("用户id") String wxId,
            @RequestParam(value = "wechatUserId", required = false) @ApiParam("微信id") Long wechatUserId,
            @RequestParam("currentPage") @ApiParam("当前页") Integer currentPage,
            @RequestParam("numPerPage") @ApiParam("每页条数") Integer numPerPage
    ) throws BizException, PermissionException {
        return new ResponseDto<>(searchBiz.getUserSearchList4Sale(wxId,  Cookie.getId(userInfo, Cookie._WECHAT_USER_ID), currentPage,numPerPage));
    }


    @ApiOperation("删除用户搜索记录4sale")
    @GetMapping("deleteUserSearchList4Sale")
    public ResponseDto<?> deleteUserSearchList4Sale(
            @RequestParam(value = "wxId", required = false) @ApiParam("用户id") String wxId,
            @RequestParam(value = "wechatUserId", required = false) @ApiParam("微信id") Long wechatUserId
    ) throws BizException, PermissionException {
        searchBiz.deleteUserSearchList4Sale(wxId,wechatUserId);
        return new ResponseDto<>();
    }




}
