/**
 * 
 */
package com.pcloud.book.mq.config;

import com.pcloud.book.base.exception.BookBizException;
import com.pcloud.book.mq.topic.ChooseDefaultChannelListener;
import com.pcloud.book.mq.topic.RegisterPartyListener;
import com.pcloud.book.mq.topic.DeleteVideoListener;
import com.pcloud.book.util.properties.BookProps;
import com.pcloud.common.exceptions.BizException;
import org.apache.activemq.ActiveMQConnection;
import org.apache.activemq.ActiveMQConnectionFactory;
import org.apache.activemq.command.ActiveMQQueue;
import org.apache.activemq.command.ActiveMQTopic;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.context.embedded.AnnotationConfigEmbeddedWebApplicationContext;
import org.springframework.context.ApplicationListener;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.stereotype.Component;

import javax.jms.*;
import java.io.Serializable;

/**
 * @描述：
 * 
 * @作者：songx
 * @创建时间：2017年2月13日,上午11:28:54 @版本：1.0
 */
@Component("mqConfig")
public class MqConfig implements ApplicationListener<ContextRefreshedEvent> {

	/**
	 * 
	 */
	private final static Logger LOGGER = LoggerFactory.getLogger(MqConfig.class);

	private static Session session; // 消息上下文
	@Autowired
	private ActiveMQConnectionFactory connectionFactory;

	@Autowired
	private DeleteVideoListener deleteVideoListener; // 清除录音

	private static MessageProducer bookDeleteMsgProducer; // 书籍删除消息生产者
	
	@Autowired
	private RegisterPartyListener registerPartyListener;	// 注册当事人Topic监听

	private static MessageProducer bookThawProducer; // 书籍解冻消息生产者

	private Boolean bool = false;
	/**
	 * 初始化
	 */
	@Override
	public void onApplicationEvent(ContextRefreshedEvent arg0) {
		if (!bool && arg0.getSource() instanceof AnnotationConfigEmbeddedWebApplicationContext) {
			LOGGER.info("【消息队列】初始化消息发送队列,<START>");
			bool = true;
			try {
				ActiveMQConnection conn = (ActiveMQConnection) connectionFactory.createConnection();
				conn.start();
				session = conn.createSession(false, Session.AUTO_ACKNOWLEDGE);
			} catch (Exception e) {
				LOGGER.error("【消息队列】初始化消息发送队列,<ERROR>:" + e.getMessage(), e);
				throw new BookBizException(BookBizException.SEND_TOPIC_ERROR, "初始化消息发送队列失败");
			}
			// 创建生产者
			bookDeleteMsgProducer = this.createProducer(BookProps.getBookDeleteProducer());
			bookThawProducer = this.createProducer(BookProps.getBookThawProducer());
			// 创建消费者,开始监听
			this.createConsumer(deleteVideoListener, BookProps.getDeleteVideoConsumer());
			this.createConsumer(registerPartyListener, BookProps.getRegisterPartyConsumer());
			LOGGER.info("【消息队列】初始化消息发送队列,<END>");
		}
	}

	/**
	 * 创建生产者
	 */
	private MessageProducer createProducer(String producerName) throws BizException {
		try {
			return session.createProducer(new ActiveMQTopic(producerName));
		} catch (Exception e) {
			LOGGER.error("【消息队列】初始化生产者Topic,<ERROR>.[" + producerName + "]:" + e.getMessage(), e);
		}
		return null;
	}

	/**
	 * 创建消费者监听
	 */
	private void createConsumer(MessageListener messageListener, String consumerName) throws BizException {
		try {
			Queue queue = new ActiveMQQueue(consumerName);
			MessageConsumer consumer = session.createConsumer(queue);
			consumer.setMessageListener(messageListener);
		} catch (Exception e) {
			LOGGER.error("【消息队列】初始化消费者监听,<ERROR>.[" + consumerName + "]:" + e.getMessage(), e);
		}
	}

	/**
	 * 发送书籍删除Topic
	 * 
	 * @param object
	 */
	public static void sendBookDeleteTopic(Serializable object) throws BizException {
		sendMsg(bookDeleteMsgProducer, object, BookProps.getBookDeleteProducer());
	}

	/**
	 * 发送图书解冻Topic
	 *
	 * @param object
	 */
	public static void sendBookThawTopic(Serializable object) throws BizException {
		sendMsg(bookThawProducer, object, BookProps.getBookThawProducer());
	}

	/**
	 * 发送消息
	 * 
	 * @param producer
	 * @param object
	 */
	private static void sendMsg(MessageProducer producer, Serializable object, String producerName)
			throws BizException {
		LOGGER.info("【消息队列】发送Topic消息,<START>.[producerName]=" + producerName);
		try {
			ObjectMessage message = session.createObjectMessage(object);
			producer.send(message);
		} catch (Exception e) {
			LOGGER.error("【消息队列】发送Topic消息[" + producerName + "],<ERROR>:" + e.getMessage(), e);
		}
		LOGGER.info("【消息队列】发送Topic消息,<END>");
	}
}
