package com.pcloud.book.personalstage.biz.impl;

import com.pcloud.book.base.exception.BookBizException;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.pcloud.book.book.entity.Book;
import com.pcloud.book.consumer.wechatgroup.WechatGroupConsr;
import com.pcloud.book.group.biz.WeixinQrcodeBiz;
import com.pcloud.book.guide.biz.PcloudGuideBiz;
import com.pcloud.book.keywords.enums.ReplyTypeEnum;
import com.pcloud.book.mq.delay.DelayMessageSender;
import com.pcloud.book.pcloudkeyword.dao.PcloudRobotClassifyDao;
import com.pcloud.book.pcloudkeyword.dao.PcloudRobotDao;
import com.pcloud.book.pcloudkeyword.entity.PcloudRobot;
import com.pcloud.book.pcloudkeyword.entity.PcloudRobotClassify;
import com.pcloud.book.personalstage.biz.PersonalStageBiz;
import com.pcloud.book.personalstage.biz.PersonalStageJumpBiz;
import com.pcloud.book.personalstage.constant.PersonalStageConstant;
import com.pcloud.book.personalstage.dao.*;
import com.pcloud.book.personalstage.dto.LinkupDelayDTO;
import com.pcloud.book.personalstage.dto.PersonalStageJumpDto;
import com.pcloud.book.personalstage.dto.PersonalStageJumpKeywordDto;
import com.pcloud.book.personalstage.entity.PersonalStage;
import com.pcloud.book.personalstage.entity.PersonalStageJump;
import com.pcloud.book.personalstage.entity.PersonalStageJumpEmail;
import com.pcloud.book.personalstage.entity.PersonalStageJumpKeyword;
import com.pcloud.book.personalstage.entity.PersonalStageJumpLinkup;
import com.pcloud.book.personalstage.entity.PersonalStageUser;
import com.pcloud.book.personalstage.enums.JumpTypeEnum;
import com.pcloud.book.personalstage.enums.PersonalStageUserStateEnum;
import com.pcloud.book.personalstage.vo.request.BaseStageJumpRequestVO;
import com.pcloud.book.personalstage.vo.request.CreateStageJumpRequestVO;
import com.pcloud.book.personalstage.vo.request.UpdateStageJumpRequestVO;
import com.pcloud.book.util.common.YesOrNoEnums;
import com.pcloud.common.core.aspect.ParamLog;
import com.pcloud.common.core.biz.MessageBiz;
import com.pcloud.common.core.dto.SendEmailDto;
import com.pcloud.common.core.mq.DelayQueueDTO;
import com.pcloud.common.exceptions.BizException;
import com.pcloud.common.page.PageBeanNew;
import com.pcloud.common.page.PageParam;
import com.pcloud.common.utils.DateNewUtils;
import com.pcloud.common.utils.ListUtils;
import com.pcloud.common.utils.NumberUtil;
import com.pcloud.common.utils.httpclient.UrlUtils;
import com.pcloud.common.utils.string.StringUtil;
import com.pcloud.wechatgroup.group.dto.GroupRobotDTO;
import com.pcloud.wechatgroup.group.dto.GroupUserDTO;
import com.sdk.wxgroup.SendMessageTypeEnum;
import com.sdk.wxgroup.SendPicMessageVO;
import com.sdk.wxgroup.SendTextMessageVO;
import com.sdk.wxgroup.WxGroupSDK;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.stream.Collectors;

@Component("personalStageJumpBiz")
public class PersonalStageJumpBizImpl implements PersonalStageJumpBiz {

    private static final Logger LOGGER = LoggerFactory.getLogger(PersonalStageJumpBizImpl.class);

    @Autowired
    private PersonalStageBiz personalStageBiz;
    @Autowired
    private PersonalStageJumpDao personalStageJumpDao;
    @Autowired
    private PersonalStageJumpKeywordDao personalStageJumpKeywordDao;
    @Autowired
    private PersonalStageJumpEmailDao personalStageJumpEmailDao;
    @Autowired
    private PersonalStageJumpLinkupDao personalStageJumpLinkupDao;
    @Autowired
    private PersonalStageUserDao personalStageUserDao;
    @Autowired
    private WechatGroupConsr wechatGroupConsr;
    @Autowired
    private WeixinQrcodeBiz weixinQrcodeBiz;
    @Autowired
    private PcloudRobotDao pcloudRobotDao;
    @Autowired
    private PcloudRobotClassifyDao pcloudRobotClassifyDao;
    @Autowired
    private MessageBiz messageBiz;
    @Autowired
    private PcloudGuideBiz pcloudGuideBiz;
    @Autowired
    private DelayMessageSender delayMessageSender;
    @Autowired
    private PersonalStageDao personalStageDao;
    @Autowired
    private PersonalStageReplyDao personalStageReplyDao;
    @Value("${system.env}")
    private String envStr;

    @Value("${wechat.group.link.prefix}")
    private String wechatLinkPrefix;
    //需求单链接替换符
    private final static String PERSONAL_STAGE_PAPER_TEMPLATE = "${PROGRESS_URL&amp;ProgressId}";
    //需求单链接替换符
    private final static String PERSONAL_STAGE_PAPER_TEMPLATE1 = "${PROGRESS_URL&ProgressId}";
    //需求定制单路由  https://wechat666.raysgo.com/personalCenter/questionNaire?paperId=679
    private final static String PERSONAL_STAGE_PAPER = "/personalCenter/questionNaire?paperId=";

    @Override
    @ParamLog("新增阶段跳转")
    @Transactional(rollbackFor = Exception.class)
    public void createPersonalStageJump(CreateStageJumpRequestVO vo) {
        //校验关键词是否重复
        if (!JumpTypeEnum.PAY_TRIGGER.key.equals(vo.getJumpType()) && !JumpTypeEnum.PAPER_TRIGGER.key.equals(vo.getJumpType())){
            this.checkKeywords(vo.getKeywords(), vo.getPersonalStageId(), null);
        }
        if (JumpTypeEnum.PAPER_TRIGGER.key.equals(vo.getJumpType())){
            this.checkPaper(vo.getPersonalStageId(), null);
        }
        if (vo.getOpenEmail()){
            this.checkEmail(vo.getEmails());
        }
        PersonalStageJump personalStageJump = CreateStageJumpRequestVO.valueToJumpEntity(vo);
        personalStageJumpDao.insert(personalStageJump);
        List<PersonalStageJumpKeyword> jumpKeywords = BaseStageJumpRequestVO.valueToJumpKeywords(vo, personalStageJump.getId());
        if (!ListUtils.isEmpty(jumpKeywords)) {
            personalStageJumpKeywordDao.batchInsert(jumpKeywords);
        }
        List<PersonalStageJumpEmail> jumpEmails = BaseStageJumpRequestVO.valueToJumpEmails(vo, personalStageJump.getId());
        if (!ListUtils.isEmpty(jumpEmails)) {
            personalStageJumpEmailDao.batchInsert(jumpEmails);
        }
        List<PersonalStageJumpLinkup> jumpLinkups = BaseStageJumpRequestVO.valueToJumpLinkups(vo, personalStageJump.getId());
        if (!ListUtils.isEmpty(jumpLinkups)){
            personalStageJumpLinkupDao.batchInsert(jumpLinkups);
        }
    }

    @Override
    @ParamLog("获取跳转设置列表")
    public PageBeanNew getJumpList(Long personalStageId, Integer currentPage, Integer numPerPage) {
        if(!NumberUtil.isNumber(personalStageId) || personalStageId <= 0){
            throw new BizException(BizException.PARAM_IS_NULL.getCode(), "personalStageId 值不能为空");
        }
        HashMap<String, Object> map = Maps.newHashMap();
        map.put("personalStageId", personalStageId);
        PageBeanNew<PersonalStageJumpDto> pageBeanNew = personalStageJumpDao.listPageNew(new PageParam(currentPage, numPerPage), map, "getJumpList");
        List<PersonalStageJumpDto> recordList = pageBeanNew.getRecordList();
        if(ListUtils.isEmpty(recordList)){
            return pageBeanNew;
        }
        this.fillJumpKeywords(recordList);
        this.fillJumpEmails(recordList);
        this.fillJumpLinkups(recordList);
        return pageBeanNew;
    }

    @ParamLog("填充邮件")
    private void fillJumpEmails(List<PersonalStageJumpDto> recordList) {
        List<Long> jumpIds = recordList.stream().map(x -> x.getId()).collect(Collectors.toList());
        HashMap<String, Object> map = Maps.newHashMap();
        map.put("jumpIds", jumpIds);
        List<PersonalStageJumpEmail> jumpEmails = personalStageJumpEmailDao.listBy(map);
        if(ListUtils.isEmpty(jumpEmails)){
            return;
        }
        Map<Long, List<PersonalStageJumpEmail>> mapJumpEmails = jumpEmails.stream().collect(Collectors.groupingBy(x -> x.getPersonalStageJumpId()));
        for (PersonalStageJumpDto jumpDto : recordList){
            jumpDto.setJumpEmails(mapJumpEmails.getOrDefault(jumpDto.getId(), Lists.newArrayList()));
        }
    }

    @ParamLog("填充跳转衔接语")
    private void fillJumpLinkups(List<PersonalStageJumpDto> recordList) {
        List<Long> jumpIds = recordList.stream().map(x -> x.getId()).collect(Collectors.toList());
        HashMap<String, Object> map = Maps.newHashMap();
        map.put("jumpIds", jumpIds);
        List<PersonalStageJumpLinkup> linkups = personalStageJumpLinkupDao.listBy(map);
        if(ListUtils.isEmpty(linkups)){
            return;
        }
        Map<Long, List<PersonalStageJumpLinkup>> mapJumpLinkups = linkups.stream().collect(Collectors.groupingBy(x -> x.getPersonalStageJumpId()));
        for (PersonalStageJumpDto jumpDto : recordList){
            jumpDto.setJumpLinkups(mapJumpLinkups.getOrDefault(jumpDto.getId(), Lists.newArrayList()));
        }
    }

    @ParamLog("填充跳转关键字列表")
    private void fillJumpKeywords(List<PersonalStageJumpDto> recordList) {
        List<Long> jumpIds = recordList.stream().map(x -> x.getId()).collect(Collectors.toList());
        HashMap<String, Object> map = Maps.newHashMap();
        map.put("jumpIds", jumpIds);
        List<PersonalStageJumpKeyword> keywords = personalStageJumpKeywordDao.listBy(map);
        if(ListUtils.isEmpty(keywords)){
            return;
        }
        Map<Long, List<PersonalStageJumpKeyword>> mapJumpKeyword = keywords.stream().collect(Collectors.groupingBy(x -> x.getPersonalStageJumpId()));
        for (PersonalStageJumpDto jumpDto : recordList){
            jumpDto.setJumpKeywords(mapJumpKeyword.getOrDefault(jumpDto.getId(), Lists.newArrayList()));
        }
    }

    @Override
    @ParamLog("修改跳转阶段")
    public void updatePersonalStageJump(UpdateStageJumpRequestVO vo) {
        //校验关键词是否重复
        if (!JumpTypeEnum.PAY_TRIGGER.key.equals(vo.getJumpType()) && !JumpTypeEnum.PAPER_TRIGGER.key.equals(vo.getJumpType())){
            this.checkKeywords(vo.getKeywords(), vo.getPersonalStageId(), vo.getPersonalStageJumpId());
        }
        if (JumpTypeEnum.PAPER_TRIGGER.key.equals(vo.getJumpType())){
            this.checkPaper(vo.getPersonalStageId(), vo.getPersonalStageJumpId());
        }
        if (vo.getOpenEmail()){
            this.checkEmail(vo.getEmails());
        }
        PersonalStageJump personalStageJump = UpdateStageJumpRequestVO.valueToJumpEntity(vo);
        personalStageJumpDao.update(personalStageJump);
        //删除原有的关键词
        personalStageJumpKeywordDao.deleteByJumpId(vo.getPersonalStageJumpId());
        //删除原有的邮件地址
        personalStageJumpEmailDao.deleteByJumpId(vo.getPersonalStageJumpId());
        //删除原有的衔接语
        personalStageJumpLinkupDao.deleteByJumpId(vo.getPersonalStageJumpId());
        List<PersonalStageJumpKeyword> jumpKeywords = BaseStageJumpRequestVO.valueToJumpKeywords(vo, personalStageJump.getId());
        if (!ListUtils.isEmpty(jumpKeywords)){
            personalStageJumpKeywordDao.batchInsert(jumpKeywords);
        }
        List<PersonalStageJumpEmail> jumpEmails = BaseStageJumpRequestVO.valueToJumpEmails(vo, personalStageJump.getId());
        if (!ListUtils.isEmpty(jumpEmails)){
            personalStageJumpEmailDao.batchInsert(jumpEmails);
        }
        List<PersonalStageJumpLinkup> jumpLinkups = BaseStageJumpRequestVO.valueToJumpLinkups(vo, personalStageJump.getId());
        if (!ListUtils.isEmpty(jumpLinkups)){
            personalStageJumpLinkupDao.batchInsert(jumpLinkups);
        }
    }

    @ParamLog("校验跳转关键词")
    private void checkKeywords(List<String> keywords, Long personalStageId, Long personalStageJumpId) {
        if (ListUtils.isEmpty(keywords)){
            throw new BookBizException(BookBizException.ERROR, "跳转关键词不能为空");
        }
        List<String> words = keywords.stream().distinct().collect(Collectors.toList());
        if (keywords.size() != words.size()) {
            throw new BookBizException(BookBizException.ERROR, "跳转关键词不能重复");
        }
        PersonalStageJumpKeyword personalStageJumpKeyword = personalStageJumpKeywordDao.getStageKeywordByKeywords(keywords, personalStageId);
        if (null != personalStageJumpKeyword) {
            if (null == personalStageJumpId
                    || !personalStageJumpId.equals(personalStageJumpKeyword.getPersonalStageJumpId())) {
                //如果personalStageJumpId为空，则为新增时校验,否则为修改时校验
                throw new BookBizException(BookBizException.ERROR, "跳转关键词不能重复");
            }
        }
    }

    @ParamLog("校验邮件地址")
    private void checkEmail(List<String> emails){
        if (ListUtils.isEmpty(emails)){
            throw new BookBizException(BookBizException.ERROR, "邮件地址不能为空");
        }
        List<String> fEmail = emails.stream().filter(e -> !StringUtil.isEmpty(e)).collect(Collectors.toList());
        if (ListUtils.isEmpty(fEmail) || emails.size() != fEmail.size()){
            throw new BookBizException(BookBizException.ERROR, "邮件地址不能为空");
        }
    }

    @ParamLog("校验需求单跳转是否唯一")
    private void checkPaper(Long personalStageId, Long personalStageJumpId){
        //获取阶段信息
        PersonalStage personalStage = personalStageDao.getById(personalStageId);
        if (null == personalStage || null == personalStage.getPaperId()){
            throw new BookBizException(BookBizException.ERROR, "阶段未配置需求定制单");
        }
        //获取阶段的需求单跳转
        List<PersonalStageJump> personalStageJumps = personalStageJumpDao.getByJumpType(personalStageId, JumpTypeEnum.PAPER_TRIGGER.key);
        if (ListUtils.isEmpty(personalStageJumps)){
            return;
        }
        if (null == personalStageJumpId || personalStageJumpId.equals(personalStageJumps.get(0).getId())){
            return;
        }
        throw new BookBizException(BookBizException.ERROR, "已有其他阶段跳转配置了需求单跳转");
    }

    @Override
    @ParamLog("根据id获取跳转对象")
    public PersonalStageJumpDto getJump(Long jumpId) {
        if(!NumberUtil.isNumber(jumpId) || jumpId <= 0){
            throw new BizException(BizException.PARAM_IS_NULL.getCode(), "jumpId 值不能为空");
        }
        PersonalStageJumpDto jumpDto = personalStageJumpDao.getDtoById(jumpId);
        if(jumpDto == null){
            jumpDto = new PersonalStageJumpDto();
            jumpDto.setJumpKeywords(Lists.newArrayList());
            jumpDto.setJumpLinkups(Lists.newArrayList());
            return jumpDto;
        }
        this.fillJumpKeywords(Lists.newArrayList(jumpDto));
        this.fillJumpEmails(Lists.newArrayList(jumpDto));
        this.fillJumpLinkups(Lists.newArrayList(jumpDto));
        return jumpDto;
    }

    @Override
    @ParamLog("删除跳转")
    public void deleteJump(Long jumpId) {
        if(!NumberUtil.isNumber(jumpId) || jumpId <= 0){
            throw new BizException(BizException.PARAM_IS_NULL.getCode(), "jumpId 值不能为空");
        }
        // 删除跳转
        personalStageJumpDao.deleteById(jumpId);
        // 删除关键字
        personalStageJumpKeywordDao.deleteByJumpId(jumpId);
        // 删除邮件
        personalStageJumpEmailDao.deleteByJumpId(jumpId);
    }

    @Override
    @ParamLog("开启关闭邮件提醒")
    public Long updateEmailStatus(Long jumpId, Integer openEmail) {
        if(!NumberUtil.isNumber(jumpId) || jumpId <= 0){
            throw new BizException(BizException.PARAM_IS_NULL.getCode(), "jumpId 值不能为空");
        }
        PersonalStageJump jump = new PersonalStageJump();
        jump.setId(jumpId);
        jump.setOpenEmail(YesOrNoEnums.YES.getValue().equals(openEmail));
        return personalStageJumpDao.update(jump);
    }

    /**
     *
     * @param userWxId
     * @param robotWxId
     * @param content
     * @return  false:未能跳转到下个阶段，需要后续的其它处理；
     *          true:命中关键字，已经跳转到下个阶段，无需后续处理；
     */
    @Override
    @ParamLog("处理用户阶段跳转逻辑")
    @Transactional(rollbackFor = Exception.class, propagation = Propagation.REQUIRES_NEW)
    public Boolean handlePersonalStageJump(String userWxId, String robotWxId, String content, JumpTypeEnum jumpTypeEnum) {
        // 查询用户的当前状态信息
        PersonalStageUser currentStageUser = personalStageUserDao.getLast(userWxId, robotWxId, null);
        // 如果用户没有阶段状态，则不处理
        if(currentStageUser == null || currentStageUser.getId() <= 0) {
            return false;
        }
        // TODO 判断消息的阶段是否和用户当前阶段匹配，如果不匹配需要当做已处理状态返回，以免触发后续其它操作
        //  (此操作需要建立在用户的消息有阶段id的情况下)
        // if(personalStageId != currentStageUser.getPersonalStageId()) { return true;  }
        // 查询命中关键字跳转下一阶段的id
        PersonalStageJumpKeywordDto jumpKeywordDto = personalStageJumpKeywordDao.getByKeyword(currentStageUser.getPersonalStageId(), content, jumpTypeEnum.key);
        if(null == jumpKeywordDto || StringUtil.isEmpty(jumpKeywordDto.getKeyword())){
            // 未能命中关键词
            return false;
        }
        // 如果下一个阶段是空，则已完成
        if(null == jumpKeywordDto.getAfterPersonalStageId() || jumpKeywordDto.getAfterPersonalStageId() <=0){
            // 更新到完成状态
            // updateStageToComplete(currentStageUser);
            return true;
        }
        GroupRobotDTO groupRobotDTO = wechatGroupConsr.getGroupRobotByWxId(robotWxId);
        String ip = weixinQrcodeBiz.getRobotIpByGeneration(groupRobotDTO.getVersion());
        // 发送邮件
        sendEmail(userWxId, robotWxId, content, jumpKeywordDto);
        // 将用户置为下个阶段
        PersonalStageUser nextPersonalStageUser = personalStageBiz.nextStageAddStageUserAndWakeupDelay(robotWxId, userWxId, ip, jumpKeywordDto.getAfterPersonalStageId());
        // 发送内容衔接语
        sendJumpLinkups(userWxId, robotWxId, ip, jumpKeywordDto.getPersonalStageJumpId(),nextPersonalStageUser.getId());
        // 停止发送引导语
        pcloudGuideBiz.stopPcloudGuidePush(robotWxId, userWxId);
        return true;
    }

    @ParamLog("处理衔接语跳转延时")
    @Override
    public void dealDelayLinkup(DelayQueueDTO dto) {
        LinkupDelayDTO linkupDelayDTO=(LinkupDelayDTO)dto.getMsg();
        PersonalStageJumpLinkup jumpLinkup = personalStageJumpLinkupDao.getById(linkupDelayDTO.getPersonalStageJumpLinkupId());
        if (jumpLinkup==null){
            LOGGER.info("没有该跳转衔接语！");
            return;
        }
        List<PersonalStageJumpLinkup> list = new ArrayList<>();
        list.add(jumpLinkup);
        this.replacePaperUrl(list, linkupDelayDTO.getRobotId(), linkupDelayDTO.getWxId(), jumpLinkup.getPersonalStageJumpId());
        jumpLinkup = list.get(0);
        if(ReplyTypeEnum.TEXT.value.equals( jumpLinkup.getReplyType())){
            SendTextMessageVO sendTextMessageVO = new SendTextMessageVO();
            sendTextMessageVO.setContent(personalStageBiz.replaceProjectProgressUrl(jumpLinkup.getLinkupContent(),linkupDelayDTO.getRobotId(),linkupDelayDTO.getWxId(),linkupDelayDTO.getPersonalStageUserId()));
            sendTextMessageVO.setAltId(linkupDelayDTO.getRobotId());
            sendTextMessageVO.setWxGroupId(dto.getKey());
            sendTextMessageVO.setIp(linkupDelayDTO.getIp());
            sendTextMessageVO.setCode(SendMessageTypeEnum.SELF.getCode());
            WxGroupSDK.sendTextMessage(sendTextMessageVO);
        } else if(ReplyTypeEnum.IMAGE.value.equals( jumpLinkup.getReplyType())) {
            SendPicMessageVO sendPicMessageVO = new SendPicMessageVO();
            sendPicMessageVO.setWxGroupId(dto.getKey());
            sendPicMessageVO.setAltId(linkupDelayDTO.getRobotId());
            sendPicMessageVO.setPicUrl(jumpLinkup.getPicUrl());
            sendPicMessageVO.setIp(linkupDelayDTO.getIp());
            sendPicMessageVO.setCode(SendMessageTypeEnum.SELF.getCode());
            WxGroupSDK.sendPicMessage(sendPicMessageVO);
        }
    }

    @ParamLog("处理用户提交表单阶段跳转")
    public boolean handlePersonalStagePaperJump(String userWxId, String robotWxId, Long paperId){
        if (null == paperId){
            return false;
        }
        // 查询用户的当前状态信息
        PersonalStageUser currentStageUser = personalStageUserDao.getLast(userWxId, robotWxId, null);
        // 如果用户没有阶段状态，则不处理
        if(currentStageUser == null || currentStageUser.getId() <= 0) {
            return false;
        }
        Long stageId = currentStageUser.getPersonalStageId();
        PersonalStage personalStage = personalStageDao.getById(stageId);
        Long jumpPaperId = personalStage.getPaperId();
        if (!paperId.equals(jumpPaperId)){
            //如果不是当前阶段的需求单id，则不处理
            return false;
        }
        //获取当前阶段的表单跳转
        List<PersonalStageJump> personalStageJump = personalStageJumpDao.getByJumpType(stageId, JumpTypeEnum.PAPER_TRIGGER.key);
        if (ListUtils.isEmpty(personalStageJump)){
            return false;
        }
        PersonalStageJump paperJump = personalStageJump.get(0);
        // 如果下一个阶段是空，则已完成
        if(null == paperJump.getAfterPersonalStageId() || paperJump.getAfterPersonalStageId() <=0){
            // 更新到完成状态
            updateStageToComplete(currentStageUser);
            return true;
        }
        GroupRobotDTO groupRobotDTO = wechatGroupConsr.getGroupRobotByWxId(robotWxId);
        String ip = weixinQrcodeBiz.getRobotIpByGeneration(groupRobotDTO.getVersion());
        // 发送邮件
        String content = "读者提交了反馈意见，请到“个人号管理——需求定制单”中查看该读者的需求并出相应方案";
        PersonalStageJumpKeywordDto jumpKeywordDto = new PersonalStageJumpKeywordDto();
        jumpKeywordDto.setAfterPersonalStageId(paperJump.getAfterPersonalStageId());
        jumpKeywordDto.setPersonalStageJumpId(paperJump.getId());
        sendEmail(userWxId, robotWxId, content, jumpKeywordDto);
        // 将用户置为下个阶段
        PersonalStageUser nextPersonalStageUser = personalStageBiz.nextStageAddStageUserAndWakeupDelay(robotWxId, userWxId, ip, paperJump.getAfterPersonalStageId());
        // 发送内容衔接语
        sendJumpLinkups(userWxId, robotWxId, ip, paperJump.getId(), nextPersonalStageUser.getId());
        // 停止发送引导语
        pcloudGuideBiz.stopPcloudGuidePush(robotWxId, userWxId);
        return true;
    }

    @Override
    public void sendPaperEmail(String userWxId, String robotWxId) {
        GroupUserDTO wxUserInfo = wechatGroupConsr.getWxUserInfoByWxUserId(userWxId);
        if(wxUserInfo==null){
            LOGGER.info("未找到用户信息，无法发送阶段跳转邮件");
        }
        PcloudRobot pcloudRobot = pcloudRobotDao.getByWxId(robotWxId);
        PcloudRobotClassify robotClassify = pcloudRobotClassifyDao.getById(pcloudRobot.getRobotType().longValue());
        if (robotClassify == null|| robotClassify.getKeywordClassifyId() == null){
            LOGGER.info("小号分类id为空，无法发送阶段跳转邮件");
            return;
        }
        List<String> emailList = new ArrayList<>();
        if ("pro".equalsIgnoreCase(envStr)){
            emailList.add("leibin@dcrays.cn");
            emailList.add("zhoulei@dcrays.cn");
        }else {
            emailList.add("hyy_8062@163.com");
            emailList.add("zhoulei@dcrays.cn");
            emailList.add("312949619@qq.com");
        }
        SendEmailDto sendEmailDto = new SendEmailDto();
        sendEmailDto.setTypeCode("feedback_notice_email");
        Map<String, Object> content = new HashMap<>();
        content.put("nickname", wxUserInfo.getNickName());
        content.put("wxUserId", userWxId);
        content.put("robotClassifyName", robotClassify.getClassifyName());
        content.put("robotNumber", pcloudRobot.getUniqueNumber());
        content.put("robotId", robotWxId);
        content.put("time", DateNewUtils.getLongDateStr());
        sendEmailDto.setContent(content);
        for (String email: emailList){
            sendEmailDto.setToEmail(email);
            messageBiz.sendEmail(sendEmailDto);
        }
    }

    @ParamLog("发送阶段跳转邮件")
    private void sendEmail(String userWxId, String robotWxId, String content, PersonalStageJumpKeywordDto jumpKeywordDto) {
        try {
            List<PersonalStageJumpEmail> jumpEmails = personalStageJumpEmailDao.getByJumpId(jumpKeywordDto.getPersonalStageJumpId());
            if(ListUtils.isEmpty(jumpEmails)){
                return;
            }
            GroupUserDTO wxUserInfo = wechatGroupConsr.getWxUserInfoByWxUserId(userWxId);
            if(wxUserInfo==null){
                LOGGER.info("未找到用户信息，无法发送阶段跳转邮件");
            }
            PcloudRobot pcloudRobot = pcloudRobotDao.getByWxId(robotWxId);
            PcloudRobotClassify robotClassify = pcloudRobotClassifyDao.getById(pcloudRobot.getRobotType().longValue());
            if (robotClassify == null|| robotClassify.getKeywordClassifyId() == null){
                LOGGER.info("小号分类id为空，无法发送阶段跳转邮件");
                return;
            }
            PersonalStage personalStage = personalStageBiz.getPersonalStage(jumpKeywordDto.getAfterPersonalStageId());
            if (personalStage == null || StringUtil.isEmpty(personalStage.getName())){
                LOGGER.info("未找到阶段信息，无法发送阶段跳转邮件");
                return;
            }
            for (PersonalStageJumpEmail email : jumpEmails){
                sendStageChangeEmail(email.getEmail(), content, wxUserInfo.getNickName(), userWxId, robotClassify.getClassifyName(), pcloudRobot.getUniqueNumber(), robotWxId, personalStage.getName());
            }
        } catch (Exception e){
            LOGGER.info("阶段跳转邮件发送失败：" + e.getMessage(), e);
        }
    }

    /**
     * 发送阶段流转邮件
     * @param toEmail 接收人邮件
     * @param customizedService 定制服务需求（即触发阶段流转的关键词）
     * @param nickname 读者昵称
     * @param wxUserId 读者ID
     * @param robotClassifyName 小号分类名称
     * @param robotNumber 小号编号
     * @param robotId 小号ID
     * @param afterStageName 流转后阶段名称
     */
    public void sendStageChangeEmail(String toEmail, String customizedService, String nickname, String wxUserId,
                                     String robotClassifyName, String robotNumber, String robotId, String afterStageName) {
        try {
            SendEmailDto sendEmailDto = new SendEmailDto();
            sendEmailDto.setToEmail(toEmail);
            sendEmailDto.setTypeCode("stage_change_notice");
            Map<String, Object> content = new HashMap<>();
            content.put("customizedService", customizedService);
            content.put("nickname", nickname);
            content.put("wxUserId", wxUserId);
            content.put("robotClassifyName", robotClassifyName);
            content.put("robotNumber", robotNumber);
            content.put("robotId", robotId);
            content.put("afterStageName", afterStageName);
            content.put("time", DateNewUtils.getLongDateStr());
            sendEmailDto.setContent(content);
            messageBiz.sendEmail(sendEmailDto);
        } catch (Exception e) {
            LOGGER.error("发送阶段流转邮件出错：" + e.getMessage(), e);
        }
    }

    @ParamLog("发送内容衔接语")
    private void sendJumpLinkups(String userWxId, String robotWxId, String ip, Long jumpId, Long personalStageUserId) {
        List<PersonalStageJumpLinkup> linkups = personalStageJumpLinkupDao.getByJumpId(jumpId);
        if(ListUtils.isEmpty(linkups)){
            return;
        }
        this.replacePaperUrl(linkups, robotWxId, userWxId, jumpId);
        for (PersonalStageJumpLinkup jumpLinkup : linkups){
            if (jumpLinkup.getToStageStartTime()!=null&&jumpLinkup.getToStageStartTime()>0){
                LinkupDelayDTO linkupDelayDTO=new LinkupDelayDTO();
                linkupDelayDTO.setPersonalStageJumpLinkupId(jumpLinkup.getId());
                linkupDelayDTO.setRobotId(robotWxId);
                linkupDelayDTO.setWxId(userWxId);
                linkupDelayDTO.setIp(ip);
                linkupDelayDTO.setPersonalStageUserId(personalStageUserId);
                DelayQueueDTO delayQueueDTO = DelayQueueDTO.builder().key(userWxId).type(PersonalStageConstant.PERSONALSTAGE_DELAY_LINKUP).msg(linkupDelayDTO).timeout(jumpLinkup.getToStageStartTime()*1000).build();
                delayMessageSender.send(delayQueueDTO);
                LOGGER.info("增加跳转衔接语延时"+delayQueueDTO.toString());
            }else {
                if(ReplyTypeEnum.TEXT.value.equals( jumpLinkup.getReplyType())){
                    SendTextMessageVO sendTextMessageVO = new SendTextMessageVO();
                    sendTextMessageVO.setContent(personalStageBiz.replaceProjectProgressUrl(jumpLinkup.getLinkupContent(),robotWxId,userWxId,personalStageUserId));
                    sendTextMessageVO.setAltId(robotWxId);
                    sendTextMessageVO.setWxGroupId(userWxId);
                    sendTextMessageVO.setIp(ip);
                    sendTextMessageVO.setCode(SendMessageTypeEnum.SELF.getCode());
                    WxGroupSDK.sendTextMessage(sendTextMessageVO);
                } else if(ReplyTypeEnum.IMAGE.value.equals( jumpLinkup.getReplyType())) {
                    SendPicMessageVO sendPicMessageVO = new SendPicMessageVO();
                    sendPicMessageVO.setWxGroupId(userWxId);
                    sendPicMessageVO.setAltId(robotWxId);
                    sendPicMessageVO.setPicUrl(jumpLinkup.getPicUrl());
                    sendPicMessageVO.setIp(ip);
                    sendPicMessageVO.setCode(SendMessageTypeEnum.SELF.getCode());
                    WxGroupSDK.sendPicMessage(sendPicMessageVO);
                }
            }
        }
    }

    @ParamLog("更新到完成状态")
    private void updateStageToComplete(PersonalStageUser currentStageUser) {
        PersonalStageUser updateStageUser = new PersonalStageUser();
        updateStageUser.setId(currentStageUser.getId());
        updateStageUser.setState(PersonalStageUserStateEnum.FINISH.value);
        updateStageUser.setUpdateTime(new Date());
        personalStageUserDao.update(updateStageUser);
    }

    @ParamLog("替换需求定制单链接")
    private void replacePaperUrl(List<PersonalStageJumpLinkup> linkups, String robotId, String userWxId, Long jumpId){
        PersonalStageJumpDto stageJump = personalStageJumpDao.getDtoById(jumpId);
        if (null == stageJump){
            return;
        }
        PersonalStage personalStage = personalStageDao.getById(stageJump.getPersonalStageId());
        if (null == personalStage){
            return;
        }
        for (PersonalStageJumpLinkup linkup : linkups){
            if(ReplyTypeEnum.TEXT.value.equals(linkup.getReplyType())){
                if (null == personalStage.getPaperId()){
                    linkup.setLinkupContent(linkup.getLinkupContent().replace(PERSONAL_STAGE_PAPER_TEMPLATE, ""));
                    linkup.setLinkupContent(linkup.getLinkupContent().replace(PERSONAL_STAGE_PAPER_TEMPLATE1, ""));
                    continue;
                }
                String longUrl = wechatLinkPrefix.concat(PERSONAL_STAGE_PAPER).concat(personalStage.getPaperId().toString()).concat("&wxId=" + userWxId + "&robotWxId=" + robotId);
                //由于前端给的替换符中的“&”可能会是“&amp;”,所以两种替换符都要换掉
                String shortLink = UrlUtils.getShortUrl4Own(longUrl);
                linkup.setLinkupContent(linkup.getLinkupContent().replace(PERSONAL_STAGE_PAPER_TEMPLATE, shortLink));
                linkup.setLinkupContent(linkup.getLinkupContent().replace(PERSONAL_STAGE_PAPER_TEMPLATE1, shortLink));
            }
        }
    }
}
