package com.pcloud.book.group.facade.impl;

import java.util.HashMap;
import java.util.Map;
import org.codehaus.jackson.JsonParseException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import com.pcloud.book.base.exception.BookBizException;
import com.pcloud.book.group.biz.BookGroupBiz;
import com.pcloud.book.group.entity.BookGroup;
import com.pcloud.book.group.facade.BookGroupFacade;
import com.pcloud.book.group.vo.UpdateRankTypeVO;
import com.pcloud.common.dto.ResponseDto;
import com.pcloud.common.exceptions.BizException;
import com.pcloud.common.page.PageBean;
import com.pcloud.common.page.PageParam;
import com.pcloud.common.permission.PermissionException;
import com.pcloud.common.utils.SessionUtil;
import com.pcloud.common.utils.string.StringUtil;

/**
 * Description 社群书群二维码接口实现类
 * Created by PENG on 2019/4/17.
 */
@RestController("bookGroupFacade")
@RequestMapping("bookGroup")
public class BookGroupFacadeImpl implements BookGroupFacade {

    @Autowired
    private BookGroupBiz bookGroupBiz;

    /**
     * 获取社群书群二维码信息
     */
    @Override
    @RequestMapping(value = "getBookGroupInfo", method = RequestMethod.GET)
    public ResponseDto<?> getBookGroupInfo(@RequestHeader("token") String token, @RequestParam(value = "bookGroupId", required = false) Long bookGroupId)
            throws BizException, PermissionException, JsonParseException {
        if (null == bookGroupId) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        SessionUtil.getToken4Redis(token);
        return new ResponseDto<>(bookGroupBiz.getBookGroupInfo(bookGroupId));
    }

    /**
     * 获取社群书群二维码信息
     */
    @Override
    @RequestMapping(value = "getBookGroupInfo4Wechat", method = RequestMethod.GET)
    public ResponseDto<?> getBookGroupInfo4Wechat(@RequestParam(value = "bookGroupId", required = false) Long bookGroupId)
            throws BizException, PermissionException, JsonParseException {
        if (null == bookGroupId) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        return new ResponseDto<>(bookGroupBiz.getBookGroupInfo4Wechat(bookGroupId));
    }

    /**
     * 获取社群书群二维码信息
     */
    @Override
    @RequestMapping(value = "getBookGroupInfoByBookId", method = RequestMethod.GET)
    public ResponseDto<?> getBookGroupInfoByBookId(@RequestHeader("token") String token, @RequestParam(value = "bookId", required = false) Long bookId,
            @RequestParam(value = "channelId", required = false) Long channelId) throws BizException, PermissionException, JsonParseException {
        if (null == bookId || null == channelId) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        return new ResponseDto<>(bookGroupBiz.getBookGroupInfoByBookId(bookId, channelId, adviserId));
    }

    /**
     * 更新群二维码信息
     */
    @Override
    @RequestMapping(value = "updateBookGroup", method = RequestMethod.POST)
    public ResponseDto<?> updateBookGroup(@RequestHeader("token") String token, @RequestBody BookGroup bookGroup)
            throws BizException, PermissionException, JsonParseException {
        if (null == bookGroup) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        bookGroupBiz.updateBookGroup(bookGroup);
        return new ResponseDto<>();
    }

    /**
     * 根据社群码ID获取书名
     */
    @Override
    @RequestMapping(value = "getBookNameByBookGroupId", method = RequestMethod.GET)
    public ResponseDto<?> getBookNameByBookGroupId(@RequestParam(value = "bookGroupId", required = false) Long bookGroupId)
            throws BizException, PermissionException, JsonParseException {
        if (null == bookGroupId) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        return new ResponseDto<>(bookGroupBiz.getBookNameByBookGroupId(bookGroupId));
    }

    /**
     * 获取社群书列表(编辑)
     */
    @Override
    @RequestMapping(value = "listBookGroup4Adviser", method = RequestMethod.GET)
    public ResponseDto<PageBean> listBookGroup4Adviser(@RequestHeader("token") String token,
            @RequestParam(value = "name", required = false) String name,
            @RequestParam(value = "bookName", required = false) String bookName,
            @RequestParam(value = "isbn", required = false) String isbn,
            @RequestParam(value = "channelId", required = false) Long channelId,
            @RequestParam(value = "typeCode", required = false) String typeCode,
            @RequestParam(value = "currentPage", required = false) Integer currentPage,
            @RequestParam(value = "numPerPage", required = false) Integer numPerPage,
            @RequestParam(value = "isMainEditor", required = false) Integer isMainEditor,
            @RequestParam(value = "isFundSupport", required = false) Integer isFundSupport,
            @RequestParam(value = "bookId", required = false) Integer bookId,
            @RequestParam(value = "startDate", required = false) String startDate,
            @RequestParam(value = "endDate", required = false) String endDate)
            throws BizException, PermissionException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (currentPage == null || numPerPage == null || currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("adviserId", adviserId);
        paramMap.put("isMainEditor", isMainEditor);
        paramMap.put("isFundSupport", isFundSupport);
        paramMap.put("channelId", channelId);
        paramMap.put("name", name != null && "".equals(name.trim()) ? null : name);
        paramMap.put("isbn", isbn != null && "".equals(isbn.trim()) ? null : isbn);
        paramMap.put("bookName", bookName != null && "".equals(bookName.trim()) ? null : bookName);
        paramMap.put("typeCode", typeCode != null && "".equals(typeCode.trim()) ? null : typeCode);
        paramMap.put("bookId", bookId);
        if (!StringUtil.isEmpty(startDate) && !StringUtil.isEmpty(endDate)) {
            paramMap.put("startDate", startDate + " 00:00:00");
            paramMap.put("endDate", endDate + " 23:59:59");
        }
        PageBean pageBean = bookGroupBiz.listBookGroup4Adviser(paramMap, new PageParam(currentPage, numPerPage), adviserId);
        return new ResponseDto<>(pageBean);
    }

    /**
     * 获取社群书列表(编辑)
     */
    @Override
    @RequestMapping(value = "listSimpleBookGroup4Adviser", method = RequestMethod.GET)
    public ResponseDto<?> listSimpleBookGroup4Adviser(@RequestHeader("token") String token,
            @RequestParam(value = "name", required = false) String name)
            throws BizException, PermissionException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("adviserId", adviserId);
        paramMap.put("name", name != null && "".equals(name.trim()) ? null : name);
        return new ResponseDto<>(bookGroupBiz.listSimpleBookGroup4Adviser(paramMap));
    }

    /**
     * 获取未创建社群码的书刊列表
     */
    @Override
    @RequestMapping(value = "listBook4CreateBookGroup", method = RequestMethod.GET)
    public ResponseDto<?> listBook4CreateBookGroup(@RequestHeader("token") String token,
            @RequestParam(value = "name", required = false) String name,
            @RequestParam(value = "currentPage", required = false) Integer currentPage,
            @RequestParam(value = "numPerPage", required = false) Integer numPerPage)
            throws BizException, PermissionException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (currentPage == null || numPerPage == null || currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("adviserId", adviserId);
        paramMap.put("name", name != null && "".equals(name.trim()) ? null : name);
        PageBean pageBean = bookGroupBiz.listBook4CreateBookGroup(paramMap, new PageParam(currentPage, numPerPage), adviserId);
        return new ResponseDto<>(pageBean);
    }

    @Override
    @PostMapping("updateRankType")
    public ResponseDto<?> updateRankType(@RequestHeader("token") String token, @RequestBody UpdateRankTypeVO updateRankTypeVO) throws BizException {
        bookGroupBiz.updateRankType(updateRankTypeVO.getBookGroupId(), updateRankTypeVO.getRankType());
        return new ResponseDto<>();
    }

    @Override
    @GetMapping("getRankType")
    public ResponseDto<Integer> getRankType(@RequestHeader("token") String token, @RequestParam("bookGroupId") Long bookGroupId) throws BizException {
        Integer rankType = bookGroupBiz.getRankType(bookGroupId);
        return new ResponseDto<>(rankType == null ? 0 : rankType);
    }

    /**
     * 编辑获取社群码总数量
     */
    @Override
    @RequestMapping(value = "getTotalBookGroupCount", method = RequestMethod.GET)
    public ResponseDto<?> getTotalBookGroupCount(@RequestHeader("token") String token) throws BizException, PermissionException, JsonParseException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        return new ResponseDto<>(bookGroupBiz.getTotalBookGroupCount(adviserId));
    }
}
