package com.pcloud.book.keywords.biz.impl;

import com.pcloud.book.base.exception.BookBizException;
import com.pcloud.book.consumer.channel.QrcodeSceneConsr;
import com.pcloud.book.consumer.resource.ProductConsr;
import com.pcloud.book.consumer.wechatgroup.WechatGroupConsr;
import com.pcloud.book.group.biz.BookGroupClassifyBiz;
import com.pcloud.book.group.dao.BookGroupDao;
import com.pcloud.book.group.dto.BookGroupDTO;
import com.pcloud.book.group.dto.GroupClassifyQrcodeDTO;
import com.pcloud.book.group.tools.SendWeixinRequestTools;
import com.pcloud.book.keywords.biz.BookGuideBiz;
import com.pcloud.book.keywords.biz.BookKeywordBiz;
import com.pcloud.book.keywords.dao.BookGuideDao;
import com.pcloud.book.keywords.dao.BookGuideReplyDao;
import com.pcloud.book.keywords.dto.GuideDTO;
import com.pcloud.book.keywords.dto.KeywordDTO;
import com.pcloud.book.keywords.entity.BookGuide;
import com.pcloud.book.keywords.entity.BookGuideReply;
import com.pcloud.book.keywords.enums.ReplyTypeEnum;
import com.pcloud.book.keywords.vo.GuideVO;
import com.pcloud.book.keywords.vo.ReplyMessageVO;
import com.pcloud.book.keywords.vo.SetGuideVO;
import com.pcloud.book.keywords.vo.UpdateGuideVO;
import com.pcloud.channelcenter.wechat.dto.AccountSettingDto;
import com.pcloud.common.core.aspect.ParamLog;
import com.pcloud.common.utils.ListUtils;
import com.pcloud.common.utils.cache.redis.JedisClusterUtils;
import com.pcloud.common.utils.string.StringUtil;

import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataIntegrityViolationException;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;

/**
 * @author lily
 * @date 2019/4/23 15:47
 */
@Component("bookGuideBiz")
public class BookGuideBizImpl implements BookGuideBiz {
    @Autowired
    private BookGuideDao bookGuideDao;
    @Autowired
    private BookGroupDao bookGroupDao;
    @Autowired
    private BookGuideReplyDao bookGuideReplyDao;
    @Autowired
    private BookGroupClassifyBiz bookGroupClassifyBiz;
    @Autowired
    private QrcodeSceneConsr qrcodeSceneConsr;
    @Autowired
    private BookKeywordBiz bookKeywordBiz;
    @Autowired
    private WechatGroupConsr wechatGroupConsr;
    @Autowired
    private ProductConsr productConsr;

    @Override
    @ParamLog("新增欢迎语")
    @Transactional(rollbackFor = Exception.class)
    public void insertGuide(SetGuideVO setGuideVO, Long partyId) {
        if (setGuideVO.getClassifyId() == null) {
            setGuideVO.setClassifyId(0L);
        }
        checkReplyParam(setGuideVO.getReplyMessages());
        //获取图书相关信息
        BookGroupDTO bookGroupDTO = bookGroupDao.getDTOById(setGuideVO.getBookGroupId());
        if (bookGroupDTO == null) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "该社群码不存在！");
        }
        BookGuide bookGuide = new BookGuide();
        BeanUtils.copyProperties(setGuideVO, bookGuide);
        bookGuide.setBookId(bookGroupDTO.getBookId());
        bookGuide.setChannelId(bookGroupDTO.getChannelId());
        bookGuide.setCreateUser(partyId);
        try {
            bookGuideDao.insert(bookGuide);
        } catch (DataIntegrityViolationException e) {
            throw new BookBizException(BookBizException.DB_DML_FAIL, "请勿重复设置！");
        }
        List<BookGuideReply> bookGuideReplies = setToBookGuideReply(setGuideVO.getReplyMessages(), bookGuide.getId(), partyId, bookGroupDTO.getChannelId());
        bookGuideReplyDao.insert(bookGuideReplies);
    }

    @ParamLog("设置回复参数")
    private List<BookGuideReply> setToBookGuideReply(List<ReplyMessageVO> replyMessages, Long bookGuideId, Long createUser, Long channelId) {
        List<BookGuideReply> bookGuideReplies = new ArrayList<>();
        List<Long> productIds = new ArrayList<>();
        for (ReplyMessageVO replyMessageVO : replyMessages) {
            BookGuideReply bookGuideReply = new BookGuideReply();
            BeanUtils.copyProperties(replyMessageVO, bookGuideReply);
            bookGuideReply.setBookGuideId(bookGuideId);
            bookGuideReply.setCreateUser(createUser);
            bookGuideReplies.add(bookGuideReply);
            if(ReplyTypeEnum.APP.value.equals(replyMessageVO.getReplyType()) && "PRODUCT".equalsIgnoreCase(replyMessageVO.getServeType())){
                productIds.add(replyMessageVO.getServeId());
            }
        }
        productConsr.productAutoOnShelves(channelId, productIds);
        return bookGuideReplies;
    }

    @ParamLog("校验参数")
    private void checkReplyParam(List<ReplyMessageVO> replyMessages) {
        if (ListUtils.isEmpty(replyMessages)) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "请设置入群欢迎语");
        }
        if (replyMessages.size() > 5) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "入群欢迎语超出5个限制");
        }
    }

    @Override
    @ParamLog("更新欢迎语")
    @Transactional(rollbackFor = Exception.class)
    public void updateGuide(UpdateGuideVO updateGuideVO, Long partyId) {
        checkReplyParam(updateGuideVO.getReplyMessages());
        //获取欢迎语基本信息
        BookGuide bookGuideOld = bookGuideDao.getById(updateGuideVO.getId());
        if(bookGuideOld == null){
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "欢迎语不存在");
        }
        BookGuide bookGuide = new BookGuide();
        BeanUtils.copyProperties(updateGuideVO, bookGuide);
        bookGuideDao.update(bookGuide);
        //更新回复消息
        bookGuideReplyDao.deleteByGuideId(updateGuideVO.getId());
        List<BookGuideReply> bookGuideReplies = setToBookGuideReply(updateGuideVO.getReplyMessages(), bookGuide.getId(), partyId, bookGuideOld.getChannelId());
        bookGuideReplyDao.insert(bookGuideReplies);
    }

    @Override
    @ParamLog("根据社群码获取欢迎语")
    public GuideVO getGuideByBookGroup(Long bookGroupId) {
        GuideVO guideVO = bookGuideDao.getGuideByBookGroup(bookGroupId);
        if (guideVO == null) {
            return new GuideVO();
        }
        //获取社群码回复内容
        List<ReplyMessageVO> messages = bookGuideReplyDao.getReplyMessage(guideVO.getId());
        guideVO.setReplyMessages(messages);
        return guideVO;
    }

    @Override
    @ParamLog("根据分类获取欢迎语")
    public GuideVO getGuideByClassify(Long classifyId) {
        GuideVO guideVO = bookGuideDao.getGuideByClassify(classifyId);
        if (guideVO == null) {
            return new GuideVO();
        }
        //获取社群码回复内容
        List<ReplyMessageVO> messages = bookGuideReplyDao.getReplyMessage(guideVO.getId());
        guideVO.setReplyMessages(messages);
        return guideVO;
    }


    @Override
    @ParamLog("推送欢迎语消息")
    public void sendGuideMessage(String weixinGroupId) {
        //通过群id获取对应基本信息
        GroupClassifyQrcodeDTO classifyQrcodeInfo = bookGroupClassifyBiz.getClassifyQrcodeInfo(weixinGroupId);
        if(classifyQrcodeInfo == null){
            return;
        }
        //获取该书是否设置推送公众号
        GuideDTO bookGuide = getBaseInfo(null, classifyQrcodeInfo.getBookGroupId());
        //获取该分类是否设置过欢迎语
        GuideDTO classifyGuide = getBaseInfo(classifyQrcodeInfo.getClassifyId(), classifyQrcodeInfo.getBookGroupId());
        //获取推送消息机器人
        String robotId = wechatGroupConsr.getRobotIdByGroupId(weixinGroupId);
        //String robotId = "wxid_e6zsl8oohcnl22";
        List<ReplyMessageVO> replyMessages = null;
        if (classifyGuide != null) {
            replyMessages = bookGuideReplyDao.getReplyMessage(classifyGuide.getId());
        } else if (bookGuide != null) {
            replyMessages = bookGuideReplyDao.getReplyMessage(bookGuide.getId());
        }
        if (ListUtils.isEmpty(replyMessages)) {
            return;
        }
        //获取公众号基本信息
        AccountSettingDto wechatInfo = qrcodeSceneConsr.getWechatInfo(classifyQrcodeInfo.getChannelId());
        //处理链接
        handleUrl(replyMessages, wechatInfo, classifyQrcodeInfo);
        //推送消息
        SendWeixinRequestTools.sendGuideMessage(replyMessages, robotId, weixinGroupId);
        //获取关键词信息
        List<KeywordDTO> keywords = bookKeywordBiz.listFiveKeyword(classifyQrcodeInfo.getClassifyId(), classifyQrcodeInfo.getBookGroupId());
        SendWeixinRequestTools.sendKeywordsInfo(keywords, robotId, weixinGroupId);
        if (bookGuide != null && bookGuide.getIsRecommend() != null && bookGuide.getIsRecommend() == 1) {
            //推送公众号消息
            SendWeixinRequestTools.sendAccountMessage(weixinGroupId, bookGuide.getRecommendLanguage(), robotId, wechatInfo);
        }
    }

    @ParamLog("处理地址")
    private void handleUrl(List<ReplyMessageVO> replyMessages, AccountSettingDto accountSettingDto, GroupClassifyQrcodeDTO groupClassifyQrcodeDTO) {
        if (ListUtils.isEmpty(replyMessages)) {
            return;
        }
        for (ReplyMessageVO replyMessageVO : replyMessages) {
            if (ReplyTypeEnum.APP.value.equals(replyMessageVO.getReplyType()) || ReplyTypeEnum.LINK.value.equals(replyMessageVO.getReplyType())) {
                String linkUrl = SendWeixinRequestTools.splitUrlNew(accountSettingDto, replyMessageVO.getLinkUrl(), groupClassifyQrcodeDTO.getBookGroupId(), groupClassifyQrcodeDTO.getClassifyId(), groupClassifyQrcodeDTO.getGroupQrcodeId());
                replyMessageVO.setLinkUrl(linkUrl);
            }
        }
    }

    @ParamLog("获取欢迎设置基本信息")
    private GuideDTO getBaseInfo(Long classifyId, Long bookGroupId) {
        if (classifyId == null) {
            classifyId = 0L;
        }
        return bookGuideDao.getBaseInfo(classifyId, bookGroupId);
    }
}
