package com.pcloud.book.giftcoupon.biz.impl;

import com.pcloud.book.base.exception.BookBizException;
import com.pcloud.book.giftcoupon.biz.GiftCouponPackageBiz;
import com.pcloud.book.giftcoupon.check.GiftParamCheck;
import com.pcloud.book.giftcoupon.dao.GiftCouponPackageDao;
import com.pcloud.book.giftcoupon.dao.GiftReceiveDao;
import com.pcloud.book.giftcoupon.dto.GiftPackageDTO;
import com.pcloud.book.giftcoupon.dto.GiftReceiveDTO;
import com.pcloud.book.giftcoupon.dto.MyGiftPackageDTO;
import com.pcloud.book.giftcoupon.entity.GiftCouponPackage;
import com.pcloud.book.giftcoupon.entity.GiftReceive;
import com.pcloud.common.core.aspect.ParamLog;
import com.pcloud.common.page.PageBeanNew;
import com.pcloud.common.page.PageParam;
import com.pcloud.common.utils.ListUtils;

import org.apache.commons.collections.MapUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@Component("giftCouponPackageBiz")
public class GiftCouponPackageBizImpl implements GiftCouponPackageBiz {

    @Autowired
    private GiftCouponPackageDao giftCouponPackageDao;
    @Autowired
    private GiftReceiveDao giftReceiveDao;
    @Autowired
    private GiftParamCheck giftParamCheck;

    @Transactional(rollbackFor = Exception.class)
    @ParamLog(value = "新增专享礼包", isAfterReturn = false)
    @Override
    public void createGiftPackage(GiftCouponPackage giftCouponPackage) {
        giftParamCheck.checkGiftAddParam(giftCouponPackage);
        giftCouponPackageDao.insert(giftCouponPackage);
    }

    @Transactional(rollbackFor = Exception.class)
    @ParamLog(value = "修改专享礼包", isAfterReturn = false)
    @Override
    public void updateGiftPackage(GiftCouponPackage giftCouponPackage) {
        giftParamCheck.checkGiftAddParam(giftCouponPackage);
        giftCouponPackageDao.update(giftCouponPackage);
    }

    @Override
    public GiftCouponPackage getGiftPackage(Long id) {
        return giftCouponPackageDao.getById(id);
    }

    @Override
    public void deleteGiftPackage(Long id) {
        //todo 删除应该有限制
        giftCouponPackageDao.deleteById(id);
    }

    @Override
    public PageBeanNew<GiftPackageDTO> list4GiftPackage(String title, Integer state, Integer currentPage, Integer numPerPage) {
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("title", title);
        paramMap.put("state", state);
        PageBeanNew<GiftPackageDTO> pageBeanNew=giftCouponPackageDao.listPageNew(new PageParam(currentPage, numPerPage), paramMap, "list4GiftPackage");
        if (pageBeanNew == null || ListUtils.isEmpty(pageBeanNew.getRecordList())) {
            return new PageBeanNew<>(currentPage, numPerPage, new ArrayList<>());
        }
        buildReceiveNum(pageBeanNew.getRecordList());
        return pageBeanNew;
    }

    private void buildReceiveNum(List<GiftPackageDTO> recordList) {
        List<Long> giftIds = recordList.stream().filter(a -> null != a.getId()).map(a -> a.getId()).distinct().collect(Collectors.toList());
        List<GiftReceiveDTO> giftReceiveNumList = ListUtils.isEmpty(giftIds)?new ArrayList<>():giftReceiveDao.getGiftReceiveNumList(giftIds);
        Map<Long, Integer> receiveNumMap = new HashMap<>();
        if(!ListUtils.isEmpty(giftReceiveNumList)){
           receiveNumMap= giftReceiveNumList.stream().collect(Collectors.toMap(a -> a.getGiftPackageId(), a -> a.getReceiveNum(), (k1, k2) -> k2));
        }
        for (GiftPackageDTO giftPackageDTO : recordList) {
            giftPackageDTO.setReceiveNum(0);
            if(MapUtils.isNotEmpty(receiveNumMap) && receiveNumMap.containsKey(giftPackageDTO.getId())){
                giftPackageDTO.setReceiveNum(receiveNumMap.get(giftPackageDTO.getId()));
            }
        }
    }

    @Transactional(rollbackFor = Exception.class)
    @ParamLog("新增专享礼包")
    @Override
    public void createGiftReceive(GiftReceive giftReceive) {
        giftParamCheck.checkGiftReceiveAddParam(giftReceive);
        GiftCouponPackage giftCouponPackage = giftCouponPackageDao.getById(giftReceive.getGiftPackageId());
        if(null==giftCouponPackage){
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "当前奖券包不存在");
        }
        GiftReceive receive = giftReceiveDao.getGiftReceive(giftReceive.getWechatUserId(), giftReceive.getGiftPackageId());
        if(null==receive){
            giftReceive.setReceiveNum(1);
            giftReceive.setUsedNum(0);
            giftReceiveDao.insert(giftReceive);
        }else{
           if(receive.getReceiveNum()>=giftCouponPackage.getReceiveLimit()){
               throw new BookBizException(BookBizException.PARAM_IS_NULL, "领取数量已超过奖券包最大领取数量");
            }
            receive.setReceiveNum(receive.getReceiveNum()+1);
            giftReceiveDao.update(receive);
        }
    }

    @Override
    public PageBeanNew<MyGiftPackageDTO> list4MyGiftPackage(Integer state,Long wechatUserId,Integer currentPage, Integer numPerPage) {
        //state=1 已领取  state=2 已过期
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("wechatUserId", wechatUserId);
        paramMap.put("state", state);
        PageBeanNew<MyGiftPackageDTO> pageBeanNew=giftReceiveDao.listPageNew(new PageParam(currentPage, numPerPage), paramMap, "list4MyGiftPackage");
        if (pageBeanNew == null || ListUtils.isEmpty(pageBeanNew.getRecordList())) {
            return new PageBeanNew<>(currentPage, numPerPage, new ArrayList<>());
        }
        return pageBeanNew;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void useGiftCoupon(Long wechatUserId, Long giftPackageId) {
        GiftReceive giftReceive = giftReceiveDao.getGiftReceive(wechatUserId, giftPackageId);
        if(null==giftReceive){
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "当前奖券包不存在");
        }
        if(giftReceive.getReceiveNum()-giftReceive.getUsedNum()<=0){
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "当前奖券包已经使用完了");
        }
        //todo 可能有其他操作,如果点击不跳转页面，需要控制重复点击
        giftReceiveDao.useGiftCoupon(wechatUserId,giftPackageId);
    }
}
