package com.pcloud.book.rightsSetting.facade;

import com.pcloud.book.base.exception.BookBizException;
import com.pcloud.book.rightsSetting.biz.RightsSettingBiz;
import com.pcloud.book.rightsSetting.dto.RightsSettingDto;
import com.pcloud.book.rightsSetting.entity.RightsSetting;
import com.pcloud.book.rightsSetting.entity.RightsSettingClassify;
import com.pcloud.book.rightsSetting.entity.RightsSettingItem;
import com.pcloud.common.dto.ResponseDto;
import com.pcloud.common.page.PageBeanNew;
import com.pcloud.common.permission.PermissionException;
import com.pcloud.common.utils.SessionUtil;
import com.pcloud.common.utils.cookie.Cookie;
import com.pcloud.common.utils.string.StringUtil;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.CookieValue;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;

import javax.validation.constraints.Min;

import io.swagger.annotations.ApiOperation;
import lombok.SneakyThrows;

@RequestMapping("rightsSetting")
@RestController("rightsSettingFacede")
public class RightsSettingFacede {

    @Autowired
    private RightsSettingBiz rightsSettingBiz;

    @PostMapping("addRightsSetting")
    public ResponseDto<Long> addRightsSetting( @RequestHeader("token") String token, @RequestBody RightsSetting rightsSetting ) throws PermissionException {
        SessionUtil.getToken4Redis(token);
        return new ResponseDto<Long>(rightsSettingBiz.addRightsSetting(rightsSetting));
    }

    @GetMapping("deleteRightsSetting")
    public ResponseDto<?> deleteRightsSetting( @RequestHeader("token") String token, @RequestParam("id") Long id ) throws PermissionException {
        SessionUtil.getToken4Redis(token);
        rightsSettingBiz.deleteRightsSetting(id);
        return new ResponseDto<>();
    }

    @PostMapping("updateRightsSetting")
    public ResponseDto<?> updateRightsSetting( @RequestHeader("token") String token, @RequestBody RightsSetting rightsSetting ) throws PermissionException {
        SessionUtil.getToken4Redis(token);
        rightsSettingBiz.updateRightsSetting(rightsSetting);
        return new ResponseDto<>();
    }

    @GetMapping("getRightsSetting")
    public ResponseDto<RightsSetting> getRightsSetting( @RequestHeader("token") String token, @RequestParam("id") Long id ) throws PermissionException {
        SessionUtil.getToken4Redis(token);
        return new ResponseDto<RightsSetting>(rightsSettingBiz.getRightsSetting(id));
    }

    @GetMapping("listRightsSettingPage")
    public ResponseDto<PageBeanNew<RightsSettingDto>> listRightsSettingPage( @RequestHeader("token") String token,
                                                                             @RequestParam("currentPage") Integer currentPage,
                                                                             @RequestParam("numPerPage") Integer numPerPage,
                                                                             @RequestParam(value = "firstClassify", required = false) Long firstClassify,
                                                                             @RequestParam(value = "secondClassify", required = false) Long secondClassify,
                                                                             @RequestParam(value = "gradeLabelId", required = false) Long gradeLabelId,
                                                                             @RequestParam(value = "subjectLabelId", required = false) Long subjectLabelId, @RequestParam(value = "rightsSettingType", required = false, defaultValue = "1") Integer rightsSettingType,
                                                                             @RequestParam(value = "bookQuery", required = false) String bookQuery

    ) throws PermissionException {
        SessionUtil.getToken4Redis(token);
        RightsSetting rightsSetting = new RightsSetting();
        rightsSetting.setFirstClassify(firstClassify);
        rightsSetting.setSecondClassify(secondClassify);
        rightsSetting.setGradeLabelId(gradeLabelId);
        rightsSetting.setSubjectLabelId(subjectLabelId);
        rightsSetting.setRightsSettingType(rightsSettingType);
        rightsSetting.setBookQuery(bookQuery);
        return new ResponseDto<PageBeanNew<RightsSettingDto>>(rightsSettingBiz.listRightsSettingPage(currentPage, numPerPage, rightsSetting));
    }

    @ApiOperation("获取读者数量-每天自增")
    @GetMapping("getUserCount")
    public ResponseDto<Integer> getUserCount() {
        return new ResponseDto<>(rightsSettingBiz.getUserCount());
    }

    @ApiOperation("新增每周/长期权益")
    @PostMapping("addRightsSettingItem")
    public ResponseDto<?> addRightsSettingItem( @RequestBody List<RightsSettingItem> rightsSettingItems,
                                                @RequestParam("rightsSettingId") Long rightsSettingId, @RequestParam("rightsType") String rightsType ) {
        rightsSettingBiz.batchInsertRightsSettingItem(rightsSettingItems, rightsSettingId, rightsType);
        return new ResponseDto<>();
    }

    @ApiOperation("小程序首页权益基本信息查询")
    @GetMapping("getRightsSettingByBookId4AppletHome")
    public ResponseDto<?> getRightsSettingByBookId4AppletHome(
            @CookieValue("userInfo") String userInfo, @RequestParam("adviserId") Long adviserId,
            @RequestParam("bookId") Long bookId, @RequestParam("channelId") Long channelId ) {
        Cookie.getId(userInfo, Cookie._WECHAT_USER_ID);
        if (null == adviserId || null == bookId || null == channelId) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数缺失！");
        }
        return new ResponseDto<>(rightsSettingBiz.getRightsSettingByBookId4AppletHome(bookId, adviserId, channelId));
    }

    @ApiOperation("小程序首页本周/长期权益查询")
    @GetMapping("getItemByRightsSettingId4AppletHome")
    public ResponseDto<?> getItemByRightsSettingId4AppletHome(
            @CookieValue("userInfo") String userInfo,
            @RequestParam("rightsSettingId") Long rightsSettingId,
            @RequestParam(value = "rightsType", required = false) String rightsType ) {
        Long wechatUserId = Cookie.getId(userInfo, Cookie._WECHAT_USER_ID);
        if (null == rightsSettingId || StringUtil.isEmpty(rightsType)) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数缺失！");
        }
        return new ResponseDto<>(rightsSettingBiz.getItemByRightsSettingId4AppletHome(rightsSettingId, wechatUserId, rightsType));
    }

    @ApiOperation("所有权益分类")
    @GetMapping("getAllRightsClassify")
    public ResponseDto<List<RightsSettingClassify>> getAllRightsClassify(
            @RequestParam(value = "rightsType", required = false) String rightsType ) {
        return new ResponseDto<>(rightsSettingBiz.getAllRightsClassify(rightsType));
    }

    @ApiOperation("根据权益id获取即享权益项")
    @GetMapping("getRightsSettingRightsNowItems")
    public ResponseDto<RightsSetting> getRightsSettingRightsNowItems(
            @CookieValue("userInfo") String userInfo,
            @RequestParam("rightsSettingId") Long rightsSettingId
    ) throws PermissionException {
        Long wechatUserId = Cookie.getId(userInfo, Cookie._WECHAT_USER_ID);
        if (null == rightsSettingId) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数缺失！");
        }
        return new ResponseDto<>(rightsSettingBiz.getRightsSettingRightsNowItems(rightsSettingId, wechatUserId));
    }

    @SneakyThrows
    @ApiOperation("根据权益id和类型获取即享权益项")
    @GetMapping("getRightsSettingRightsNowItemsByType")
    public ResponseDto<RightsSetting> getRightsSettingRightsNowItemsByType( @CookieValue("userInfo") String userInfo,
                                                                      @RequestParam("rightsSettingId") @Min(value = 1,
                                                                              message = "权益id不能小于1") Long rightsSettingId,
                                                                      @RequestParam("rightsSettingType") @Min(value = 1,
                                                                              message = "权益类型不能小于1") Integer rightsSettingType ) {
        Long wechatUserId = Cookie.getId(userInfo, Cookie._WECHAT_USER_ID);
        return new ResponseDto<>(rightsSettingBiz.getRightsSettingRightsNowItemsByType(rightsSettingId, wechatUserId, rightsSettingType));
    }

    @ApiOperation("根据权益id获取即享权益项专享礼包")
    @GetMapping("getRightsSettingGiftCoupons")
    public ResponseDto<RightsSetting> getRightsSettingGiftCoupons(
            @CookieValue("userInfo") String userInfo,
            @RequestParam("rightsSettingId") Long rightsSettingId
    ) throws PermissionException {
        Long wechatUserId = Cookie.getId(userInfo, Cookie._WECHAT_USER_ID);
        if (null == rightsSettingId) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数缺失！");
        }
        return new ResponseDto<>(rightsSettingBiz.getRightsSettingGiftCoupons(rightsSettingId, wechatUserId));
    }

    @ApiOperation("根据权益id获取即享权益项补充咨询")
    @GetMapping("getFillRightsSettingApplets")
    public ResponseDto<?> getFillRightsSettingApplets(
            @CookieValue("userInfo") String userInfo,
            @RequestParam("rightsSettingId") Long rightsSettingId
    ) throws PermissionException {
        Long wechatUserId = Cookie.getId(userInfo, Cookie._WECHAT_USER_ID);
        if (null == rightsSettingId) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数缺失！");
        }
        return new ResponseDto<>(rightsSettingBiz.getFillRightsSettingApplets(rightsSettingId, wechatUserId, null, null));
    }

    @ApiOperation("根据权益id获取即享权益配套资料分组集合")
    @GetMapping("getRightsItemGroups")
    public ResponseDto<?> getRightsItemGroups(
            @CookieValue("userInfo")String userInfo,
            @RequestParam("rightsSettingId") Long rightsSettingId,
            @RequestParam("adviserId") Long adviserId,
            @RequestParam("bookId")Long bookId,
            @RequestParam("channelId")Long channelId
    ) throws PermissionException {
        Cookie.getId(userInfo,Cookie._WECHAT_USER_ID);
        if (null==rightsSettingId){
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数缺失！");
        }
        return new ResponseDto<>(rightsSettingBiz.getRightsItemGroups(rightsSettingId, adviserId, bookId, channelId, true));
    }
}