package com.pcloud.book.mq.topic;

import com.pcloud.book.consumer.wechatgroup.WechatGroupConsr;
import com.pcloud.book.group.biz.WeixinQrcodeBiz;
import com.pcloud.book.group.dao.GroupQrcodeDao;
import com.pcloud.book.group.dto.BookWxQrcodeDTO;
import com.pcloud.book.group.entity.GroupQrcode;
import com.pcloud.book.mq.config.MQTopicConumer;
import com.pcloud.common.core.aspect.ParamLog;
import com.pcloud.common.exceptions.BizException;
import com.pcloud.wechatgroup.message.dto.GroupNameDTO;
import com.sdk.wxgroup.ChangeNameVO;
import com.sdk.wxgroup.SendTextMessageVO;
import com.sdk.wxgroup.WxGroupSDK;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.amqp.rabbit.annotation.RabbitHandler;
import org.springframework.amqp.rabbit.annotation.RabbitListener;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Collections;
import java.util.Map;
import java.util.Optional;

@Component
@RabbitListener(queues = MQTopicConumer.UPDATE_WXGROUP_NAME)
public class UpdateWXGroupNameListener {

    private static final Logger LOGGER = LoggerFactory.getLogger(UpdateWXGroupNameListener.class);

    @Autowired
    private GroupQrcodeDao groupQrcodeDao;
    @Autowired
    private WechatGroupConsr wechatGroupConsr;
    @Autowired
    private WeixinQrcodeBiz weixinQrcodeBiz;

    @ParamLog("接收修改群名称topic")
    @RabbitHandler
    public void onMessage(GroupNameDTO groupNameDTO) throws BizException {
        try {
            String wechatGroupId = groupNameDTO.getWechatGroupId();
            if (groupNameDTO == null || wechatGroupId == null
                    || groupNameDTO.getNewGroupName() == null || groupNameDTO.getNickName() == null)
                return;
            GroupQrcode groupQrcode = groupQrcodeDao.getGroupQrcodeByGroupId(wechatGroupId);
            if (groupQrcode == null) {
                LOGGER.info("没有找到该群" + wechatGroupId);
                return;
            }
            String groupName = groupQrcode.getGroupName();
            // 如果数据库微信群名称超过16，就截断，然后重新设置数据库的名称，因为wxgroup那边设置的时候截断了
            if (groupName.length() > 15) {
                groupName = groupName.substring(0, 15);
                groupQrcodeDao.updateGroupName(groupQrcode.getId(), groupName);
            }
            if (!groupNameDTO.getNewGroupName().equals(groupName)) {
                ChangeNameVO changeNameVO = new ChangeNameVO();
                changeNameVO.setName(groupName);
                changeNameVO.setWxGroupId(wechatGroupId);
                String altId = groupNameDTO.getWxId();
                changeNameVO.setAltId(altId);
                changeNameVO.setIp(groupNameDTO.getIp());
                LOGGER.info("修改群名称" + changeNameVO.toString());
                WxGroupSDK.changeGroupName(changeNameVO);
                SendTextMessageVO sendTextMessageVO = new SendTextMessageVO();
                sendTextMessageVO.setContent("@"+groupNameDTO.getNickName()+"，为方便群友入群分享好的助学方法，请不要随意修改群名。如再修改，会将您移出群聊。谢谢配合。");
                sendTextMessageVO.setAltId(altId);
                sendTextMessageVO.setWxGroupId(wechatGroupId);
                LOGGER.info("发送警告文本消息"+sendTextMessageVO.toString());
                sendTextMessageVO.setIp(findIp(wechatGroupId));
                WxGroupSDK.sendTextMessage(sendTextMessageVO);
            }
        } catch (Exception e) {
            LOGGER.error("接收修改群名称失败" + e.getMessage(), e);
        }
    }

    @ParamLog("获取ip")
    private String findIp(String wechatGroupId) {
        Map<String, BookWxQrcodeDTO> groupVersion = weixinQrcodeBiz.getGroupVersion(Collections.singletonList(wechatGroupId));
        String ip = Optional.ofNullable(groupVersion.get(wechatGroupId)).orElse(new BookWxQrcodeDTO()).getWechatGroupIp();
        return ip;
    }
}
