package com.pcloud.common.utils.word;

import java.io.ByteArrayInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.math.BigInteger;
import java.util.List;

import org.apache.poi.POIXMLDocument;
import org.apache.poi.openxml4j.exceptions.InvalidFormatException;
import org.apache.poi.openxml4j.opc.OPCPackage;
import org.apache.poi.util.Units;
import org.apache.poi.wp.usermodel.HeaderFooterType;
import org.apache.poi.xwpf.usermodel.XWPFDocument;
import org.apache.poi.xwpf.usermodel.XWPFFooter;
import org.apache.poi.xwpf.usermodel.XWPFHeader;
import org.apache.poi.xwpf.usermodel.XWPFParagraph;
import org.apache.poi.xwpf.usermodel.XWPFPicture;
import org.apache.poi.xwpf.usermodel.XWPFPictureData;
import org.apache.poi.xwpf.usermodel.XWPFRun;
import org.apache.poi.xwpf.usermodel.XWPFTable;
import org.apache.poi.xwpf.usermodel.XWPFTableCell;
import org.apache.poi.xwpf.usermodel.XWPFTableRow;
import org.openxmlformats.schemas.wordprocessingml.x2006.main.CTTblWidth;
import org.openxmlformats.schemas.wordprocessingml.x2006.main.STTblWidth;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.pcloud.common.constant.FilePathConstant;
import com.pcloud.common.constant.WordConstant;
import com.pcloud.common.exceptions.BizException;
import com.pcloud.common.exceptions.FileException;
import com.pcloud.common.utils.FileUtils;
import com.pcloud.common.utils.ListUtils;
import com.pcloud.common.utils.UUIDUitl;
import com.pcloud.common.utils.string.StringUtil;
import com.pcloud.common.utils.word.dto.ParagraphDTO;
import com.pcloud.common.utils.word.dto.RunDTO;
import com.pcloud.common.utils.word.dto.RunImageDTO;
import com.pcloud.common.utils.word.dto.RunTableDTO;
import com.pcloud.common.utils.word.dto.RunTextDTO;
import com.pcloud.common.utils.word.dto.WordDTO;
import com.pcloud.common.utils.word.enums.ParagraphType;

/**
 * 重新定义word工具类
 *
 * @author：songx
 * @date：2019/5/27,10:06
 */
public class WordNewUtils {

    private static final Logger LOGGER = LoggerFactory.getLogger(WordNewUtils.class);

//    public static void main(String[] args) {
//        List<ParagraphDTO> paragraphDTOS = new ArrayList<>();
//        // 页眉
//        paragraphDTOS.add(getHeader1());
//        paragraphDTOS.add(getHeader2());
//        // 标题
//        paragraphDTOS.add(getParagraph1());
//        // 内容
//        paragraphDTOS.add(getParagraph2());
//        paragraphDTOS.add(getParagraph3());
//        paragraphDTOS.add(getParagraph4());
//        paragraphDTOS.add(getParagraph4());
//        // 页脚
//        paragraphDTOS.add(getFooter1());
//        try {
//            System.out.println(create(paragraphDTOS));
//        } catch (Exception e) {
//            LOGGER.error(e.getMessage(), e);
//        }
//        System.out.println("Game over!");
//    }
//
//    private static ParagraphDTO getHeader1() {
//        RunTextDTO runTextDTO = RunTextDTO
//                .newText("?????的个人专属错题本                                                共40道错题", 9, "微软雅黑");
//        return ParagraphDTO.newHeader(ParagraphType.text, runTextDTO);
//    }
//
//    private static ParagraphDTO getHeader2() {
//        RunTextDTO runTextDTO = RunTextDTO
//                .newText("来源《书名书名书名书名书名书名书名书书名书名书名书书名书名书名书》", 9, "微软雅黑");
//        return ParagraphDTO.newHeader(ParagraphType.text, runTextDTO);
//    }
//
//    private static ParagraphDTO getParagraph1() {
//        RunTextDTO runTextDTO = RunTextDTO
//                .newText("个人专属错题本", 18, "微软雅黑", true);
//        return ParagraphDTO.newParagraph(ParagraphType.text, ParagraphAlignment.CENTER, runTextDTO, 1);
//    }
//
//    private static ParagraphDTO getParagraph2() {
//        RunTextDTO runTextDTO = RunTextDTO
//                .newText("完成下面错题练习，扫描二维码核对答案，并记录练习结果：", 11, "微软雅黑", true);
//        return ParagraphDTO.newParagraph(ParagraphType.text, runTextDTO, 1);
//    }
//
//    private static ParagraphDTO getParagraph3() {
//        RunImageDTO runImageDTO = RunImageDTO
//                .newImage("I:\\TEST\\image\\02.jpg", 80, 80, null, listAfterTexts3(11, "微软雅黑"));
//        return ParagraphDTO.newParagraph(ParagraphType.image, TextAlignment.TOP, runImageDTO, 1);
//    }
//
//    private static ParagraphDTO getParagraph4() {
//        RunImageDTO runImageDTO = RunImageDTO
//                .newImage("I:\\TEST\\image\\2.jpg", 200, 500);
//        return ParagraphDTO.newParagraph(ParagraphType.image, runImageDTO, 3);
//    }
//
//    private static ParagraphDTO getFooter1() {
//        RunTextDTO runTextDTO = RunTextDTO.newText("反复练习，攻克错题", 11, "微软雅黑");
//        return ParagraphDTO.newFooter(ParagraphType.text, ParagraphAlignment.CENTER, runTextDTO);
//    }
//
//    /**
//     * 组装图片后面的文字
//     *
//     * @param fontSize
//     * @param fontFamily
//     * @return
//     */
//    private static List<RunTextDTO> listAfterTexts3(int fontSize, String fontFamily) {
//        List<RunTextDTO> afterRuns = Lists.newArrayList();
//        afterRuns.add(RunTextDTO.newText("  答对", fontSize, fontFamily));
//        afterRuns.add(RunTextDTO.newText("  10  ", fontSize, fontFamily, UnderlinePatterns.SINGLE));
//        afterRuns.add(RunTextDTO.newText("道题，答错", fontSize, fontFamily));
//        afterRuns.add(RunTextDTO.newText("  11  ", fontSize, fontFamily, UnderlinePatterns.SINGLE));
//        afterRuns.add(RunTextDTO.newText("道题，日期", fontSize, fontFamily));
//        afterRuns.add(RunTextDTO.newText("  " + LocalDateUtils.getDateNow() + "  ", fontSize, fontFamily,
//                UnderlinePatterns.SINGLE));
//        return afterRuns;
//    }

    /**
     * 创建WORD
     *
     * @param wordDTO
     * @return
     * @throws Exception
     */
    public static String create(WordDTO wordDTO) throws FileException {
        LOGGER.info("【WORD】创建WORD文档,<START>");
        if (wordDTO == null || ListUtils.isEmpty(wordDTO.getParagraphs())) {
            return null;
        }
        String wordTemplate = null;
        String outPath = null;
        CustomXWPFDocument document = null;
        FileOutputStream fos = null;
        try {
            wordTemplate = FilePathConstant.WORD + UUIDUitl.taskName() + ".docx";
            FileUtils.downloadFileFromUrl(WordConstant.TEMPLATE_URL, wordTemplate);
            OPCPackage pack = POIXMLDocument.openPackage(wordTemplate);
            document = new CustomXWPFDocument(pack);
            // 添加内容到文档中
            addParagraphs(wordDTO, document);
            // 生成目标word
            outPath = FilePathConstant.WORD + UUIDUitl.taskName() + ".docx";
            fos = new FileOutputStream(outPath);
            document.write(fos);
            fos.flush();
        } catch (Exception e) {
            LOGGER.error("【WORD】创建WORD文档失败:" + e.getMessage(), e);
            throw new FileException(FileException.FILE_CONTENT_ERROR, e.getMessage());
        } finally {
            close(document, fos);
            FileUtils.deleteFile(wordTemplate);
        }
        LOGGER.info("【WORD】创建WORD文档,<END>.[outPath]=" + outPath);
        return outPath;
    }

    /**
     * 添加内容到WORD中
     *
     * @param wordDTO
     * @param document
     * @throws BizException
     */
    private static void addParagraphs(WordDTO wordDTO, CustomXWPFDocument document) {
        List<ParagraphDTO> paragraphDTOS = wordDTO.getParagraphs();
        for (ParagraphDTO paragraphDTO : paragraphDTOS) {
            ParagraphType paragraphType = paragraphDTO.getParagraphType();
            if (ParagraphType.table.equals(paragraphType)) {
                addTable(paragraphDTO, document);
            } else if (ParagraphType.image.equals(paragraphType)) {
                addImage(wordDTO, paragraphDTO, document);
            } else if (ParagraphType.text.equals(paragraphType)) {
                addText(wordDTO, paragraphDTO, document);
            }

        }
    }

    /**
     * 获取段落
     *
     * @param paragraphDTO
     * @param document
     * @return
     */
    private static XWPFParagraph getParagraph(ParagraphDTO paragraphDTO, CustomXWPFDocument document) {
        if (paragraphDTO.isHeader()) {
            XWPFHeader xwpfHeader = document.createHeader(HeaderFooterType.DEFAULT);
            return xwpfHeader.createParagraph();
        } else if (paragraphDTO.isFooter()) {
            XWPFFooter xwpfFooter = document.createFooter(HeaderFooterType.DEFAULT);
            return xwpfFooter.createParagraph();
        } else {
            return document.createParagraph();
        }
    }

    /**
     * 添加文本块
     *
     * @param wordDTO
     * @param paragraphDTO
     * @throws BizException
     */
    private static void addText(WordDTO wordDTO, ParagraphDTO paragraphDTO, CustomXWPFDocument document)
        throws BizException {
        XWPFParagraph paragraph = getParagraph(paragraphDTO, document);
        // 设置段落样式
        setParagraphStyle(paragraphDTO, paragraph);
        // 设置段落块样式
        if (paragraphDTO.getRun() != null) {
            setTextStyle(wordDTO, (RunTextDTO) paragraphDTO.getRun(), paragraph);
        } else if (!ListUtils.isEmpty(paragraphDTO.getRuns())) {
            for (RunDTO runDTO : paragraphDTO.getRuns()) {
                setTextStyle(wordDTO, (RunTextDTO) runDTO, paragraph);
            }
        }
        // 添加空行
        addEmptyText(paragraphDTO.getAfterEmpty(), document);
    }

    /**
     * 添加空行
     *
     * @param afterEmpty
     * @param document
     * @throws BizException
     */
    private static void addEmptyText(int afterEmpty, CustomXWPFDocument document) throws BizException {
        for (int i = 0; i < afterEmpty; i++) {
            setTextStyle(null, RunTextDTO.newEmpty(), document.createParagraph());
        }
    }

    /**
     * 设置段落样式
     *
     * @param paragraphDTO
     * @param paragraph
     */
    private static void setParagraphStyle(ParagraphDTO paragraphDTO, XWPFParagraph paragraph) {
        // 与前一行的间距
        if (paragraphDTO.getSpacingBefore() > 0) {
            paragraph.setSpacingBeforeLines(paragraphDTO.getSpacingBefore());
        }
        // 与下一行的间距
        if (paragraphDTO.getSpacingAfter() > 0) {
            paragraph.setSpacingAfterLines(paragraphDTO.getSpacingAfter());
        }
        // 对齐
        if (paragraphDTO.getParagraphAlignment() != null) {
            paragraph.setFontAlignment(paragraphDTO.getParagraphAlignment().getValue());
        }
        // 水平居中
        if (paragraphDTO.getTextAlignment() != null) {
            paragraph.setVerticalAlignment(paragraphDTO.getTextAlignment());
        }
    }

    /**
     * 设置文本样式
     *
     * @param wordDTO
     * @param runTextDTO
     * @param paragraph
     */
    private static void setTextStyle(WordDTO wordDTO, RunTextDTO runTextDTO, XWPFParagraph paragraph) {
        XWPFRun xwpfRun = paragraph.createRun();
        xwpfRun.setText(runTextDTO.getText());
        // 是否加粗
        xwpfRun.setBold(runTextDTO.isBold());
        // 字体大小

        if (runTextDTO.getFontSize() > 0) {
            xwpfRun.setFontSize(runTextDTO.getFontSize());
        } else if (wordDTO != null && wordDTO.getFontSize() > 0) {
            xwpfRun.setFontSize(wordDTO.getFontSize());
        }
        // 字体
        if (runTextDTO.getFontFamily() != null) {
            xwpfRun.setFontFamily(runTextDTO.getFontFamily());
        } else if (wordDTO != null && wordDTO.getFontFamily() != null) {
            xwpfRun.setFontFamily(wordDTO.getFontFamily());
        }
        // 下划线
        if (runTextDTO.getUnderlinePatterns() != null) {
            xwpfRun.setUnderline(runTextDTO.getUnderlinePatterns());
        }
    }

    /**
     * 添加表格
     *
     * @param paragraphDTO
     * @throws BizException
     */
    private static void addTable(ParagraphDTO paragraphDTO, CustomXWPFDocument document) throws BizException {
        XWPFParagraph paragraph = document.createParagraph();
        // 设置段落样式
        setParagraphStyle(paragraphDTO, paragraph);
        // 创建表格
        if (paragraphDTO.getRun() != null) {
            createTable((RunTableDTO) paragraphDTO.getRun(), document);
        } else if (!ListUtils.isEmpty(paragraphDTO.getRuns())) {
            for (RunDTO runDTO : paragraphDTO.getRuns()) {
                createTable((RunTableDTO) runDTO, document);
            }
        }
        // 添加空行
        addEmptyText(paragraphDTO.getAfterEmpty(), document);
    }

    /**
     * 创建表格
     *
     * @param runTableDTO
     * @param document
     */
    private static void createTable(RunTableDTO runTableDTO, CustomXWPFDocument document) {
        String[][] contents = runTableDTO.getContents();
        if (contents == null || contents.length < 1) {
            return;
        }
        int rowNum = contents.length; // 总行数
        int colNum = runTableDTO.getColumnWidths().length; // 总列数
        if (colNum < 1) {
            throw new FileException(FileException.FILE_CONTENT_ERROR, "表格列数与内容列数不一致");
        }
        // 创建表格
        XWPFTable ComTable = document.createTable();
        // 列宽自动分割
        CTTblWidth comTableWidth = ComTable.getCTTbl().addNewTblPr().addNewTblW();
        comTableWidth.setType(STTblWidth.DXA);
        comTableWidth.setW(BigInteger.valueOf(9072));
        // 设置行和列
        for (int i = 0; i < rowNum; i++) {
            String[] cols = contents[i];
            // 创建行
            XWPFTableRow comTableRow = i == 0 ? ComTable.getRow(0) : ComTable.createRow();
            for (int j = 0; j < colNum; j++) {
                String text = (j >= cols.length || StringUtil.isEmpty(cols[j])) ? "" : cols[j];
                // 填充单元格内容
                XWPFTableCell cell = null;
                if (i == 0) {
                    cell = j == 0 ? comTableRow.getCell(0) : comTableRow.addNewTableCell();
                } else {
                    cell = comTableRow.getCell(j);
                }
                cell.setText(text);
                cell.setVerticalAlignment(runTableDTO.getVertAlign());
            }
            // 行高
            if (runTableDTO.getHeight() > 0) {
                comTableRow.setHeight(runTableDTO.getHeight());
            }
        }
    }

    /**
     * 添加图片和文本
     *
     * @param paragraphDTO
     * @param document
     */
    private static void addImage(WordDTO wordDTO, ParagraphDTO paragraphDTO, CustomXWPFDocument document) {
        XWPFParagraph paragraph = null;
        if (paragraphDTO.isHeader()) {
            XWPFHeader header = document.createHeader(HeaderFooterType.DEFAULT);
            paragraph = header.createParagraph();
            addImageHeader(wordDTO, paragraphDTO, paragraph, header);
        } else {
            paragraph = document.createParagraph();
            addImageContent(wordDTO, paragraphDTO, paragraph, document);
        }
        // 设置段落样式
        setParagraphStyle(paragraphDTO, paragraph);
        // 添加空行
        addEmptyText(paragraphDTO.getAfterEmpty(), document);
    }

    /**
     * 页眉中插入图片
     *
     * @param wordDTO
     * @param paragraphDTO
     * @param paragraph
     * @param header
     */
    private static void addImageHeader(WordDTO wordDTO, ParagraphDTO paragraphDTO, XWPFParagraph paragraph,
        XWPFHeader header) {
        // 创建图片
        if (paragraphDTO.getRun() != null) {
            createImageHeader(wordDTO, (RunImageDTO) paragraphDTO.getRun(), paragraph, header);
        } else if (!ListUtils.isEmpty(paragraphDTO.getRuns())) {
            for (RunDTO runDTO : paragraphDTO.getRuns()) {
                createImageHeader(wordDTO, (RunImageDTO) runDTO, paragraph, header);
            }
        }
    }

    /**
     * 内容中插入图片
     *
     * @param wordDTO
     * @param paragraphDTO
     * @param paragraph
     * @param document
     */
    private static void addImageContent(WordDTO wordDTO, ParagraphDTO paragraphDTO, XWPFParagraph paragraph,
        CustomXWPFDocument document) {
        // 创建图片
        if (paragraphDTO.getRun() != null) {
            createImageContent(wordDTO, (RunImageDTO) paragraphDTO.getRun(), paragraph, document);
        } else if (!ListUtils.isEmpty(paragraphDTO.getRuns())) {
            for (RunDTO runDTO : paragraphDTO.getRuns()) {
                createImageContent(wordDTO, (RunImageDTO) runDTO, paragraph, document);
            }
        }
    }

    /**
     * 创建图片和文本
     *
     * @param wordDTO
     * @param runImageDTO
     * @param paragraph
     * @param document
     */
    private static void createImageContent(WordDTO wordDTO, RunImageDTO runImageDTO, XWPFParagraph paragraph,
        CustomXWPFDocument document) {
        // 前置文本
        addImageBefore(wordDTO, runImageDTO, paragraph);
        try {
            // 图片
            String relationId = addPictureContent(runImageDTO, document);
            if (!StringUtil.isEmpty(relationId)) {
                int id = document.getAllPictures().size() - 1;
                document.createPicture(paragraph, relationId, id, runImageDTO.getWidth(), runImageDTO.getHeight());
            }
        } catch (Exception e) {
            LOGGER.error("【WORD】创建WORD文档失败:" + e.getMessage(), e);
        }
        // 后置文本
        addImageAfter(wordDTO, runImageDTO, paragraph);
    }

    /**
     * 创建图片
     *
     * @param wordDTO
     * @param runImageDTO
     * @param paragraph
     * @param header
     */
    private static void createImageHeader(WordDTO wordDTO, RunImageDTO runImageDTO, XWPFParagraph paragraph,
        XWPFHeader header) {
        // 前置文本
        addImageBefore(wordDTO, runImageDTO, paragraph);
        // 图片
        addPictureHeader(runImageDTO, paragraph, header);
        // 后置文本
        addImageAfter(wordDTO, runImageDTO, paragraph);
    }

    /**
     * 图片前置文本
     *
     * @param wordDTO
     * @param runImageDTO
     * @param paragraph
     */
    private static void addImageBefore(WordDTO wordDTO, RunImageDTO runImageDTO, XWPFParagraph paragraph) {
        List<RunTextDTO> beforeTexts = runImageDTO.getBeforeTexts();
        if (!ListUtils.isEmpty(beforeTexts)) {
            for (RunTextDTO runTextDTO : beforeTexts) {
                setTextStyle(wordDTO, runTextDTO, paragraph);
            }
        }
    }

    /**
     * 图片后置文本
     *
     * @param wordDTO
     * @param runImageDTO
     * @param paragraph
     */
    private static void addImageAfter(WordDTO wordDTO, RunImageDTO runImageDTO, XWPFParagraph paragraph) {
        List<RunTextDTO> afterTexts = runImageDTO.getAfterTexts();
        if (!ListUtils.isEmpty(afterTexts)) {
            for (RunTextDTO runTextDTO : afterTexts) {
                setTextStyle(wordDTO, runTextDTO, paragraph);
            }
        }
    }

    /**
     * 添加页眉图片
     *
     * @param runImageDTO
     * @param paragraph
     * @param header
     */
    private static void addPictureHeader(RunImageDTO runImageDTO, XWPFParagraph paragraph, XWPFHeader header) {
        InputStream is = null;
        String imageUrl = runImageDTO.getImageUrl();
        if (!StringUtil.isEmpty(imageUrl)) {
            if (imageUrl.startsWith("http")) {
                is = new ByteArrayInputStream(FileUtils.downloadByteFromUrl(imageUrl));
            } else {
                is = new ByteArrayInputStream(FileUtils.file2byte(imageUrl));
            }
        } else if (runImageDTO.getImageByte() != null) {
            is = new ByteArrayInputStream(runImageDTO.getImageByte());
        }
        XWPFRun run = paragraph.createRun();
        try {
            XWPFPicture picture = run
                .addPicture(is, XWPFDocument.PICTURE_TYPE_JPEG, "123", Units.toEMU(runImageDTO.getWidth()),
                    Units.toEMU(runImageDTO.getHeight()));
            //这段必须有，不然打开的logo图片不显示
            String blipID = "";
            for (XWPFPictureData picturedata : header.getAllPackagePictures()) {
                blipID = header.getRelationId(picturedata);
            }
            picture.getCTPicture().getBlipFill().getBlip().setEmbed(blipID);
            run.addTab();
        } catch (Exception e) {
            LOGGER.error("【WORD】WORD页眉图片生成失败:" + e.getMessage(), e);
        } finally {
            try {
                if (is != null) {
                    is.close();
                    is = null;
                }
            } catch (Exception e) {
            }
        }
    }

    /**
     * 添加图片数据
     *
     * @param runImageDTO
     * @param document
     * @return
     * @throws InvalidFormatException
     */
    private static String addPictureContent(RunImageDTO runImageDTO, CustomXWPFDocument document)
        throws InvalidFormatException {
        String relationId = null;
        String imageUrl = runImageDTO.getImageUrl();
        if (!StringUtil.isEmpty(imageUrl)) {
            if (imageUrl.startsWith("http")) {
                relationId = document.addPictureData(FileUtils.downloadByteFromUrl(imageUrl),
                    XWPFDocument.PICTURE_TYPE_JPEG);
            } else {
                relationId = document.addPictureData(FileUtils.file2byte(imageUrl),
                    XWPFDocument.PICTURE_TYPE_JPEG);
            }
        } else if (runImageDTO.getImageByte() != null) {
            relationId = document.addPictureData(runImageDTO.getImageByte(), XWPFDocument.PICTURE_TYPE_JPEG);
        }
        return relationId;
    }

    /**
     * 根据图片类型获取对应的图片类型代码
     *
     * @param picType
     * @return
     */
    public static int getPictureType(String picType) {
        int res = CustomXWPFDocument.PICTURE_TYPE_PICT;
        if (picType != null) {
            if (picType.equalsIgnoreCase("png")) {
                res = CustomXWPFDocument.PICTURE_TYPE_PNG;
            } else if (picType.equalsIgnoreCase("dib")) {
                res = CustomXWPFDocument.PICTURE_TYPE_DIB;
            } else if (picType.equalsIgnoreCase("emf")) {
                res = CustomXWPFDocument.PICTURE_TYPE_EMF;
            } else if (picType.equalsIgnoreCase("jpg") || picType.equalsIgnoreCase("jpeg")) {
                res = CustomXWPFDocument.PICTURE_TYPE_JPEG;
            } else if (picType.equalsIgnoreCase("wmf")) {
                res = CustomXWPFDocument.PICTURE_TYPE_WMF;
            }
        }
        return res;
    }

    /**
     * 结束关闭文件流
     *
     * @param document
     * @param os
     */
    public static void close(CustomXWPFDocument document, OutputStream os) {
        try {
            if (document != null) {
                document.close();
                document = null;
            }
            if (os != null) {
                os.close();
                os = null;
            }
        } catch (Exception e) {
            LOGGER.error("【WORD】WORD文件流关闭失败:" + e.getMessage(), e);
        }
    }

}
