package com.pcloud.book.consumer.data;

import cn.hutool.core.collection.CollUtil;
import com.pcloud.common.dto.ResponseDto;
import com.pcloud.common.utils.ResponseHandleUtil;
import com.pcloud.data.domain.dto.BookStatisParamDTO;
import com.pcloud.data.domain.vo.AllAdviserBookStatisVO;
import com.pcloud.data.domain.vo.AllAdviserWorksStatisVO;
import com.pcloud.data.domain.vo.AllQrcodeWorksStatisVO;
import com.pcloud.data.domain.vo.CalDataOverviewVO;
import com.pcloud.data.domain.vo.CalQrcodeTargetVO;
import com.pcloud.data.domain.vo.CalScanHourTrendTargetVO;
import com.pcloud.data.domain.vo.CalUserInfoCityVO;
import com.pcloud.data.domain.vo.CalUserInfoProvinceVO;
import com.pcloud.data.domain.vo.CalUserInfoSexVO;
import com.pcloud.data.domain.vo.DailyQrcodeSummaryVO;
import com.pcloud.data.domain.vo.MidUserInfoMergeVO;
import com.pcloud.data.provider.BookStatisService;
import com.pcloud.data.provider.CalGroupTrackTargetService;
import lombok.extern.slf4j.Slf4j;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @ClassName com.pcloud.analysisengine.consumer.data.DataService
 * @Author zhuyajie
 * @Description 行为轨迹
 * @Date 2021/9/30 17:50
 * @Version 1.0
 **/
@Slf4j
@Component
public class DataConsr {

    private static final Logger LOGGER = LoggerFactory.getLogger(DataConsr.class);

    @Autowired
    private BookStatisService bookStatisService;
    @Autowired
    private CalGroupTrackTargetService calGroupTrackTargetService;

    /**
     * 获取书刊统计信息（书刊总收益、二维码数量、扫码量、扫码人、浏览量）
     */
    public AllAdviserBookStatisVO getBookStatisByBookId(Long bookId, Long adviserId, Long channelId){
        try {
            ResponseEntity<ResponseDto<AllAdviserBookStatisVO>> responseEntity = bookStatisService.getBookStatisByBookId(bookId, adviserId, channelId);
            return ResponseHandleUtil.parseResponse(responseEntity, AllAdviserBookStatisVO.class);
        } catch (Exception e){
            LOGGER.warn("调用数据统计接口 [DataConsr.getBookStatisByBookId] 失败", e);
        }
        return null;
    }

    /**
     * 获取书刊统计信息（书刊总收益、二维码数量、扫码量、扫码人、浏览量）
     */
    public Map<String, AllAdviserBookStatisVO> getBookStatisByBookIds(List<Long> bookIds, List<Long> adviserIds, List<Long> channelIds){
        try {
            BookStatisParamDTO paramDTO = new BookStatisParamDTO();
            paramDTO.setBookIds(bookIds);
            paramDTO.setAdviserIds(adviserIds);
            paramDTO.setChannelIds(channelIds);
            ResponseEntity<ResponseDto<Map<String, AllAdviserBookStatisVO>>> responseEntity = bookStatisService.getBookStatisByBookIds(paramDTO);
            return ResponseHandleUtil.parseMap(responseEntity, String.class, AllAdviserBookStatisVO.class);
        } catch (Exception e){
            LOGGER.warn("调用数据统计接口 [DataConsr.getBookStatisByBookId] 失败", e);
        }
        return new HashMap<>();
    }

    /**
     * 二维码统计（二维码扫码量、浏览量、读者量、平均浏览时长、销售额、购买次数）
     */
    public CalQrcodeTargetVO getQrcodeStatisByQrcodeId(Long bookId, Long qrcodeId, Long adviserId, Long channelId){
        try {
            ResponseEntity<ResponseDto<CalQrcodeTargetVO>> responseEntity = bookStatisService.getQrcodeStatisByQrcodeId(bookId, qrcodeId, adviserId, channelId);
            return ResponseHandleUtil.parseResponse(responseEntity, CalQrcodeTargetVO.class);
        } catch (Exception e){
            LOGGER.warn("调用数据统计接口 [DataConsr.getQrcodeStatisByQrcodeId] 失败", e);
        }
        return null;
    }

    /**
     * 获取二维码下用户扫码地区分布图
     */
    public List<MidUserInfoMergeVO> getQrcodeScanUserAreaList(Long bookId, Long adviserId, Long channelId, Long qrcodeId, Integer currentPage, Integer numPerPage){
        try {
            ResponseEntity<ResponseDto<List<MidUserInfoMergeVO>>> responseEntity = bookStatisService.getQrcodeScanUserAreaList(bookId, adviserId, channelId, qrcodeId, currentPage, numPerPage);
            List<MidUserInfoMergeVO> list = ResponseHandleUtil.parseList(responseEntity, MidUserInfoMergeVO.class);
            return list != null ? list : CollUtil.toList();
        } catch (Exception e){
            LOGGER.warn("调用数据统计接口 [DataConsr.getQrcodeScanUserAreaList] 失败", e);
        }
        return CollUtil.toList();
    }

    /**
     * 获取二维码下用户省份分布图
     */
    public List<CalUserInfoProvinceVO> getQrcodeUserProvinceTop5(Long bookId, Long adviserId, Long channelId, Long qrcodeId, Integer top, Integer loadCity){
        try {
            ResponseEntity<ResponseDto<List<CalUserInfoProvinceVO>>> responseEntity = bookStatisService.getQrcodeUserProvinceTop5(bookId, adviserId, channelId, qrcodeId, top, loadCity);
            List<CalUserInfoProvinceVO> list = ResponseHandleUtil.parseList(responseEntity, CalUserInfoProvinceVO.class);
            return list != null ? list : CollUtil.toList();
        } catch (Exception e){
            LOGGER.warn("调用数据统计接口 [DataConsr.getQrcodeUserProvinceTop5] 失败", e);
        }
        return CollUtil.toList();
    }

    /**
     * 获取二维码下用户城市分布图
     */
    public List<CalUserInfoCityVO> getQrcodeUserCityTop5(Long bookId, Long adviserId, Long channelId, Long qrcodeId, String province, Integer top){
        try {
            ResponseEntity<ResponseDto<List<CalUserInfoCityVO>>> responseEntity = bookStatisService.getQrcodeUserCityTop5(bookId, adviserId, channelId, qrcodeId, province, top);
            List<CalUserInfoCityVO> list = ResponseHandleUtil.parseList(responseEntity, CalUserInfoCityVO.class);
            return list != null ? list : CollUtil.toList();
        } catch (Exception e){
            LOGGER.warn("调用数据统计接口 [DataConsr.getQrcodeUserCityTop5] 失败", e);
        }
        return CollUtil.toList();
    }

    /**
     * 获取二维码下用户城市分布图
     */
    public CalUserInfoSexVO getQrcodeUserSex(Long bookId, Long adviserId, Long channelId, Long qrcodeId){
        try {
            ResponseEntity<ResponseDto<CalUserInfoSexVO>> responseEntity = bookStatisService.getQrcodeUserSex(bookId, adviserId, channelId, qrcodeId);
            return ResponseHandleUtil.parseResponse(responseEntity, CalUserInfoSexVO.class);
        } catch (Exception e){
            LOGGER.warn("调用数据统计接口 [DataConsr.getQrcodeUserSex] 失败", e);
        }
        return null;
    }

    /**
     * 获取二维码访问趋势
     */
    public List<DailyQrcodeSummaryVO> getQrcodeScanUserChart(Long bookId, Long adviserId, Long channelId, Long qrcodeId, String beginDate, String endDate){
        try {
            ResponseEntity<ResponseDto<List<DailyQrcodeSummaryVO>>> responseEntity = bookStatisService.getQrcodeScanUserChart(bookId, adviserId, channelId, qrcodeId, beginDate, endDate);
            List<DailyQrcodeSummaryVO> list = ResponseHandleUtil.parseList(responseEntity, DailyQrcodeSummaryVO.class);
            return list != null ? list : CollUtil.toList();
        } catch (Exception e){
            LOGGER.warn("调用数据统计接口 [DataConsr.getQrcodeScanUserChart] 失败", e);
        }
        return CollUtil.toList();
    }

    /**
     * 获取二维码24小时访问趋势
     */
    public List<CalScanHourTrendTargetVO> getQrcodeScanUser24HourChart(Long bookId, Long adviserId, Long channelId, Long qrcodeId){
        try {
            ResponseEntity<ResponseDto<List<CalScanHourTrendTargetVO>>> responseEntity = bookStatisService.getQrcodeScanUser24HourChart(bookId, adviserId, channelId, qrcodeId);
            List<CalScanHourTrendTargetVO> list = ResponseHandleUtil.parseList(responseEntity, CalScanHourTrendTargetVO.class);
            return list != null ? list : CollUtil.toList();
        } catch (Exception e){
            LOGGER.warn("调用数据统计接口 [DataConsr.getQrcodeScanUser24HourChart] 失败", e);
        }
        return CollUtil.toList();
    }

    /**
     * 获取二维码下资源访问趋势（浏览量、浏览人数、人均浏览时长）
     */
    public List<AllQrcodeWorksStatisVO> getQrcodeServeStatisChart(Long bookId, Long adviserId, Long channelId, Long qrcodeId, Long typeId, String typeCode, String beginDate, String endDate ){
        try {
            ResponseEntity<ResponseDto<List<AllQrcodeWorksStatisVO>>> responseEntity = bookStatisService.getQrcodeServeStatisChart(bookId, adviserId, channelId, qrcodeId, typeId, typeCode, beginDate, endDate);
            List<AllQrcodeWorksStatisVO> list = ResponseHandleUtil.parseList(responseEntity, AllQrcodeWorksStatisVO.class);
            return list != null ? list : CollUtil.toList();
        } catch (Exception e){
            LOGGER.warn("调用数据统计接口 [DataConsr.getQrcodeServeStatisChart] 失败", e);
        }
        return CollUtil.toList();
    }

    /**
     * 编辑应用排行统计（应用浏览量、读者量）
     */
    public List<AllAdviserWorksStatisVO> getAdviserAppStatis(Long adviserId){
        try {
            ResponseEntity<ResponseDto<List<AllAdviserWorksStatisVO>>> responseEntity = bookStatisService.getAdviserAppStatis(adviserId);
            List<AllAdviserWorksStatisVO> list = ResponseHandleUtil.parseList(responseEntity, AllAdviserWorksStatisVO.class);
            return list != null ? list : CollUtil.toList();
        } catch (Exception e){
            LOGGER.warn("调用数据统计接口 [DataConsr.getAdviserAppStatis] 失败", e);
        }
        return CollUtil.toList();
    }

    /**
     * 获取今日数据概览
     */
    public CalDataOverviewVO getTodayDataView(Long adviserId){
        try {
            ResponseEntity<ResponseDto<CalDataOverviewVO>> responseEntity = bookStatisService.getTodayDataView(adviserId);
            return ResponseHandleUtil.parseResponse(responseEntity, CalDataOverviewVO.class);
        } catch (Exception e){
            LOGGER.warn("调用数据统计接口 [DataConsr.getTodayDataView] 失败", e);
        }
        return null;
    }
}
