package com.pcloud.book.keywords.biz.impl;

import com.pcloud.book.base.exception.BookBizException;
import com.pcloud.book.consumer.content.ResourceConsr;
import com.pcloud.book.keywords.biz.NotKeywordBiz;
import com.pcloud.book.keywords.dao.NotKeywordItemDao;
import com.pcloud.book.keywords.dao.NotKeywordTemplateDao;
import com.pcloud.book.keywords.entity.NotKeywordItem;
import com.pcloud.book.keywords.entity.NotKeywordTemplate;
import com.pcloud.book.keywords.enums.ReplyTypeEnum;
import com.pcloud.common.core.aspect.ParamLog;
import com.pcloud.common.page.PageBeanNew;
import com.pcloud.common.page.PageParam;
import com.pcloud.common.utils.ListUtils;
import com.pcloud.common.utils.string.StringUtil;
import com.pcloud.contentcenter.resource.dto.ResourceDTO;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@Component("notKeywordBiz")
public class NotKeywordBizImpl implements NotKeywordBiz {

    @Autowired
    private NotKeywordTemplateDao notKeywordTemplateDao;
    @Autowired
    private NotKeywordItemDao notKeywordItemDao;
    @Autowired
    private ResourceConsr resourceConsr;


    @ParamLog("新增非关键词回复模板")
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void createNotKeywordTemplate(NotKeywordTemplate notKeywordTemplate) {
        if (notKeywordTemplate==null|| ListUtils.isEmpty(notKeywordTemplate.getNotKeywordItems())){
            throw new BookBizException(BookBizException.PARAM_IS_NULL,"参数为空！");
        }
        for (NotKeywordItem notKeywordItem:notKeywordTemplate.getNotKeywordItems()){
            if (notKeywordItem.getType()==null){
                throw new BookBizException(BookBizException.PARAM_IS_NULL,"类型不能为空！");
            }
            if (ReplyTypeEnum.TEXT.value.equals(notKeywordItem.getType())&& StringUtil.isEmpty(notKeywordItem.getContent())){
                throw new BookBizException(BookBizException.PARAM_IS_NULL,"内容不能为空！");
            }
            if (ReplyTypeEnum.RESOURCE.value.equals(notKeywordItem.getType())&& notKeywordItem.getResourceId()==null){
                throw new BookBizException(BookBizException.PARAM_IS_NULL,"内容不能为空！");
            }
        }
        notKeywordTemplateDao.insert(notKeywordTemplate);
        for (NotKeywordItem notKeywordItem:notKeywordTemplate.getNotKeywordItems()){
            notKeywordItem.setTemplateId(notKeywordTemplate.getId());
        }
        notKeywordItemDao.batchInsert(notKeywordTemplate.getNotKeywordItems());
    }

    @ParamLog("删除非关键词回复模板")
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void deleteNotKeywordTemplate(Long id) {
        notKeywordTemplateDao.deleteById(id);
        notKeywordItemDao.deleteByTemplateId(id);
    }

    @ParamLog("获取非关键词回复模板列表")
    @Override
    public PageBeanNew<NotKeywordTemplate> getNotKeywordTemplateList(Integer currentPage, Integer numPerPage) {
        if (currentPage==null||currentPage<0||numPerPage==null||numPerPage<=0){
            throw new BookBizException(BookBizException.PARAM_IS_NULL,"分页参数有误！");
        }
        PageBeanNew<NotKeywordTemplate> page = notKeywordTemplateDao.listPageNew(new PageParam(currentPage, numPerPage), new HashMap<>(), "getNotKeywordTemplateList");
        if (ListUtils.isEmpty(page.getRecordList())){
            return page;
        }
        List<NotKeywordTemplate> list=page.getRecordList();
        List<Long> templateIds=list.stream().filter(s->s.getId()!=null).map(NotKeywordTemplate::getId).collect(Collectors.toList());
        List<NotKeywordItem> notKeywordItems=notKeywordItemDao.getListByTemplateIds(templateIds);
        Map<Long,List<NotKeywordItem>> map=new HashMap<>();
        if (!ListUtils.isEmpty(notKeywordItems)){
            fillResource(notKeywordItems);
            map=notKeywordItems.stream().collect(Collectors.groupingBy(NotKeywordItem::getTemplateId));
            for (NotKeywordTemplate notKeywordTemplate:list){
                notKeywordTemplate.setNotKeywordItems(map.get(notKeywordTemplate.getId()));
            }
        }
        return page;
    }

    @ParamLog("获取非关键词回复模板")
    @Override
    public NotKeywordTemplate getNotKeywordTemplate(Long id) {
        NotKeywordTemplate notKeywordTemplate = notKeywordTemplateDao.getById(id);
        List<NotKeywordItem> list = notKeywordItemDao.getListByTemplateIds(Arrays.asList(notKeywordTemplate.getId()));
        if (!ListUtils.isEmpty(list)){
            fillResource(list);
        }
        notKeywordTemplate.setNotKeywordItems(list);
        return notKeywordTemplate;
    }

    @ParamLog("填充资源项")
    private void fillResource(List<NotKeywordItem> list) {
        if (ListUtils.isEmpty(list)){
            return;
        }
        List<Long> resourceIds=list.stream().filter(s->s.getResourceId()!=null).map(NotKeywordItem::getResourceId).collect(Collectors.toList());
        Map<Long, ResourceDTO> resourceDTOMap=new HashMap<>();
        if (!ListUtils.isEmpty(resourceIds)){
            resourceDTOMap = resourceConsr.mapByIds(resourceIds);
        }
        for (NotKeywordItem notKeywordItem:list){
            if (ReplyTypeEnum.RESOURCE.value.equals(notKeywordItem.getType())){
                ResourceDTO resourceDTO = resourceDTOMap.get(notKeywordItem.getResourceId());
                if (resourceDTO!=null){
                    notKeywordItem.setResourceName(resourceDTO.getResourceName());
                    notKeywordItem.setResourceUrl(resourceDTO.getFileUrl());
                    notKeywordItem.setResourceTypeCode(resourceDTO.getTypeCode());
                    notKeywordItem.setResourceTypeName(resourceDTO.getTypeName());
                    notKeywordItem.setFileType(resourceDTO.getFileType());
                    notKeywordItem.setResourcePdfItems(resourceDTO.getResourcePdfItems());
                    notKeywordItem.setResourceOfficeItemDTOs(resourceDTO.getResourceOfficeItemDTOs());
                }
            }
        }
    }
}
