package com.pcloud.book.skill.biz.impl;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.pcloud.book.base.exception.BookBizException;
import com.pcloud.book.consumer.label.LabelConsr;
import com.pcloud.book.skill.biz.PcloudResourceBiz;
import com.pcloud.book.skill.dao.PcloudResourceDao;
import com.pcloud.book.skill.entity.PcloudResource;
import com.pcloud.book.skill.facade.response.QueryPcloudResponseVO;
import com.pcloud.common.core.aspect.ParamLog;
import com.pcloud.common.page.PageBeanNew;
import com.pcloud.common.page.PageParam;
import com.pcloud.common.utils.string.StringUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Service
public class PcloudResourceBizImpl implements PcloudResourceBiz {

    @Autowired
    private PcloudResourceDao pcloudResourceDao;
    @Autowired
    private LabelConsr labelConsr;


    @ParamLog("创建资源")
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void savePcloudResource(PcloudResource pcloudResource) {
        checkPcloudResource(pcloudResource);
        // 处理资源盒子和列表关系
        // pcloudResource.getItemList()

        pcloudResourceDao.insert(pcloudResource);
    }

    @ParamLog("更新资源")
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void updatePcloudResource(PcloudResource pcloudResource) {
        if (pcloudResource.getId() == null){
            throw new BookBizException(BookBizException.PARAM_IS_NULL,"id不能为空！");
        }
        checkPcloudResource(pcloudResource);
        // 处理资源盒子和列表关系
        // pcloudResource.getItemList()
        pcloudResourceDao.update(pcloudResource);
    }

    @ParamLog("删除资源")
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void deletePcloudResource(Long id) {
        if (id==null){
            throw new BookBizException(BookBizException.PARAM_IS_NULL,"参数为空！");
        }
        pcloudResourceDao.deletePcloudResource(id);
        // 删除关系
        // pcloudResourceItemDao.deleteByPcloudResourceId(id);
    }

    @ParamLog("获取资源列表")
    @Override
    public PageBeanNew getPcloudResourceList(String name, Long proLabelId, Long depLabelId, Long purLabelId, Integer currentPage, Integer numPerPage) {
        HashMap<String, Object> map = Maps.newHashMap();
        map.put("name", name);
        map.put("proLabelId", proLabelId);
        map.put("depLabelId", depLabelId);
        map.put("purLabelId", purLabelId);
        PageBeanNew<QueryPcloudResponseVO> pageBeanNew = pcloudResourceDao.listPageNew(new PageParam(currentPage, numPerPage), map, "getPcloudResourceList");
        if (null == pageBeanNew || CollectionUtils.isEmpty(pageBeanNew.getRecordList())) {
            return pageBeanNew;
        }
        // 组装标签名称
        fillLabels(pageBeanNew.getRecordList());
        return pageBeanNew;
    }

    @ParamLog("获取资源")
    @Override
    public QueryPcloudResponseVO getPcloudResource(Long id) {
        if (id == null){
            throw new BookBizException(BookBizException.PARAM_IS_NULL,"参数为空！");
        }
        QueryPcloudResponseVO vo = pcloudResourceDao.getPcloudResource(id);
        if(vo == null){
            return new QueryPcloudResponseVO();
        }
        // 组装标签名称
        fillLabels(Lists.newArrayList(vo));
        return vo;
    }

    @ParamLog("组装标签名称")
    private void fillLabels(List<QueryPcloudResponseVO> recordList) {
        List<Long> labelId = Lists.newArrayList();
        for (QueryPcloudResponseVO vo : recordList) {
            labelId.add(vo.getProLabelId());
            labelId.add(vo.getPurLabelId());
            labelId.add(vo.getDepLabelId());
        }
        Map<Long, String> labelName = labelConsr.getLabelName(labelId);
        for (QueryPcloudResponseVO vo : recordList) {
            vo.setPurLabelName(labelName.get(vo.getPurLabelId()));
            vo.setProLabelName(labelName.get(vo.getProLabelId()));
            vo.setDepLabelName(labelName.get(vo.getDepLabelId()));
        }
    }

    @ParamLog("检测参数")
    private void checkPcloudResource(PcloudResource pcloudResource) {
        if (StringUtil.isBlank(pcloudResource.getName()) || null == pcloudResource.getProLabelId() || null == pcloudResource.getDepLabelId() || null == pcloudResource.getPurLabelId()) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "请求参数缺失");
        }
    }
}
