package com.pcloud.book.advertising.facade.impl;

import com.pcloud.book.advertising.biz.AdvertisingSpaceBiz;
import com.pcloud.book.advertising.dto.AdvertisingSpaceDTO;
import com.pcloud.book.advertising.dto.TestParamDTO;
import com.pcloud.book.advertising.entity.*;
import com.pcloud.book.advertising.facade.AdvertisingSpaceFacade;
import com.pcloud.book.base.exception.BookBizException;
import com.pcloud.common.dto.ResponseDto;
import com.pcloud.common.exceptions.BizException;
import com.pcloud.common.page.PageBean;
import com.pcloud.common.page.PageParam;
import com.pcloud.common.permission.PermissionException;
import com.pcloud.common.utils.SessionUtil;
import com.pcloud.common.utils.cookie.Cookie;
import com.pcloud.common.utils.string.StringUtil;

import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.codehaus.jackson.JsonParseException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

/**
 * Description 广告位接口层接口实现类
 * @author PENG
 * @date 2019/2/28
 */
@RestController("advertisingSpaceFacade")
@RequestMapping("/advertising")
public class AdvertisingSpaceFacadeImpl implements AdvertisingSpaceFacade {

    @Autowired
    private AdvertisingSpaceBiz advertisingSpaceBiz;

    /**
     * 创建广告位
     */
    @Override
    @RequestMapping(value = "create", method = RequestMethod.POST)
    public ResponseDto<?> create(@RequestHeader("token") String token, @RequestBody AdvertisingSpace advertisingSpace)
            throws PermissionException, BizException, JsonParseException {
        if (null == advertisingSpace) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        SessionUtil.getToken4Redis(token);
        Map<String, Object> result = new HashMap<>();
        result.put("adId", advertisingSpaceBiz.create(advertisingSpace));
        return new ResponseDto<>(result);
    }

    /**
     * 修改广告位
     */
    @Override
    @RequestMapping(value = "update", method = RequestMethod.POST)
    public ResponseDto<?> update(@RequestHeader("token") String token, @RequestBody AdvertisingSpace advertisingSpace)
            throws PermissionException, BizException, JsonParseException {
        if (null == advertisingSpace) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        SessionUtil.getToken4Redis(token);
        advertisingSpaceBiz.update(advertisingSpace);
        return new ResponseDto<>();
    }

    /**
     * 获取广告位信息
     */
    @Override
    @RequestMapping(value = "get", method = RequestMethod.GET)
    public ResponseDto<?> get(@RequestHeader("token") String token, @RequestParam(value = "adId", required = false) Long adId)
            throws PermissionException, BizException, JsonParseException {
        if (null == adId) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        SessionUtil.getToken4Redis(token);
        return new ResponseDto<>(advertisingSpaceBiz.get(adId));
    }

    /**
     * 客户端获取广告位信息
     */
    @Override
    @RequestMapping(value = "getAdInfo4Wechat", method = RequestMethod.GET)
    public ResponseDto<AdvertisingSpaceDTO> getAdInfo4Wechat(@CookieValue("userInfo") String userInfo, @RequestParam(value = "adId", required = false) Long adId) {
        if (null == adId) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        Cookie.getUserInfo(userInfo);
        return new ResponseDto<>(advertisingSpaceBiz.get(adId));
    }

    /**
     * 停用广告位
     */
    @Override
    @RequestMapping(value = "stop", method = RequestMethod.POST)
    public ResponseDto<?> stop(@RequestHeader("token") String token, @RequestBody AdvertisingSpace advertisingSpace)
            throws PermissionException, BizException, JsonParseException {
        if (null == advertisingSpace) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        SessionUtil.getToken4Redis(token);
        advertisingSpaceBiz.stop(advertisingSpace);
        return new ResponseDto<>();
    }

    /**
     * 投放书刊
     */
    @Override
    @RequestMapping(value = "distributeBook", method = RequestMethod.POST)
    public ResponseDto<?> distributeBook(@RequestHeader("token") String token, @RequestBody AdvertisingDistributionBook advertisingDistributionBook)
            throws PermissionException, BizException, JsonParseException {
        if (null == advertisingDistributionBook) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        advertisingSpaceBiz.distributeBook(advertisingDistributionBook);
        return new ResponseDto<>();
    }

    /**
     * 投放微信群
     */
    @Override
    @RequestMapping(value = "distributeWechatGroup", method = RequestMethod.POST)
    public ResponseDto<?> distributeWechatGroup(@RequestHeader("token") String token, @RequestBody AdvertisingDistributionBook advertisingDistributionBook)
            throws PermissionException, BizException, JsonParseException {
        if (null == advertisingDistributionBook) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        advertisingSpaceBiz.distributeWechatGroup(advertisingDistributionBook, false);
        return new ResponseDto<>();
    }

    /**
     * 获取出版社书刊权限
     */
    @Override
    @RequestMapping(value = "getAgentPermission", method = RequestMethod.GET)
    public ResponseDto<?> getAgentPermission(@RequestHeader("token") String token)
            throws PermissionException, BizException, JsonParseException {
        Long agentId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        Map<String, Object> result = new HashMap<>();
        result.put("isOpen", advertisingSpaceBiz.getAgentPermission(agentId));
        return new ResponseDto<>(result);
    }

    /**
     * 设置出版社书刊权限
     */
    @Override
    @RequestMapping(value = "setAgentPermission", method = RequestMethod.POST)
    public ResponseDto<?> setAgentPermission(@RequestHeader("token") String token, @RequestBody AdvertisingAgentPermission advertisingAgentPermission)
            throws PermissionException, BizException, JsonParseException {
        if (null == advertisingAgentPermission) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        Long agentId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        advertisingAgentPermission.setAgentId(agentId);
        advertisingSpaceBiz.setAgentPermission(advertisingAgentPermission);
        return new ResponseDto<>();
    }

    /**
     * 设置编辑书刊权限
     */
    @Override
    @RequestMapping(value = "setAdviserPermission", method = RequestMethod.POST)
    public ResponseDto<?> setAdviserPermission(@RequestHeader("token") String token, @RequestBody AdvertisingAdviserPermission advertisingAdviserPermission)
            throws PermissionException, BizException, JsonParseException {
        if (null == advertisingAdviserPermission) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        advertisingAdviserPermission.setAdviserId(adviserId);
        if (null == advertisingAdviserPermission.getIsBatchOperation()) {
            advertisingAdviserPermission.setIsBatchOperation(false);
        }
        advertisingSpaceBiz.setAdviserPermission(advertisingAdviserPermission);
        return new ResponseDto<>();
    }

    /**
     * 编辑端获取书刊列表
     */
    @Override
    @RequestMapping(value = "listBook4Adviser", method = RequestMethod.GET)
    public ResponseDto<?> listBook4Adviser(@RequestHeader("token") String token, @RequestParam(value = "name", required = false) String name,
            @RequestParam(value = "currentPage", required = false) Integer currentPage, @RequestParam(value = "numPerPage", required = false) Integer numPerPage)
            throws PermissionException, BizException, JsonParseException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (currentPage == null || numPerPage == null || currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("adviserId", adviserId);
        paramMap.put("name", StringUtil.isEmpty(name) ? null : name);
        return new ResponseDto<>(advertisingSpaceBiz.listBook4Adviser(paramMap, new PageParam(currentPage, numPerPage), adviserId));
    }

    /**
     * 平台端获取某编辑书刊列表
     */
    @Override
    @RequestMapping(value = "listBook4Platform", method = RequestMethod.GET)
    public ResponseDto<?> listBook4Platform(@RequestHeader("token") String token, @RequestParam(value = "adviserId", required = false) Long adviserId,
            @RequestParam(value = "name", required = false) String name, @RequestParam(value = "currentPage", required = false) Integer currentPage,
            @RequestParam(value = "numPerPage", required = false) Integer numPerPage, @RequestParam(value = "bookGroupOnly", required = false) Boolean bookGroupOnly)
            throws PermissionException, BizException, JsonParseException {
        if (currentPage == null || numPerPage == null || currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        /*if (null == adviserId) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }*/
        if (null == bookGroupOnly) {
            bookGroupOnly = false;
        }
        SessionUtil.getToken4Redis(token);
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("adviserId", adviserId);
        paramMap.put("name", StringUtil.isEmpty(name) ? null : name);
        paramMap.put("bookGroupOnly", bookGroupOnly);
        return new ResponseDto<>(advertisingSpaceBiz.listBook4Platform(paramMap, new PageParam(currentPage, numPerPage), adviserId));
    }

    /**
     * 平台端获取社群书微信群列表
     */
    @Override
    @RequestMapping(value = "listGroupQrcode4Platform", method = RequestMethod.GET)
    public ResponseDto<?> listGroupQrcode4Platform(
            @RequestHeader("token") String token, @RequestParam(value = "name", required = false) String name,
            @RequestParam(value = "currentPage", required = false) Integer currentPage, @RequestParam(value = "numPerPage", required = false) Integer numPerPage,
            @RequestParam(value = "proLabelId", required = false) Long proLabelId, @RequestParam(value = "depLabelId", required = false) Long depLabelId,
            @RequestParam(value = "purLabelId", required = false) Long purLabelId)
            throws PermissionException, BizException, JsonParseException {
        if (currentPage == null || numPerPage == null || currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        SessionUtil.getToken4Redis(token);
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("name", StringUtil.isEmpty(name) ? null : name);
        paramMap.put("proLabelId",proLabelId);
        paramMap.put("depLabelId",depLabelId);
        paramMap.put("purLabelId",purLabelId);
        return new ResponseDto<>(advertisingSpaceBiz.listGroupQrcode4Platform(paramMap, new PageParam(currentPage, numPerPage)));
    }

    /**
     * 获取广告位列表
     */
    @Override
    @RequestMapping(value = "list", method = RequestMethod.GET)
    public ResponseDto<?> list(@RequestHeader("token") String token, @RequestParam(value = "name", required = false) String name,
                               @RequestParam(value = "currentPage", required = false) Integer currentPage, @RequestParam(value = "numPerPage", required = false) Integer numPerPage,
                               @RequestParam(value = "filterClose", required = false) Boolean filterClose,
                               @RequestParam(value = "filterTest", required = false) Boolean filterTest,
                               @RequestParam(value = "adPosition", required = false) String adPosition)
            throws PermissionException, BizException, JsonParseException {
        if (currentPage == null || numPerPage == null || currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        if (null == filterClose) {
            filterClose = true;
        }
        if (null == filterTest) {
            filterTest = true;
        }
        return new ResponseDto<>(advertisingSpaceBiz.list(name, new PageParam(currentPage, numPerPage), filterClose, filterTest, adPosition));
    }

    /**
     * 获取广告位投放书刊列表
     */
    @Override
    @RequestMapping(value = "distributionBookList", method = RequestMethod.GET)
    public ResponseDto<?> distributionBookList(@RequestHeader("token") String token, @RequestParam(value = "adId", required = false) Long adId,
            @RequestParam(value = "name", required = false) String name, @RequestParam(value = "currentPage", required = false) Integer currentPage,
            @RequestParam(value = "numPerPage", required = false) Integer numPerPage) throws PermissionException, BizException, JsonParseException {
        if (currentPage == null || numPerPage == null || currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        if (null == adId) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        SessionUtil.getToken4Redis(token);
        return new ResponseDto<>(advertisingSpaceBiz.distributionBookList(adId, name, new PageParam(currentPage, numPerPage)));
    }

    /**
     * 获取广告位投放微信群列表
     */
    @Override
    @RequestMapping(value = "distributionWechatGroupList", method = RequestMethod.GET)
    public ResponseDto<?> distributionWechatGroupList(@RequestHeader("token") String token, @RequestParam(value = "adId", required = false) Long adId,
            @RequestParam(value = "name", required = false) String name, @RequestParam(value = "currentPage", required = false) Integer currentPage,
            @RequestParam(value = "numPerPage", required = false) Integer numPerPage) throws PermissionException, BizException, JsonParseException {
        if (currentPage == null || numPerPage == null || currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        if (null == adId) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        SessionUtil.getToken4Redis(token);
        return new ResponseDto<>(advertisingSpaceBiz.distributionWechatGroupList(adId, name, new PageParam(currentPage, numPerPage)));
    }

    /**
     * 获取书刊配置的广告位信息
     */
    @Override
    @RequestMapping(value = "get4Wechat", method = RequestMethod.GET)
    public ResponseDto<?> get4Wechat(@CookieValue("userInfo") String userInfo, @RequestParam(value = "adPositionDetail", required = false) String adPositionDetail)
            throws PermissionException, BizException, JsonParseException {
        if (StringUtil.isEmpty(adPositionDetail)) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        Long sceneId = Cookie.getId(userInfo, Cookie._SCENE_ID);
        Long adviserId = Cookie.getId(userInfo, Cookie._ADVISER_ID);
        Long channelId = Cookie.getId(userInfo, Cookie._CHANNEL_ID);
        Long bookGroupId = Cookie.getId(userInfo, Cookie.BOOK_GROUP_ID);
        if (null == bookGroupId && (null == sceneId || null == adviserId || null == channelId)) {
            return new ResponseDto<>(new ArrayList<>());
        }
        return new ResponseDto<>(advertisingSpaceBiz.get4Wechat(sceneId, adviserId, channelId, adPositionDetail, bookGroupId));
    }

    /**
     * 获取社群书配置的广告位信息
     */
    @Override
    @RequestMapping(value = "getBookGroupAd4Wechat", method = RequestMethod.GET)
    public ResponseDto<?> getBookGroupAd4Wechat(@CookieValue("userInfo") String userInfo, @RequestParam(value = "adPositionDetail", required = false) String adPositionDetail)
            throws PermissionException, BizException, JsonParseException {
        if (StringUtil.isEmpty(adPositionDetail)) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        Long bookGroupId = Cookie.getId(userInfo, Cookie.BOOK_GROUP_ID);
        if (null == bookGroupId) {
            return new ResponseDto<>(new AdvertisingSpaceDTO());
        }
        return new ResponseDto<>(advertisingSpaceBiz.getBookGroupAd4Wechat(bookGroupId, adPositionDetail));
    }

    /**
     * 曝光量埋点
     */
    @Override
    @RequestMapping(value = "addExposureTrack", method = RequestMethod.POST)
    public ResponseDto<?> addExposureTrack(@CookieValue("userInfo") String userInfo, @RequestBody AdvertisingExposureRecord advertisingExposureRecord)
            throws PermissionException, BizException, JsonParseException {
        if (null == advertisingExposureRecord || null == advertisingExposureRecord.getAdId() || null == advertisingExposureRecord.getBookId()) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        Long sceneId = Cookie.getId(userInfo, Cookie._SCENE_ID);
        Long adviserId = Cookie.getId(userInfo, Cookie._ADVISER_ID);
        Long channelId = Cookie.getId(userInfo, Cookie._CHANNEL_ID);
        Long officialAccountsId = Cookie.getId(userInfo, Cookie._OFFICIAL_ACCOUNTS_ID);
        Long wechatUserId = Cookie.getId(userInfo, Cookie._WECHAT_USER_ID);
        advertisingExposureRecord.setSceneId(sceneId);
        advertisingExposureRecord.setAdviserId(adviserId);
        advertisingExposureRecord.setChannelId(channelId);
        advertisingExposureRecord.setOfficialAccountId(officialAccountsId);
        advertisingExposureRecord.setWechatUserId(wechatUserId);
        advertisingSpaceBiz.addExposureTrack(advertisingExposureRecord);
        return new ResponseDto<>();
    }

    /**
     * 点击量埋点
     */
    @Override
    @RequestMapping(value = "addClickTrack", method = RequestMethod.POST)
    public ResponseDto<?> addClickTrack(@CookieValue("userInfo") String userInfo, @RequestBody AdvertisingClickRecord advertisingClickRecord)
            throws PermissionException, BizException, JsonParseException {
        Long sceneId = Cookie.getId(userInfo, Cookie._SCENE_ID);
        Long adviserId = Cookie.getId(userInfo, Cookie._ADVISER_ID);
        Long channelId = Cookie.getId(userInfo, Cookie._CHANNEL_ID);
        Long officialAccountsId = Cookie.getId(userInfo, Cookie._OFFICIAL_ACCOUNTS_ID);
        Long wechatUserId = Cookie.getId(userInfo, Cookie._WECHAT_USER_ID);
        Long bookGroupId = Cookie.getId(userInfo, Cookie.BOOK_GROUP_ID);
        if (null == advertisingClickRecord || null == advertisingClickRecord.getAdId()) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        if ((null == advertisingClickRecord.getBookId() && !"WECHAT_GROUP".equals(advertisingClickRecord.getFromType()))
                || (null == bookGroupId && "WECHAT_GROUP".equals(advertisingClickRecord.getFromType()))) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        advertisingClickRecord.setBookGroupId(bookGroupId);
        advertisingClickRecord.setSceneId(sceneId);
        advertisingClickRecord.setAdviserId(adviserId);
        advertisingClickRecord.setChannelId(channelId);
        advertisingClickRecord.setOfficialAccountId(officialAccountsId);
        advertisingClickRecord.setWechatUserId(wechatUserId);
        advertisingSpaceBiz.addClickTrack(advertisingClickRecord);
        return new ResponseDto<>();
    }

    /**
     * 添加广告主
     */
    @Override
    @RequestMapping(value = "createMaster", method = RequestMethod.POST)
    public ResponseDto<?> createMaster(@RequestHeader("token") String token, @RequestBody AdvertisingMaster advertisingMaster)
            throws PermissionException, BizException, JsonParseException {
        if (null == advertisingMaster) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        SessionUtil.getToken4Redis(token);
        advertisingSpaceBiz.createMaster(advertisingMaster);
        return new ResponseDto<>();
    }

    /**
     * 修改广告主
     */
    @Override
    @RequestMapping(value = "updateMaster", method = RequestMethod.POST)
    public ResponseDto<?> updateMaster(@RequestHeader("token") String token, @RequestBody AdvertisingMaster advertisingMaster)
            throws PermissionException, BizException, JsonParseException {
        if (null == advertisingMaster) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        SessionUtil.getToken4Redis(token);
        advertisingSpaceBiz.updateMaster(advertisingMaster);
        return new ResponseDto<>();
    }

    /**
     * 获取广告主列表
     */
    @Override
    @RequestMapping(value = "getMasterList", method = RequestMethod.GET)
    public ResponseDto<?> getMasterList(@RequestHeader("token") String token, @RequestParam(value = "masterName", required = false) String masterName,
            @RequestParam(value = "currentPage", required = false) Integer currentPage, @RequestParam(value = "numPerPage", required = false) Integer numPerPage)
            throws PermissionException, BizException, JsonParseException {
        if (currentPage == null || numPerPage == null || currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        SessionUtil.getToken4Redis(token);
        return new ResponseDto<>(advertisingSpaceBiz.getMasterList(new PageParam(currentPage, numPerPage), masterName));
    }

    /**
     * 获取所有广告主
     */
    @Override
    @RequestMapping(value = "getAllMaster", method = RequestMethod.GET)
    public ResponseDto<?> getAllMaster(@RequestHeader("token") String token) throws PermissionException, BizException, JsonParseException {
        SessionUtil.getToken4Redis(token);
        return new ResponseDto<>(advertisingSpaceBiz.getAllMaster());
    }

    /**
     * 获取广告主信息
     */
    @Override
    @RequestMapping(value = "getMasterInfo", method = RequestMethod.GET)
    public ResponseDto<?> getMasterInfo(@RequestHeader("token") String token, @RequestParam(value = "masterId", required = false) Long masterId)
            throws PermissionException, BizException, JsonParseException {
        if (null == masterId) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        SessionUtil.getToken4Redis(token);
        return new ResponseDto<>(advertisingSpaceBiz.getMasterInfo(masterId));
    }

    /**
     * 获取概览数据
     */
    @Override
    @RequestMapping(value = "overviewStatis", method = RequestMethod.GET)
    public ResponseDto<?> overviewStatis(@RequestHeader("token") String token) throws PermissionException, BizException, JsonParseException {
        SessionUtil.getToken4Redis(token);
        return new ResponseDto<>(advertisingSpaceBiz.overviewStatis());
    }

    /**
     * 获取流量趋势
     */
    @Override
    @RequestMapping(value = "getFlowStatisTrend", method = RequestMethod.GET)
    public ResponseDto<?> getFlowStatisTrend(@RequestHeader("token") String token, @RequestParam(value = "day", required = false) Integer day,
            @RequestParam(value = "startDate", required = false) String startDate, @RequestParam(value = "endDate", required = false) String endDate)
            throws PermissionException, BizException, JsonParseException {
        SessionUtil.getToken4Redis(token);
        return new ResponseDto<>(advertisingSpaceBiz.getFlowStatisTrend(day, startDate, endDate));
    }

    /**
     * 获取收益趋势
     */
    @Override
    @RequestMapping(value = "getIncomeStatisTrend", method = RequestMethod.GET)
    public ResponseDto<?> getIncomeStatisTrend(@RequestHeader("token") String token, @RequestParam(value = "day", required = false) Integer day,
            @RequestParam(value = "startDate", required = false) String startDate, @RequestParam(value = "endDate", required = false) String endDate)
            throws PermissionException, BizException, JsonParseException {
        SessionUtil.getToken4Redis(token);
        return new ResponseDto<>(advertisingSpaceBiz.getIncomeStatisTrend(day, startDate, endDate));
    }

    /**
     * 获取广告位投放书刊列表
     */
    @Override
    @RequestMapping(value = "advertisingBookList", method = RequestMethod.GET)
    public ResponseDto<?> advertisingBookList(@RequestHeader("token") String token, @RequestParam(value = "name", required = false) String name,
            @RequestParam(value = "currentPage", required = false) Integer currentPage, @RequestParam(value = "numPerPage", required = false) Integer numPerPage)
            throws PermissionException, BizException, JsonParseException {
        if (currentPage == null || numPerPage == null || currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        SessionUtil.getToken4Redis(token);
        return new ResponseDto<>(advertisingSpaceBiz.advertisingBookList(name, new PageParam(currentPage, numPerPage)));
    }

    /**
     * 获取广告位投放微信群列表
     */
    @Override
    @RequestMapping(value = "advertisingWechatGroupList", method = RequestMethod.GET)
    public ResponseDto<?> advertisingWechatGroupList(@RequestHeader("token") String token, @RequestParam(value = "name", required = false) String name,
            @RequestParam(value = "currentPage", required = false) Integer currentPage, @RequestParam(value = "numPerPage", required = false) Integer numPerPage)
            throws PermissionException, BizException, JsonParseException {
        if (currentPage == null || numPerPage == null || currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        SessionUtil.getToken4Redis(token);
        return new ResponseDto<>(advertisingSpaceBiz.advertisingWechatGroupList(name, new PageParam(currentPage, numPerPage)));
    }

    /**
     * 获取广告位投放微信群曝光量前几位
     */
    @Override
    @RequestMapping(value = "advertisingWechatGroupRankTop", method = RequestMethod.GET)
    public ResponseDto<?> advertisingWechatGroupRankTop(@RequestHeader("token") String token) throws PermissionException, BizException, JsonParseException {
        SessionUtil.getToken4Redis(token);
        return new ResponseDto<>(advertisingSpaceBiz.advertisingWechatGroupRankTop());
    }

    /**
     * 书刊广告位明细
     */
    @Override
    @RequestMapping(value = "advertisingDetail4Book", method = RequestMethod.GET)
    public ResponseDto<?> advertisingDetail4Book(@RequestHeader("token") String token, @RequestParam(value = "bookId", required = false) Long bookId,
            @RequestParam(value = "channelId", required = false) Long channelId, @RequestParam(value = "adviserId", required = false) Long adviserId,
            @RequestParam(value = "statisMonth", required = false) String statisMonth, @RequestParam(value = "currentPage", required = false) Integer currentPage,
            @RequestParam(value = "numPerPage", required = false) Integer numPerPage) throws PermissionException, BizException, JsonParseException {
        if (null == bookId || null == adviserId || null == channelId || null == statisMonth) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数有误！");
        }
        if (currentPage == null || numPerPage == null || currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        SessionUtil.getToken4Redis(token);
        return new ResponseDto<>(advertisingSpaceBiz.advertisingDetail4Book(bookId, channelId, adviserId, statisMonth, new PageParam(currentPage, numPerPage)));
    }

    /**
     * 导出书刊广告位明细
     */
    @Override
    @RequestMapping(value = "exportAdvertisingDetail4Book", method = RequestMethod.GET)
    public ResponseDto<?> exportAdvertisingDetail4Book(@RequestHeader("token") String token, @RequestParam(value = "bookId", required = false) Long bookId,
            @RequestParam(value = "channelId", required = false) Long channelId, @RequestParam(value = "adviserId", required = false) Long adviserId,
            @RequestParam(value = "statisMonth", required = false) String statisMonth) throws PermissionException, BizException, JsonParseException {
        if (null == bookId || null == adviserId || null == channelId || null == statisMonth) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数有误！");
        }
        SessionUtil.getToken4Redis(token);
        return new ResponseDto<>(advertisingSpaceBiz.exportAdvertisingDetail4Book(bookId, channelId, adviserId, statisMonth));
    }

    /**
     * 微信群广告位明细
     */
    @Override
    @RequestMapping(value = "advertisingDetail4WechatGroup", method = RequestMethod.GET)
    public ResponseDto<?> advertisingDetail4WechatGroup(@RequestHeader("token") String token, @RequestParam(value = "qrcodeId", required = false) Long qrcodeId,
            @RequestParam(value = "statisMonth", required = false) String statisMonth, @RequestParam(value = "currentPage", required = false) Integer currentPage,
            @RequestParam(value = "numPerPage", required = false) Integer numPerPage) throws PermissionException, BizException, JsonParseException {
        if (null == qrcodeId || null == statisMonth) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数有误！");
        }
        if (currentPage == null || numPerPage == null || currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        SessionUtil.getToken4Redis(token);
        return new ResponseDto<>(advertisingSpaceBiz.advertisingDetail4WechatGroup(qrcodeId, statisMonth, new PageParam(currentPage, numPerPage)));
    }

    /**
     * 导出微信群广告位明细
     */
    @Override
    @RequestMapping(value = "exportAdvertisingDetail4WechatGroup", method = RequestMethod.GET)
    public ResponseDto<?> exportAdvertisingDetail4WechatGroup(@RequestHeader("token") String token, @RequestParam(value = "qrcodeId", required = false) Long qrcodeId,
            @RequestParam(value = "statisMonth", required = false) String statisMonth) throws PermissionException, BizException, JsonParseException {
        if (null == qrcodeId || null == statisMonth) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数有误！");
        }
        SessionUtil.getToken4Redis(token);
        return new ResponseDto<>(advertisingSpaceBiz.exportAdvertisingDetail4WechatGroup(qrcodeId, statisMonth));
    }

    /**
     * 微信群广告位点击读者列表
     */
    @Override
    @RequestMapping(value = "clickUserList4AdvertisingWechatGroup", method = RequestMethod.GET)
    public ResponseDto<?> clickUserList4AdvertisingWechatGroup(@RequestHeader("token") String token, @RequestParam(value = "qrcodeId", required = false) Long qrcodeId,
            @RequestParam(value = "adId", required = false) Long adId, @RequestParam(value = "statisMonth", required = false) String statisMonth,
            @RequestParam(value = "currentPage", required = false) Integer currentPage, @RequestParam(value = "numPerPage", required = false) Integer numPerPage)
            throws PermissionException, BizException, JsonParseException {
        if (null == qrcodeId || null == adId) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数有误！");
        }
        if (currentPage == null || numPerPage == null || currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        SessionUtil.getToken4Redis(token);
        return new ResponseDto<>(advertisingSpaceBiz.clickUserList4AdvertisingWechatGroup(qrcodeId, adId, statisMonth, new PageParam(currentPage, numPerPage)));
    }

    /**
     * 导出微信群广告位点击读者列表
     */
    @Override
    @RequestMapping(value = "exportClickUserList4AdvertisingWechatGroup", method = RequestMethod.GET)
    public ResponseDto<?> exportClickUserList4AdvertisingWechatGroup(@RequestHeader("token") String token, @RequestParam(value = "qrcodeId", required = false) Long qrcodeId,
            @RequestParam(value = "adId", required = false) Long adId, @RequestParam(value = "statisMonth", required = false) String statisMonth)
            throws PermissionException, BizException, JsonParseException {
        if (null == qrcodeId || null == adId) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数有误！");
        }
        SessionUtil.getToken4Redis(token);
        return new ResponseDto<>(advertisingSpaceBiz.exportClickUserList4AdvertisingWechatGroup(qrcodeId, adId, statisMonth));
    }

    /**
     * 广告主广告位明细
     */
    @Override
    @RequestMapping(value = "advertisingDetail4Master", method = RequestMethod.GET)
    public ResponseDto<?> advertisingDetail4Master(@RequestHeader("token") String token, @RequestParam(value = "masterId", required = false) Long masterId,
            @RequestParam(value = "statisMonth", required = false) String statisMonth, @RequestParam(value = "currentPage", required = false) Integer currentPage,
            @RequestParam(value = "numPerPage", required = false) Integer numPerPage) throws PermissionException, BizException, JsonParseException {
        if (null == masterId || null == statisMonth) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数有误！");
        }
        if (currentPage == null || numPerPage == null || currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        SessionUtil.getToken4Redis(token);
        return new ResponseDto<>(advertisingSpaceBiz.advertisingDetail4Master(masterId, statisMonth, new PageParam(currentPage, numPerPage)));
    }

    /**
     * 导出广告主广告位明细
     */
    @Override
    @RequestMapping(value = "exportAdvertisingDetail4Master", method = RequestMethod.GET)
    public ResponseDto<?> exportAdvertisingDetail4Master(@RequestHeader("token") String token, @RequestParam(value = "masterId", required = false) Long masterId,
            @RequestParam(value = "statisMonth", required = false) String statisMonth) throws PermissionException, BizException, JsonParseException {
        if (null == masterId || null == statisMonth) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数有误！");
        }
        SessionUtil.getToken4Redis(token);
        return new ResponseDto<>(advertisingSpaceBiz.exportAdvertisingDetail4Master(masterId, statisMonth));
    }

    /**
     * 获取微信群信息
     */
    @Override
    @RequestMapping(value = "getGroupQrcodeInfo", method = RequestMethod.GET)
    public ResponseDto<?> getGroupQrcodeInfo(@RequestHeader("token") String token, @RequestParam(value = "qrcodeId", required = false) Long qrcodeId)
            throws PermissionException, BizException, JsonParseException {
        if (null == qrcodeId) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        SessionUtil.getToken4Redis(token);
        return new ResponseDto<>(advertisingSpaceBiz.getGroupQrcodeInfo(qrcodeId));
    }

    @Override
    @PostMapping("updateTestStatus")
    public ResponseDto<?> updateTestStatus(@RequestHeader("token") String token, @RequestBody TestParamDTO testParamDTO) throws PermissionException {
        SessionUtil.getToken4Redis(token);
        if (null == testParamDTO || null == testParamDTO.getId() || null == testParamDTO.getTest()) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "缺少参数");
        }
        advertisingSpaceBiz.updateTestStatus(testParamDTO.getTest(), testParamDTO.getId());
        return new ResponseDto<>();
    }


    @ApiOperation("创建广告位计划")
    @PostMapping("createAdvertisingPlan")
    @Override
    public ResponseDto<?> createAdvertisingPlan(
            @RequestHeader("token") String token,
            @RequestBody @ApiParam("广告位计划") AdvertisingPlan advertisingPlan
    ) throws PermissionException, BizException, JsonParseException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (null == advertisingPlan) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        advertisingPlan.setCreateUser(partyId);
        advertisingPlan.setUpdateUser(partyId);
        return new ResponseDto<>(advertisingSpaceBiz.createAdvertisingPlan(advertisingPlan));
    }

    @ApiOperation("修改广告位计划")
    @PostMapping("updateAdvertisingPlan")
    @Override
    public ResponseDto<?> updateAdvertisingPlan(
            @RequestHeader("token") String token,
            @RequestBody @ApiParam("广告位计划") AdvertisingPlan advertisingPlan
    ) throws PermissionException, BizException, JsonParseException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (null == advertisingPlan) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        if (advertisingPlan.getId() == null) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "计划id不能为空！");
        }
        advertisingPlan.setUpdateUser(partyId);
        advertisingSpaceBiz.updateAdvertisingPlan(advertisingPlan);
        return new ResponseDto<>();
    }

    @ApiOperation("删除广告位计划")
    @GetMapping("deleteAdvertisingPlan")
    @Override
    public ResponseDto<?> deleteAdvertisingPlan(
            @RequestHeader("token") String token,
            @RequestParam @ApiParam("广告位计划id") Long advertisingPlanId
    ) throws PermissionException, BizException, JsonParseException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (null == advertisingPlanId) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        advertisingSpaceBiz.deleteAdvertisingPlan(advertisingPlanId,partyId);
        return new ResponseDto<>();
    }

    @ApiOperation("获取广告位计划")
    @GetMapping("getAdvertisingPlan")
    @Override
    public ResponseDto<?> getAdvertisingPlan(
            @RequestHeader("token") String token,
            @RequestParam @ApiParam("广告位计划id") Long advertisingPlanId
    ) throws PermissionException, BizException, JsonParseException {
        SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (null == advertisingPlanId) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        return new ResponseDto<>(advertisingSpaceBiz.getAdvertisingPlan(advertisingPlanId));
    }

    @ApiOperation("获取广告位计划列表")
    @GetMapping("getAdvertisingPlanList")
    @Override
    public ResponseDto<?> getAdvertisingPlanList(
            @RequestHeader("token") String token,
            @RequestParam(value = "keyword", required = false) String keyword,
            @RequestParam("currentPage") Integer currentPage,
            @RequestParam("numPerPage") Integer numPerPage
    ) throws PermissionException, BizException, JsonParseException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (currentPage==null||currentPage<0||numPerPage==null||numPerPage<=0){
            throw new BookBizException(BookBizException.PARAM_IS_ERROR,"分页参数错误");
        }
        return new ResponseDto<>(advertisingSpaceBiz.getAdvertisingPlanList(keyword,partyId,currentPage,numPerPage));
    }

    /**
     * 完课量埋点
     *
     * @param userInfo               用户身份信息
     * @param advertisingPilotRecord 广告位试听实体
     */
    @Override
    @RequestMapping(value = "addPilotTrack", method = RequestMethod.POST)
    public ResponseDto<?> addPilotTrack(@CookieValue("userInfo")String userInfo, @RequestBody @ApiParam AdvertisingPilotRecord advertisingPilotRecord) throws PermissionException, BizException, JsonParseException {
        if (null == advertisingPilotRecord || null == advertisingPilotRecord.getAdId()) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        if (StringUtil.isEmpty(advertisingPilotRecord.getFromType()) || null == advertisingPilotRecord.getFromId()) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        Long sceneId = Cookie.getId(userInfo, Cookie._SCENE_ID);
        Long adviserId = Cookie.getId(userInfo, Cookie._ADVISER_ID);
        Long channelId = Cookie.getId(userInfo, Cookie._CHANNEL_ID);
        Long officialAccountsId = Cookie.getId(userInfo, Cookie._OFFICIAL_ACCOUNTS_ID);
        Long wechatUserId = Cookie.getId(userInfo, Cookie._WECHAT_USER_ID);
        Long bookGroupId = Cookie.getId(userInfo, Cookie.BOOK_GROUP_ID);
        if ((null == advertisingPilotRecord.getBookId() && !"WECHAT_GROUP".equals(advertisingPilotRecord.getFromType()))
                || (null == bookGroupId && "WECHAT_GROUP".equals(advertisingPilotRecord.getFromType()))) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        advertisingPilotRecord.setBookGroupId(bookGroupId);
        advertisingPilotRecord.setSceneId(sceneId);
        advertisingPilotRecord.setAdviserId(adviserId);
        advertisingPilotRecord.setChannelId(channelId);
        advertisingPilotRecord.setOfficialAccountId(officialAccountsId);
        advertisingPilotRecord.setWechatUserId(wechatUserId);
        advertisingSpaceBiz.addPilotTrack(advertisingPilotRecord);
        return new ResponseDto<>();
    }

    /**
     * 浏览时长埋点
     * @param userInfo
     * @param advertisingBrowseRecord
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @Override
    @RequestMapping(value = "addBrowseTrack", method = RequestMethod.POST)
    public ResponseDto<?> addBrowseTrack(@CookieValue("userInfo") String userInfo, @RequestBody @ApiParam AdvertisingBrowseRecord advertisingBrowseRecord)
            throws PermissionException, BizException, JsonParseException{
        if (null == advertisingBrowseRecord || null == advertisingBrowseRecord.getAdId()) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        if (StringUtil.isEmpty(advertisingBrowseRecord.getFromType()) || null == advertisingBrowseRecord.getFromId()) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        Long sceneId = Cookie.getId(userInfo, Cookie._SCENE_ID);
        Long adviserId = Cookie.getId(userInfo, Cookie._ADVISER_ID);
        Long channelId = Cookie.getId(userInfo, Cookie._CHANNEL_ID);
        Long officialAccountsId = Cookie.getId(userInfo, Cookie._OFFICIAL_ACCOUNTS_ID);
        Long wechatUserId = Cookie.getId(userInfo, Cookie._WECHAT_USER_ID);
        Long bookGroupId = Cookie.getId(userInfo, Cookie.BOOK_GROUP_ID);
        if ((null == advertisingBrowseRecord.getBookId() && !"WECHAT_GROUP".equals(advertisingBrowseRecord.getFromType()))
                || (null == bookGroupId && "WECHAT_GROUP".equals(advertisingBrowseRecord.getFromType()))) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        if (null == advertisingBrowseRecord.getBrowseTimes()){
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "缺少浏览时长！");
        }
        advertisingBrowseRecord.setBookGroupId(bookGroupId);
        advertisingBrowseRecord.setSceneId(sceneId);
        advertisingBrowseRecord.setAdviserId(adviserId);
        advertisingBrowseRecord.setChannelId(channelId);
        advertisingBrowseRecord.setOfficialAccountId(officialAccountsId);
        advertisingBrowseRecord.setWechatUserId(wechatUserId);
        return new ResponseDto<>(advertisingSpaceBiz.addBrowseTrack(advertisingBrowseRecord));
    }

    /**
     * 微信群广告位明细-投放详情
     */
    @Override
    @RequestMapping(value = "advertisingDistributionDetail4WechatGroup", method = RequestMethod.GET)
    public ResponseDto<?> advertisingDistributionDetail4WechatGroup(
            @RequestHeader("token") String token, @RequestParam(value = "qrcodeId", required = false) Long qrcodeId,
            @RequestParam(value = "statisMonth", required = false) String statisMonth, @RequestParam(value = "currentPage", required = false) Integer currentPage,
            @RequestParam(value = "numPerPage", required = false) Integer numPerPage, @RequestParam(value = "adId", required = false) Long adId) throws PermissionException, BizException, JsonParseException {
        if (null == qrcodeId || null == adId) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数有误！");
        }
        if (currentPage == null || numPerPage == null || currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        SessionUtil.getToken4Redis(token);
        PageBean pageBean = advertisingSpaceBiz.advertisingDistributionDetail4WechatGroup(qrcodeId, statisMonth, adId, new PageParam(currentPage, numPerPage));
        return new ResponseDto<>(pageBean);
    }


}
