package com.pcloud.book.advertising.facade;

import com.pcloud.book.advertising.dto.AdvertisingSpaceDTO;
import com.pcloud.book.advertising.dto.Book4AdvertisingMasterDTO;
import com.pcloud.book.advertising.dto.GroupMasterAddDTO;
import com.pcloud.book.advertising.dto.GroupMasterDTO;
import com.pcloud.book.advertising.dto.TestParamDTO;
import com.pcloud.book.advertising.entity.AdvertisingAdviserPermission;
import com.pcloud.book.advertising.entity.AdvertisingAgentPermission;
import com.pcloud.book.advertising.entity.AdvertisingBrowseRecord;
import com.pcloud.book.advertising.entity.AdvertisingClickRecord;
import com.pcloud.book.advertising.entity.AdvertisingDistributionBook;
import com.pcloud.book.advertising.entity.AdvertisingExposureRecord;
import com.pcloud.book.advertising.entity.AdvertisingMaster;
import com.pcloud.book.advertising.entity.AdvertisingPilotRecord;
import com.pcloud.book.advertising.entity.AdvertisingPlan;
import com.pcloud.book.advertising.entity.AdvertisingSpace;
import com.pcloud.common.dto.ResponseDto;
import com.pcloud.common.exceptions.BizException;
import com.pcloud.common.page.PageBeanNew;
import com.pcloud.common.permission.PermissionException;

import org.codehaus.jackson.JsonParseException;
import org.springframework.cloud.netflix.feign.FeignClient;
import org.springframework.web.bind.annotation.CookieValue;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;

import java.util.List;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;

/**
 * Description 广告位接口层接口
 * @author PENG
 * @date 2019/2/28
 */
@FeignClient(value = "pcloud-service-book", qualifier = "advertisingSpaceFacadeCloud", path = "book/v1.0/advertising")
@Api(description = "广告位接口层接口")
public interface AdvertisingSpaceFacade {

    /**
     * 创建广告位
     * @param token            用户身份信息
     * @param advertisingSpace 广告位实体
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "创建广告位", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "token", value = "用户身份信息", required = false, dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "advertisingSpace", value = "广告位实体", required = false, dataType = "AdvertisingSpace", paramType = "body")
    })
    @RequestMapping(value = "create", method = RequestMethod.POST)
    ResponseDto<?> create(@RequestHeader("token") String token, @RequestBody AdvertisingSpace advertisingSpace)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 修改广告位
     * @param token            用户身份信息
     * @param advertisingSpace 广告位实体
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "修改广告位", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "token", value = "用户身份信息", required = false, dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "advertisingSpace", value = "广告位实体", required = false, dataType = "AdvertisingSpace", paramType = "body")
    })
    @RequestMapping(value = "update", method = RequestMethod.POST)
    ResponseDto<?> update(@RequestHeader("token") String token, @RequestBody AdvertisingSpace advertisingSpace)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 获取广告位信息
     * @param token 用户身份信息
     * @param adId  广告位ID
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "获取广告位信息", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "token", value = "用户身份信息", required = false, dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "adId", value = "广告位ID", required = false, dataType = "Long", paramType = "query")
    })
    @RequestMapping(value = "get", method = RequestMethod.GET)
    ResponseDto<?> get(@RequestHeader("token") String token, @RequestParam(value = "adId", required = false) Long adId)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 客户端获取广告位信息
     * @param userInfo 用户身份信息
     * @param adId  广告位ID
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "客户端获取广告位信息", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userInfo", value = "用户身份信息", dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "adId", value = "广告位ID", required = false, dataType = "Long", paramType = "query")
    })
    @RequestMapping(value = "getAdInfo4Wechat", method = RequestMethod.GET)
    ResponseDto<AdvertisingSpaceDTO> getAdInfo4Wechat(@CookieValue("userInfo") String userInfo, @RequestParam(value = "adId", required = false) Long adId);
    /**
     * 停用广告位
     * @param token            用户身份信息
     * @param advertisingSpace 广告位实体
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "停用广告位", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "token", value = "用户身份信息", required = false, dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "advertisingSpace", value = "广告位实体", required = false, dataType = "AdvertisingSpace", paramType = "body")
    })
    @RequestMapping(value = "stop", method = RequestMethod.POST)
    ResponseDto<?> stop(@RequestHeader("token") String token, @RequestBody AdvertisingSpace advertisingSpace)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 投放书刊
     * @param token                       用户身份信息
     * @param advertisingDistributionBook 广告位书刊实体
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "投放书刊", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "token", value = "用户身份信息", required = false, dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "advertisingDistributionBook", value = "广告位书刊实体", required = false, dataType = "AdvertisingDistributionBook", paramType = "body")
    })
    @RequestMapping(value = "distributeBook", method = RequestMethod.POST)
    ResponseDto<?> distributeBook(@RequestHeader("token") String token, @RequestBody AdvertisingDistributionBook advertisingDistributionBook)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 投放微信群
     * @param token                       用户身份信息
     * @param advertisingDistributionBook 广告位书刊实体
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "投放微信群", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "token", value = "用户身份信息", required = false, dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "advertisingDistributionBook", value = "广告位书刊实体", required = false, dataType = "AdvertisingDistributionBook", paramType = "body")
    })
    @RequestMapping(value = "distributeWechatGroup", method = RequestMethod.POST)
    ResponseDto<?> distributeWechatGroup(@RequestHeader("token") String token, @RequestBody AdvertisingDistributionBook advertisingDistributionBook)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 获取出版社书刊权限
     * @param token 用户身份信息
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "获取出版社书刊权限", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "token", value = "用户身份信息", required = false, dataType = "string", paramType = "header")
    })
    @RequestMapping(value = "getAgentPermission", method = RequestMethod.GET)
    ResponseDto<?> getAgentPermission(@RequestHeader("token") String token)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 设置出版社书刊权限
     * @param token                      用户身份信息
     * @param advertisingAgentPermission 广告位出版权限实体
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "设置出版社书刊权限", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "token", value = "用户身份信息", required = false, dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "advertisingAgentPermission", value = "广告位出版权限实体", required = false, dataType = "AdvertisingAgentPermission", paramType = "body")
    })
    @RequestMapping(value = "setAgentPermission", method = RequestMethod.POST)
    ResponseDto<?> setAgentPermission(@RequestHeader("token") String token, @RequestBody AdvertisingAgentPermission advertisingAgentPermission)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 设置编辑书刊权限
     * @param token                        用户身份信息
     * @param advertisingAdviserPermission 广告位编辑权限实体
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "设置编辑书刊权限", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "token", value = "用户身份信息", required = false, dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "advertisingAdviserPermission", value = "广告位编辑权限实体", required = false, dataType = "AdvertisingAdviserPermission", paramType = "body")
    })
    @RequestMapping(value = "setAdviserPermission", method = RequestMethod.POST)
    ResponseDto<?> setAdviserPermission(@RequestHeader("token") String token, @RequestBody AdvertisingAdviserPermission advertisingAdviserPermission)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 编辑端获取书刊列表
     * @param token       用户身份信息
     * @param name        书籍名称
     * @param currentPage 当前页
     * @param numPerPage  每页条数
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "编辑端获取书刊列表", httpMethod = "GET")
    @ApiImplicitParams({@ApiImplicitParam(name = "token", value = "token", dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "name", value = "书籍名称", dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "currentPage", value = "当前页", dataType = "int", paramType = "query"),
            @ApiImplicitParam(name = "numPerPage", value = "每页条数", dataType = "int", paramType = "query")
    })
    @RequestMapping(value = "listBook4Adviser", method = RequestMethod.GET)
    ResponseDto<?> listBook4Adviser(@RequestHeader("token") String token, @RequestParam(value = "name", required = false) String name,
            @RequestParam(value = "currentPage", required = false) Integer currentPage, @RequestParam(value = "numPerPage", required = false) Integer numPerPage)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 平台端获取某编辑书刊列表
     * @param token         用户身份信息
     * @param adviserId     编辑ID
     * @param name          书籍名称
     * @param currentPage   当前页
     * @param numPerPage    每页条数
     * @param bookGroupOnly 是否社群书
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "平台端获取某编辑书刊列表", httpMethod = "GET")
    @ApiImplicitParams({@ApiImplicitParam(name = "token", value = "token", dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "adviserId", value = "编辑ID", dataType = "Long", paramType = "query"),
            @ApiImplicitParam(name = "name", value = "书籍名称", dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "currentPage", value = "当前页", dataType = "int", paramType = "query"),
            @ApiImplicitParam(name = "numPerPage", value = "每页条数", dataType = "int", paramType = "query"),
            @ApiImplicitParam(name = "bookGroupOnly", value = "是否社群书", dataType = "boolean", paramType = "query")
    })
    @RequestMapping(value = "listBook4Platform", method = RequestMethod.GET)
    ResponseDto<?> listBook4Platform(@RequestHeader("token") String token, @RequestParam(value = "adviserId", required = false) Long adviserId,
            @RequestParam(value = "name", required = false) String name, @RequestParam(value = "currentPage", required = false) Integer currentPage,
            @RequestParam(value = "numPerPage", required = false) Integer numPerPage, @RequestParam(value = "bookGroupOnly", required = false) Boolean bookGroupOnly)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 平台端获取社群书微信群列表
     * @param token       用户身份信息
     * @param name        书籍名称
     * @param currentPage 当前页
     * @param numPerPage  每页条数
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "平台端获取社群书微信群列表", httpMethod = "GET")
    @ApiImplicitParams({@ApiImplicitParam(name = "token", value = "token", dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "name", value = "书籍名称", dataType = "string",required = false, paramType = "query"),
            @ApiImplicitParam(name = "currentPage", value = "当前页", dataType = "int", paramType = "query"),
            @ApiImplicitParam(name = "numPerPage", value = "每页条数", dataType = "int", paramType = "query"),
            @ApiImplicitParam(name = "proLabelId", value = "专业标签id", dataType = "long",required = false, paramType = "query"),
            @ApiImplicitParam(name = "depLabelId", value = "深度标签id", dataType = "long",required = false, paramType = "query"),
            @ApiImplicitParam(name = "purLabelId", value = "目的标签id", dataType = "string",required = false, paramType = "query")
    })
    @RequestMapping(value = "listGroupQrcode4Platform", method = RequestMethod.GET)
    ResponseDto<?> listGroupQrcode4Platform(
            @RequestHeader("token") String token, @RequestParam(value = "name", required = false) String name,
            @RequestParam(value = "currentPage", required = false) Integer currentPage, @RequestParam(value = "numPerPage", required = false) Integer numPerPage,
            @RequestParam(value = "proLabelId", required = false) Long proLabelId, @RequestParam(value = "depLabelId", required = false) Long depLabelId,
            @RequestParam(value = "purLabelId", required = false) Long purLabelId)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 获取广告位列表
     * @param token       用户身份信息
     * @param name        广告位名称
     * @param currentPage 当前页
     * @param numPerPage  每页条数
     * @param filterClose 过滤已关闭
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "获取广告位列表", httpMethod = "GET")
    @ApiImplicitParams({@ApiImplicitParam(name = "token", value = "token", dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "name", value = "广告位名称", dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "currentPage", value = "当前页", dataType = "int", paramType = "query"),
            @ApiImplicitParam(name = "numPerPage", value = "每页条数", dataType = "int", paramType = "query"),
            @ApiImplicitParam(name = "filterClose", value = "过滤已关闭", dataType = "int", paramType = "query"),
            @ApiImplicitParam(name = "filterTest", value = "过滤测试数据", dataType = "int", paramType = "query"),
            @ApiImplicitParam(name = "adPosition", value = "广告位位置", dataType = "string", paramType = "query")
    })
    @RequestMapping(value = "list", method = RequestMethod.GET)
    ResponseDto<?> list(@RequestHeader("token") String token, @RequestParam(value = "name", required = false) String name,
                        @RequestParam(value = "currentPage", required = false) Integer currentPage, @RequestParam(value = "numPerPage", required = false) Integer numPerPage,
                        @RequestParam(value = "filterClose", required = false) Boolean filterClose,
                        @RequestParam(value = "filterTest", required = false) Boolean filterTest,
                        @RequestParam(value = "adPosition", required = false) String adPosition)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 获取广告位投放书刊列表
     * @param token       用户身份信息
     * @param adId        广告位ID
     * @param name        书刊名称
     * @param currentPage 当前页
     * @param numPerPage  每页条数
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "获取广告位投放书刊列表", httpMethod = "GET")
    @ApiImplicitParams({@ApiImplicitParam(name = "token", value = "token", dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "adId", value = "广告位ID", dataType = "Long", paramType = "query"),
            @ApiImplicitParam(name = "name", value = "书刊名称", dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "currentPage", value = "当前页", dataType = "int", paramType = "query"),
            @ApiImplicitParam(name = "numPerPage", value = "每页条数", dataType = "int", paramType = "query")
    })
    @RequestMapping(value = "distributionBookList", method = RequestMethod.GET)
    ResponseDto<?> distributionBookList(@RequestHeader("token") String token, @RequestParam(value = "adId", required = false) Long adId,
            @RequestParam(value = "name", required = false) String name, @RequestParam(value = "currentPage", required = false) Integer currentPage,
            @RequestParam(value = "numPerPage", required = false) Integer numPerPage) throws PermissionException, BizException, JsonParseException;

    /**
     * 获取广告位投放微信群列表
     * @param token       用户身份信息
     * @param adId        广告位ID
     * @param name        名称
     * @param currentPage 当前页
     * @param numPerPage  每页条数
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "获取广告位投放微信群列表", httpMethod = "GET")
    @ApiImplicitParams({@ApiImplicitParam(name = "token", value = "token", dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "adId", value = "广告位ID", dataType = "Long", paramType = "query"),
            @ApiImplicitParam(name = "name", value = "名称", dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "currentPage", value = "当前页", dataType = "int", paramType = "query"),
            @ApiImplicitParam(name = "numPerPage", value = "每页条数", dataType = "int", paramType = "query")
    })
    @RequestMapping(value = "distributionWechatGroupList", method = RequestMethod.GET)
    ResponseDto<?> distributionWechatGroupList(@RequestHeader("token") String token, @RequestParam(value = "adId", required = false) Long adId,
            @RequestParam(value = "name", required = false) String name, @RequestParam(value = "currentPage", required = false) Integer currentPage,
            @RequestParam(value = "numPerPage", required = false) Integer numPerPage) throws PermissionException, BizException, JsonParseException;

    /**
     * 获取书刊配置的广告位信息
     * @param userInfo         用户身份信息
     * @param adPositionDetail 广告具体位置
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "获取书刊配置的广告位信息", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userInfo", value = "用户身份信息", dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "adPositionDetail", value = "广告具体位置", dataType = "string", paramType = "query"),
    })
    @RequestMapping(value = "get4Wechat", method = RequestMethod.GET)
    ResponseDto<?> get4Wechat(@CookieValue("userInfo") String userInfo, @RequestParam(value = "adPositionDetail", required = false) String adPositionDetail)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 获取社群书配置的广告位信息
     * @param userInfo         用户身份信息
     * @param adPositionDetail 广告具体位置
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "获取社群书配置的广告位信息", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userInfo", value = "用户身份信息", dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "adPositionDetail", value = "广告具体位置", dataType = "string", paramType = "query"),
    })
    @RequestMapping(value = "getBookGroupAd4Wechat", method = RequestMethod.GET)
    ResponseDto<?> getBookGroupAd4Wechat(@CookieValue("userInfo") String userInfo, @RequestParam(value = "adPositionDetail", required = false) String adPositionDetail)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 曝光量埋点
     * @param userInfo                  用户身份信息
     * @param advertisingExposureRecord 广告位曝光实体
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "曝光量埋点", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userInfo", value = "用户身份信息", required = false, dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "advertisingExposureRecord", value = "广告位曝光实体", required = false, dataType = "AdvertisingExposureRecord", paramType = "body")
    })
    @RequestMapping(value = "addExposureTrack", method = RequestMethod.POST)
    ResponseDto<?> addExposureTrack(@CookieValue("userInfo") String userInfo, @RequestBody AdvertisingExposureRecord advertisingExposureRecord)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 点击量埋点
     * @param userInfo               用户身份信息
     * @param advertisingClickRecord 广告位点击实体
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "点击量埋点", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userInfo", value = "用户身份信息", required = false, dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "advertisingClickRecord", value = "广告位点击实体", required = false, dataType = "AdvertisingClickRecord", paramType = "body")
    })
    @RequestMapping(value = "addClickTrack", method = RequestMethod.POST)
    ResponseDto<?> addClickTrack(@CookieValue("userInfo") String userInfo, @RequestBody AdvertisingClickRecord advertisingClickRecord)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 添加广告主
     * @param token             用户身份信息
     * @param advertisingMaster 广告主实体
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "添加广告主", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "token", value = "用户身份信息", required = false, dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "advertisingMaster", value = "广告主实体", required = false, dataType = "AdvertisingMaster", paramType = "body")
    })
    @RequestMapping(value = "createMaster", method = RequestMethod.POST)
    ResponseDto<?> createMaster(@RequestHeader("token") String token, @RequestBody AdvertisingMaster advertisingMaster)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 修改广告主
     * @param token             用户身份信息
     * @param advertisingMaster 广告主实体
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "修改广告主", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "token", value = "用户身份信息", required = false, dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "advertisingMaster", value = "广告主实体", required = false, dataType = "AdvertisingMaster", paramType = "body")
    })
    @RequestMapping(value = "updateMaster", method = RequestMethod.POST)
    ResponseDto<?> updateMaster(@RequestHeader("token") String token, @RequestBody AdvertisingMaster advertisingMaster)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 获取广告主列表
     * @param token       用户身份信息
     * @param masterName  广告主名称
     * @param currentPage 当前页
     * @param numPerPage  每页条数
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "获取广告主列表", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "token", value = "用户身份信息", required = false, dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "masterName", value = "广告主名称", required = false, dataType = "String", paramType = "query"),
            @ApiImplicitParam(name = "currentPage", value = "当前页", dataType = "int", paramType = "query"),
            @ApiImplicitParam(name = "numPerPage", value = "每页条数", dataType = "int", paramType = "query")
    })
    @RequestMapping(value = "getMasterList", method = RequestMethod.GET)
    ResponseDto<?> getMasterList(@RequestHeader("token") String token, @RequestParam(value = "masterName", required = false) String masterName,
            @RequestParam(value = "currentPage", required = false) Integer currentPage, @RequestParam(value = "numPerPage", required = false) Integer numPerPage)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 获取所有广告主
     * @param token 用户身份信息
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "获取所有广告主", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "token", value = "用户身份信息", required = false, dataType = "string", paramType = "header"),
    })
    @RequestMapping(value = "getAllMaster", method = RequestMethod.GET)
    ResponseDto<?> getAllMaster(@RequestHeader("token") String token) throws PermissionException, BizException, JsonParseException;

    /**
     * 获取广告主信息
     * @param token    用户身份信息
     * @param masterId 广告主ID
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "获取广告主信息", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "token", value = "用户身份信息", required = false, dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "masterId", value = "广告主ID", required = false, dataType = "Long", paramType = "query"),
    })
    @RequestMapping(value = "getMasterInfo", method = RequestMethod.GET)
    ResponseDto<?> getMasterInfo(@RequestHeader("token") String token, @RequestParam(value = "masterId", required = false) Long masterId)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 获取概览数据
     * @param token 用户身份信息
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "获取概览数据", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "token", value = "用户身份信息", required = false, dataType = "string", paramType = "header"),
    })
    @RequestMapping(value = "overviewStatis", method = RequestMethod.GET)
    ResponseDto<?> overviewStatis(@RequestHeader("token") String token) throws PermissionException, BizException, JsonParseException;

    /**
     * 获取流量趋势
     * @param token     用户身份信息
     * @param day       天数
     * @param startDate 开始时间
     * @param endDate   结束时间
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "获取流量趋势", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "token", value = "用户身份信息", required = false, dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "day", value = "天数", required = false, dataType = "Integer", paramType = "query"),
            @ApiImplicitParam(name = "startDate", value = "开始时间", required = false, dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "endDate", value = "结束时间", required = false, dataType = "string", paramType = "query"),
    })
    @RequestMapping(value = "getFlowStatisTrend", method = RequestMethod.GET)
    ResponseDto<?> getFlowStatisTrend(@RequestHeader("token") String token, @RequestParam(value = "day", required = false) Integer day,
            @RequestParam(value = "startDate", required = false) String startDate, @RequestParam(value = "endDate", required = false) String endDate)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 获取收益趋势
     * @param token     用户身份信息
     * @param day       天数
     * @param startDate 开始时间
     * @param endDate   结束时间
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "获取收益趋势", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "token", value = "用户身份信息", required = false, dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "day", value = "天数", required = false, dataType = "Integer", paramType = "query"),
            @ApiImplicitParam(name = "startDate", value = "开始时间", required = false, dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "endDate", value = "结束时间", required = false, dataType = "string", paramType = "query"),
    })
    @RequestMapping(value = "getIncomeStatisTrend", method = RequestMethod.GET)
    ResponseDto<?> getIncomeStatisTrend(@RequestHeader("token") String token, @RequestParam(value = "day", required = false) Integer day,
            @RequestParam(value = "startDate", required = false) String startDate, @RequestParam(value = "endDate", required = false) String endDate)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 获取广告位投放书刊列表
     * @param token       用户身份信息
     * @param name        书刊名称
     * @param currentPage 当前页
     * @param numPerPage  每页条数
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "获取广告位投放书刊列表", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "token", value = "token", dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "name", value = "书刊名称", dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "currentPage", value = "当前页", dataType = "int", paramType = "query"),
            @ApiImplicitParam(name = "numPerPage", value = "每页条数", dataType = "int", paramType = "query")
    })
    @RequestMapping(value = "advertisingBookList", method = RequestMethod.GET)
    ResponseDto<?> advertisingBookList(@RequestHeader("token") String token, @RequestParam(value = "name", required = false) String name,
            @RequestParam(value = "currentPage", required = false) Integer currentPage, @RequestParam(value = "numPerPage", required = false) Integer numPerPage)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 获取广告位投放微信群列表
     * @param token       用户身份信息
     * @param name        书刊名称
     * @param currentPage 当前页
     * @param numPerPage  每页条数
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "获取广告位投放微信群列表", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "token", value = "token", dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "name", value = "书刊名称", dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "currentPage", value = "当前页", dataType = "int", paramType = "query"),
            @ApiImplicitParam(name = "numPerPage", value = "每页条数", dataType = "int", paramType = "query")
    })
    @RequestMapping(value = "advertisingWechatGroupList", method = RequestMethod.GET)
    ResponseDto<?> advertisingWechatGroupList(@RequestHeader("token") String token, @RequestParam(value = "name", required = false) String name,
            @RequestParam(value = "currentPage", required = false) Integer currentPage, @RequestParam(value = "numPerPage", required = false) Integer numPerPage)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 获取广告位投放微信群曝光量前几位
     * @param token 用户身份信息
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "获取广告位投放微信群曝光量前几位", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "token", value = "token", dataType = "string", paramType = "header")
    })
    @RequestMapping(value = "advertisingWechatGroupRankTop", method = RequestMethod.GET)
    ResponseDto<?> advertisingWechatGroupRankTop(@RequestHeader("token") String token) throws PermissionException, BizException, JsonParseException;

    /**
     * 书刊广告位明细
     * @param token       用户身份信息
     * @param bookId      书刊ID
     * @param channelId   运营ID
     * @param adviserId   编辑ID
     * @param statisMonth 统计月份
     * @param currentPage 当前页
     * @param numPerPage  每页条数
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "书刊广告位明细", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "token", value = "token", dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "bookId", value = "书刊ID", dataType = "Long", paramType = "query"),
            @ApiImplicitParam(name = "channelId", value = "运营ID", dataType = "Long", paramType = "query"),
            @ApiImplicitParam(name = "adviserId", value = "编辑ID", dataType = "Long", paramType = "query"),
            @ApiImplicitParam(name = "statisMonth", value = "统计月份", dataType = "String", paramType = "query"),
            @ApiImplicitParam(name = "currentPage", value = "当前页", dataType = "int", paramType = "query"),
            @ApiImplicitParam(name = "numPerPage", value = "每页条数", dataType = "int", paramType = "query")
    })
    @RequestMapping(value = "advertisingDetail4Book", method = RequestMethod.GET)
    ResponseDto<?> advertisingDetail4Book(@RequestHeader("token") String token, @RequestParam(value = "bookId", required = false) Long bookId,
            @RequestParam(value = "channelId", required = false) Long channelId, @RequestParam(value = "adviserId", required = false) Long adviserId,
            @RequestParam(value = "statisMonth", required = false) String statisMonth, @RequestParam(value = "currentPage", required = false) Integer currentPage,
            @RequestParam(value = "numPerPage", required = false) Integer numPerPage) throws PermissionException, BizException, JsonParseException;

    /**
     * 导出书刊广告位明细
     * @param token       用户身份信息
     * @param bookId      书刊ID
     * @param channelId   运营ID
     * @param adviserId   编辑ID
     * @param statisMonth 统计月份
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "导出书刊广告位明细", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "token", value = "token", dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "bookId", value = "书刊ID", dataType = "Long", paramType = "query"),
            @ApiImplicitParam(name = "channelId", value = "运营ID", dataType = "Long", paramType = "query"),
            @ApiImplicitParam(name = "adviserId", value = "编辑ID", dataType = "Long", paramType = "query"),
            @ApiImplicitParam(name = "statisMonth", value = "统计月份", dataType = "String", paramType = "query")
    })
    @RequestMapping(value = "exportAdvertisingDetail4Book", method = RequestMethod.GET)
    ResponseDto<?> exportAdvertisingDetail4Book(@RequestHeader("token") String token, @RequestParam(value = "bookId", required = false) Long bookId,
            @RequestParam(value = "channelId", required = false) Long channelId, @RequestParam(value = "adviserId", required = false) Long adviserId,
            @RequestParam(value = "statisMonth", required = false) String statisMonth) throws PermissionException, BizException, JsonParseException;

    /**
     * 微信群广告位明细
     * @param token       用户身份信息
     * @param qrcodeId    微信群ID
     * @param statisMonth 统计月份
     * @param currentPage 当前页
     * @param numPerPage  每页条数
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "微信群广告位明细", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "token", value = "token", dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "qrcodeId", value = "微信群ID", dataType = "Long", paramType = "query"),
            @ApiImplicitParam(name = "statisMonth", value = "统计月份", dataType = "String", paramType = "query"),
            @ApiImplicitParam(name = "currentPage", value = "当前页", dataType = "int", paramType = "query"),
            @ApiImplicitParam(name = "numPerPage", value = "每页条数", dataType = "int", paramType = "query")
    })
    @RequestMapping(value = "advertisingDetail4WechatGroup", method = RequestMethod.GET)
    ResponseDto<?> advertisingDetail4WechatGroup(@RequestHeader("token") String token, @RequestParam(value = "qrcodeId", required = false) Long qrcodeId,
            @RequestParam(value = "statisMonth", required = false) String statisMonth, @RequestParam(value = "currentPage", required = false) Integer currentPage,
            @RequestParam(value = "numPerPage", required = false) Integer numPerPage) throws PermissionException, BizException, JsonParseException;

    /**
     * 导出微信群广告位明细
     * @param token       用户身份信息
     * @param qrcodeId    微信群ID
     * @param statisMonth 统计月份
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "导出微信群广告位明细", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "token", value = "token", dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "qrcodeId", value = "微信群ID", dataType = "Long", paramType = "query"),
            @ApiImplicitParam(name = "statisMonth", value = "统计月份", dataType = "String", paramType = "query")
    })
    @RequestMapping(value = "exportAdvertisingDetail4WechatGroup", method = RequestMethod.GET)
    ResponseDto<?> exportAdvertisingDetail4WechatGroup(@RequestHeader("token") String token, @RequestParam(value = "qrcodeId", required = false) Long qrcodeId,
            @RequestParam(value = "statisMonth", required = false) String statisMonth) throws PermissionException, BizException, JsonParseException;

    /**
     * 微信群广告位点击读者列表
     * @param token       用户身份信息
     * @param qrcodeId    微信群ID
     * @param adId        广告位ID
     * @param statisMonth 统计月份
     * @param currentPage 当前页
     * @param numPerPage  每页条数
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "微信群广告位点击读者列表", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "token", value = "token", dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "qrcodeId", value = "微信群ID", dataType = "Long", paramType = "query"),
            @ApiImplicitParam(name = "adId", value = "广告位ID", dataType = "Long", paramType = "query"),
            @ApiImplicitParam(name = "statisMonth", value = "统计月份", dataType = "String", paramType = "query"),
            @ApiImplicitParam(name = "currentPage", value = "当前页", dataType = "Integer", paramType = "query"),
            @ApiImplicitParam(name = "numPerPage", value = "每页条数", dataType = "Integer", paramType = "query"),
    })
    @RequestMapping(value = "clickUserList4AdvertisingWechatGroup", method = RequestMethod.GET)
    ResponseDto<?> clickUserList4AdvertisingWechatGroup(@RequestHeader("token") String token, @RequestParam(value = "qrcodeId", required = false) Long qrcodeId,
            @RequestParam(value = "adId", required = false) Long adId, @RequestParam(value = "statisMonth", required = false) String statisMonth,
            @RequestParam(value = "currentPage", required = false) Integer currentPage, @RequestParam(value = "numPerPage", required = false) Integer numPerPage)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 导出微信群广告位点击读者列表
     * @param token       用户身份信息
     * @param qrcodeId    微信群ID
     * @param adId        广告位ID
     * @param statisMonth 统计月份
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "导出微信群广告位点击读者列表", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "token", value = "token", dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "qrcodeId", value = "微信群ID", dataType = "Long", paramType = "query"),
            @ApiImplicitParam(name = "adId", value = "广告位ID", dataType = "Long", paramType = "query"),
            @ApiImplicitParam(name = "statisMonth", value = "统计月份", dataType = "String", paramType = "query"),
    })
    @RequestMapping(value = "exportClickUserList4AdvertisingWechatGroup", method = RequestMethod.GET)
    ResponseDto<?> exportClickUserList4AdvertisingWechatGroup(@RequestHeader("token") String token, @RequestParam(value = "qrcodeId", required = false) Long qrcodeId,
            @RequestParam(value = "adId", required = false) Long adId, @RequestParam(value = "statisMonth", required = false) String statisMonth)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 广告主广告位明细
     * @param token       用户身份信息
     * @param masterId    广告主ID
     * @param statisMonth 统计月份
     * @param currentPage 当前页
     * @param numPerPage  每页条数
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "广告主广告位明细", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "token", value = "token", dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "masterId", value = "广告主ID", dataType = "Long", paramType = "query"),
            @ApiImplicitParam(name = "statisMonth", value = "统计月份", dataType = "String", paramType = "query"),
            @ApiImplicitParam(name = "currentPage", value = "当前页", dataType = "int", paramType = "query"),
            @ApiImplicitParam(name = "numPerPage", value = "每页条数", dataType = "int", paramType = "query")
    })
    @RequestMapping(value = "advertisingDetail4Master", method = RequestMethod.GET)
    ResponseDto<?> advertisingDetail4Master(@RequestHeader("token") String token, @RequestParam(value = "masterId", required = false) Long masterId,
            @RequestParam(value = "statisMonth", required = false) String statisMonth, @RequestParam(value = "currentPage", required = false) Integer currentPage,
            @RequestParam(value = "numPerPage", required = false) Integer numPerPage) throws PermissionException, BizException, JsonParseException;

    /**
     * 导出广告主广告位明细
     * @param token       用户身份信息
     * @param masterId    广告主ID
     * @param statisMonth 统计月份
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "导出广告主广告位明细", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "token", value = "token", dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "masterId", value = "广告主ID", dataType = "Long", paramType = "query"),
            @ApiImplicitParam(name = "statisMonth", value = "统计月份", dataType = "String", paramType = "query")
    })
    @RequestMapping(value = "exportAdvertisingDetail4Master", method = RequestMethod.GET)
    ResponseDto<?> exportAdvertisingDetail4Master(@RequestHeader("token") String token, @RequestParam(value = "masterId", required = false) Long masterId,
            @RequestParam(value = "statisMonth", required = false) String statisMonth) throws PermissionException, BizException, JsonParseException;

    /**
     * 获取微信群信息
     * @param token    用户身份信息
     * @param qrcodeId 微信群ID
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "获取微信群信息", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "token", value = "token", dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "qrcodeId", value = "微信群ID", dataType = "Long", paramType = "query")
    })
    @RequestMapping(value = "getGroupQrcodeInfo", method = RequestMethod.GET)
    ResponseDto<?> getGroupQrcodeInfo(@RequestHeader("token") String token, @RequestParam(value = "qrcodeId", required = false) Long qrcodeId)
            throws PermissionException, BizException, JsonParseException;

    @ApiOperation(value = "更新广告位测试状态", httpMethod = "POST")
    @PostMapping("updateTestStatus")
    public ResponseDto<?> updateTestStatus(@RequestHeader("token") String token, @RequestBody @ApiParam TestParamDTO testParamDTO) throws PermissionException;


    @ApiOperation("创建广告位计划")
    @PostMapping("createAdvertisingPlan")
    ResponseDto<?> createAdvertisingPlan(
            @RequestHeader("token") String token,
            @RequestBody @ApiParam("广告位计划") AdvertisingPlan advertisingPlan
    ) throws PermissionException, BizException, JsonParseException;

    @ApiOperation("修改广告位计划")
    @PostMapping("updateAdvertisingPlan")
    ResponseDto<?> updateAdvertisingPlan(
            @RequestHeader("token") String token,
            @RequestBody @ApiParam("广告位计划") AdvertisingPlan advertisingPlan
    ) throws PermissionException, BizException, JsonParseException;

    @ApiOperation("删除广告位计划")
    @GetMapping("deleteAdvertisingPlan")
    ResponseDto<?> deleteAdvertisingPlan(
            @RequestHeader("token") String token,
            @RequestParam @ApiParam("广告位计划id") Long advertisingPlanId
    ) throws PermissionException, BizException, JsonParseException;

    @ApiOperation("获取广告位计划")
    @GetMapping("getAdvertisingPlan")
    ResponseDto<?> getAdvertisingPlan(
            @RequestHeader("token") String token,
            @RequestParam @ApiParam("广告位计划id") Long advertisingPlanId
    ) throws PermissionException, BizException, JsonParseException;

    @ApiOperation("获取广告位计划列表")
    @GetMapping("getAdvertisingPlanList")
    ResponseDto<?> getAdvertisingPlanList(
            @RequestHeader("token") String token,
            @RequestParam(value = "keyword", required = false) String keyword,
            @RequestParam("currentPage") Integer currentPage,
            @RequestParam("numPerPage") Integer numPerPage
    ) throws PermissionException, BizException, JsonParseException;


    /**
     * 完课量埋点
     * @param userInfo               用户身份信息
     * @param advertisingPilotRecord 广告位试听实体
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = " 完课量埋点", httpMethod = "POST")
    @RequestMapping(value = "addPilotTrack", method = RequestMethod.POST)
    ResponseDto<?> addPilotTrack(@CookieValue("userInfo") String userInfo, @RequestBody @ApiParam AdvertisingPilotRecord advertisingPilotRecord)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 浏览时长埋点
     * @param userInfo
     * @param advertisingBrowseRecord
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = " 浏览时长埋点", httpMethod = "POST")
    @RequestMapping(value = "addBrowseTrack", method = RequestMethod.POST)
    ResponseDto<?> addBrowseTrack(@CookieValue("userInfo") String userInfo, @RequestBody @ApiParam AdvertisingBrowseRecord advertisingBrowseRecord)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 微信群广告位明细-投放详情
     * @param token
     * @param qrcodeId
     * @param statisMonth
     * @param currentPage
     * @param numPerPage
     * @param adId
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "微信群广告位明细-投放详情", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "token", value = "token", dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "qrcodeId", value = "微信群ID", dataType = "Long", paramType = "query"),
            @ApiImplicitParam(name = "statisMonth", value = "统计月份", dataType = "String", paramType = "query"),
            @ApiImplicitParam(name = "currentPage", value = "当前页", dataType = "int", paramType = "query"),
            @ApiImplicitParam(name = "numPerPage", value = "每页条数", dataType = "int", paramType = "query"),
            @ApiImplicitParam(name = "adId", value = "广告id", dataType = "Long", paramType = "query")
    })
    @RequestMapping(value = "advertisingDistributionDetail4WechatGroup", method = RequestMethod.GET)
    ResponseDto<?> advertisingDistributionDetail4WechatGroup(
            @RequestHeader("token") String token, @RequestParam(value = "qrcodeId", required = false) Long qrcodeId,
            @RequestParam(value = "statisMonth", required = false) String statisMonth, @RequestParam(value = "currentPage", required = false) Integer currentPage,
            @RequestParam(value = "numPerPage", required = false) Integer numPerPage, @RequestParam(value = "adId", required = false) Long adId) throws PermissionException, BizException, JsonParseException;


    /**
     * 微信群广告主标记列表
     * @param token
     * @param currentPage
     * @param numPerPage
     * @param name
     * @param proLabelId
     * @param depLabelId
     * @param purLabelId
     * @param masterId
     * @return
     * @throws PermissionException
     */
    @ApiOperation(value = "微信群广告主标记列表", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "currentPage", value = "当前页", dataType = "integer", paramType = "query"),
            @ApiImplicitParam(name = "numPerPage", value = "每页数量", dataType = "integer", paramType = "query"),
            @ApiImplicitParam(name = "name", value = "书籍名称", dataType = "string", required = false, paramType = "query"),
            @ApiImplicitParam(name = "proLabelId", value = "专业标签id", dataType = "long", required = false, paramType = "query"),
            @ApiImplicitParam(name = "depLabelId", value = "深度标签id", dataType = "long", required = false, paramType = "query"),
            @ApiImplicitParam(name = "purLabelId", value = "目的标签id", dataType = "string", required = false, paramType = "query"),
            @ApiImplicitParam(name = "masterId", value = "广告主id", dataType = "string", required = false, paramType = "query")
    })
    @RequestMapping(value = "listPageGroupMaster", method = RequestMethod.GET)
    public ResponseDto<PageBeanNew<GroupMasterDTO>> listPageGroupMaster(
            @RequestHeader("token") String token, @RequestParam("currentPage") Integer currentPage, @RequestParam("numPerPage") Integer numPerPage,
            @RequestParam(value = "name", required = false) String name, @RequestParam(value = "proLabelId", required = false) Long proLabelId,
            @RequestParam(value = "depLabelId", required = false) Long depLabelId, @RequestParam(value = "purLabelId", required = false) Long purLabelId,
            @RequestParam(value = "masterId", required = false) Long masterId) throws PermissionException;

    @ApiOperation(value = "获取品牌方广告主", httpMethod = "GET")
    @RequestMapping(value = "getBrandMaster", method = RequestMethod.GET)
    ResponseDto<?> getBrandMaster(@RequestHeader("token") String token) throws PermissionException, BizException, JsonParseException;


    /**
     * 微信群标记广告主
     * @param token
     * @param groupMasterAddDTO
     * @return
     * @throws PermissionException
     */
    @ApiOperation(value = "微信群标记广告主", httpMethod = "POST")
    @PostMapping("setGroupMaster")
    public ResponseDto<?> setGroupMaster(@RequestHeader("token") String token, @RequestBody @ApiParam GroupMasterAddDTO groupMasterAddDTO) throws PermissionException;


    /**
     * 广告主标记书刊列表
     * @param token
     * @param currentPage
     * @param numPerPage
     * @param name
     * @return
     * @throws PermissionException
     */
    @ApiOperation(value = "广告主标记书刊列表", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "currentPage", value = "当前页", dataType = "integer", paramType = "query"),
            @ApiImplicitParam(name = "numPerPage", value = "每页数量", dataType = "integer", paramType = "query"),
            @ApiImplicitParam(name = "name", value = "书籍名称", dataType = "string", required = false, paramType = "query")
    })
    @RequestMapping(value = "listPageBook4AdMaster", method = RequestMethod.GET)
    public ResponseDto<PageBeanNew<Book4AdvertisingMasterDTO>> listPageBook4AdMaster(
            @RequestHeader("token") String token, @RequestParam("currentPage") Integer currentPage, @RequestParam("numPerPage") Integer numPerPage,
            @RequestParam(value = "name", required = false) String name) throws PermissionException;

    /**
     * 品牌方广告主-微信群列表
     * @param token
     * @param currentPage
     * @param numPerPage
     * @param name
     * @param proLabelId
     * @param depLabelId
     * @param purLabelId
     * @param startTime
     * @param endTime
     * @return
     * @throws PermissionException
     */
    @ApiOperation(value = "品牌方广告主-微信群列表", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "currentPage", value = "当前页", dataType = "integer", paramType = "query"),
            @ApiImplicitParam(name = "numPerPage", value = "每页数量", dataType = "integer", paramType = "query"),
            @ApiImplicitParam(name = "name", value = "书籍名称", dataType = "string", required = false, paramType = "query"),
            @ApiImplicitParam(name = "proLabelId", value = "专业标签id", dataType = "long", required = false, paramType = "query"),
            @ApiImplicitParam(name = "depLabelId", value = "深度标签id", dataType = "long", required = false, paramType = "query"),
            @ApiImplicitParam(name = "purLabelId", value = "目的标签id", dataType = "string", required = false, paramType = "query"),
            @ApiImplicitParam(name = "startTime", value = "startTime", dataType = "string", required = false, paramType = "query"),
            @ApiImplicitParam(name = "endTime", value = "endTime", dataType = "string", required = false, paramType = "query")
    })
    @RequestMapping(value = "listPageGroupList4Master", method = RequestMethod.GET)
    public ResponseDto<PageBeanNew<GroupMasterDTO>> listPageGroupList4Master(
            @RequestHeader("token") String token, @RequestParam("currentPage") Integer currentPage, @RequestParam("numPerPage") Integer numPerPage,
            @RequestParam(value = "name", required = false) String name, @RequestParam(value = "proLabelId", required = false) Long proLabelId,
            @RequestParam(value = "depLabelId", required = false) Long depLabelId, @RequestParam(value = "purLabelId", required = false) Long purLabelId,
            @RequestParam(value = "startTime", required = false) String startTime, @RequestParam(value = "endTime", required = false) String endTime) throws PermissionException;


    /**
     * 导出-微信群广告主标记列表
     * @param token
     * @param name
     * @param proLabelId
     * @param depLabelId
     * @param purLabelId
     * @param masterId
     * @return
     * @throws PermissionException
     */
    @ApiOperation(value = "导出-微信群广告主标记列表", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "name", value = "书籍名称", dataType = "string", required = false, paramType = "query"),
            @ApiImplicitParam(name = "proLabelId", value = "专业标签id", dataType = "long", required = false, paramType = "query"),
            @ApiImplicitParam(name = "depLabelId", value = "深度标签id", dataType = "long", required = false, paramType = "query"),
            @ApiImplicitParam(name = "purLabelId", value = "目的标签id", dataType = "string", required = false, paramType = "query"),
            @ApiImplicitParam(name = "masterId", value = "广告主id", dataType = "string", required = false, paramType = "query")
    })
    @RequestMapping(value = "exportGroupMaster", method = RequestMethod.GET)
    public ResponseDto<?> exportGroupMaster(
            @RequestHeader("token") String token,
            @RequestParam(value = "name", required = false) String name, @RequestParam(value = "proLabelId", required = false) Long proLabelId,
            @RequestParam(value = "depLabelId", required = false) Long depLabelId, @RequestParam(value = "purLabelId", required = false) Long purLabelId,
            @RequestParam(value = "masterId", required = false) Long masterId) throws PermissionException;

    /**
     * 品牌方广告主-微信群列表-导出
     * @param token
     * @param name
     * @param proLabelId
     * @param depLabelId
     * @param purLabelId
     * @param startTime
     * @param endTime
     * @return
     * @throws PermissionException
     */
    @ApiOperation(value = "品牌方广告主-微信群列表-导出", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "name", value = "书籍名称", dataType = "string", required = false, paramType = "query"),
            @ApiImplicitParam(name = "proLabelId", value = "专业标签id", dataType = "long", required = false, paramType = "query"),
            @ApiImplicitParam(name = "depLabelId", value = "深度标签id", dataType = "long", required = false, paramType = "query"),
            @ApiImplicitParam(name = "purLabelId", value = "目的标签id", dataType = "string", required = false, paramType = "query"),
            @ApiImplicitParam(name = "startTime", value = "startTime", dataType = "string", required = false, paramType = "query"),
            @ApiImplicitParam(name = "endTime", value = "endTime", dataType = "string", required = false, paramType = "query")
    })
    @RequestMapping(value = "exportGroupList4Master", method = RequestMethod.GET)
    public ResponseDto<PageBeanNew<GroupMasterDTO>> exportGroupList4Master(
            @RequestHeader("token") String token,
            @RequestParam(value = "name", required = false) String name, @RequestParam(value = "proLabelId", required = false) Long proLabelId,
            @RequestParam(value = "depLabelId", required = false) Long depLabelId, @RequestParam(value = "purLabelId", required = false) Long purLabelId,
            @RequestParam(value = "startTime", required = false) String startTime, @RequestParam(value = "endTime", required = false) String endTime) throws PermissionException;

    /**
     * 品牌方广告主-微信群基本信息列表
     * @param token
     * @param currentPage
     * @param numPerPage
     * @param name
     * @return
     * @throws PermissionException
     */
    @ApiOperation(value = "品牌方广告主-微信群基本信息列表", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "currentPage", value = "当前页", dataType = "integer", paramType = "query"),
            @ApiImplicitParam(name = "numPerPage", value = "每页数量", dataType = "integer", paramType = "query"),
            @ApiImplicitParam(name = "name", value = "书籍名称", dataType = "string", required = false, paramType = "query")
    })
    @RequestMapping(value = "listPageGroupBaseInfo4Master", method = RequestMethod.GET)
    public ResponseDto<PageBeanNew<GroupMasterDTO>> listPageGroupBaseInfo4Master(
            @RequestHeader("token") String token, @RequestParam("currentPage") Integer currentPage,
            @RequestParam("numPerPage") Integer numPerPage, @RequestParam(value = "name", required = false) String name) throws PermissionException;

    /**
     * 广告主已标记的书刊id
     * @param token
     * @param masterId
     * @return
     * @throws PermissionException
     */
    @ApiOperation(value = "广告主已标记的书刊id", httpMethod = "GET")
    @ApiImplicitParam(name = "masterId", value = "广告主id", dataType = "long", paramType = "query")
    @GetMapping("getTagBookIds4AdMatser")
    public ResponseDto<List<Long>> getTagBookIds4AdMatser(
            @RequestHeader("token") String token, @RequestParam(value = "masterId", required = false) Long masterId) throws PermissionException;
}
