package com.pcloud.book.copyright.biz.impl;

import com.alibaba.fastjson.JSONObject;
import com.pcloud.book.base.enums.BookStatusEnum;
import com.pcloud.book.base.exception.BookBizException;
import com.pcloud.book.book.biz.BookBiz;
import com.pcloud.book.book.dto.BookDto;
import com.pcloud.book.consumer.channel.QrcodeSceneConsr;
import com.pcloud.book.consumer.reader.ReaderConsr;
import com.pcloud.book.copyright.biz.BookAuthCodeBiz;
import com.pcloud.book.copyright.biz.BookAuthInfoBiz;
import com.pcloud.book.copyright.biz.BookAuthServeBiz;
import com.pcloud.book.copyright.biz.BookAuthUserBiz;
import com.pcloud.book.copyright.constants.ImportAuthCodeStatus;
import com.pcloud.book.copyright.dao.BookAuthCodeDao;
import com.pcloud.book.copyright.dao.BookAuthCodeImportRecordDao;
import com.pcloud.book.copyright.entity.BookAuthCode;
import com.pcloud.book.copyright.entity.BookAuthCodeImportRecord;
import com.pcloud.book.copyright.entity.BookAuthUser;
import com.pcloud.book.copyright.tools.ExcelUtil;
import com.pcloud.book.copyright.vo.BookAuthInfoVO;
import com.pcloud.book.copyright.vo.CheckCodeParam;
import com.pcloud.book.copyright.vo.CheckIsAuthServeParam;
import com.pcloud.book.copyright.vo.FileVO;
import com.pcloud.book.copyright.vo.ImportRecordVO;
import com.pcloud.common.core.aspect.ParamLog;
import com.pcloud.common.core.biz.MessageBiz;
import com.pcloud.common.core.constant.SystemCode;
import com.pcloud.common.core.dto.SendNotifyDto;
import com.pcloud.common.core.enums.NotifyOriginTypeEnum;
import com.pcloud.common.exceptions.BizException;
import com.pcloud.common.page.PageBeanNew;
import com.pcloud.common.page.PageParam;
import com.pcloud.common.utils.string.StringUtil;
import com.pcloud.readercenter.wechat.entity.WechatUser;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * @author lily
 * @date 2018/12/4 19:11
 */
@Component("bookAuthCodeBiz")
public class BookAuthCodeBizImpl implements BookAuthCodeBiz {
    private static final Logger LOGGER = LoggerFactory.getLogger(BookAuthInfoBizImpl.class);
    private static final ExecutorService EXECUTOR_SERVICE = Executors.newFixedThreadPool(2);
    @Autowired
    private BookAuthCodeDao bookAuthCodeDao;

    @Autowired
    private BookAuthUserBiz bookAuthUserBiz;

    @Autowired
    private BookAuthInfoBiz bookAuthInfoBiz;

    @Autowired
    private BookAuthServeBiz bookAuthServeBiz;
    @Autowired
    private QrcodeSceneConsr qrcodeSceneConsr;
    @Autowired
    private ReaderConsr readerConsr;
    @Autowired
    private BookAuthCodeImportRecordDao bookAuthCodeImportRecordDao;
    @Autowired
    private BookBiz bookBiz;
    @Autowired
    private MessageBiz messageBiz;

    @Override
    @ParamLog(value = "批量插入授权码", isBefore = false)
    public void insert(List<BookAuthCode> bookAuthCodes) {
        int insertLength = bookAuthCodes.size();
        int i = 0;
        while (insertLength > 5000) {
            bookAuthCodeDao.insert(bookAuthCodes.subList(i, i + 5000));
            i = i + 5000;
            insertLength = insertLength - 5000;
        }
        if (insertLength > 0) {
            bookAuthCodeDao.insert(bookAuthCodes.subList(i, i + insertLength));
        }
    }

    @Override
    public Integer getMaxBatchNum(Long bookId, Long channelId, Long adviserId) {
        Integer batchNum = bookAuthCodeDao.getMaxBatchNum(bookId, channelId, adviserId);
        return batchNum;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Integer checkCode(CheckCodeParam checkCodeParam, Long channelId, Long wechatUserId) {
        String code = checkCodeParam.getCode();
        if (StringUtils.isEmpty(code) || code.length() != 15) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "此码错误，请重新验证");
        }
        BookAuthInfoVO authBookInfo = bookAuthInfoBiz.getAuthBookInfo(checkCodeParam.getBookId(), channelId, checkCodeParam.getAdviserId(), checkCodeParam.getSceneId());
        if (authBookInfo == null || BookStatusEnum.NO_SET.value.equals(authBookInfo.getBookStatus())) {
            return BookStatusEnum.CodeUseTypeEnum.RIGHT.value;
        }
        Long bookId = authBookInfo.getBookId();
        Long adviserId = authBookInfo.getAdviserId();
        //校验用户是否已经授权过
        Boolean isHaveAuth = bookAuthUserBiz.checkIsHaveAuth(bookId, channelId, adviserId, wechatUserId);
        if (isHaveAuth) {
            return BookStatusEnum.CodeUseTypeEnum.RIGHT.value;
        }
        Boolean isHaveCode = bookAuthCodeDao.getIsHaveCode(bookId, channelId, adviserId, code);
        Integer count = bookAuthCodeDao.updateUseCount(bookId, channelId, adviserId, code, authBookInfo.getCodeUseCount());
        if (count < 1 && isHaveCode) {
            return BookStatusEnum.CodeUseTypeEnum.HAVE_USE.value;
        } else if (count < 1 && !isHaveCode) {
            return BookStatusEnum.CodeUseTypeEnum.NOT_RIGHT.value;
        } else {
            //新增一条校验成功记录
            addUserRecord(bookId, channelId, adviserId, wechatUserId);
            return BookStatusEnum.CodeUseTypeEnum.RIGHT.value;
        }
    }

    @Override
    @ParamLog("校验服务是否需要验证")
    public Boolean checkServe(CheckIsAuthServeParam checkIsAuthServeParam, Long channelId, Long wechatUserId) {
        Long serveId = checkIsAuthServeParam.getServeId();
        String serveType = checkIsAuthServeParam.getServeType();
        if (serveId == null || serveType == null) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "请选择要进入的服务");
        }
        //校验图书是否开启保护
        BookAuthInfoVO authBookInfo = bookAuthInfoBiz.getAuthBookInfo(checkIsAuthServeParam.getBookId(), channelId, checkIsAuthServeParam.getAdviserId(), checkIsAuthServeParam.getSceneId());
        if (authBookInfo == null || BookStatusEnum.NO_SET.value.equals(authBookInfo.getBookStatus())) {
            return false;
        }
        Long bookId = authBookInfo.getBookId();
        Long adviserId = authBookInfo.getAdviserId();
        List<Long> serveIds = new ArrayList<>();
        serveIds.add(serveId);
        Map<String, Boolean> isHaveServe = qrcodeSceneConsr.listIsInBook(bookId, channelId, adviserId, serveIds);
        if (isHaveServe == null) return false;
        //校验服务是否需要授权
        Boolean isNeedAuth = bookAuthServeBiz.checkIsNeedAuth(bookId, channelId, adviserId, serveId, serveType);
        if (!isNeedAuth) {
            return false;
        }
        //校验用户是否已经授权过
        Boolean isHaveAuth = bookAuthUserBiz.checkIsHaveAuth(bookId, channelId, adviserId, wechatUserId);
        if (isHaveAuth) {
            return false;
        } else {
            return true;
        }
    }

    @Override
    @ParamLog("导入授权码")
    public void importCode(FileVO fileVO, Long adviserId) throws BizException {
        //新增导入记录
        BookAuthCodeImportRecord bookAuthCodeImportRecord = new BookAuthCodeImportRecord();
        bookAuthCodeImportRecord.setBookId(fileVO.getBookId());
        bookAuthCodeImportRecord.setAdviserId(adviserId);
        bookAuthCodeImportRecord.setChannelId(fileVO.getChannelId());
        bookAuthCodeImportRecord.setCreateUser(adviserId);
        bookAuthCodeImportRecord.setFileName(fileVO.getFileName());
        bookAuthCodeImportRecord.setFileUrl(fileVO.getFileUrl());
        bookAuthCodeImportRecordDao.insert(bookAuthCodeImportRecord);
        EXECUTOR_SERVICE.execute(() -> {
            Integer isFinish = 1;
            try {
                String[][] data = ExcelUtil.getData(fileVO.getFileUrl(), 3);
                if (data != null) {
                    List<BookAuthCode> bookAuthCodes = new ArrayList<>();
                    for (int i = 0, size = data.length; i < size; i++) {
                        BookAuthCode bookAuthCode = new BookAuthCode();
                        bookAuthCode.setBookId(fileVO.getBookId());
                        bookAuthCode.setChannelId(fileVO.getChannelId());
                        bookAuthCode.setAdviserId(adviserId);
                        String fullCode = data[i][1];
                        bookAuthCode.setFullCode(fullCode);
                        if (fullCode == null || fullCode.length() < 15) {
                            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "授权码错误");
                        }
                        bookAuthCode.setBatchNum(fullCode.substring(0, 4));
                        bookAuthCode.setAuthCode(fullCode.substring(4, data[i][1].length()));
                        bookAuthCode.setUseCount(0);
                        bookAuthCode.setCreatedUser(adviserId);
                        bookAuthCodes.add(bookAuthCode);
                    }
                    insert(bookAuthCodes);
                }else{
                    isFinish = ImportAuthCodeStatus.FAIL.value;
                }
            } catch (Exception e) {
                isFinish = ImportAuthCodeStatus.FAIL.value;
                LOGGER.error("导入失败" + e.getMessage(), e);
            }finally {
                bookAuthCodeImportRecordDao.updateState(bookAuthCodeImportRecord.getId(), isFinish);
                //发送站内信
                sendImportRecord(isFinish, fileVO.getBookId(),fileVO.getChannelId(), adviserId);
            }
        });
    }

    private void sendImportRecord(Integer isFinish, Long bookId, Long channelId, Long adviserId) {
        BookDto bookDto = bookBiz.getBaseById(bookId);
        SendNotifyDto sendNotifyDto = new SendNotifyDto();
        sendNotifyDto.setCreatedTime(new Date());
        JSONObject content = new JSONObject();
        if (isFinish == ImportAuthCodeStatus.SUCCESS.value) {
            content.put("result", "成功");
        } else {
            content.put("result", "失败");
        }
        content.put("bookName", "[" + bookDto.getBookName() + "]");
        sendNotifyDto.setNotifyContent(content.toJSONString());
        sendNotifyDto.setToId(adviserId);
        sendNotifyDto.setFromId(adviserId);
        sendNotifyDto.setResourceId(bookId + "-" + channelId);
        sendNotifyDto.setOriginType(NotifyOriginTypeEnum.GENUINE.value);
        sendNotifyDto.setSystemCode(SystemCode.adviser.code);
        sendNotifyDto.setTypeCode("genuine_qrcord_import_result");
        messageBiz.sendLetter(sendNotifyDto);
    }

    @Override
    @ParamLog("导入记录")
    public PageBeanNew<ImportRecordVO> importCodeRecord(Long bookId, Long channelId, Long adviserId, Integer currentPage, Integer numPerPage) {
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("bookId", bookId);
        paramMap.put("channelId", channelId);
        paramMap.put("adviserId", adviserId);
        PageParam pageParam = new PageParam(currentPage, numPerPage);
        return bookAuthCodeImportRecordDao.listPageNew(pageParam, paramMap, "importCodeRecord");
    }

    @ParamLog("新增一条授权成功记录")
    private void addUserRecord(Long bookId, Long channelId, Long adviserId, Long wechatUserId) {
        BookAuthUser bookAuthUser = new BookAuthUser();
        bookAuthUser.setBookId(bookId);
        bookAuthUser.setChannelId(channelId);
        bookAuthUser.setAdviserId(adviserId);
        bookAuthUser.setWechatUserId(wechatUserId);
        bookAuthUser.setAuthCode(BookStatusEnum.AuthCodeTypeEnum.BY_CODE.value);
        WechatUser wechatUser = readerConsr.getWechatUser(wechatUserId);
        bookAuthUser.setProvince(wechatUser == null || StringUtil.isEmpty(wechatUser.getWechatUserProvince()) ? "未知" : wechatUser.getWechatUserProvince());
        bookAuthUser.setCity(wechatUser == null || StringUtil.isEmpty(wechatUser.getWechatUserCity()) ? "未知" : wechatUser.getWechatUserCity());
        bookAuthUserBiz.insert(bookAuthUser);
    }

}
