package com.pcloud.book.reading.facade;

import com.pcloud.book.base.exception.BookBizException;
import com.pcloud.book.reading.biz.ReadingActivityBiz;
import com.pcloud.book.reading.check.ParamCheck;
import com.pcloud.book.reading.dao.ReadingActivityDao;
import com.pcloud.book.reading.dto.AssignGroupDTO;
import com.pcloud.book.reading.dto.ReadingActivityAdditionParamDTO;
import com.pcloud.book.reading.dto.ReadingActivityBaseParamDTO;
import com.pcloud.book.reading.dto.ReadingActivityDTO;
import com.pcloud.common.core.aspect.ParamLog;
import com.pcloud.common.dto.ResponseDto;
import com.pcloud.common.page.PageBeanNew;
import com.pcloud.common.permission.PermissionException;
import com.pcloud.common.utils.SessionUtil;
import com.pcloud.common.utils.cookie.Cookie;
import com.sdk.wxgroup.CollectionVO;
import com.pcloud.common.utils.string.StringUtil;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.CookieValue;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;

/**
 * @描述：共读活动外部接口
 * @作者：zhuyajie
 * @创建时间：17:45 2020/1/14
 * @版本：1.0
 */
@Api("共读活动")
@RestController("readingActivityFacade")
@RequestMapping("readingActivity")
@Slf4j
public class ReadingActivityFacade {

    @Autowired
    private ReadingActivityBiz readingActivityBiz;
    @Autowired
    private ParamCheck paramCheck;
    @Autowired
    ReadingActivityDao readingActivityDao;

    @PostMapping("receiveMoney")
    public ResponseDto<Void> receiveMoney(@RequestBody CollectionVO vo) {
        readingActivityBiz.receiveMoney(vo);
        return new ResponseDto<>();
    }

    @ApiOperation("创建共读活动")
    @PostMapping("add")
    public ResponseDto<?> add(@RequestHeader("token") String token, @RequestBody @ApiParam ReadingActivityBaseParamDTO readingActivityBaseParamDTO) throws PermissionException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        paramCheck.checkAddParam(readingActivityBaseParamDTO);
        readingActivityBaseParamDTO.setCreateUser(partyId);
        Long id = readingActivityBiz.add(readingActivityBaseParamDTO);
        return new ResponseDto<>(id);
    }

    @ApiOperation("修改共读活动")
    @PostMapping("update")
    public ResponseDto<?> update(@RequestHeader("token") String token, @RequestBody @ApiParam ReadingActivityBaseParamDTO readingActivityBaseParamDTO) throws PermissionException {
        SessionUtil.getToken4Redis(token);
        paramCheck.checkAddParam(readingActivityBaseParamDTO);
        paramCheck.checkUpdateParam(readingActivityBaseParamDTO);
        readingActivityBiz.update(readingActivityBaseParamDTO);
        return new ResponseDto<>();
    }

    @ApiOperation("共读活动高级设置")
    @PostMapping("updateAdditionSetting")
    public ResponseDto<?> updateAdditionSetting(@RequestHeader("token") String token, @RequestBody @ApiParam ReadingActivityAdditionParamDTO readingActivityAdditionParamDTO) throws PermissionException {
        SessionUtil.getToken4Redis(token);
        paramCheck.checkAdditionParam(readingActivityAdditionParamDTO);
        readingActivityBiz.updateAdditionSetting(readingActivityAdditionParamDTO);
        return new ResponseDto<>();
    }

    @ApiOperation("根据id查询")
    @GetMapping("getById")
    public ResponseDto<ReadingActivityDTO> getById(@RequestHeader("token") String token, @RequestParam("id") Long id) throws PermissionException {
        SessionUtil.getToken4Redis(token);
        if (null == id) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "id为空");
        }
        ReadingActivityDTO activityDTO = readingActivityBiz.getById(id);
        return new ResponseDto<>(activityDTO);
    }

    @ApiOperation("查询列表")
    @GetMapping("getList")
    public ResponseDto<PageBeanNew<ReadingActivityDTO>> getList(
            @RequestHeader("token") String token,
            @RequestParam("currentPage") Integer currentPage,
            @RequestParam("numPerPage") Integer numPerPage
    ) throws PermissionException {
        SessionUtil.getToken4Redis(token);
        PageBeanNew<ReadingActivityDTO> page = readingActivityBiz.getList(currentPage, numPerPage);
        return new ResponseDto<>(page);
    }

    @ApiOperation("分配群")
    @PostMapping("assignGroup")
    @ParamLog("接收分配群参数")
    public ResponseDto<?> assignGroup(@RequestBody @ApiParam AssignGroupDTO assignGroupDTO) {
        if (null == assignGroupDTO || StringUtil.isEmpty(assignGroupDTO.getWxGroupId()) || StringUtil.isEmpty(assignGroupDTO.getMasterId())) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数缺失");
        }
        readingActivityBiz.assignGroup(assignGroupDTO);
        return new ResponseDto<>();
    }

    @ApiOperation("获取活动开启的总天数")
    @GetMapping("getReadingActivityDays4Wechat")
    public ResponseDto<?> getReadingActivityDays4Wechat(@CookieValue("userInfo") String userInfo, @RequestParam("readingActivityId") Long readingActivityId) throws PermissionException {
        if (null == readingActivityId) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "id为空");
        }
        Long wechatUserId = Cookie.getId(userInfo, Cookie._WECHAT_USER_ID);
        return new ResponseDto<>(readingActivityBiz.getReadingActivityDays4Wechat(readingActivityId));
    }
}
