package com.pcloud.common.core.rule;

import com.dcg.util.constant.GrayConstants;
import com.netflix.loadbalancer.ILoadBalancer;
import com.netflix.loadbalancer.Server;
import com.netflix.loadbalancer.ZoneAvoidanceRule;
import com.pcloud.common.utils.cache.redis.JedisClusterUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ThreadLocalRandom;

/**
 * 自定义灰度发布规则 2019-11-19 by david
 */
@Slf4j
public class GrayRule extends ZoneAvoidanceRule {

    /**
     * 在choose方法中，自定义规则，返回的Server就是具体选择出来的服务
     *
     * @param key 服务key
     * @return 可用server
     */
    @Override
    public Server choose(Object key) {
        ILoadBalancer loadBalancer = this.getLoadBalancer();
        List<Server> allServers = loadBalancer.getAllServers();
        List<Server> serverList = this.getPredicate().getEligibleServers(allServers, key);
        log.info("[gray choose] key：{}； allServers：{}； serverList：{}", key, allServers, serverList);
        if (CollectionUtils.isEmpty(serverList)) {
            log.warn("=====GrayRule choose serverList isEmpty key:{}=====", key);
            return null;
        }
        // 灰度开关
        String switchValue = JedisClusterUtils.get(GrayConstants.GRAY_SWITCH);
        if (StringUtils.isBlank(switchValue) || "0".equals(switchValue)) {
            return getRandom(serverList);
        }
        // 灰度服务列表
        final Map<String, String> grayAddress = JedisClusterUtils.hgetAll(GrayConstants.GRAY_ADDRESS);
        if (CollectionUtils.isEmpty(grayAddress)) {
            log.info("[choose] : grayAddress isEmpty return serverList:{}", serverList);
            return getRandom(serverList);
        }
        List<String> grayServers = new ArrayList<>(grayAddress.keySet());
        // 查找非灰度服务并返回
        List<Server> noGrayServerList = new ArrayList<>();
        for (Server server : serverList) {
            // 过滤正在发布的服务
            if (grayServers.contains(server.getHostPort())) {
                continue;
            }
            noGrayServerList.add(server);
        }
        if (!noGrayServerList.isEmpty()) {
            return getRandom(noGrayServerList);
        }
        return null;
    }

    /**
     * 随机返回一个可用服务
     *
     * @param serverList 服务列表
     * @return 随机获取的服务
     */
    private static Server getRandom(List<Server> serverList) {
        if (CollectionUtils.isEmpty(serverList)) {
            return null;
        }
        ThreadLocalRandom random = ThreadLocalRandom.current();
        return serverList.get(random.nextInt(serverList.size()));
    }

}
