package com.pcloud.common.core.zipkin;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.TypeReference;
import com.pcloud.common.utils.string.StringUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.bootstrap.config.PropertySourceLocator;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.core.env.Environment;
import org.springframework.core.env.PropertySource;
import org.springframework.stereotype.Component;

import java.util.*;

/**
 * 刷新本地采样率
 */
@Slf4j
@Component
public class RefreshSamplerLocalProperties implements ApplicationContextAware, InitializingBean {

    @Autowired(required = false)
    private List<PropertySourceLocator> propertySourceLocators = new ArrayList<>();

    @Autowired(required = false)
    private PercentageLocalSampler percentageLocalSampler;

    //线程休眠时间
    private static final Long REFRESH_SLEEP_TIME = 60 * 1000L;
    //个性化采样率map的key
    private static final String SPRING_ZIPKIN_URISAMPLEMAPJSONSTR = "spring.zipkin.uriSampleMapJsonStr";
    //个性化设置全局采样率的key
    private static final String SPRING_ZIPKIN_GLOBALPERCENTAGE = "spring.zipkin.globalPercentage";
    //容器上下文
    private ApplicationContext applicationContext;
    //个性化采样率默认值
    private static final String DEFAULT_URISAMPLEMAPJSONSTR = "{}";
    //全局采样率默认值
    private static final Float DEFAULT_GLOBALPERCENTAGE = 0.01f;

    private Environment environment;

    private static final String STATUS_DELIMITER="-";

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
        environment = applicationContext.getEnvironment();
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        new RefreshRequest().start();

    }

    public void setPropertySourceLocators(
            Collection<PropertySourceLocator> propertySourceLocators) {
        this.propertySourceLocators = new ArrayList<>(propertySourceLocators);
    }

    /**
     * 读取git配置文件线程
     */
    class RefreshRequest extends Thread {
        @Override
        public void run() {
            while (true) {
                try {
                    for (PropertySourceLocator locator : propertySourceLocators) {
                        PropertySource<?> source = null;
                        source = locator.locate(environment);
                        if (source == null) {
                            continue;
                        }
                        float globalPercentage = source.getProperty(SPRING_ZIPKIN_GLOBALPERCENTAGE) != null ? Float.parseFloat(source.getProperty(SPRING_ZIPKIN_GLOBALPERCENTAGE).toString()) : DEFAULT_GLOBALPERCENTAGE;
                        String uriSampleMapJsonStr = source.getProperty(SPRING_ZIPKIN_URISAMPLEMAPJSONSTR) != null ? source.getProperty(SPRING_ZIPKIN_URISAMPLEMAPJSONSTR).toString() : DEFAULT_URISAMPLEMAPJSONSTR;
                        log.warn("读取git配置文件 globalPercentage {} ,globalPercentageJsonStr {}", globalPercentage, uriSampleMapJsonStr);
                        if (!uriSampleMapJsonStr.equals(DEFAULT_URISAMPLEMAPJSONSTR)) {
                            loadSamplerProperties(globalPercentage,uriSampleMapJsonStr);
                        }
                    }
                    Thread.sleep(REFRESH_SLEEP_TIME);
                } catch (Exception e) {
                    log.error("刷新采样率异常 {}", e);
                }
            }
        }
    }

    /**
     * 刷新PercentageLocalSampler
     *
     * @param globalPercentage    全局采样率
     * @param uriSampleMapJsonStr url正则表达式和个性化采样率
     */
    public void loadSamplerProperties(Float globalPercentage, String uriSampleMapJsonStr) {
        //获取容器的percentageLocalSampler
        if (null == percentageLocalSampler) {
            percentageLocalSampler = (PercentageLocalSampler) applicationContext.getBean("percentageLocalSampler");
        }
        SamplerLocalProperties samplerLocalProperties = percentageLocalSampler.getConfiguration();
        SamplerLocalProperties samplerRemoteLocalProperties = new SamplerLocalProperties();
        //全局采样率配置更新
        if (globalPercentage != null && globalPercentage != DEFAULT_GLOBALPERCENTAGE) {
            samplerRemoteLocalProperties.setPercentage(globalPercentage);
        }
        if (!StringUtil.isEmpty(uriSampleMapJsonStr) && !uriSampleMapJsonStr.equals(DEFAULT_URISAMPLEMAPJSONSTR)) {
            List<UriSampleProperties> uriSamples = new ArrayList<>();
            Map<Float, ArrayList<String>> uriSampleMap = (HashMap<Float, ArrayList<String>>) JSON.parseObject(uriSampleMapJsonStr, new TypeReference<HashMap<Float, ArrayList<String>>>() {
            });
            if (uriSampleMap != null && uriSampleMap.size() > 0) {
                for (Map.Entry<Float, ArrayList<String>> entry : uriSampleMap.entrySet()) {
                    float percentage = entry.getKey();
                    for (String uri : entry.getValue()) {
                        UriSampleProperties uriSampleProperties = new UriSampleProperties();
                        uriSampleProperties.setUriPercentage(percentage);
                        uriSampleProperties.setUriRegex(uri);
                        uriSamples.add(uriSampleProperties);
                    }
                }
            }
            samplerRemoteLocalProperties.setUriSamples(uriSamples);
        }
        if(!getPercentageSamplerHashCode(samplerRemoteLocalProperties).equals(getPercentageSamplerHashCode(samplerLocalProperties))){
            log.warn("更新采样率 globalPercentage {} ,globalPercentageJsonStr {}", globalPercentage, uriSampleMapJsonStr);
            percentageLocalSampler.initPercentage(samplerRemoteLocalProperties);
        }
    }
    /**
     * 计算计算hash值
     *
     * @param samplerLocalProperties
     * @return
     */
    public static String getPercentageSamplerHashCode(SamplerLocalProperties samplerLocalProperties) {
        StringBuffer reconcileHashCode = new StringBuffer((samplerLocalProperties.getPercentage() + STATUS_DELIMITER));
        for (UriSampleProperties uriSampleProperties : samplerLocalProperties.getUriSamples()) {
            reconcileHashCode.append(uriSampleProperties.getUriRegex().hashCode() + STATUS_DELIMITER);
            reconcileHashCode.append((uriSampleProperties.getUriPercentage() + STATUS_DELIMITER));
        }
        return reconcileHashCode.toString();
    }
}
