package com.pcloud.book.advertising.facade;

import com.pcloud.book.advertising.biz.LinkRevertBiz;
import com.pcloud.book.advertising.dto.LinkRevertRecordDTO;
import com.pcloud.book.advertising.entity.AdvertisingLinkClick;
import com.pcloud.book.advertising.entity.AdvertisingLinkRevert;
import com.pcloud.book.base.exception.BookBizException;
import com.pcloud.common.dto.ResponseDto;
import com.pcloud.common.page.PageBeanNew;
import com.pcloud.common.permission.PermissionException;
import com.pcloud.common.utils.SessionUtil;
import com.pcloud.common.utils.cookie.Cookie;
import com.pcloud.common.utils.string.StringUtil;
import com.pcloud.wechatgroup.message.enums.IsSystem;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.CookieValue;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.Map;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;

/**
 * @描述：链接转换
 * @作者：zhuyajie
 * @创建时间：16:57 2019/10/10
 * @版本：1.0
 */
@Api(description = "链接转换外部接口")
@RequestMapping(value = "/linkRevert")
@RestController("linkRevertFacade")
public class LinkRevertFacade {

    @Autowired
    private LinkRevertBiz linkRevertBiz;

    @ApiOperation(value = "添加链接转换记录", httpMethod = "POST")
    @PostMapping("addRevertRecord")
    public ResponseDto<Long> addRevertRecord(@RequestHeader("token") String token, @RequestBody @ApiParam AdvertisingLinkRevert advertisingLinkRevert) throws PermissionException {
        if (null == advertisingLinkRevert) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数为空");
        }
        if (null == advertisingLinkRevert.getBrandId() || StringUtil.isEmpty(advertisingLinkRevert.getLink())) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数缺失");
        }
        Map<String, Object> map = SessionUtil.getToken4Redis(token);
        String isSystem = (String) map.get(SessionUtil.IS_SYSTEM);
        Long partyId = (Long) map.get(SessionUtil.PARTY_ID);
        if (IsSystem.NOT_SYSTEM.code.equals(isSystem)) {
            partyId = (Long) map.get(SessionUtil.MEMBER_ID);
        }
        advertisingLinkRevert.setCreateUser(partyId);
        Long id = linkRevertBiz.addRevertRecord(advertisingLinkRevert);
        return new ResponseDto<>(id);
    }

    @ApiOperation(value = "删除转换记录", httpMethod = "GET")
    @GetMapping("deleteRevertRecordById")
    public ResponseDto<?> deleteRevertRecordById(@RequestHeader("token") String token, @RequestParam("id") Long id) throws PermissionException {
        SessionUtil.getToken4Redis(token);
        if (null == id) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "id为空");
        }
        linkRevertBiz.deleteRevertRecordById(id);
        return new ResponseDto<>();
    }

    @ApiOperation(value = "转换记录列表查询", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "currentPage", value = "当前页", dataType = "integer", paramType = "query"),
            @ApiImplicitParam(name = "numPerPage", value = "每页数量", dataType = "integer", paramType = "query"),
            @ApiImplicitParam(name = "brandId", value = "品牌id", dataType = "long", required = false, paramType = "query")
    })
    @GetMapping("listPageRevertRecord")
    public ResponseDto<PageBeanNew<LinkRevertRecordDTO>> listPageRevertRecord(
            @RequestHeader("token") String token, @RequestParam(value = "brandId", required = false) Long brandId,
            @RequestParam("currentPage") Integer currentPage,
            @RequestParam("numPerPage") Integer numPerPage) throws PermissionException {
        SessionUtil.getToken4Redis(token);
        if (currentPage == null || numPerPage == null || currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        PageBeanNew<LinkRevertRecordDTO> pageBean = linkRevertBiz.listPageRevertRecord(brandId, currentPage, numPerPage);
        return new ResponseDto<>(pageBean);
    }

    @ApiOperation(value = "转换记录列表导出", httpMethod = "GET")
    @ApiImplicitParam(name = "brandId", value = "品牌id", dataType = "long", required = false, paramType = "query")
    @GetMapping("exportRevertRecordList")
    public ResponseDto<?> exportRevertRecordList(@RequestHeader("token") String token, @RequestParam(value = "brandId", required = false) Long brandId) throws PermissionException {
        Map<String, Object> map = SessionUtil.getToken4Redis(token);
        String isSystem = (String) map.get(SessionUtil.IS_SYSTEM);
        Long partyId = (Long) map.get(SessionUtil.PARTY_ID);
        if (IsSystem.NOT_SYSTEM.code.equals(isSystem)) {
            partyId = (Long) map.get(SessionUtil.MEMBER_ID);
        }
        linkRevertBiz.exportRevertRecordList(brandId, partyId);
        return new ResponseDto<>();
    }

    @ApiOperation(value = "添加链接点击记录", httpMethod = "POST")
    @PostMapping("addClickRecord")
    public ResponseDto<?> addClickRecord(@CookieValue("userInfo") String userInfo, @RequestBody @ApiParam AdvertisingLinkClick advertisingLinkClick) {
        if (null == advertisingLinkClick) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数为空");
        }
        if (StringUtil.isEmpty(advertisingLinkClick.getLink())) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数缺失");
        }
        Long userId = Cookie.getId(userInfo, Cookie._WECHAT_USER_ID);
        advertisingLinkClick.setUserId(userId);
        linkRevertBiz.addClickRecord(advertisingLinkClick);
        return new ResponseDto<>();
    }

    @ApiOperation(value = "根据id查记录", httpMethod = "GET")
    @GetMapping("getById")
    public ResponseDto<AdvertisingLinkRevert> getById(@RequestParam("id") Long id) {
        if (null == id) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "id为空");
        }
        AdvertisingLinkRevert revert = linkRevertBiz.getById(id);
        return new ResponseDto<>(revert);
    }

    @ApiOperation("修改链接数据变化值")
    @PostMapping("updateLinkBrandData")
    public ResponseDto<?> updateLinkBrandData(@RequestHeader("token") String token, @RequestBody @ApiParam AdvertisingLinkRevert linkRevert) throws PermissionException {
        SessionUtil.getToken4Redis(token);
        if (null==linkRevert || null == linkRevert.getId()){
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "缺少参数");
        }
        linkRevert.setSendCountChange(linkRevert.getSendCountChange()==null?0:linkRevert.getSendCountChange());
        linkRevert.setClickCountChange(linkRevert.getClickCountChange()==null?0:linkRevert.getClickCountChange());
        linkRevert.setClickUserCountChange(linkRevert.getClickUserCountChange()==null?0:linkRevert.getClickUserCountChange());
        linkRevertBiz.updateLinkBrandData(linkRevert);
        return new ResponseDto<>();
    }

    @ApiOperation(value = "品牌方-转换记录列表查询", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "currentPage", value = "当前页", dataType = "integer", paramType = "query"),
            @ApiImplicitParam(name = "numPerPage", value = "每页数量", dataType = "integer", paramType = "query")
    })
    @GetMapping("listPageRevertRecord4Brand")
    public ResponseDto<PageBeanNew<LinkRevertRecordDTO>> listPageRevertRecord4Brand(
            @RequestHeader("token") String token,
            @RequestParam("currentPage") Integer currentPage,
            @RequestParam("numPerPage") Integer numPerPage) throws PermissionException {
        Long userId = (Long) SessionUtil.getVlaue(token, SessionUtil.USER_ID);
        if (currentPage == null || numPerPage == null || currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        //品牌方登录，userId=brandId
        PageBeanNew<LinkRevertRecordDTO> pageBean = linkRevertBiz.listPageRevertRecord(userId, currentPage, numPerPage);
        return new ResponseDto<>(pageBean);
    }

}
