package com.pcloud.book.group.facade.impl;

import cn.hutool.core.date.DatePattern;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.map.MapUtil;
import com.pcloud.book.base.exception.BookBizException;
import com.pcloud.book.book.constant.BookConstant;
import com.pcloud.book.book.dto.MapResourceTotalCountDTO;
import com.pcloud.book.group.biz.BookGroupBiz;
import com.pcloud.book.group.biz.BookGroupClassifyBiz;
import com.pcloud.book.group.dao.BookGroupDao;
import com.pcloud.book.group.dto.BackgroundGroupQrcodeDTO;
import com.pcloud.book.group.dto.BookAppletSceneDTO;
import com.pcloud.book.group.dto.BookGroupCreateDTO;
import com.pcloud.book.group.dto.BookGroupDTO;
import com.pcloud.book.group.dto.BookServeQueryDTO;
import com.pcloud.book.group.dto.ChangeQrCodeTypeDto;
import com.pcloud.book.group.dto.GroupStoreMyPayDto;
import com.pcloud.book.group.dto.HotAppDTO;
import com.pcloud.book.group.dto.SelfBookGroupStParamDTO;
import com.pcloud.book.group.dto.UserSelectParamDTO;
import com.pcloud.book.group.entity.BookGroup;
import com.pcloud.book.group.entity.BookGroupClassify;
import com.pcloud.book.group.entity.BookGroupServe;
import com.pcloud.book.group.entity.WxWechatUserCorrelation;
import com.pcloud.book.group.enums.JoinGroupTypeEnum;
import com.pcloud.book.group.enums.UseTypeEnum;
import com.pcloud.book.group.facade.BookGroupFacade;
import com.pcloud.book.group.vo.BookGroupAnalysisParam;
import com.pcloud.book.group.vo.BookGroupAnalysisVO;
import com.pcloud.book.group.vo.FriendsVO;
import com.pcloud.book.group.vo.GroupIncomeStaticParamVO;
import com.pcloud.book.group.vo.GroupScanTrendParamVO;
import com.pcloud.book.group.vo.GroupScanTrendVO;
import com.pcloud.book.group.vo.GroupStatisticVO;
import com.pcloud.book.group.vo.ListBook4ChannelVO;
import com.pcloud.book.group.vo.ListBookGroup4ChannelParamVO;
import com.pcloud.book.group.vo.ResourcesStatisticVO;
import com.pcloud.book.group.vo.TotalRescourceDataVO;
import com.pcloud.book.group.vo.UpdateRankTypeVO;
import com.pcloud.book.group.vo.UserBookInfoVO;
import com.pcloud.book.group.vo.WxGroupStatisticVO;
import com.pcloud.book.keywords.vo.ListKeywordVO;
import com.pcloud.book.record.biz.BookBrowseRecordBiz;
import com.pcloud.book.record.entity.BookBrowseRecord;
import com.pcloud.common.core.constant.SystemCode;
import com.pcloud.common.dto.ResponseDto;
import com.pcloud.common.exceptions.BizException;
import com.pcloud.common.page.PageBean;
import com.pcloud.common.page.PageBeanNew;
import com.pcloud.common.page.PageParam;
import com.pcloud.common.permission.PermissionException;
import com.pcloud.common.utils.SessionUtil;
import com.pcloud.common.utils.cache.redis.JedisClusterUtils;
import com.pcloud.common.utils.cookie.Cookie;
import com.pcloud.common.utils.nginx.NginxUtils;
import com.pcloud.common.utils.string.StringUtil;
import com.pcloud.wechatgroup.base.exception.WechatGroupBizException;
import com.pcloud.wechatgroup.message.enums.IsSystem;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.codehaus.jackson.JsonParseException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.CookieValue;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletRequest;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Description 社群书群二维码接口实现类 Created by PENG on 2019/4/17.
 */
@Slf4j
@RestController("bookGroupFacade")
@RequestMapping("bookGroup")
public class BookGroupFacadeImpl implements BookGroupFacade {

    @Autowired
    private BookGroupBiz bookGroupBiz;
    @Autowired
    private BookGroupClassifyBiz bookGroupClassifyBiz;
    @Autowired
    private BookGroupDao bookGroupDao;
    @Autowired
    private BookBrowseRecordBiz bookBrowseRecordBiz;

    @SneakyThrows
    @PostMapping("updateNoISBNBookGeneratorQrcode")
    public ResponseDto<String> updateNoISBNBookGeneratorQrcode(@RequestParam("file") MultipartFile file ) {
        return new ResponseDto<>(bookGroupBiz.updateNoISBNBookGeneratorQrcode(file));
    }

    /**
     * 获取社群书群二维码信息
     */
    @Override
    @RequestMapping(value = "getBookGroupInfo", method = RequestMethod.GET)
    public ResponseDto<?> getBookGroupInfo( @RequestHeader("token") String token, @RequestParam(value = "bookGroupId", required = false) Long bookGroupId )
            throws BizException, PermissionException, JsonParseException {
        if (null == bookGroupId) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        SessionUtil.getToken4Redis(token);
        return new ResponseDto<>(bookGroupBiz.getBookGroupInfo(bookGroupId));
    }

    /**
     * 获取社群书名和封面
     */
    @Override
    @RequestMapping(value = "getBookInfo", method = RequestMethod.GET)
    public ResponseDto<?> getBookInfo( @CookieValue("userInfo") String userInfo, @RequestParam(value = "bookGroupId", required = false) Long bookGroupId )
            throws BizException, PermissionException, JsonParseException {
        if (null == bookGroupId) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        Cookie.getUserInfo(userInfo);
        return new ResponseDto<>(bookGroupBiz.getBookInfo(bookGroupId));
    }

    /**
     * 获取社群书群二维码信息
     */
    @Override
    @RequestMapping(value = "getBookGroupInfo4Wechat", method = RequestMethod.GET)
    public ResponseDto<?> getBookGroupInfo4Wechat( @RequestParam(value = "bookGroupId", required = false) Long bookGroupId, HttpServletRequest request)
            throws BizException, PermissionException, JsonParseException {
        if (null == bookGroupId) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        String requestIp = NginxUtils.getClientIp(request);
        // 新增埋点
        BookBrowseRecord bookBrowseRecord = new BookBrowseRecord();
        BookGroupDTO bookGroupDTO = bookGroupDao.getBookBaseInfoById(bookGroupId);
        bookBrowseRecord.setBookId(bookGroupDTO.getBookId());
        bookBrowseRecord.setIp(requestIp);
        bookBrowseRecordBiz.addBookRecord4BookId(bookBrowseRecord);
        return new ResponseDto<>(bookGroupBiz.getBookGroupInfo4Wechat(bookGroupId));
    }

    /**
     * 获取社群书群二维码信息
     */
    @Override
    @RequestMapping(value = "getBookGroupInfoByBookId", method = RequestMethod.GET)
    public ResponseDto<?> getBookGroupInfoByBookId( @RequestHeader("token") String token, @RequestParam(value = "bookId", required = false) Long bookId,
                                                    @RequestParam(value = "channelId", required = false) Long channelId,
                                                    @RequestParam(value = "addType", required = false) Integer addType,
                                                    @RequestParam(value = "sceneId", required = false) Long sceneId,
                                                    @RequestParam(value = "joinGroupType", required = false) Integer joinGroupType,
                                                    @RequestParam(value = "jumpType", required = false) Integer jumpType,
                                                    @RequestParam(value = "jumpUrl", required = false) String jumpUrl
    ) throws BizException, PermissionException, JsonParseException {
        if (null == bookId || null == channelId) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        if (bookId.equals(0L)){
            return new ResponseDto<>();
        }
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        return new ResponseDto<>(bookGroupBiz.getBookGroupInfoByBookId(bookId, channelId, adviserId, addType, sceneId, joinGroupType, jumpType, jumpUrl));
    }

    /**
     * 是否有社群码
     */
    @Override
    @RequestMapping(value = "isHaveQrcode", method = RequestMethod.GET)
    public ResponseDto<Boolean> isHaveQrcode( @RequestHeader("token") String token, @RequestParam(value = "bookId", required = false) Long bookId,
                                              @RequestParam(value = "channelId", required = false) Long channelId ) throws BizException, PermissionException, JsonParseException {
        if (null == bookId || null == channelId) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        return new ResponseDto<Boolean>(bookGroupBiz.isHaveQrcode(bookId, channelId, adviserId));
    }


    /**
     * 更新群二维码信息
     */
    @Override
    @RequestMapping(value = "updateBookGroup", method = RequestMethod.POST)
    public ResponseDto<?> updateBookGroup( @RequestHeader("token") String token, @RequestBody BookGroup bookGroup )
            throws BizException, PermissionException, JsonParseException {
        if (null == bookGroup) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        bookGroupBiz.updateBookGroup(bookGroup);
        return new ResponseDto<>();
    }

    /**
     * 群二维码下载带背景图
     */
    @RequestMapping(value = "getBackgroundGroupQr", method = RequestMethod.POST)
    public ResponseDto<?> getBackgroundGroupQr( @RequestHeader("token") String token, @RequestBody BackgroundGroupQrcodeDTO backgroundGroupQrcodeDTO )
            throws BizException, PermissionException, JsonParseException {
        if (null == backgroundGroupQrcodeDTO || null == backgroundGroupQrcodeDTO.getBackgroundType() || null == backgroundGroupQrcodeDTO.getBookGroupId()) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        BackgroundGroupQrcodeDTO backgroundGroupQr = bookGroupBiz.getBackgroundGroupQr(backgroundGroupQrcodeDTO);
        return new ResponseDto<>(backgroundGroupQr);
    }

    /**
     * 根据社群码ID获取书名
     */
    @Override
    @RequestMapping(value = "getBookNameByBookGroupId", method = RequestMethod.GET)
    public ResponseDto<?> getBookNameByBookGroupId( @RequestParam(value = "bookGroupId", required = false) Long bookGroupId )
            throws BizException, PermissionException, JsonParseException {
        if (null == bookGroupId) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        return new ResponseDto<>(bookGroupBiz.getBookNameByBookGroupId(bookGroupId));
    }

    /**
     * 获取社群书列表(编辑)
     */
    @Override
    @RequestMapping(value = "listBookGroup4Adviser", method = RequestMethod.GET)
    public ResponseDto<PageBean> listBookGroup4Adviser( @RequestHeader("token") String token,
                                                        @RequestParam(value = "name", required = false) String name,
                                                        @RequestParam(value = "bookName", required = false) String bookName,
                                                        @RequestParam(value = "isbn", required = false) String isbn,
                                                        @RequestParam(value = "channelId", required = false) Long channelId,
                                                        @RequestParam(value = "typeCode", required = false) String typeCode,
                                                        @RequestParam(value = "currentPage", required = false) Integer currentPage,
                                                        @RequestParam(value = "numPerPage", required = false) Integer numPerPage,
                                                        @RequestParam(value = "isMainEditor", required = false) Integer isMainEditor,
                                                        @RequestParam(value = "isFundSupport", required = false) Integer isFundSupport,
                                                        @RequestParam(value = "bookId", required = false) Integer bookId,
                                                        @RequestParam(value = "joinGroupType", required = false) String joinGroupType,
                                                        @RequestParam(value = "startDate", required = false) String startDate,
                                                        @RequestParam(value = "endDate", required = false) String endDate,
                                                        @RequestParam(value = "hasServe", required = false) Boolean hasServe,
                                                        @RequestParam(value = "isPrint", required = false) Boolean isPrint,
                                                        @RequestParam(value = "proLabelId", required = false) @ApiParam("专业标签") Long proLabelId,
                                                        @RequestParam(value = "depLabelId", required = false) @ApiParam("深度标签") Long depLabelId,
                                                        @RequestParam(value = "purLabelId", required = false) @ApiParam("目的") Long purLabelId,
                                                        @RequestParam(value = "firstClassifyId", required = false) @ApiParam("分类") Long firstClassifyId,
                                                        @RequestParam(value = "gradeLabelId", required = false) @ApiParam("年级") Long gradeLabelId,
                                                        @RequestParam(value = "subjectLabelId", required = false) @ApiParam("科目") Long subjectLabelId,
                                                        @RequestParam(value = "jumpType", required = false) Integer jumpType)
            throws BizException, PermissionException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (currentPage == null || numPerPage == null || currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("adviserId", adviserId);
        paramMap.put("isMainEditor", isMainEditor);
        paramMap.put("isFundSupport", isFundSupport);
        paramMap.put("channelId", channelId);
        paramMap.put("name", name != null && "".equals(name.trim()) ? null : name);
        paramMap.put("isbn", isbn != null && "".equals(isbn.trim()) ? null : isbn);
        paramMap.put("bookName", bookName != null && "".equals(bookName.trim()) ? null : bookName);
        paramMap.put("typeCode", typeCode != null && "".equals(typeCode.trim()) ? null : typeCode);
        paramMap.put("bookId", bookId);
        List<Integer> joinGroupTypes = new ArrayList<>();
        if ("GROUP".equals(joinGroupType)) {
            joinGroupTypes.add(JoinGroupTypeEnum.GROUP_QRCODE.getCode());
        }
        if ("ROBOT".equals(joinGroupType)) {
            joinGroupTypes.add(JoinGroupTypeEnum.ROBOT.getCode());
            joinGroupTypes.add(JoinGroupTypeEnum.AI_ROBOT.getCode());
            joinGroupTypes.add(JoinGroupTypeEnum.XIAORUI.getCode());
        }
        paramMap.put("joinGroupTypes", joinGroupTypes);
        if (!StringUtil.isEmpty(startDate) && !StringUtil.isEmpty(endDate)) {
            paramMap.put("startDate", startDate + " 00:00:00");
            paramMap.put("endDate", endDate + " 23:59:59");
        }
        paramMap.put("hasServe", hasServe);
        paramMap.put("isPrint", isPrint);
        paramMap.put("jumpType", jumpType);
        PageBean pageBean = bookGroupBiz.listBookGroup4Adviser(paramMap, new PageParam(currentPage, numPerPage), adviserId);
        return new ResponseDto<>(pageBean);
    }

    @Override
    @RequestMapping(value = "listBookGroup4Channel", method = RequestMethod.POST)
    public ResponseDto<PageBeanNew<ListBook4ChannelVO>> listBookGroup4Channel( @RequestHeader("token") String token,
                                                        @RequestBody ListBookGroup4ChannelParamVO listBookGroup4ChannelParamVO ) throws BizException, PermissionException {
        Long channelId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        PageBeanNew<ListBook4ChannelVO> pageBean = bookGroupBiz.listBookGroup4Channel(channelId, listBookGroup4ChannelParamVO);
        return new ResponseDto<>(pageBean);
    }

    @Override
    public ResponseDto<PageBeanNew<ListBook4ChannelVO>> listBookGroup4AdviserKStore(String token, ListBookGroup4ChannelParamVO listBookGroup4ChannelParamVO) throws BizException, PermissionException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        PageBeanNew<ListBook4ChannelVO> pageBean = bookGroupBiz.listBookGroup4AdviserKStore(adviserId, listBookGroup4ChannelParamVO);
        return new ResponseDto<>(pageBean);
    }

    @Override
    @RequestMapping(value = "getUserBrowseGroup4KnowLedgeMall", method = RequestMethod.GET)
    public ResponseDto<?> getUserBrowseGroup4KnowLedgeMall( @CookieValue("userInfo") String userInfo,
                                                            @RequestParam(value = "currentPage", required = false) Integer currentPage,
                                                            @RequestParam(value = "numPerPage", required = false) Integer numPerPage ) throws BizException, PermissionException {
        Long channelId = Cookie.getId(userInfo, Cookie._CHANNEL_ID);
        Long wechatUserId = Cookie.getId(userInfo, Cookie._WECHAT_USER_ID);
        PageBeanNew<GroupStoreMyPayDto> resultInfos = bookGroupBiz.getUserBrowseGroup4KnowLedgeMall(channelId, wechatUserId, currentPage, numPerPage);
        return new ResponseDto<>(resultInfos);
    }

    /**
     * 获取社群书列表(编辑)
     */
    @Override
    @RequestMapping(value = "listSimpleBookGroup4Adviser", method = RequestMethod.GET)
    public ResponseDto<?> listSimpleBookGroup4Adviser( @RequestHeader("token") String token,
                                                       @RequestParam(value = "name", required = false) String name,
                                                       @RequestParam(value = "currentPage", required = false) Integer currentPage,
                                                       @RequestParam(value = "numPerPage", required = false) Integer numPerPage )
            throws BizException, PermissionException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("adviserId", adviserId);
        paramMap.put("name", name != null && "".equals(name.trim()) ? null : name);
        if ((null == currentPage && null != numPerPage) || (null != currentPage && null == numPerPage)) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        if (null != currentPage) {
            return new ResponseDto<>(bookGroupBiz.listSimpleBookGroup4Adviser(paramMap, new PageParam(currentPage, numPerPage)));
        } else {
            return new ResponseDto<>(bookGroupBiz.listSimpleBookGroup4Adviser(paramMap));
        }
    }

    /**
     * 获取未创建社群码的书刊列表
     */
    @Override
    @RequestMapping(value = "listBook4CreateBookGroup", method = RequestMethod.GET)
    public ResponseDto<?> listBook4CreateBookGroup( @RequestHeader("token") String token,
                                                    @RequestParam(value = "name", required = false) String name,
                                                    @RequestParam(value = "currentPage", required = false) Integer currentPage,
                                                    @RequestParam(value = "numPerPage", required = false) Integer numPerPage )
            throws BizException, PermissionException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (currentPage == null || numPerPage == null || currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("adviserId", adviserId);
        paramMap.put("name", name != null && "".equals(name.trim()) ? null : name);
        PageBean pageBean = bookGroupBiz.listBook4CreateBookGroup(paramMap, new PageParam(currentPage, numPerPage), adviserId);
        return new ResponseDto<>(pageBean);
    }

    @Override
    @PostMapping("updateRankType")
    public ResponseDto<?> updateRankType( @RequestHeader("token") String token, @RequestBody UpdateRankTypeVO updateRankTypeVO ) throws BizException {
        bookGroupBiz.updateRankType(updateRankTypeVO.getBookGroupId(), updateRankTypeVO.getRankType());
        return new ResponseDto<>();
    }

    @Override
    @GetMapping("getRankType")
    public ResponseDto<Integer> getRankType( @RequestHeader("token") String token, @RequestParam("bookGroupId") Long bookGroupId ) throws BizException {
        Integer rankType = bookGroupBiz.getRankType(bookGroupId);
        return new ResponseDto<>(rankType == null ? 0 : rankType);
    }

    /**
     * 编辑获取社群码总数量
     */
    @Override
    @RequestMapping(value = "getTotalBookGroupCount", method = RequestMethod.GET)
    public ResponseDto<?> getTotalBookGroupCount( @RequestHeader("token") String token ) throws BizException, PermissionException, JsonParseException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        return new ResponseDto<>(bookGroupBiz.getTotalBookGroupCount(adviserId));
    }

    /**
     * 获取社群书分类列表(编辑)
     */
    @Override
    @RequestMapping(value = "listSimpleBookGroupClassify4Adviser", method = RequestMethod.POST)
    public ResponseDto<?> listSimpleBookGroupClassify4Adviser( @RequestHeader("token") String token,
                                                               @RequestBody BookGroupClassify bookGroupClassify )
            throws BizException, PermissionException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (null == bookGroupClassify) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数有误");
        }
        if (null == bookGroupClassify.getCurrentPage() || null == bookGroupClassify.getNumPerPage()) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "分页参数有误");
        }
        PageParam pageParam = new PageParam(bookGroupClassify.getCurrentPage(), bookGroupClassify.getNumPerPage());
        return new ResponseDto<>(bookGroupClassifyBiz.listSimpleBookGroupClassify4Adviser(adviserId, bookGroupClassify, pageParam));
    }

    @Override
    @RequestMapping(value = "isSpecialAgent", method = RequestMethod.GET)
    public ResponseDto<Boolean> isSpecialAgent( @RequestHeader("token") String token ) throws BizException, PermissionException {
        Long agentId = (Long) SessionUtil.getVlaue(token, SessionUtil.TENANT_ID);
        return new ResponseDto<>(bookGroupBiz.isSpecialAgent(agentId));
    }

    @ApiOperation("获取应用作品统计")
    @GetMapping("getAppStatistics")
    @Override
    public ResponseDto<?> getAppStatistics(
            @RequestHeader("token") String token,
            @RequestParam("currentPage") Integer currentPage,
            @RequestParam("numPerPage") Integer numPerPage,
            @RequestParam("bookGroupId") Long bookGroupId,
            @RequestParam(value = "qrcodeId", required = false) Long qrcodeId
    ) throws BizException, PermissionException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (currentPage == null || currentPage < 0 || numPerPage == null || numPerPage <= 0) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "分页参数错误！");
        }
        if (bookGroupId == null) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "社群码id不能为空！");
        }
        return new ResponseDto<>(bookGroupBiz.getAppStatistics(currentPage, numPerPage, bookGroupId, qrcodeId, partyId));
    }

    @Override
    @ApiOperation(value = "导出社群码(编辑)", httpMethod = "GET")
    @RequestMapping(value = "exportGroupQrcode4Adviser", method = RequestMethod.GET)
    public ResponseDto<?> exportGroupQrcode4Adviser( @RequestHeader("token") String token,
                                                     @RequestParam(value = "name", required = false) String name,
                                                     @RequestParam(value = "bookName", required = false) String bookName,
                                                     @RequestParam(value = "isbn", required = false) String isbn,
                                                     @RequestParam(value = "channelId", required = false) Long channelId,
                                                     @RequestParam(value = "typeCode", required = false) String typeCode,
                                                     @RequestParam(value = "isMainEditor", required = false) Integer isMainEditor,
                                                     @RequestParam(value = "isFundSupport", required = false) Integer isFundSupport,
                                                     @RequestParam(value = "bookId", required = false) Integer bookId,
                                                     @RequestParam(value = "startDate", required = false) String startDate,
                                                     @RequestParam(value = "endDate", required = false) String endDate,
                                                     @RequestParam(value = "joinGroupType", required = false) String joinGroupType,
                                                     @RequestParam(value = "hasServe", required = false) Boolean hasServe,
                                                     @RequestParam(value = "isPrint", required = false) Boolean isPrint,
                                                     @RequestParam(value = "jumpType", required = false) Integer jumpType)
            throws BizException, PermissionException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("adviserId", adviserId);
        paramMap.put("isMainEditor", isMainEditor);
        paramMap.put("isFundSupport", isFundSupport);
        paramMap.put("channelId", channelId);
        paramMap.put("name", name != null && "".equals(name.trim()) ? null : name);
        paramMap.put("isbn", isbn != null && "".equals(isbn.trim()) ? null : isbn);
        paramMap.put("bookName", bookName != null && "".equals(bookName.trim()) ? null : bookName);
        paramMap.put("typeCode", typeCode != null && "".equals(typeCode.trim()) ? null : typeCode);
        paramMap.put("bookId", bookId);
        if (!StringUtil.isEmpty(startDate) && !StringUtil.isEmpty(endDate)) {
            paramMap.put("startDate", startDate + " 00:00:00");
            paramMap.put("endDate", endDate + " 23:59:59");
        }
        List<Integer> joinGroupTypes = new ArrayList<>();
        if ("GROUP".equals(joinGroupType)) {
            joinGroupTypes.add(JoinGroupTypeEnum.GROUP_QRCODE.getCode());
        }
        if ("ROBOT".equals(joinGroupType)) {
            joinGroupTypes.add(JoinGroupTypeEnum.ROBOT.getCode());
            joinGroupTypes.add(JoinGroupTypeEnum.AI_ROBOT.getCode());
            joinGroupTypes.add(JoinGroupTypeEnum.XIAORUI.getCode());
        }
        paramMap.put("joinGroupTypes", joinGroupTypes);
        paramMap.put("hasServe", hasServe);
        paramMap.put("isPrint", isPrint);
        paramMap.put("jumpType", jumpType);
        bookGroupBiz.exportGroupQrcode4Adviser(paramMap, adviserId);
        return new ResponseDto<>();
    }

    @ApiOperation("根据bookId获取社群书分类等统计")
    @GetMapping("getBookGroupStatistics")
    @Override
    public ResponseDto<?> getBookGroupStatistics(
            @RequestHeader("token") String token,
            @RequestParam("bookId") Long bookId
    ) throws BizException, PermissionException, JsonParseException {
        SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (bookId == null) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "bookId不能为空！");
        }
        return new ResponseDto<>(bookGroupBiz.getBookGroupStatistics(bookId));
    }

    @ApiOperation("根据bookId获取社群书群分类资源统计信息")
    @GetMapping("getBookGroupKeywordStatistics")
    @Override
    public ResponseDto<?> getBookGroupKeywordStatistics(
            @RequestHeader("token") String token,
            @RequestParam("bookId") Long bookId
    ) throws BizException, PermissionException, JsonParseException {
        SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (bookId == null) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "bookId不能为空！");
        }
        return new ResponseDto<>(bookGroupBiz.getBookGroupKeywordStatistics(bookId));
    }

    @ApiOperation("更新是否显示书名")
    @GetMapping("updateIsShowBookName")
    @Override
    public ResponseDto<?> updateIsShowBookName(
            @RequestHeader("token") String token,
            @RequestParam("isShowBookName") Boolean isShowBookName
    ) throws BizException, PermissionException, JsonParseException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (isShowBookName == null) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "isShowBookName不能为空！");
        }
        bookGroupBiz.updateIsShowBookName(isShowBookName, partyId);
        return new ResponseDto<>();
    }

    @ApiOperation("获取是否显示书名")
    @GetMapping("getIsShowBookName")
    @Override
    public ResponseDto<?> getIsShowBookName(
            @RequestHeader("token") String token
    ) throws BizException, PermissionException, JsonParseException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        return new ResponseDto<>(bookGroupBiz.getIsShowBookName(partyId));
    }


    @ApiOperation("删除社群码")
    @GetMapping("deleteBookGroup")
    @Override
    public ResponseDto<?> deleteBookGroup(
            @RequestHeader("token") String token,
            @RequestParam("bookGroupId") Long bookGroupId
    ) throws BizException, PermissionException, JsonParseException {
        SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        bookGroupBiz.deleteBookGroup(bookGroupId);
        return new ResponseDto<>();
    }

    @GetMapping("deleteBookGroup4Adviser")
    @Override
    public ResponseDto<?> deleteBookGroup4Adviser(
            @RequestHeader("token") String token,
            @RequestParam("bookGroupId") Long bookGroupId
    ) throws BizException, PermissionException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        log.info("===deleteBookGroup4Adviser partyId :{}; bookGroupId:{}===", partyId, bookGroupId);
        bookGroupBiz.deleteBookGroup4Adviser(bookGroupId);
        return new ResponseDto<>();
    }

    @ApiOperation("根据社群书id获取个人二维码")
    @GetMapping("getOwnAltQrcodeInfoDTOByBookGroupId")
    @Override
    public ResponseDto<?> getOwnAltQrcodeInfoDTOByBookGroupId(
            @CookieValue("userInfo") String userInfo,
            @RequestParam("bookGroupId") Long bookGroupId
    ) throws BizException, PermissionException, JsonParseException {
        Long wechatUserId = Cookie.getId(userInfo, Cookie._WECHAT_USER_ID);
        if (bookGroupId == null) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "社群码id不能为空！");
        }
        return new ResponseDto<>(bookGroupBiz.getOwnAltQrcodeInfoDTOByBookGroupId(wechatUserId, bookGroupId));
    }

    @ApiOperation("根据分类id获取个人二维码")
    @GetMapping("getOwnAltQrcodeInfoDTOByClassifyId")
    @Override
    public ResponseDto<?> getOwnAltQrcodeInfoDTOByClassifyId(
            @CookieValue("userInfo") String userInfo,
            @RequestParam("classifyId") Long classifyId
    ) throws BizException, PermissionException, JsonParseException {
        Long wechatUserId = Cookie.getId(userInfo, Cookie._WECHAT_USER_ID);
        if (classifyId == null) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "分类不能为空！");
        }
        return new ResponseDto<>(bookGroupBiz.getOwnAltQrcodeInfoDTOByClassifyId(wechatUserId, classifyId));
    }

    @ApiOperation("根据群二维码id获取个人二维码")
    @GetMapping("getOwnAltQrcodeInfoDTOByQrcodeId")
    @Override
    public ResponseDto<?> getOwnAltQrcodeInfoDTOByQrcodeId(
            @CookieValue("userInfo") String userInfo,
            @RequestParam("qrcodeId") Long qrcodeId
    ) throws BizException, PermissionException, JsonParseException {
        Long wechatUserId = Cookie.getId(userInfo, Cookie._WECHAT_USER_ID);
        if (qrcodeId == null) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "分类不能为空！");
        }
        return new ResponseDto<>(bookGroupBiz.getOwnAltQrcodeInfoDTOByQrcodeId(wechatUserId, qrcodeId));
    }

    @ApiOperation("获取所有的大类")
    @GetMapping("getAllLargeTemplet")
    @Override
    public ResponseDto<?> getAllLargeTemplet(
            @RequestHeader("token") String token
    ) throws BizException, PermissionException, JsonParseException {
        SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        return new ResponseDto<>(bookGroupBiz.getAllLargTemplet());
    }

    @Override
    @RequestMapping(value = "getBookGroupStatisByBookGroupId", method = RequestMethod.GET)
    public ResponseDto<?> getBookGroupStatisByBookGroupId( @RequestParam("bookGroupId") Long bookGroupId ) throws BizException {
        return new ResponseDto<>(bookGroupBiz.getBookGroupStatisByBookGroupId(bookGroupId));
    }

    @Override
    @RequestMapping(value = "getGroupIncomeStatic", method = RequestMethod.POST)
    public ResponseDto<PageBeanNew<ResourcesStatisticVO>> getGroupIncomeStatic( @RequestHeader("token") String token,
                                                                                @RequestBody GroupIncomeStaticParamVO groupIncomeStaticParamVO ) throws BizException, PermissionException {
        Map<String, Object> map = SessionUtil.getToken4Redis(token);
        Long adviserId = (Long) map.get(SessionUtil.PARTY_ID);
        PageBeanNew<ResourcesStatisticVO> groupIncomeStatic = bookGroupBiz.getGroupIncomeStatic(groupIncomeStaticParamVO, adviserId);
        return new ResponseDto<>(groupIncomeStatic);
    }

    @Override
    @RequestMapping(value = "getGroupScanTrend", method = RequestMethod.POST)
    public ResponseDto<List<GroupScanTrendVO>> getGroupScanTrend( @RequestHeader("token") String token,
                                                                  @RequestBody GroupScanTrendParamVO groupScanTrendParamVO ) throws BizException, PermissionException {
        Map<String, Object> map = SessionUtil.getToken4Redis(token);
        Long adviserId = (Long) map.get(SessionUtil.PARTY_ID);
        List<GroupScanTrendVO> groupScanTrend = bookGroupBiz.getGroupScanTrend(groupScanTrendParamVO, adviserId);
        return new ResponseDto<>(groupScanTrend);
    }

    @Override
    @RequestMapping(value = "getClassifyStatistic", method = RequestMethod.GET)
    public ResponseDto<?> getClassifyStatistic( @RequestHeader("token") String token,
                                                @RequestParam(value = "bookGroupId", required = true) Long bookGroupId,
                                                @RequestParam(value = "currentPage", required = true) Integer currentPage,
                                                @RequestParam(value = "numPerPage", required = true) Integer numPerPage ) throws BizException, PermissionException {
        Map<String, Object> map = SessionUtil.getToken4Redis(token);
        Long adviserId = (Long) map.get(SessionUtil.PARTY_ID);
        PageBeanNew<GroupStatisticVO> groupQrcodeStatistic = bookGroupBiz.getClassifyStatistic(bookGroupId, adviserId, currentPage, numPerPage);
        return new ResponseDto<>(groupQrcodeStatistic);
    }

    @Override
    @RequestMapping(value = "getQrcodeStatistic", method = RequestMethod.GET)
    public ResponseDto<?> getQrcodeStatistic( @RequestHeader("token") String token,
                                              @RequestParam(value = "bookGroupId", required = true) Long bookGroupId,
                                              @RequestParam("classifyId") Long classifyId ) throws BizException, PermissionException {
        Map<String, Object> map = SessionUtil.getToken4Redis(token);
        Long adviserId = (Long) map.get(SessionUtil.PARTY_ID);
        List<WxGroupStatisticVO> qrcodeStatistic = bookGroupBiz.getQrcodeStatistic(bookGroupId, classifyId, adviserId);
        return new ResponseDto<>(qrcodeStatistic);
    }

    @Override
    @RequestMapping(value = "getTotalRescourceData", method = RequestMethod.GET)
    public ResponseDto<?> getTotalRescourceData( @RequestHeader("token") String token, @RequestParam(value = "bookGroupId", required = true) Long bookGroupId,
                                                 @RequestParam(value = "classifyId", required = false) Long classifyId, @RequestParam(value = "qrcodeId", required = false) Long qrcodeId,
                                                 @RequestParam("type") String type ) throws BizException, PermissionException {
        Map<String, Object> map = SessionUtil.getToken4Redis(token);
        Long adviserId = (Long) map.get(SessionUtil.PARTY_ID);
        TotalRescourceDataVO totalRescourceData = bookGroupBiz.getTotalRescourceData(bookGroupId, adviserId, classifyId, qrcodeId, type);
        return new ResponseDto<>(totalRescourceData);
    }

    @Override
    @RequestMapping(value = "exportRescourceIncomeData", method = RequestMethod.GET)
    public ResponseDto<?> exportRescourceIncomeData( @RequestHeader("token") String token,
                                                     @RequestParam(value = "bookGroupId", required = true) Long bookGroupId ) throws BizException, PermissionException {
        Map<String, Object> map = SessionUtil.getToken4Redis(token);
        Long adviserId = (Long) map.get(SessionUtil.PARTY_ID);
        Map<String, String> resultMap = bookGroupBiz.exportRescourceIncomeData(bookGroupId, adviserId);
        return new ResponseDto<>(resultMap);
    }

    @Override
    @GetMapping("listPageFriendsStatistic")
    public ResponseDto<PageBeanNew<FriendsVO>> listPageFriendsStatistic(
            @RequestHeader("token") String token, @RequestParam(value = "bookGroupId", required = true) Long bookGroupId,
            @RequestParam(value = "currentPage", required = true) Integer currentPage,
            @RequestParam(value = "numPerPage", required = true) Integer numPerPage ) throws BizException, PermissionException {
        SessionUtil.getToken4Redis(token);
        PageBeanNew<FriendsVO> pageBeanNew = bookGroupBiz.listPageFriendsStatistic(bookGroupId, currentPage, numPerPage);
        return new ResponseDto<>(pageBeanNew);
    }

    @ApiOperation("批量新增资源配置")
    @PostMapping("batchAddBookGroupServe")
    @Override
    public ResponseDto<?> batchAddBookGroupServe(
            @RequestHeader("token") String token,
            @RequestBody List<BookGroupServe> bookGroupServes
    ) throws BizException, PermissionException, JsonParseException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        bookGroupBiz.batchAddBookGroupServe(partyId, bookGroupServes);
        return new ResponseDto<>();
    }

    @ApiOperation("批量更新资源配置")
    @PostMapping("batchUpdateBookGroupServe")
    @Override
    public ResponseDto<?> batchUpdateBookGroupServe(
            @RequestHeader("token") String token,
            @RequestBody List<BookGroupServe> bookGroupServes
    ) throws BizException, PermissionException, JsonParseException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        bookGroupBiz.batchUpdateBookGroupServe(partyId, bookGroupServes);
        return new ResponseDto<>();
    }

    @ApiOperation("删除资源配置")
    @GetMapping("deleteBookGroupServe")
    @Override
    public ResponseDto<?> deleteBookGroupServe(
            @RequestHeader("token") String token,
            @RequestParam("id") Long id
    ) throws BizException, PermissionException, JsonParseException {
        SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        bookGroupBiz.deleteBookGroupServe(id);
        return new ResponseDto<>();
    }

    @ApiOperation("获取资源配置集合")
    @GetMapping("getBookGroupServeList")
    @Override
    public ResponseDto<?> getBookGroupServeList(
            @RequestHeader("token") String token,
            @RequestParam("bookGroupId") Long bookGroupId
    ) throws BizException, PermissionException, JsonParseException {
        SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        return new ResponseDto<>(bookGroupBiz.getBookGroupServeList(bookGroupId));
    }

    @ApiOperation("客户端获取资源配置集合")
    @GetMapping("getBookGroupServeList4Wechat")
    @Override
    public ResponseDto<?> getBookGroupServeList4Wechat(
            @CookieValue("userInfo") String userInfo,
            @RequestParam("bookGroupId") Long bookGroupId,
            @RequestParam("currentPage") Integer currentPage,
            @RequestParam("numPerPage") Integer numPerPage
    ) throws BizException, PermissionException, JsonParseException {
        Cookie.getId(userInfo, Cookie._WECHAT_USER_ID);
        return new ResponseDto<>(bookGroupBiz.getBookGroupServeList4Wechat(bookGroupId, currentPage, numPerPage));
    }

    @ApiOperation("获取资源配置集合(特殊二维码用)")
    @GetMapping("getBookGroupServeListByBookGroupId")
    @Override
    public ResponseDto<?> getBookGroupServeListByBookGroupId(
            @RequestParam("bookGroupId") Long bookGroupId
    ) throws BizException, PermissionException, JsonParseException {
        return new ResponseDto<>(bookGroupBiz.getBookGroupServeListByBookGroupId(bookGroupId));
    }

    @ApiOperation("获取社群码暗号")
    @GetMapping("getBookGroupCipher")
    @Override
    public ResponseDto<?> getBookGroupCipher(
            @RequestHeader("token") String token
    ) throws BizException, PermissionException, JsonParseException {
        SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        return new ResponseDto<>(bookGroupBiz.getBookGroupCipher());
    }

    @Override
    @PostMapping("listPageBookGroupAnalysis")
    public ResponseDto<PageBeanNew<BookGroupAnalysisVO>> listPageBookGroupAnalysis(
            @RequestHeader("token") String token, @RequestBody @ApiParam BookGroupAnalysisParam bookGroupAnalysisParam ) throws PermissionException {
        SessionUtil.getToken4Redis(token);
        if (null == bookGroupAnalysisParam) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数为空");
        }
        if (null == bookGroupAnalysisParam.getCurrentPage() || null == bookGroupAnalysisParam.getNumPerPage()) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "缺少分页参数");
        }
        PageBeanNew<BookGroupAnalysisVO> pageBeanNew = bookGroupBiz.listPageBookGroupAnalysis(bookGroupAnalysisParam);
        return new ResponseDto<>(pageBeanNew);
    }

    @Override
    @PostMapping("exportBookGroupAnalysis")
    public ResponseDto<?> exportBookGroupAnalysis(
            @RequestHeader("token") String token, @RequestBody @ApiParam BookGroupAnalysisParam bookGroupAnalysisParam ) throws PermissionException {
        Map<String, Object> map = SessionUtil.getToken4Redis(token);
        String isSystem = (String) map.get(SessionUtil.IS_SYSTEM);
        Long partyId = (Long) map.get(SessionUtil.PARTY_ID);
        if (IsSystem.NOT_SYSTEM.code.equals(isSystem)) {
            partyId = (Long) map.get(SessionUtil.MEMBER_ID);
        }
        bookGroupBiz.exportBookGroupAnalysis(bookGroupAnalysisParam, partyId);
        return new ResponseDto<>();
    }

    @ApiOperation("更新是否邀请入群")
    @PostMapping("updateIsInviteGroup")
    @Override
    public ResponseDto<?> updateIsInviteGroup(
            @RequestHeader("token") String token,
            @RequestBody BookGroup bookGroup
    ) throws BizException, PermissionException, JsonParseException {
        SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        bookGroupBiz.updateIsInviteGroup(bookGroup);
        return new ResponseDto<>();
    }

    @ApiOperation("更新1v1是否有更新")
    @PostMapping("updateIsSomeUpdate")
    @Override
    public ResponseDto<?> updateIsSomeUpdate(
            @RequestHeader("token") String token,
            @RequestBody BookGroup bookGroup
    ) throws BizException, PermissionException, JsonParseException {
        SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        bookGroupBiz.updateIsSomeUpdate(bookGroup);
        return new ResponseDto<>();
    }

    @ApiOperation("新增推送更新")
    @PostMapping("addPushBookGroupUpdate")
    @Override
    public ResponseDto<?> addPushBookGroupUpdate(
            @RequestHeader("token") String token,
            @RequestBody BookGroup bookGroup
    ) throws BizException, PermissionException, JsonParseException {
        SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        bookGroupBiz.addPushBookGroupUpdate(bookGroup);
        return new ResponseDto<>();
    }

    @ApiOperation("获取是否能推送更新的各个状态")
    @GetMapping("getStatesForPushUpdate")
    @Override
    public ResponseDto<?> getStatesForPushUpdate(
            @RequestHeader("token") String token,
            @RequestParam("bookGroupId") Long bookGroupId
    ) throws BizException, PermissionException, JsonParseException {
        SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        return new ResponseDto<>(bookGroupBiz.getStatesForPushUpdate(bookGroupId));
    }

    @ApiOperation("根据bookId获取社群书基本信息")
    @GetMapping("getBookGroupDTOByBookId")
    @Override
    public ResponseDto<?> getBookGroupDTOByBookId(
            @RequestHeader("token") String token,
            @RequestParam("bookId") Long bookId,
            @RequestParam("channelId") Long channelId
    ) throws PermissionException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        return new ResponseDto<>(bookGroupBiz.getDTOByBookId(bookId, channelId, partyId));
    }

    @ApiOperation("根据类型获取社群书数量")
    @GetMapping("getBookGroupCountByJoinGroupType")
    @Override
    public ResponseDto<?> getBookGroupCountByJoinGroupType(
            @RequestHeader("token") String token,
            @RequestParam(value = "joinGroupType", required = false) Integer joinGroupType
    ) throws PermissionException {
        String systemCode = (String) SessionUtil.getVlaue(token, SessionUtil.SYSTEM_CODE);
        if (!SystemCode.pcloud.code.equalsIgnoreCase(systemCode)) {
            throw new BookBizException(BookBizException.ERROR, "角色错误！");
        }
        return new ResponseDto<>(bookGroupBiz.getBookGroupCountByJoinGroupType(joinGroupType));
    }

    @ApiOperation("获取个人号累计加好友人数")
    @GetMapping("getAddFriendCount")
    @Override
    public ResponseDto<?> getAddFriendCount(
            @RequestHeader("token") String token
    ) throws PermissionException {
        String systemCode = (String) SessionUtil.getVlaue(token, SessionUtil.SYSTEM_CODE);
        if (!SystemCode.pcloud.code.equalsIgnoreCase(systemCode)) {
            throw new PermissionException(PermissionException.PERMISSION_NOT_FOUND);
        }
        return new ResponseDto<>(bookGroupBiz.getAddFriendCount());
    }

    @ApiOperation("根据时间段获取个人号加好友人数")
    @GetMapping("getAddFriendCountByTime")
    @Override
    public ResponseDto<?> getAddFriendCountByTime(
            @RequestHeader("token") String token,
            @RequestParam("timeType") @ApiParam("时间类型（1按天，2按小时）") Integer timeType,
            @RequestParam("startTime") @ApiParam("开始时间") String startTime,
            @RequestParam("endTime") @ApiParam("结束时间") String endTime
    ) throws PermissionException {
        String systemCode = (String) SessionUtil.getVlaue(token, SessionUtil.SYSTEM_CODE);
        if (!SystemCode.pcloud.code.equalsIgnoreCase(systemCode)) {
            throw new PermissionException(PermissionException.PERMISSION_NOT_FOUND);
        }
        return new ResponseDto<>(bookGroupBiz.getAddFriendCountByTime(timeType, startTime, endTime));
    }

    @ApiOperation("获取出版社创建社群书数量排行")
    @GetMapping("getTopAgentCreateBookGroup")
    @Override
    public ResponseDto<?> getTopAgentCreateBookGroup(
            @RequestHeader("token") String token,
            @RequestParam(value = "joinGroupType", required = false) Integer joinGroupType,
            @RequestParam("top") Integer top,
            @RequestParam("isSpecial") Boolean isSpecial
    ) throws PermissionException {
        String systemCode = (String) SessionUtil.getVlaue(token, SessionUtil.SYSTEM_CODE);
        if (!SystemCode.pcloud.code.equalsIgnoreCase(systemCode)) {
            throw new PermissionException(PermissionException.PERMISSION_NOT_FOUND);
        }
        return new ResponseDto<>(bookGroupBiz.getTopAgentCreateBookGroup(joinGroupType, top, isSpecial));
    }

    @ApiOperation("获取个人号读者专业或深度分布")
    @GetMapping("getUserLabelDistribution")
    @Override
    public ResponseDto<?> getUserLabelDistribution(
            @RequestHeader("token") String token,
            @RequestParam("largeTemplet") @ApiParam("1:K12类,2:少儿类,3:高等/职教,4:大众类,5:特殊出版社") Integer largeTemplet,
            @RequestParam("type") @ApiParam("1:专业,2:深度") Integer type
    ) throws PermissionException {
        String systemCode = (String) SessionUtil.getVlaue(token, SessionUtil.SYSTEM_CODE);
        if (!SystemCode.pcloud.code.equalsIgnoreCase(systemCode)) {
            throw new PermissionException(PermissionException.PERMISSION_NOT_FOUND);
        }
        return new ResponseDto<>(bookGroupBiz.getUserLabelDistribution(largeTemplet, type));
    }

    @ApiOperation("获取出版社个人号统计信息")
    @GetMapping("getAgentStatisticsInfo")
    @Override
    public ResponseDto<?> getAgentStatisticsInfo(
            @RequestHeader("token") String token,
            @RequestParam(value = "agentId", required = false) Long agentId,
            @RequestParam(value = "isSpecial", required = false) Boolean isSpecial,
            @RequestParam(value = "currentPage") Integer currentPage,
            @RequestParam(value = "numPerPage") Integer numPerPage ) throws PermissionException {
        String systemCode = (String) SessionUtil.getVlaue(token, SessionUtil.SYSTEM_CODE);
        if (!SystemCode.pcloud.code.equalsIgnoreCase(systemCode)) {
            throw new PermissionException(PermissionException.PERMISSION_NOT_FOUND);
        }
        return new ResponseDto<>(bookGroupBiz.getAgentStatisticsInfo(agentId, isSpecial, currentPage, numPerPage));
    }

    @ApiOperation("导出出版社个人号统计信息")
    @GetMapping("exportAgentStatisticsInfo")
    @Override
    public ResponseDto<?> exportAgentStatisticsInfo(
            @RequestHeader("token") String token,
            @RequestParam(value = "agentId", required = false) Long agentId,
            @RequestParam(value = "isSpecial", required = false) Boolean isSpecial ) throws PermissionException {
        Map<String, Object> map = SessionUtil.getToken4Redis(token);
        String isSystem = (String) map.get(SessionUtil.IS_SYSTEM);
        String systemCode = (String) map.get(SessionUtil.SYSTEM_CODE);
        Long partyId = (Long) map.get(SessionUtil.PARTY_ID);
        if (!SystemCode.pcloud.code.equalsIgnoreCase(systemCode)) {
            throw new PermissionException(PermissionException.PERMISSION_NOT_FOUND);
        }
        if (IsSystem.NOT_SYSTEM.code.equals(isSystem)) {
            partyId = (Long) map.get(SessionUtil.MEMBER_ID);
        }
        bookGroupBiz.exportAgentStatisticsInfo(partyId, agentId, isSpecial);
        return new ResponseDto<>();
    }

    @ApiOperation("按时间获取资源服务点击量趋势")
    @PostMapping("getServeClickTendencyList")
    @Override
    public ResponseDto<?> getServeClickTendencyList(
            @RequestHeader("token") String token,
            @RequestParam("bookGroupId") Long bookGroupId,
            @RequestBody List<String> dateList
    ) throws PermissionException {
        String systemCode = (String) SessionUtil.getVlaue(token, SessionUtil.SYSTEM_CODE);
        if (!SystemCode.pcloud.code.equalsIgnoreCase(systemCode)) {
            throw new PermissionException(PermissionException.PERMISSION_NOT_FOUND);
        }
        return new ResponseDto<>(bookGroupBiz.getServeClickTendencyList(bookGroupId, dateList));
    }

    @ApiOperation("个人号社群书统计")
    @PostMapping("getSelfBookGroupStatistics")
    @Override
    public ResponseDto<?> getSelfBookGroupStatistics(
            @RequestHeader("token") String token,
            @RequestBody SelfBookGroupStParamDTO selfBookGroupStParamDTO
    ) throws PermissionException {
        String systemCode = (String) SessionUtil.getVlaue(token, SessionUtil.SYSTEM_CODE);
        if (!SystemCode.pcloud.code.equalsIgnoreCase(systemCode)) {
            throw new PermissionException(PermissionException.PERMISSION_NOT_FOUND);
        }
        return new ResponseDto<>(bookGroupBiz.getSelfBookGroupStatistics(selfBookGroupStParamDTO));
    }

    @ApiOperation("导出个人号社群书统计")
    @PostMapping("exportSelfBookGroupStatistics")
    @Override
    public ResponseDto<?> exportSelfBookGroupStatistics(
            @RequestHeader("token") String token,
            @RequestBody SelfBookGroupStParamDTO selfBookGroupStParamDTO
    ) throws PermissionException {
        String systemCode = (String) SessionUtil.getVlaue(token, SessionUtil.SYSTEM_CODE);
        if (!SystemCode.pcloud.code.equalsIgnoreCase(systemCode)) {
            throw new PermissionException(PermissionException.PERMISSION_NOT_FOUND);
        }
        Map<String, Object> map = SessionUtil.getToken4Redis(token);
        String isSystem = (String) map.get(SessionUtil.IS_SYSTEM);
        Long partyId = (Long) map.get(SessionUtil.PARTY_ID);
        if (IsSystem.NOT_SYSTEM.code.equals(isSystem)) {
            partyId = (Long) map.get(SessionUtil.MEMBER_ID);
        }
        bookGroupBiz.exportSelfBookGroupStatistics(selfBookGroupStParamDTO, systemCode, partyId);
        return new ResponseDto<>();
    }


    @ApiOperation(value = "个人号群发-用户列表查询", httpMethod = "POST")
    @PostMapping("listUser4SelfPush")
    @Override
    public ResponseDto<PageBeanNew<UserBookInfoVO>> listUser4SelfPush(
            @RequestHeader("token") String token, @RequestBody @ApiParam UserSelectParamDTO userSelectParamDTO ) throws PermissionException {
        SessionUtil.getToken4Redis(token);
        if (null == userSelectParamDTO || null == userSelectParamDTO.getCurrentPage() || null == userSelectParamDTO.getNumPerPage()) {
            throw new WechatGroupBizException(WechatGroupBizException.PARAM_IS_NULL, "缺少参数");
        }
        if (StringUtil.isEmpty(userSelectParamDTO.getRobotId())) {
            throw new WechatGroupBizException(WechatGroupBizException.PARAM_IS_NULL, "请选择个人号");
        }
        PageBeanNew<UserBookInfoVO> pageBeanNew = bookGroupBiz.listUser4SelfPush(userSelectParamDTO);
        return new ResponseDto<>(pageBeanNew);
    }

    @ApiOperation(value = "编辑端，个人号群发-用户列表查询", httpMethod = "GET")
    @GetMapping("listUser4SelfPushByAdvise")
    @Override
    public ResponseDto<PageBeanNew<UserBookInfoVO>> listUser4SelfPushByAdvise( @RequestHeader("token") String token,
                                                                               @RequestParam(value = "userQuery", required = false) String userQuery,
                                                                               @RequestParam("currentPage") Integer currentPage,
                                                                               @RequestParam("numPerPage") Integer numPerPage,
                                                                               @RequestParam("bookId") Long bookId ) throws PermissionException {
        SessionUtil.getToken4Redis(token);
        if (null == currentPage || null == numPerPage) {
            throw new WechatGroupBizException(WechatGroupBizException.PARAM_IS_NULL, "缺少参数");
        }
        if (null == bookId) {
            throw new WechatGroupBizException(WechatGroupBizException.PARAM_IS_NULL, "缺少图书Id");
        }

        Map<String, Object> userSelectParam = new HashMap<>();
        userSelectParam.put("currentPage", currentPage);
        userSelectParam.put("numPerPage", numPerPage);
        userSelectParam.put("bookId", bookId);
        if (!StringUtil.isEmpty(userQuery)) {
            userSelectParam.put("userQuery", userQuery);
        }

        PageBeanNew<UserBookInfoVO> pageBeanNew = bookGroupBiz.listUser4SelfPushByAdvise(userSelectParam);
        return new ResponseDto<>(pageBeanNew);
    }


    @ApiOperation("获取随机码")
    @GetMapping("getRandomCode")
    @Override
    public ResponseDto<?> getRandomCode(
            @CookieValue("userInfo") String userInfo
    ) throws PermissionException {
        Long wechatUserId = Cookie.getId(userInfo, Cookie._WECHAT_USER_ID);
        return new ResponseDto<>(bookGroupBiz.getRandomCode(wechatUserId));
    }

    @Override
    @PostMapping("changeQrCodeType")
    public ResponseDto<?> changeQrCodeType( @RequestHeader("token") String token, @RequestBody @Validated ChangeQrCodeTypeDto changeQrCodeTypeDto ) throws PermissionException {
        Map<String, Object> token4Redis = SessionUtil.getToken4Redis(token);
        return new ResponseDto<>(bookGroupBiz.changeQrCodeType(changeQrCodeTypeDto));
    }

    @Override
    @GetMapping("getChangeQrChangeTarget")
    public ResponseDto<?> getChangeQrChangeTarget( @RequestParam("id") Long id, @RequestParam("type") String type ) {
        Long newBookGroupId = bookGroupBiz.getChangeQrChangeTarget(id, type);
        return new ResponseDto<>(newBookGroupId);
    }

    @ApiOperation("获取1v1社群书列表")
    @Override
    @RequestMapping(value = "getListBookGroup4Adviser", method = RequestMethod.GET)
    public ResponseDto<PageBeanNew<BookGroupDTO>> getListBookGroup4Adviser(
            @RequestHeader("token") String token,
            @RequestParam(value = "name", required = false) String name,
            @RequestParam("currentPage") Integer currentPage,
            @RequestParam("numPerPage") Integer numPerPage,
            @RequestParam(value = "channelId", required = false) Long channelId,
            @RequestParam("serveId") Long serveId,
            @RequestParam("serveType") String serveType )
            throws BizException, PermissionException {
        if (currentPage == null || numPerPage == null || currentPage < 0 || numPerPage <= 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("adviserId", adviserId);
        paramMap.put("name", name != null && "".equals(name.trim()) ? null : name);
        paramMap.put("joinGroupType", JoinGroupTypeEnum.ROBOT.getCode());
        paramMap.put("channelId", channelId);
        paramMap.put("serveId", serveId);
        paramMap.put("serveType", serveType);
        PageBeanNew<BookGroupDTO> pageBean = bookGroupBiz.getListBookGroup4Adviser(paramMap, new PageParam(currentPage, numPerPage));
        return new ResponseDto<>(null == pageBean ? new PageBeanNew<>() : pageBean);
    }

    @Override
    @ApiOperation("获取1v1关联社群书列表")
    @RequestMapping(value = "getAssocBookGroupByAppInfo", method = RequestMethod.GET)
    public ResponseDto<PageBeanNew<BookGroupDTO>> getAssocBookGroupByAppInfo(
            @RequestHeader("token") String token,
            @RequestParam("channelId") Long channelId,
            @RequestParam("serveId") Long serveId,
            @RequestParam("serveType") String serveType,
            @RequestParam("currentPage") Integer currentPage,
            @RequestParam("numPerPage") Integer numPerPage )
            throws BizException, PermissionException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("channelId", channelId);
        paramMap.put("serveId", serveId);
        paramMap.put("serveType", serveType);
        paramMap.put("adviserId", adviserId);
        PageBeanNew<BookGroupDTO> pageBean = bookGroupBiz.getAssocBookGroupByAppInfo(paramMap, new PageParam(currentPage, numPerPage));
        return new ResponseDto<>(null == pageBean ? new PageBeanNew<>() : pageBean);
    }

    @Override
    @GetMapping("updateSpecialState")
    public ResponseDto<?> updateSpecialState( @RequestParam("id") Long id, @RequestParam("belongSpecialAgent") Boolean belongSpecialAgent ) {
        bookGroupBiz.updateSpecialState(id, belongSpecialAgent);
        return new ResponseDto<>();
    }

    @ApiOperation("出版社根据bookId获取社群码统计信息")
    @GetMapping("getBookGroupStatisticsByAgent")
    @Override
    public ResponseDto<?> getBookGroupStatisticsByAgent(
            @RequestHeader("token") String token,
            @RequestParam("bookId") Long bookId,
            @RequestParam("adviserId") Long adviserId,
            @RequestParam("channelId") Long channelId
    ) throws BizException, PermissionException, JsonParseException {
        Long agentId = (Long)SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (bookId == null) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "bookId不能为空！");
        }
        return new ResponseDto<>(bookGroupBiz.getBookGroupStatisticsByAgent(bookId, agentId, adviserId, channelId));
    }

    @Override
    @GetMapping("getClassifyKeywordStatistic")
    public ResponseDto<List<ListKeywordVO>> getClassifyKeywordStatistic(
            @RequestHeader("token") String token, @RequestParam("classifyId") Long classifyId ) throws PermissionException {
        SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (classifyId == null) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "classifyId不能为空！");
        }
        List<ListKeywordVO> listKeywordVOS = bookGroupBiz.getClassifyKeywordStatistic(classifyId);
        return new ResponseDto<>(listKeywordVOS);
    }

    @ApiOperation("新增微信用户和rays系统用户关联关系")
    @PostMapping("createWxWechatUserCorrelation")
    @Override
    public ResponseDto<?> createWxWechatUserCorrelation(
            @CookieValue("userInfo") String userInfo,
            @RequestBody WxWechatUserCorrelation wxWechatUserCorrelation
    ) throws BizException, PermissionException {
        Long wechatUserId = Cookie.getId(userInfo, Cookie._WECHAT_USER_ID);
        bookGroupBiz.createWxWechatUserCorrelation(wechatUserId, wxWechatUserCorrelation);
        return new ResponseDto<>();
    }

    @ApiOperation(value = "共读活动-用户列表查询", httpMethod = "POST")
    @PostMapping("listUser4ReadingActivity")
    @Override
    public ResponseDto<PageBeanNew<UserBookInfoVO>> listUser4ReadingActivity( @RequestHeader("token") String token, @RequestBody @ApiParam UserSelectParamDTO userSelectParamDTO ) throws PermissionException {
        SessionUtil.getToken4Redis(token);
        if (null == userSelectParamDTO || null == userSelectParamDTO.getCurrentPage() || null == userSelectParamDTO.getNumPerPage()) {
            throw new WechatGroupBizException(WechatGroupBizException.PARAM_IS_NULL, "缺少参数");
        }
        PageBeanNew<UserBookInfoVO> pageBeanNew = bookGroupBiz.listUser4ReadingActivity(userSelectParamDTO);
        return new ResponseDto<>(pageBeanNew);
    }

    @Override
    @GetMapping("getBookGroupAppletByBookGroupId")
    public ResponseDto<BookAppletSceneDTO> getBookGroupAppletByBookGroupId( @RequestParam("bookGroupId") Long bookGroupId, HttpServletRequest request) {
        if (null == bookGroupId) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "bookGroupId不能为空！");
        }
        String requestIp = NginxUtils.getClientIp(request);
        return new ResponseDto<>(bookGroupBiz.getBookGroupAppletByBookGroupId(bookGroupId, requestIp));
    }

    @Override
    @GetMapping("getBookGroupId4OldData")
    public ResponseDto<Long> getBookGroupId4OldData( @RequestParam("raysBookId") Long raysBookId ) {
        if (null == raysBookId) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "raysBookId不能为空！");
        }
        return new ResponseDto<>(bookGroupBiz.getBookGroupId4OldData(raysBookId));
    }

    @Override
    @GetMapping("getByAppletScene")
    public ResponseDto<?> getByAppletScene( @CookieValue("userInfo") String userInfo, @RequestParam("scene") String scene ) {
        if (StringUtil.isEmpty(scene)) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "scene不能为空！");
        }
        Long wechatUserId = Cookie.getId(userInfo, Cookie._WECHAT_USER_ID);
        Long officialAccountsId = Cookie.getId(userInfo, Cookie._OFFICIAL_ACCOUNTS_ID);
        BookGroupDTO bookGroupDTO = bookGroupBiz.getByAppletScene(wechatUserId, scene, officialAccountsId);
        return new ResponseDto<>(bookGroupDTO);
    }

    @Override
    @GetMapping("getBookAndBookGroupServeList")
    public ResponseDto<?> getBookAndBookGroupServeList(
            @CookieValue("userInfo") String userInfo,
            @RequestParam("adviserId") Long adviserId, @RequestParam("bookId") Long bookId, @RequestParam("channelId") Long channelId ) {
        if (null == adviserId || null == bookId || null == channelId) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数缺失！");
        }
        Long wechatUserId = Cookie.getId(userInfo, Cookie._WECHAT_USER_ID);
        return new ResponseDto<>(bookGroupBiz.getBookAndBookGroupServeList(adviserId, bookId, channelId, wechatUserId));
    }

    @Override
    @GetMapping("getBookBaseInfo4Applet")
    public ResponseDto<?> getBookBaseInfo4Applet(
            @RequestParam(value = "adviserId", required = false) Long adviserId,
            @RequestParam("bookId") Long bookId,
            @RequestParam(value = "channelId", required = false) Long channelId ) {
        if (null == adviserId || null == bookId || null == channelId) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数缺失！");
        }
        return new ResponseDto<>(bookGroupBiz.getBookBaseInfo4Applet(adviserId, channelId, bookId));
    }

    @GetMapping("getRightsSettingId4Book")
    public ResponseDto<Map<String,Object>> getRightsSettingId4Book(
            @RequestParam("adviserId") Long adviserId,
            @RequestParam("bookId") Long bookId,
            @RequestParam("channelId") Long channelId) {
        if (null == adviserId || null == bookId || null == channelId) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数缺失！");
        }
        return new ResponseDto<>(MapUtil.of("rightsSettingId",bookGroupBiz.getRightsSettingId4Book(adviserId, channelId, bookId)));
    }


    @Override
    @GetMapping("updateBookGroupOpenWeapp")
    public ResponseDto<?> updateBookGroupOpenWeapp(
            @RequestHeader("token") String token, @RequestParam("bookGroupId") Long bookGroupId, @RequestParam("openWeapp") Boolean openWeapp){
        SessionUtil.getToken4Redis(token);
        bookGroupBiz.updateBookGroupOpenWeapp(bookGroupId,openWeapp);
        return new ResponseDto<>();
    }
    @Override
    @PostMapping("createSingleBookGroup")
    public ResponseDto<Long> createSingleBookGroup(@RequestHeader("token") String token, @RequestBody @ApiParam BookGroup bookGroup){
        Long adviserId = (Long) SessionUtil.getVlaue(token,SessionUtil.PARTY_ID);
        return new ResponseDto<>(bookGroupBiz.createSingleBookGroup(bookGroup,adviserId));
    }

    @Override
    @GetMapping("updateRelatedBookGroup")
    public ResponseDto<?> updateRelatedBookGroup(@RequestHeader("token") String token,
                                          @RequestParam("bookGroupId") Long bookGroupId, @RequestParam(value = "relatedBookGroupId",required = false) Long relatedBookGroupId) {
        SessionUtil.getToken4Redis(token);
        bookGroupBiz.updateRelatedBookGroup(bookGroupId, relatedBookGroupId);
        return new ResponseDto<>();
    }

    @Override
    @GetMapping("listSingleBookGroup4Adviser")
    public ResponseDto<?> listSingleBookGroup4Adviser(@RequestHeader("token") String token,
                                                      @RequestParam(value = "name", required = false) String name,
                                                      @RequestParam(value = "currentPage", required = true) Integer currentPage,
                                                      @RequestParam(value = "numPerPage", required = true) Integer numPerPage)
            throws BizException, PermissionException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        PageBeanNew<BookGroupDTO> pageBeanNew = bookGroupBiz.listSingleBookGroup4Adviser(name, currentPage, numPerPage, adviserId);
        return new ResponseDto<>(pageBeanNew);
    }


    @ApiOperation(value = "通过 bookGroupQrcodeId 或者 classifyId 获取群二维码永久图文素材", httpMethod = "GET")
    @RequestMapping(value = "/getGroupNewsMaterialById", method = RequestMethod.GET)
    public ResponseDto<?> getGroupNewsMaterialById(@CookieValue("userInfo") String userInfo,
                                                   @RequestParam(value = "bookGroupQrcodeId", required = false) Long bookGroupQrcodeId,
                                                   @RequestParam(value = "classifyId", required = false) Long classifyId,
                                                   @RequestParam(value = "useType", required = false, defaultValue = "0") Integer useType,
                                                   @RequestParam(value = "groupActivityId", required = false) Long groupActivityId) throws PermissionException {
        Cookie.getUserInfo(userInfo);
        if (UseTypeEnum.NORMAL_GROUP.value.equals(useType) && bookGroupQrcodeId == null && classifyId == null) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "useType == 0时 classifyId 和 bookGroupQrcodeId 不可同时为空！");
        }
        if (UseTypeEnum.THIRD_GROUP.value.equals(useType) && null == groupActivityId) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "useType == 1时 groupActivityId 不可为空！");
        }
        Map<String, Object> map = new HashMap<>(1);
        map.put("url", bookGroupBiz.getGroupNewsMaterialById(bookGroupQrcodeId, classifyId, useType, groupActivityId));
        return new ResponseDto<>(map);
    }

    @Override
    @RequestMapping(value = "listBookGroup4AppletAgent", method = RequestMethod.GET)
    public ResponseDto<PageBean> listBookGroup4AppletAgent(
            @RequestHeader("token") String token,
            @RequestParam(value = "name", required = false) String name,
            @RequestParam(value = "adviserId", required = false) Long adviserId,
            @RequestParam(value = "currentPage", required = false) Integer currentPage,
            @RequestParam(value = "numPerPage", required = false) Integer numPerPage)
            throws BizException, PermissionException{
        Map<String, Object> map = SessionUtil.getToken4Redis(token);
        Long agentId = (Long) map.get(SessionUtil.RAY_AGENT_ID);
        if (currentPage == null || numPerPage == null || currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("adviserId", adviserId);
        paramMap.put("name", name != null && "".equals(name.trim()) ? null : name);
        paramMap.put("joinGroupType", JoinGroupTypeEnum.XIAORUI.getCode());
        paramMap.put("agentId", agentId);
        PageBean pageBean = bookGroupBiz.listBookGroup4AppletAgent(paramMap, new PageParam(currentPage, numPerPage));
        return new ResponseDto<>(pageBean);
    }

    @Override
    @GetMapping("getRayBookByBookGroupId")
    public ResponseDto<?> getRayBookByBookGroupId(@RequestParam("bookGroupId") Long bookGroupId){
        return new ResponseDto<>(bookGroupBiz.getRayBookByBookGroupId(bookGroupId));
    }
    @Override
    public ResponseDto<?> getBookGroupsByBookIds(@RequestBody MapResourceTotalCountDTO requestDTO) throws BizException, PermissionException {
        if(CollectionUtils.isEmpty(requestDTO.getBookIds())){
            return new ResponseDto<>(new ArrayList<>());
        }
        return new ResponseDto<>(bookGroupBiz.getBookGroupsByBookIds(requestDTO.getBookIds(), requestDTO.getAdviserIds(), requestDTO.getChannelIds()));
    }

    @Override
    @GetMapping("getBookBase4Applet")
    public ResponseDto<?> getBookBase4Applet(
            @RequestParam(value = "sceneId", required = false) Long sceneId,
            @RequestParam(value = "bookGroupId", required = false) Long bookGroupId) {
        if (null == sceneId && null == bookGroupId) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数缺失！");
        }
        return new ResponseDto<>(bookGroupBiz.getBookBase4Applet(sceneId, bookGroupId));
    }

    @Override
    @GetMapping("getSceneQrcodeByBookGroupId")
    public ResponseDto<?> getSceneQrcodeByBookGroupId(@RequestParam("bookGroupId") Long bookGroupId){
        return new ResponseDto<>(bookGroupBiz.getSceneQrcodeByBookGroupId(bookGroupId));
    }

    @Override
    @PostMapping("createBookGroup")
    public ResponseDto<?> createBookGroup(@RequestHeader("token") String token, @RequestBody @Validated BookGroupCreateDTO bookGroupCreateDTO){
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        bookGroupCreateDTO.setAdviserId(adviserId);
        return new ResponseDto<>(bookGroupBiz.createBookGroup(bookGroupCreateDTO));
    }

    @Override
    @GetMapping("/listHotApp")
    public ResponseDto<List<HotAppDTO>> listHotApp() {
        return new ResponseDto<>(bookGroupBiz.listHotApp());
    }

    @GetMapping("/testListHotApp")
    public void testListHotApp(){
        if (JedisClusterUtils.exists(BookConstant.BOOK_HOT_APP + DateUtil.format(DateUtil.yesterday(), DatePattern.NORM_DATE_FORMAT))) return;
        JedisClusterUtils.set(BookConstant.BOOK_HOT_APP + DateUtil.format(DateUtil.yesterday(), DatePattern.NORM_DATE_FORMAT),
                "[{\"count\":32,\"increment\":0,\"typeCode\":\"MATCH_LISTEN\",\"typeName\":\"配套听力\"},{\"count\":209,\"increment\":0,\"typeCode\":\"VIDEO\",\"typeName\":\"视频\"},{\"count\":16,\"increment\":0,\"typeCode\":\"COURSE_WARE\",\"typeName\":\"课件\"},{\"count\":13,\"increment\":0,\"typeCode\":\"WORD_DICTATION\",\"typeName\":\"生词听写\"},{\"count\":81,\"increment\":0,\"typeCode\":\"AUDIO\",\"typeName\":\"音频\"},{\"count\":19,\"increment\":0,\"typeCode\":\"SPECIAL\",\"typeName\":\"专题拓展\"},{\"count\":9,\"increment\":0,\"typeCode\":\"STROKE_ORDER\",\"typeName\":\"笔顺动图\"},{\"count\":17,\"increment\":0,\"typeCode\":\"WISH\",\"typeName\":\"愿望单\"},{\"count\":13,\"increment\":0,\"typeCode\":\"QA\",\"typeName\":\"专家咨询\"},{\"count\":20,\"increment\":0,\"typeCode\":\"IMAGE\",\"typeName\":\"高清大图\"},{\"count\":3,\"increment\":0,\"typeCode\":\"INTERACT\",\"typeName\":\"签到互动墙\"},{\"count\":12,\"increment\":0,\"typeCode\":\"SUBJECTNOTE\",\"typeName\":\"错题本\"},{\"count\":0,\"increment\":0,\"typeCode\":\"ENROLL\",\"typeName\":\"编创赛报名\"},{\"count\":4,\"increment\":0,\"typeCode\":\"LISTEN\",\"typeName\":\"听力音频\"},{\"count\":18,\"increment\":0,\"typeCode\":\"ZSCORE\",\"typeName\":\"查弱项\"},{\"count\":12,\"increment\":0,\"typeCode\":\"AUDIO_MAGIC\",\"typeName\":\"伴读魔法箱\"},{\"count\":10,\"increment\":0,\"typeCode\":\"EF\",\"typeName\":\"专题问答\"},{\"count\":85,\"increment\":0,\"typeCode\":\"TEST\",\"typeName\":\"趣味测试\"},{\"count\":49,\"increment\":6,\"typeCode\":\"CLOCK\",\"typeName\":\"打卡\"},{\"count\":2,\"increment\":0,\"typeCode\":\"NAIRE\",\"typeName\":\"读者反馈\"},{\"count\":44,\"increment\":0,\"typeCode\":\"LIVE_TIMETABLE\",\"typeName\":\"直播课\"},{\"count\":17,\"increment\":0,\"typeCode\":\"ARTICLE_READING\",\"typeName\":\"课文诵读\"},{\"count\":15,\"increment\":0,\"typeCode\":\"VOTE\",\"typeName\":\"投票\"},{\"count\":11,\"increment\":0,\"typeCode\":\"EBOOK\",\"typeName\":\"热门电子书\"},{\"count\":12,\"increment\":0,\"typeCode\":\"TEST_PAPER\",\"typeName\":\"试卷\"},{\"count\":27,\"increment\":0,\"typeCode\":\"RECITE_WORD\",\"typeName\":\"背单词\"},{\"count\":9,\"increment\":0,\"typeCode\":\"BM\",\"typeName\":\"活动报名\"},{\"count\":55,\"increment\":1,\"typeCode\":\"FLAG\",\"typeName\":\"立Flag\"},{\"count\":15,\"increment\":0,\"typeCode\":\"MEMBER\",\"typeName\":\"开通会员\"},{\"count\":7,\"increment\":0,\"typeCode\":\"BOOKCARD\",\"typeName\":\"读书卡片\"},{\"count\":5,\"increment\":0,\"typeCode\":\"TUTOR\",\"typeName\":\"名师辅导\"},{\"count\":21,\"increment\":0,\"typeCode\":\"GROUP\",\"typeName\":\"读者交流圈\"},{\"count\":33,\"increment\":1,\"typeCode\":\"PDF\",\"typeName\":\"图文\"},{\"count\":35,\"increment\":1,\"typeCode\":\"TEACH_RESOURCE\",\"typeName\":\"教辅资料包\"},{\"count\":8,\"increment\":0,\"typeCode\":\"VOICE_MSG\",\"typeName\":\"作者留言\"},{\"count\":39,\"increment\":1,\"typeCode\":\"ITEM_BANK\",\"typeName\":\"在线题库\"},{\"count\":4,\"increment\":0,\"typeCode\":\"KK\",\"typeName\":\"扫扫看\"},{\"count\":100,\"increment\":1,\"typeCode\":\"ANSWER\",\"typeName\":\"答案\"},{\"count\":20,\"increment\":0,\"typeCode\":\"DRAW\",\"typeName\":\"抽奖\"},{\"count\":11,\"increment\":0,\"typeCode\":\"PBSTORY\",\"typeName\":\"绘本伴读\"},{\"count\":3,\"increment\":2,\"typeCode\":\"FOREIGN_READ\",\"typeName\":\"外刊阅读\"},{\"count\":56,\"increment\":0,\"typeCode\":\"ARTICLE\",\"typeName\":\"精品文章\"},{\"count\":27,\"increment\":0,\"typeCode\":\"COLD_KNOWLEDGE\",\"typeName\":\"冷知识\"},{\"count\":14,\"increment\":0,\"typeCode\":\"BOOK\",\"typeName\":\"资源合集\"},{\"count\":19,\"increment\":1,\"typeCode\":\"TICKET\",\"typeName\":\"准考证\"},{\"count\":17,\"increment\":0,\"typeCode\":\"ENGLISH_WALKMAN\",\"typeName\":\"英语随声听\"},{\"count\":18,\"increment\":0,\"typeCode\":\"ORAL_EVALUATION\",\"typeName\":\"口语评测\"},{\"count\":1,\"increment\":0,\"typeCode\":\"ACTIVITY\",\"typeName\":\"分享互动\"},{\"count\":12,\"increment\":0,\"typeCode\":\"BOOK_CLICK\",\"typeName\":\"点读书\"}]");
        JedisClusterUtils.del(BookConstant.BOOK_HOT_APP + DateUtil.format(new Date(), DatePattern.NORM_DATE_FORMAT));
    }

    @ApiOperation("书刊配置的RAYS应用")
    @GetMapping("getBookAndBookGroupRaysApps")
    public ResponseDto<?> getBookAndBookGroupRaysApps(
            @RequestParam("adviserId") Long adviserId, @RequestParam("bookId")Long bookId, @RequestParam("channelId") Long channelId){
        return new ResponseDto<>(bookGroupBiz.getBookAndBookGroupRaysApps(adviserId, bookId, channelId));
    }

    @ApiOperation("上分MVP-根据模块查课程资源列表")
    @PostMapping("listPageCourse4ExerciseBook")
    public ResponseDto<?> listPageCourse4ExerciseBook(@RequestBody BookServeQueryDTO bookServeQueryDTO) {
        return new ResponseDto<>(bookGroupBiz.listPageCourse4ExerciseBook(bookServeQueryDTO));
    }

    @ApiOperation("上分MVP-课程列表")
    @PostMapping("listCourse4ExerciseBook")
    public ResponseDto<?> listCourse4ExerciseBook(@RequestBody BookServeQueryDTO bookServeQueryDTO) {
        return new ResponseDto<>(bookGroupBiz.listCourse4ExerciseBook(bookServeQueryDTO));
    }

    @ApiOperation("更新社群码排序值")
    @GetMapping("updateBookGroupSeq")
    public ResponseDto<?> updateBookGroupSeq(@RequestHeader("token") String token,
                                             @RequestParam("bookGroupId") Long bookGroupId,
                                             @RequestParam("seq") Integer seq) {
        SessionUtil.getToken4Redis(token);
        bookGroupBiz.updateBookGroupSeq(bookGroupId, seq);
        return new ResponseDto<>();
    }
}
