package com.pcloud.book.book.biz.impl;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import com.pcloud.appcenter.app.dto.AppDto;
import com.pcloud.book.base.exception.BookBizException;
import com.pcloud.book.book.biz.BookKeywordWarehouseBiz;
import com.pcloud.book.book.constant.BookConstant;
import com.pcloud.book.book.dao.KeywordUseRecordDao;
import com.pcloud.book.book.dao.KeywordWarehouseDao;
import com.pcloud.book.book.dao.KeywordWarehouseDetailDao;
import com.pcloud.book.book.entity.KeywordUseRecord;
import com.pcloud.book.book.entity.KeywordWarehouse;
import com.pcloud.book.book.entity.KeywordWarehouseDetail;
import com.pcloud.book.book.vo.BookKeywordDetailVO;
import com.pcloud.book.book.vo.BookKeywordProductVO;
import com.pcloud.book.book.vo.KeywordWearhouseLabelVO;
import com.pcloud.book.book.vo.request.AddBookKeywordRequestVO;
import com.pcloud.book.book.vo.request.CheckKeywordRequestVO;
import com.pcloud.book.book.vo.request.EditBookKeywordRequestVO;
import com.pcloud.book.book.vo.request.QueryBookKeywordWarehouseRequestVO;
import com.pcloud.book.book.vo.response.BookKeywordResponseVO;
import com.pcloud.book.book.vo.response.CheckKeywordResponseVO;
import com.pcloud.book.consumer.app.AppConsr;
import com.pcloud.book.consumer.label.LabelConsr;
import com.pcloud.book.consumer.resource.ProductConsr;
import com.pcloud.book.consumer.user.AdviserConsr;
import com.pcloud.book.consumer.user.PartyConsr;
import com.pcloud.book.group.biz.BookGroupBiz;
import com.pcloud.book.group.dto.BookGroupDTO;
import com.pcloud.book.keywords.biz.BookKeywordBiz;
import com.pcloud.book.keywords.vo.ListKeywordParam;
import com.pcloud.book.keywords.vo.ListKeywordVO;
import com.pcloud.book.keywords.vo.SetKeywordVO;
import com.pcloud.common.core.aspect.ParamLog;
import com.pcloud.common.core.constant.SystemCode;
import com.pcloud.common.entity.BaseRequestVO;
import com.pcloud.common.page.PageBeanNew;
import com.pcloud.common.page.PageParam;
import com.pcloud.common.utils.BeanUtils;
import com.pcloud.common.utils.string.StringUtil;
import com.pcloud.resourcecenter.product.dto.ProductLabelDto;
import com.pcloud.usercenter.party.adviser.dto.AdviserBaseInfoDto;
import com.pcloud.usercenter.party.agent.entity.Agent;

import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import lombok.extern.slf4j.Slf4j;

@Slf4j
@Service
public class BookKeywordWarehouseBizImpl implements BookKeywordWarehouseBiz {

    @Resource
    private KeywordWarehouseDao keywordWarehouseDao;
    @Resource
    private KeywordWarehouseDetailDao keywordWarehouseDetailDao;
    @Resource
    private KeywordUseRecordDao keywordUseRecordDao;
    @Resource
    private BookKeywordBiz bookKeywordBiz;
    @Resource
    private BookGroupBiz bookGroupBiz;
    @Resource
    private PartyConsr partyConsr;
    @Resource
    private AdviserConsr adviserConsr;
    @Resource
    private LabelConsr labelConsr;
    @Resource
    private AppConsr appConsr;
    @Resource
    private ProductConsr productConsr;

    /**
     * 添加关键词的时候同步加入到仓库
     */
    @Override
    @ParamLog("[addBookKeyword SetKeywordVO]")
    @Transactional(rollbackFor = {Exception.class})
    public Long addBookKeyword(SetKeywordVO setKeywordVO) {
        if (null == setKeywordVO || 1 != setKeywordVO.getAddWarehouse()) {
            log.info("关键词添加到仓库失败，参数为空");
            return 0L;
        }
        final AddBookKeywordRequestVO vo = new AddBookKeywordRequestVO();
        vo.setKeyword(setKeywordVO.getKeywords());
        vo.setGuide(setKeywordVO.getGuide());
        vo.setMatchingRule(setKeywordVO.getMatchingRule());
        // 加入词库是属于编辑自己
        vo.setScope(BookConstant.BookKeywordScopeEnum.EDITOR.getCode());
        vo.setEditorId(setKeywordVO.getUserId());
//        vo.setAuditStatus(BookConstant.BookKeywordAuditEnum.TO_AUDIT.getCode());
        vo.setAuditStatus(BookConstant.BookKeywordAuditEnum.PASS_THE_AUDIT.getCode());
        final List<BookKeywordDetailVO> details = Lists.newArrayList();
        final List<KeywordWearhouseLabelVO> labels = Lists.newArrayList();
        final BookGroupDTO info = bookGroupBiz.getBookGroupInfo(setKeywordVO.getBookGroupId());
        if (null == info) {
            throw new BookBizException(BookBizException.ID_NOT_EXIST, "社群码不存在");
        }
        final KeywordWearhouseLabelVO v = new KeywordWearhouseLabelVO();
        if(null == info.getProLabelId() || null == info.getDepLabelId() || null == info.getPurLabelId()) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL,"请先完成社群码信息");
        }
        v.setProfessionalLabelId(info.getProLabelId());
        v.setDepthLabelId(info.getDepLabelId());
        v.setPurposeLabelId(info.getPurLabelId());
        labels.add(v);
        final BookKeywordDetailVO v1 = new BookKeywordDetailVO();
        v1.setGroupId(1);
        v1.setChannelId(setKeywordVO.getChannelId());
        v1.setContent(setKeywordVO.getContent());
        v1.setProductId(setKeywordVO.getServeId());
        v1.setProductType(setKeywordVO.getServeType());
        v1.setDescription(setKeywordVO.getDescription());
        v1.setTypeCode(setKeywordVO.getTypeCode());
        v1.setLinkUrl(setKeywordVO.getLinkUrl());
        v1.setPicUrl(setKeywordVO.getPicUrl());
        v1.setLabels(labels);
        details.add(v1);
        vo.setDetails(details);
        vo.setUserId(setKeywordVO.getUserId());
        return this.addBookKeyword(vo, 1);
    }

    @Override
    @ParamLog("[addBookKeyword]")
    @Transactional(rollbackFor = {Exception.class})
    public Long addBookKeyword(AddBookKeywordRequestVO vo, Integer useKeyword) {
        if (null == vo || vo.check()) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数错误");
        }
        final KeywordWarehouse keyword = vo.getEntity();
        keyword.setCreateUser(vo.getUserId());
        keyword.setCreateTime(keyword.getUpdateTime());
        this.keywordWarehouseDao.insert(keyword);
        final List<KeywordWarehouseDetail> detail = vo.getDetailEntity(keyword.getId());
        if(null != useKeyword && 1 == useKeyword) {
            final KeywordWarehouseDetail detail1 = detail.get(0);
            this.keywordWarehouseDetailDao.insert(detail1);
            this.useKeyword(vo.getUserId(), detail1.getId());
        } else {
            this.keywordWarehouseDetailDao.insert(detail);
        }
        this.onlineProduct(vo);
        return keyword.getId();
    }

    @Override
    @ParamLog("[editBookKeyword]")
    @Transactional(rollbackFor = {Exception.class})
    public void editBookKeyword(EditBookKeywordRequestVO vo) {
        if (null == vo || vo.check()) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数错误");
        }
        final KeywordWarehouse keyword = vo.getEntity();
        // 1、修改关键词 2、标记删除详情 3、记录详情
        this.keywordWarehouseDao.updateByPrimaryKeySelective(keyword);
        this.keywordWarehouseDetailDao.deleteDetails(vo.getKeywordId());
        this.keywordWarehouseDetailDao.insert(vo.getDetailEntity(vo.getKeywordId()));
        this.onlineProduct(vo);
    }

    /**
     * 处理需要上架的商品信息
     */
    private void onlineProduct(BaseRequestVO vo) {
        Map<Long, List<BookKeywordDetailVO>> collect;
        if (vo instanceof AddBookKeywordRequestVO) {
            final AddBookKeywordRequestVO vo1 = (AddBookKeywordRequestVO) vo;
            collect = vo1.getDetails().stream().filter(x -> "PRODUCT".equals(x.getProductType())).collect(Collectors.groupingBy(BookKeywordDetailVO::getChannelId));
        } else if (vo instanceof EditBookKeywordRequestVO) {
            final EditBookKeywordRequestVO vo1 = (EditBookKeywordRequestVO) vo;
            collect = vo1.getDetails().stream().filter(x -> "PRODUCT".equals(x.getProductType())).collect(Collectors.groupingBy(BookKeywordDetailVO::getChannelId));
        } else {
            log.info("[onlineProduct] 类型错误");
            return;
        }
        if (CollectionUtils.isEmpty(collect)) {
            log.info("[addBookKeyword] 没有商品需要上架");
            return;
        }
        for (Long channelId : collect.keySet()) {
            final List<BookKeywordDetailVO> vos = collect.get(channelId);
            if (CollectionUtils.isEmpty(vos)) {
                log.info("[addBookKeyword] channelId 没有作品：{}", channelId);
                continue;
            }
            // 上架商品
            this.productConsr.productAutoOnShelves(channelId, vos.stream().map(BookKeywordDetailVO::getProductId).collect(Collectors.toList()));
        }
    }

    @Override
    @ParamLog("[deleteBookKeyword]")
    @Transactional(rollbackFor = {Exception.class})
    public void deleteBookKeyword(Long keywordId, Long adviserId) {
        if (null == keywordId || null == adviserId) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数错误");
        }
        final KeywordWarehouse keywordWarehouse = keywordWarehouseDao.selectByPrimaryKey(keywordId);
        if (null == keywordWarehouse || 0 != keywordWarehouse.getIsDelete()) {
            log.info("[deleteBookKeyword] keywordWarehouse:{}", keywordWarehouse);
            return;
        }
        keywordWarehouse.setUpdateUser(adviserId);
        keywordWarehouse.setUpdateTime(new Date());
        keywordWarehouse.setIsDelete(1);
        this.keywordWarehouseDao.updateByPrimaryKey(keywordWarehouse);
        this.keywordWarehouseDetailDao.deleteDetails(keywordId);
    }

    @Override
    @ParamLog("[listBookKeywordWarehouse]")
    public PageBeanNew<BookKeywordResponseVO> listBookKeywordWarehouse(QueryBookKeywordWarehouseRequestVO vo) {
        if (null == vo || null == vo.getCurrentPage() || null == vo.getNumPerPage()) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "分页参数为空");
        }
        // 根据主表分页，然后组装子表数据
        PageBeanNew<BookKeywordResponseVO> warehouse = this.keywordWarehouseDao.listPageNew(new PageParam(vo.getCurrentPage(), vo.getNumPerPage()), BeanUtils.beanToMap(vo), "listBookKeywordWarehouse");
        if (null == warehouse || CollectionUtils.isEmpty(warehouse.getRecordList())) {
            return warehouse;
        }
        final List<BookKeywordResponseVO> list = warehouse.getRecordList();
        final List<Long> ids = list.stream().map(BookKeywordResponseVO::getId).collect(Collectors.toList());
        final List<KeywordWarehouseDetail> details = this.keywordWarehouseDetailDao.listByKeywordIds(ids);
        if (CollectionUtils.isEmpty(details)) {
            return warehouse;
        }
        this.processData(warehouse.getRecordList(), details);
        return warehouse;
    }

    private void processData(List<BookKeywordResponseVO> list, List<KeywordWarehouseDetail> details) {
        final Map<Long, List<KeywordWarehouseDetail>> map = details.stream().collect(Collectors.groupingBy(KeywordWarehouseDetail::getKeywordId));
        final List<Long> editIds = Lists.newArrayList();
        for (BookKeywordResponseVO vo1 : list) {
            Long id = vo1.getId();
            final List<KeywordWarehouseDetail> warehouseDetails = map.get(id);
            if (CollectionUtils.isEmpty(warehouseDetails)) {
                log.info("warehouseDetails is null");
                continue;
            }
            editIds.add(vo1.getEditorId());
            List<BookKeywordDetailVO> vos = Lists.newArrayList();
            final List<String> groupIds = warehouseDetails.stream().map(KeywordWarehouseDetail::getGroupId).distinct().collect(Collectors.toList());
            final Map<String, List<KeywordWarehouseDetail>> collect = warehouseDetails.stream().collect(Collectors.groupingBy(KeywordWarehouseDetail::getGroupId));
            // 一个关键词对应多个作品   一个作品对应多组标签
            List<Long> appIds = Lists.newArrayList();
            List<Long> productIds = Lists.newArrayList();
            for (String s : groupIds) {
                final List<KeywordWarehouseDetail> keywordWarehouseDetails = collect.get(s);
                if (CollectionUtils.isEmpty(keywordWarehouseDetails)) {
                    log.info("keywordWarehouseDetails is null");
                    continue;
                }
                BookKeywordDetailVO v = BookKeywordDetailVO.cloneByEntity(keywordWarehouseDetails.get(0));
                if ("APP".equals(v.getProductType())) {
                    appIds.add(v.getProductId());
                }
                if ("PRODUCT".equals(v.getProductType())) {
                    productIds.add(v.getProductId());
                }
                // 填充标签名称
                List<KeywordWearhouseLabelVO> labels = Lists.newArrayList();
                final List<Long> labelIds = Lists.newArrayList();
                for (KeywordWarehouseDetail detail : keywordWarehouseDetails) {
                    KeywordWearhouseLabelVO v1 = KeywordWearhouseLabelVO.cloneByEntity(detail);
                    labelIds.add(v1.getPurposeLabelId());
                    labelIds.add(v1.getDepthLabelId());
                    labelIds.add(v1.getProfessionalLabelId());
                    labels.add(v1);
                }
                final Map<Long, String> labelName = labelConsr.getLabelName(labelIds);
                for (KeywordWearhouseLabelVO label : labels) {
                    label.setPurposeLabelName(labelName.get(label.getPurposeLabelId()));
                    label.setDepthLabelName(labelName.get(label.getDepthLabelId()));
                    label.setProfessionalLabelName(labelName.get(label.getProfessionalLabelId()));
                }
                v.setLabels(labels);
                vos.add(v);
            }
            final Map<Long, ProductLabelDto> productLabelDtoMap = productConsr.getLabelByProIds(productIds);
            // 填充应用标签
            final Map<Long, AppDto> baseByIds = appConsr.mapBaseByIds(appIds);
            for (BookKeywordDetailVO detailVO : vos) {
                if("APP".equals(detailVO.getProductType())) {
                    final AppDto appDto = baseByIds.get(detailVO.getProductId());
                    if (null == appDto) {
                        log.info("[listBookKeywordWarehouse] ProductId ：{} 未找到应用", detailVO.getProductId());
                        continue;
                    }
                    detailVO.setProLabelName(appDto.getProLabelName());
                    detailVO.setDepLabelName(appDto.getDepLabelName());
                    detailVO.setPurLabelName(appDto.getPurLabelName());
                }
                if("PRODUCT".equals(detailVO.getProductType())) {
                    final ProductLabelDto labelDto = productLabelDtoMap.get(detailVO.getProductId());
                    if (null == labelDto) {
                        log.info("[listBookKeywordWarehouse] ProductId ：{} 未找到应用", detailVO.getProductId());
                        continue;
                    }
                    detailVO.setProLabelName(labelDto.getProLabelName());
                    detailVO.setDepLabelName(labelDto.getDepLabelName());
                    detailVO.setPurLabelName(labelDto.getPurLabelName());
                }

            }
            vo1.setProduct(vos);
        }
        // 填充编辑名称
        final Map<Long, String> names = adviserConsr.getNames(editIds);
        final Map<Long, AdviserBaseInfoDto> agentMap = adviserConsr.getAdviserId2AdviserInfoDtoMap(editIds);
        for (BookKeywordResponseVO vo1 : list) {
            final AdviserBaseInfoDto infoDto = agentMap.get(vo1.getEditorId());
            vo1.setEditorName(names.get(vo1.getEditorId()));
            vo1.setAgentName(null == infoDto ? "" : infoDto.getAgentName());
        }
    }


    @Override
    @ParamLog("[useKeyword]")
    @Transactional(rollbackFor = {Exception.class})
    public void useKeyword(Long adviserId, Long keywordDetailId) {
        if (Objects.isNull(adviserId) || Objects.isNull(keywordDetailId)) {
            return;
        }
        final KeywordWarehouseDetail detail = this.keywordWarehouseDetailDao.selectByPrimaryKey(keywordDetailId);
        if (Objects.isNull(detail) || detail.getIsDelete() != 0) {
            log.info("[useKeyword] warehouse is null");
            return;
        }
        final KeywordUseRecord record = this.keywordUseRecordDao.getByAdviserIdAndKeywordId(adviserId, keywordDetailId);
        final Date now = new Date();
        // 常用词库+1
        if (Objects.isNull(record)) {
            KeywordUseRecord r = new KeywordUseRecord();
            r.setAdviserId(adviserId);
            r.setKeywordId(detail.getKeywordId());
            r.setKeywordDetailId(keywordDetailId);
            r.setUseNum(1);
            r.setCreateUser(adviserId);
            r.setCreateTime(now);
            r.setUpdateUser(adviserId);
            r.setUpdateTime(now);
            r.setIsDelete(0);
            this.keywordUseRecordDao.insert(r);
        } else {
            this.keywordUseRecordDao.addOne(adviserId, record.getId());
        }
        // 总词库+1
        this.keywordWarehouseDetailDao.addOne(adviserId, detail.getId());
    }

    @Override
    @ParamLog("[getDetail]")
    public BookKeywordResponseVO getDetail(Long keywordId) {
        QueryBookKeywordWarehouseRequestVO vo = new QueryBookKeywordWarehouseRequestVO();
        vo.setId(keywordId);
        vo.setCurrentPage(0);
        vo.setNumPerPage(1);
        final PageBeanNew<BookKeywordResponseVO> keyword = this.listBookKeywordWarehouse(vo);
        if (Objects.isNull(keyword) || CollectionUtils.isEmpty(keyword.getRecordList())) {
            return new BookKeywordResponseVO();
        }
        final BookKeywordResponseVO responseVO = keyword.getRecordList().get(0);
        if (Objects.isNull(responseVO)) {
            return new BookKeywordResponseVO();
        }
        final Map<Long, Agent> byPartyId = partyConsr.getAgentByPartyId(Collections.singletonList(responseVO.getEditorId()), SystemCode.adviser.code);
        final Agent agent = Optional.ofNullable(byPartyId).orElse(Maps.newHashMap()).get(responseVO.getEditorId());
        responseVO.setAgentId(null == agent ? 0L : agent.getAgentId());
        return responseVO;
    }

    @Override
    @ParamLog("[oftenKeyword]")
    public PageBeanNew<BookKeywordProductVO> oftenKeyword(Integer currentPage, Integer numPerPage, Long bookGroupId, Long adviserId, Long depthLabelId, Long classifyId) {
        Map<String, Object> map = Maps.newHashMap();
        map.put("depthLabelId", depthLabelId);
        map.put("adviserId", adviserId);
        map.put("bookGroupId", bookGroupId);
        final PageBeanNew<BookKeywordProductVO> oftenKeyword = this.keywordUseRecordDao.listPageNew(new PageParam(currentPage, numPerPage), map, "oftenKeyword");
        if (null == oftenKeyword || CollectionUtils.isEmpty(oftenKeyword.getRecordList())) {
            return oftenKeyword;
        }
        PageBeanNew<BookKeywordProductVO> pageBeanNew = this.processIsMark(oftenKeyword, bookGroupId, classifyId);
        return this.processKeyword(pageBeanNew);
    }

    @Override
    @ParamLog("[hotKeyword]")
    public PageBeanNew<BookKeywordProductVO> hotKeyword(Integer currentPage, Integer numPerPage, Long adviserId, Long bookGroupId, String keyword, Long depthLabelId, Long classifyId, Boolean isForAdviser) {
        final Map<String, Object> map = Maps.newHashMap();
        map.put("keyword", keyword);
        map.put("depthLabelId", depthLabelId);
        map.put("bookGroupId", bookGroupId);
        map.put("adviserId", adviserId);
        map.put("isForAdviser", isForAdviser);
        final PageBeanNew<BookKeywordProductVO> hotKeyword = this.keywordWarehouseDetailDao.listPageNew(new PageParam(currentPage, numPerPage), map, "hotKeyword");
        if (null == hotKeyword || CollectionUtils.isEmpty(hotKeyword.getRecordList())) {
            return hotKeyword;
        }
        return this.processIsMark(hotKeyword, bookGroupId, classifyId);
    }

    /**
     * 标记已选择列表
     */
    private PageBeanNew<BookKeywordProductVO> processIsMark(PageBeanNew<BookKeywordProductVO> hotKeyword, Long bookGroupId, Long classifyId) {
        ListKeywordParam param = new ListKeywordParam();
        param.setBookGroupId(bookGroupId);
        param.setClassifyId(classifyId);
        param.setCurrentPage(0);
        param.setNumPerPage(200);
        final PageBeanNew<ListKeywordVO> beanNew = bookKeywordBiz.listKeywordsByClassify(param);
        if (null == beanNew || CollectionUtils.isEmpty(beanNew.getRecordList())) {
            return hotKeyword;
        }
        final Map<Long, Integer> map1 = Maps.newHashMap();
        for (ListKeywordVO vo : beanNew.getRecordList()) {
            map1.put(vo.getWarehouseId(), 1);
        }
        for (BookKeywordProductVO vo : hotKeyword.getRecordList()) {
            final Integer num = map1.get(vo.getKeywordId());
            vo.setIsMark(null == num ? 0 : 1);
        }
        return hotKeyword;
    }

    /**
     * 标记已选择列表
     */
    private PageBeanNew<BookKeywordProductVO> processKeyword(PageBeanNew<BookKeywordProductVO> hotKeyword) {
        if (null == hotKeyword || CollectionUtils.isEmpty(hotKeyword.getRecordList())) {
            return hotKeyword;
        }
        final List<Long> ids = hotKeyword.getRecordList().stream().map(BookKeywordProductVO::getKeywordId).distinct().collect(Collectors.toList());
        final List<KeywordWarehouse> list = keywordWarehouseDao.listByIds(ids);
        if (CollectionUtils.isEmpty(list)) {
            return hotKeyword;
        }
        final Map<Long, String> map1 = Maps.newHashMap();
        for (KeywordWarehouse warehouse : list) {
            map1.put(warehouse.getId(), warehouse.getKeyword());
        }
        for (BookKeywordProductVO vo : hotKeyword.getRecordList()) {
            final String keyword = map1.get(vo.getKeywordId());
            vo.setKeyword(StringUtil.isEmpty(keyword) ? "" : keyword);
        }
        return hotKeyword;
    }


    @Override
    @ParamLog("[auditKeyword]")
    public void auditKeyword(Long partyId, Long keywordId, BookConstant.BookKeywordAuditEnum auditEnum) {
        final KeywordWarehouse warehouse = this.keywordWarehouseDao.selectByPrimaryKey(keywordId);
        if (Objects.isNull(warehouse)) {
            throw new BookBizException(BookBizException.RESULT_NULL, "关键词不存在");
        }
        warehouse.setAuditStatus(auditEnum.getCode());
        if (BookConstant.BookKeywordAuditEnum.PASS_THE_AUDIT.equals(auditEnum)) {
            warehouse.setScope(BookConstant.BookKeywordScopeEnum.ALL.getCode());
            warehouse.setEditorId(0L);
        }
        this.keywordWarehouseDao.updateByPrimaryKey(warehouse);
    }

    @Override
    public List<Long> checkKeyword(CheckKeywordRequestVO vo) {
        final List<CheckKeywordResponseVO> responseVO = this.keywordWarehouseDao.checkKeyword(vo);
        if (CollectionUtils.isEmpty(responseVO)) {
            return Lists.newArrayList();
        }
        return responseVO.stream().filter(x -> x.getNum() > 0).filter(x -> !x.getKeywordId().equals(vo.getKeywordId())).map(CheckKeywordResponseVO::getDepthLabelId).collect(Collectors.toList());
    }

}
