package com.pcloud.book.push.biz.impl;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.pcloud.appcenter.app.dto.AppDto;
import com.pcloud.book.base.exception.BookBizException;
import com.pcloud.book.consumer.app.AppConsr;
import com.pcloud.book.consumer.channel.QrcodeSceneConsr;
import com.pcloud.book.consumer.resource.ProductConsr;
import com.pcloud.book.consumer.wechatgroup.WechatGroupConsr;
import com.pcloud.book.group.biz.BookGroupClassifyBiz;
import com.pcloud.book.group.dao.BookGroupClassifyDao;
import com.pcloud.book.group.dao.GroupQrcodeDao;
import com.pcloud.book.group.dto.ClassifyDTO;
import com.pcloud.book.group.dto.GroupClassifyQrcodeDTO;
import com.pcloud.book.group.entity.GroupQrcode;
import com.pcloud.book.group.tools.SendWeixinRequestTools;
import com.pcloud.book.push.biz.PushBiz;
import com.pcloud.book.push.check.PushCheck;
import com.pcloud.book.push.dao.*;
import com.pcloud.book.push.dto.PushDTO;
import com.pcloud.book.push.dto.PushGroupDTO;
import com.pcloud.book.push.dto.PushRecordDTO;
import com.pcloud.book.push.entity.*;
import com.pcloud.book.push.enums.ItemTypeEnum;
import com.pcloud.book.push.enums.PushStatusEnum;
import com.pcloud.book.push.enums.PushTypeEnum;
import com.pcloud.channelcenter.wechat.dto.AccountSettingDto;
import com.pcloud.common.core.aspect.ParamLog;
import com.pcloud.common.page.PageBeanNew;
import com.pcloud.common.page.PageParam;
import com.pcloud.common.utils.BeanUtils;
import com.pcloud.common.utils.DateUtils;
import com.pcloud.common.utils.ListUtils;
import com.pcloud.common.utils.string.StringUtil;
import com.pcloud.facade.quartz.entity.CallBackParam;
import com.pcloud.facade.quartz.entity.ScheduleJob;
import com.pcloud.facade.quartz.service.ScheduleService;
import com.pcloud.resourcecenter.product.dto.ProductDto;
import com.sdk.wxgroup.SendArticleMessageVO;
import com.sdk.wxgroup.SendPicMessageVO;
import com.sdk.wxgroup.SendTextMessageVO;
import com.sdk.wxgroup.WxGroupSDK;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.concurrent.*;
import java.util.stream.Collectors;

/**
 * @Description
 * @Author ruansiyuan
 * @Date 2019/4/17 17:46
 **/
@Component("pushBiz")
public class PushBizImpl implements PushBiz {

    private static final Logger LOGGER = LoggerFactory.getLogger(PushBizImpl.class);

    @Autowired
    private PushCheck pushCheck;
    @Autowired
    private PushDao pushDao;
    @Autowired
    private PushGroupDao pushGroupDao;
    @Autowired
    private PushItemDao pushItemDao;
    @Autowired
    private ScheduleService scheduleService;
    @Autowired
    private PushGroupRecordDao pushGroupRecordDao;
    @Autowired
    private PushRecordDao pushRecordDao;
    @Autowired
    private GroupQrcodeDao groupQrcodeDao;
    @Autowired
    private AppConsr appConsr;
    @Autowired
    private ProductConsr productConsr;
    @Autowired
    private WechatGroupConsr wechatGroupConsr;
    @Autowired
    private BookGroupClassifyBiz bookGroupClassifyBiz;
    @Autowired
    private QrcodeSceneConsr qrcodeSceneConsr;
    @Autowired
    private BookGroupClassifyDao bookGroupClassifyDao;

    private static final String PUSH_SCHEDULE_PRE="BOOK_GROUP_PUSH_";

    private static final ThreadFactory NAMED_THREAD_FACTORY = new ThreadFactoryBuilder().setNameFormat("pushBizImpl-pool-%d").build();
    private static final ExecutorService EXECUTOR_SERVICE = new ThreadPoolExecutor(5, 20, 0L, TimeUnit.MILLISECONDS,
            new LinkedBlockingQueue<>(1024), NAMED_THREAD_FACTORY, new ThreadPoolExecutor.CallerRunsPolicy());

    @Transactional(rollbackFor = Exception.class)
    @ParamLog("创建群发")
    @Override
    public void createPush(Push push) {
        //校验参数
        pushCheck.createPushParamCheck(push);
        //插入群发
        pushDao.insert(push);
        //填充关联模型字段
        fillPushGroupParam(push);
        //插入群发关联表
        pushGroupDao.batchInsert(push.getPushGroups());
        //填充消息项模型
        fillPushItemParam(push);
        //插入群发消息项
        pushItemDao.batchInsert(push.getPushItems());
        //遍历，如果含有超级作者作品，自动上架
        EXECUTOR_SERVICE.execute(() -> itemProductAutoOnShelves(push.getPushGroups(), push.getPushItems()));
        //设置定时任务
        pushQuartz(push);
    }

    /**
     * 遍历，如果含有超级作者作品，自动上架
     */
    @ParamLog("遍历，如果含有超级作者作品，自动上架")
    private void itemProductAutoOnShelves(List<PushGroup> pushGroups, List<PushItem> pushItems) {
        if (ListUtils.isEmpty(pushItems)) {
            return;
        }
        if (ListUtils.isEmpty(pushGroups)) {
            return;
        }
        List<Long> allProductIds = pushItems.stream().filter(s -> ItemTypeEnum.APP.value.equals(s.getItemType()) && s.getProductId() != null).map(PushItem::getProductId).collect(Collectors.toList());
        // 过滤判断是否是超级作者
        Map<Long, Boolean> isSuperMap = productConsr.getIsSuperByProductIdList(allProductIds);
        for (PushGroup pushGroup : pushGroups) {
            //通过群id获取对应基本信息
            ClassifyDTO classifyDTO = bookGroupClassifyDao.getById(pushGroup.getClassifyId());
            if (classifyDTO == null) {
                LOGGER.error("未找到群分类信息" + classifyDTO.toString());
                break;
            }
            Long channelId = classifyDTO.getChannelId();
            List<Long> productIds = new ArrayList<>();
            for (PushItem pushItem : pushItems) {
                if (ItemTypeEnum.APP.value.equals(pushItem.getItemType())) {
                    Long productId = pushItem.getProductId();
                    if (productId != null && isSuperMap.get(productId) != null && isSuperMap.get(productId)) {
                        productIds.add(productId);
                    }
                }
            }
            if (channelId != null && !ListUtils.isEmpty(productIds)) {
                productConsr.productAutoOnShelves(channelId, productIds);
            }
        }
    }

    /**
     * 设置任务
     * @param push
     */
    @ParamLog("设置任务")
    private void pushQuartz(Push push) {
        Integer pushType = push.getPushType();
        if (PushTypeEnum.NOW.value.equals(pushType)) {
            sendGroupMessage(push.getId());
        } else if (PushTypeEnum.ONE.value.equals(pushType)) {
            pushOne(push);
        } else if (PushTypeEnum.DAY.value.equals(pushType)) {
            pushDay(push);
        }
    }

    @Transactional(rollbackFor = Exception.class)
    @ParamLog("发送群发消息")
    @Override
    public void sendGroupMessage(Long pushId) {
        Push push=pushDao.getById(pushId);
        if (push==null){
            return;
        }
        //查询关联群
        List<PushGroup> pushGroupList=pushGroupDao.getListByPushId(pushId);
        if (ListUtils.isEmpty(pushGroupList)){
            return;
        }
        //查询消息项
        List<PushItem> pushItemList = pushItemDao.getListByPushId(pushId);
        if (ListUtils.isEmpty(pushItemList)){
            return;
        }
        //群发消息记录
        PushRecord pushRecord = new PushRecord();
        BeanUtils.copyProperties(push, pushRecord);
        pushRecord.setPushId(pushId);
        try {
            pushRecord.setPushStatus(PushStatusEnum.PUSHING.value);
            //新增群发记录
            pushRecordDao.insert(pushRecord);
            //新增细化到群的记录并发送消息
            addPushGroupRecordsAndSend(pushGroupList, pushRecord, pushItemList);
        } catch (Exception e) {
            LOGGER.error(e.getMessage(), "群发消息失败！"+e);
            pushRecordDao.updateStateById(pushRecord.getId(), PushStatusEnum.FAIL.value);
            pushGroupRecordDao.updateStateByPushRecordId(pushRecord.getId(), PushStatusEnum.FAIL.value);
        }
    }

    /**
     * 新增群发记录，细化到群
     * @param pushGroupList
     * @param pushRecord
     */
    private void addPushGroupRecordsAndSend(List<PushGroup> pushGroupList, PushRecord pushRecord, List<PushItem> pushItemList) {
        //群发消息记录，细化到群
        List<PushGroupRecord> pushGroupRecords = new ArrayList<>();
        for (PushGroup pushGroup : pushGroupList) {
            PushGroupRecord pushGroupRecord = new PushGroupRecord();
            BeanUtils.copyProperties(pushGroup, pushGroupRecord);
            pushGroupRecord.setPushRecordId(pushRecord.getId());
            pushGroupRecord.setPushStatus(pushRecord.getPushStatus());
            pushGroupRecords.add(pushGroupRecord);
            pushGroupRecordDao.insert(pushGroupRecord);
            //遍历发送消息
            for (PushItem pushItem : pushItemList) {
                sendWechatMessage(pushGroup, pushItem, pushGroupRecord.getId());
            }
        }
    }

    /**
     * 发消息
     */
    @ParamLog("发消息")
    private void sendWechatMessage(PushGroup pushGroup, PushItem pushItem, Long pushGroupRecordId) {
        //获取群信息
        GroupQrcode groupQrcode = groupQrcodeDao.getById(pushGroup.getBookGroupQrcodeId());
        if (groupQrcode == null) {
            LOGGER.error("未找到群信息！");
            return;
        }
        String groupId = groupQrcode.getWeixinGroupId();
        ClassifyDTO classifyDTO = bookGroupClassifyDao.getById(groupQrcode.getClassifyId());
        if (classifyDTO == null) {
            LOGGER.error("未找到分类信息！");
            return;
        }
        String otherUrl = "book_group_id=" + classifyDTO.getBookGroupId() + "&classify_id=" + classifyDTO.getId() + "&qrcode_id=" + pushGroup.getBookGroupQrcodeId();
        //获取机器人微信号
        String altId = wechatGroupConsr.getRobotIdByGroupId(groupId);
        if (groupQrcode != null && !StringUtil.isEmpty(groupId) && !StringUtil.isEmpty(altId)) {
            Integer itemType = pushItem.getItemType();
            if (ItemTypeEnum.TEXT.value.equals(itemType)) {
                SendTextMessageVO sendTextMessageVO = new SendTextMessageVO();
                sendTextMessageVO.setGroupId(groupId);
                sendTextMessageVO.setAltId(altId);
                sendTextMessageVO.setContent(pushItem.getTextContent());
                sendTextMessageVO.setPushGroupRecordId(pushGroupRecordId);
                WxGroupSDK.sendTextMessage(sendTextMessageVO);
            }
            if (ItemTypeEnum.LINK.value.equals(itemType)) {
                SendArticleMessageVO sendArticleMessageVO = new SendArticleMessageVO();
                sendArticleMessageVO.setAltId(altId);
                sendArticleMessageVO.setDescription(pushItem.getLinkDescription());
                sendArticleMessageVO.setGroupId(groupId);
                String url = pushItem.getLinkUrl();
                if (url.contains("?")) {
                    url = url + "&" + otherUrl;
                } else {
                    url = url + "?" + otherUrl;
                }
                sendArticleMessageVO.setLinkUrl(url);
                sendArticleMessageVO.setPicUrl(pushItem.getLinkImageUrl());
                sendArticleMessageVO.setTitle(pushItem.getLinkTitle());
                sendArticleMessageVO.setPushGroupRecordId(pushGroupRecordId);
                WxGroupSDK.sendArticleMessage(sendArticleMessageVO);
            }
            if (ItemTypeEnum.APP.value.equals(itemType)) {
                //通过群id获取对应基本信息
                GroupClassifyQrcodeDTO classifyQrcodeInfo = bookGroupClassifyBiz.getClassifyQrcodeInfo(groupId);
                if (classifyQrcodeInfo == null) {
                    return;
                }
                AccountSettingDto accountSettingDto = qrcodeSceneConsr.getWechatInfo(classifyQrcodeInfo.getChannelId());
                if (pushItem.getAppId() != null) {
                    AppDto appDto = appConsr.getBaseById(pushItem.getAppId());

                    if (appDto != null) {
                        SendArticleMessageVO sendArticleMessageVO = new SendArticleMessageVO();
                        sendArticleMessageVO.setAltId(altId);
                        sendArticleMessageVO.setDescription(appDto.getTypeName());
                        sendArticleMessageVO.setGroupId(groupId);
                        // 处理链接地址
                        String endUrl = pushItem.getAppUrl() + "&" + otherUrl;
                        String linkUrl = SendWeixinRequestTools.splitUrl(accountSettingDto, endUrl);
                        sendArticleMessageVO.setLinkUrl(linkUrl);
                        sendArticleMessageVO.setPicUrl(appDto.getSquareImg());
                        sendArticleMessageVO.setTitle(appDto.getTitle());
                        sendArticleMessageVO.setPushGroupRecordId(pushGroupRecordId);
                        WxGroupSDK.sendArticleMessage(sendArticleMessageVO);
                    }
                }
                if (pushItem.getProductId() != null) {
                    ProductDto productDto = productConsr.getProBaseById(pushItem.getProductId());
                    if (productDto != null) {
                        SendArticleMessageVO sendArticleMessageVO = new SendArticleMessageVO();
                        sendArticleMessageVO.setAltId(altId);
                        if (productDto.getProductTypeDto() != null) {
                            sendArticleMessageVO.setDescription(productDto.getProductTypeDto().getTypeName());
                        }
                        sendArticleMessageVO.setGroupId(groupId);
                        // 处理链接地址
                        String endUrl = pushItem.getProductUrl() + "&" + otherUrl;
                        String linkUrl = SendWeixinRequestTools.splitUrl(accountSettingDto, endUrl);
                        sendArticleMessageVO.setLinkUrl(linkUrl);
                        sendArticleMessageVO.setPicUrl(productDto.getCoverImg());
                        sendArticleMessageVO.setTitle(productDto.getProductName());
                        sendArticleMessageVO.setPushGroupRecordId(pushGroupRecordId);
                        WxGroupSDK.sendArticleMessage(sendArticleMessageVO);
                    }
                }
            }
            if (ItemTypeEnum.IMAGE.value.equals(itemType)) {
                SendPicMessageVO sendPicMessageVO = new SendPicMessageVO();
                sendPicMessageVO.setAltId(altId);
                sendPicMessageVO.setGroupId(groupId);
                sendPicMessageVO.setPicUrl(pushItem.getImageUrl());
                sendPicMessageVO.setPushGroupRecordId(pushGroupRecordId);
                WxGroupSDK.sendPicMessage(sendPicMessageVO);
            }
        }
    }

    @Transactional(rollbackFor = Exception.class)
    @ParamLog("更新群发")
    @Override
    public void updatePush(Push push) {
        //校验参数
        pushCheck.updatePushParamCheck(push);
        Push pushOld = pushDao.getById(push.getId());
        if (pushOld == null) {
            throw new BookBizException(BookBizException.ERROR, "未找到该群发！");
        }
        pushDao.update(push);
        //删除之前的关联表
        pushGroupDao.deleteByPushId(push.getId(), push.getUpdateUser());
        //填充关联模型字段
        fillPushGroupParam(push);
        //插入新纪录
        pushGroupDao.batchInsert(push.getPushGroups());
        //删除之前的群发消息项
        pushItemDao.deleteByPushId(push.getId(), push.getUpdateUser());
        //填充消息项模型
        fillPushItemParam(push);
        //插入群发消息项
        pushItemDao.batchInsert(push.getPushItems());
        //判断是否需要删除群发定时任务
        if (isNeedupdatepushSchedule(pushOld, push)) {
            //删除之前的群发定时任务，重新设置群发
            deletePushSchedule(push.getId());
            pushQuartz(push);
        }
    }

    /**
     * 删除群发定时任务
     */
    @ParamLog("删除群发定时任务")
    private void deletePushSchedule(Long pushId) {
        try {
            scheduleService.deleteJob(PUSH_SCHEDULE_PRE + pushId, "book");
        } catch (Exception e) {
            LOGGER.error("删除定时任务失败");
        }
    }

    @Transactional(rollbackFor = Exception.class)
    @ParamLog("删除群发")
    @Override
    public void deletePush(Long pushId, Long partyId) {
        if (pushId == null) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "群发id不能为空！");
        }
        Push pushOld = pushDao.getById(pushId);
        if (pushOld == null) {
            throw new BookBizException(BookBizException.ERROR, "未找到该群发！");
        }
        //删除群发
        pushDao.deleteByPushId(pushId,partyId);
        //删除群发关联
        pushGroupDao.deleteByPushId(pushId,partyId);
        //删除群发消息项
        pushItemDao.deleteByPushId(pushId,partyId);
        //删除定时任务
        deletePushSchedule(pushId);
    }

    @Transactional(rollbackFor = Exception.class)
    @ParamLog("批量新增群发关联")
    @Override
    public void createPushGroupBatch(List<PushGroup> pushGroups, Long partyId) {
        pushCheck.createPushGroupBatchParamCheck(pushGroups);
        Long pushId = pushGroups.get(0).getPushId();
        Push push = pushDao.getByPushId(pushId);
        if (push == null) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "没有该群发或已被删除！");
        }
        for (PushGroup pushGroup : pushGroups) {
            pushGroup.setCreateUser(partyId);
            pushGroup.setUpdateUser(partyId);
        }
        pushGroupDao.batchInsert(pushGroups);
    }

    @Transactional(rollbackFor = Exception.class)
    @ParamLog("批量删除群发关联")
    @Override
    public void deletePushGroupBatch(List<Long> pushGroupIds, Long partyId) {
        if (ListUtils.isEmpty(pushGroupIds)) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "群发关联id集合不能为空！");
        }
        pushGroupDao.deleteByIds(pushGroupIds, partyId);
    }

    @Transactional(rollbackFor = Exception.class)
    @ParamLog("新增群发消息项")
    @Override
    public void createPushItem(PushItem pushItem) {
        pushCheck.createPushItemParamCheck(pushItem);
        Integer count = pushItemDao.getCountByPushId(pushItem.getPushId());
        if (count >= 4) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "群发消息数量不能超过4条！");
        }
        Integer maxSeqNum = pushItemDao.getMaxSeqNumByPushId(pushItem.getPushId());
        pushItem.setSeqNum(maxSeqNum+1);
        pushItemDao.insert(pushItem);
    }

    @Transactional(rollbackFor = Exception.class)
    @ParamLog("删除群发消息项")
    @Override
    public void deletePushItem(Long pushItemId, Long partyId) {
        if (pushItemId == null) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "群发项id不能为空！");
        }
        pushItemDao.deleteByPushItemId(pushItemId,partyId);
    }

    @Transactional(rollbackFor = Exception.class)
    @ParamLog("群发项排序")
    @Override
    public void sortPushItems(List<Long> pushItemIds, Long partyId) {
        if (ListUtils.isEmpty(pushItemIds)){
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "群发项id集合不能为空！");
        }
        pushItemDao.sortPushItems(pushItemIds,partyId);
    }

    @ParamLog("分页获取群发记录")
    @Override
    public PageBeanNew<PushDTO> getTimingPushList( Long partyId, Integer currentPage, Integer numPerPage) {
        PageParam pageParam = new PageParam(currentPage, numPerPage);
        Map<String, Object> map = new HashMap<>();
        map.put("partyId", partyId);
        PageBeanNew<PushDTO> pageBeanNew = pushDao.listPageNew(pageParam, map, "getTimingPushList");
        List<PushDTO> pushDTOS = pageBeanNew.getRecordList();
        fillPushDTOList(pushDTOS);
        return pageBeanNew;
    }

    @Transactional(rollbackFor = Exception.class)
    @ParamLog("更新群发状态为成功")
    @Override
    public void updatePushStateSuccess(Integer pushGroupRecordId) {
        PushGroupRecord pushGroupRecord = pushGroupRecordDao.getById(pushGroupRecordId);
        if (pushGroupRecord != null) {
            //更新细化到群的状态
            pushGroupRecordDao.updateStateById(pushGroupRecordId, PushStatusEnum.SUCCESS.value);
            Long pushRecordId = pushGroupRecord.getPushRecordId();
            //查询群发关联表
            List<PushGroupRecord> pushGroupRecords = pushGroupRecordDao.getListByPushRecordId(pushRecordId);
            if (!ListUtils.isEmpty(pushGroupRecords)) {
                Integer successCount = 0;
                for (PushGroupRecord pushGroupRecordIn : pushGroupRecords) {
                    if (PushStatusEnum.SUCCESS.value.equals(pushGroupRecordIn.getPushStatus())) {
                        successCount = successCount + 1;
                    }
                }
                if (successCount == pushGroupRecords.size()) {
                    //如果全部成功才成功，更新群发状态
                    pushRecordDao.updateStateById(pushRecordId, PushStatusEnum.SUCCESS.value);
                }
            }
        }
    }

    @Transactional(rollbackFor = Exception.class)
    @ParamLog("修改群发时间")
    @Override
    public void updatePushTime(Push push) {
        if (push.getId()==null){
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "群发id不能为空！");
        }
        if (push.getPushType() == null) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "群发类型不能为空！");
        }
        pushCheck.checkPushTime(push);
        Push pushOld = pushDao.getById(push.getId());
        if (pushOld == null) {
            throw new BookBizException(BookBizException.ERROR, "未找到该群发！");
        }
        //修改数据库
        pushDao.updatePushTime(push);
        //判断是否需要删除群发定时任务
        if (isNeedupdatepushSchedule(pushOld, push)) {
            //删除之前的群发定时任务，重新设置群发
            deletePushSchedule(push.getId());
            pushQuartz(push);
        }

    }

    @ParamLog("获取群发关联集合")
    @Override
    public PageBeanNew<PushGroupDTO> getPushGroupList(Long pushId, Integer currentPage, Integer numPerPage) {
        PageParam pageParam = new PageParam(currentPage, numPerPage);
        Map<String, Object> map = new HashMap<>();
        map.put("pushId",pushId);
        PageBeanNew<PushGroupDTO> pageBeanNew = pushGroupDao.listPageNew(pageParam, map, "getPushGroupList");
        return pageBeanNew;
    }

    @ParamLog("获取群发记录集合")
    @Override
    public PageBeanNew<PushRecordDTO> getPushRecordList(Long partyId, Integer currentPage, Integer numPerPage) {
        PageParam pageParam = new PageParam(currentPage, numPerPage);
        Map<String, Object> map = new HashMap<>();
        map.put("partyId",partyId);
        PageBeanNew<PushRecordDTO> pageBeanNew = pushRecordDao.listPageNew(pageParam, map, "getPushRecordList");
        List<PushRecordDTO> pushRecordDTOS = pageBeanNew.getRecordList();
        fillPushRecordDTOList(pushRecordDTOS);
        return pageBeanNew;
    }

    /**
     * 填充群发记录
     *
     * @param pushRecordDTOS
     */
    private void fillPushRecordDTOList(List<PushRecordDTO> pushRecordDTOS) {
        if (ListUtils.isEmpty(pushRecordDTOS)) {
            return;
        }
        List<Long> pushIds = pushRecordDTOS.stream().map(PushRecordDTO::getPushId).distinct().collect(Collectors.toList());
        List<PushItem> pushItems = pushItemDao.getListByPushIds(pushIds);
        Map<Long, List<PushItem>> pushItemMap = new HashMap<>();
        List<Long> appIds = new ArrayList<>();
        List<Long> productIds = new ArrayList<>();
        Map<Long, AppDto> appDtoMap = new HashMap<>();
        Map<Long, ProductDto> productDtoMap = new HashMap<>();
        fillSomeMap(pushItems, pushItemMap, appIds, productIds);
        if (!ListUtils.isEmpty(appIds)) {
            appDtoMap = appConsr.mapByIds(appIds);
        }
        if (!ListUtils.isEmpty(productIds)) {
            productDtoMap = productConsr.getProBasesByIds(productIds);
        }

        for (PushRecordDTO pushRecordDTO : pushRecordDTOS) {
            List<PushItem> pushItemsForOne = pushItemMap.get(pushRecordDTO.getPushId());
            if (!ListUtils.isEmpty(pushItemsForOne)) {
                pushRecordDTO.setPushItems(pushItemsForOne);
                pushRecordDTO.setItemCount(pushItemsForOne.size());
                pushRecordDTO.setItemTypes(fillAppItemAndGetItemTypes(pushItemsForOne,appDtoMap,productDtoMap));
            }
        }
    }

    /**
     *
     * @param pushItems
     * @param appDtoMap
     * @param productDtoMap
     * @return
     */
    private List<String> fillAppItemAndGetItemTypes(List<PushItem> pushItems, Map<Long, AppDto> appDtoMap, Map<Long, ProductDto> productDtoMap) {
        List<String> itemTypes = new ArrayList<>();
        if (ListUtils.isEmpty(pushItems)){
            return itemTypes;
        }
        for (PushItem pushItem : pushItems) {
            if (ItemTypeEnum.TEXT.value.equals(pushItem.getItemType())) {
                if (!itemTypes.contains(changeToItemTypeName(ItemTypeEnum.TEXT.value))) {
                    itemTypes.add(changeToItemTypeName(ItemTypeEnum.TEXT.value));
                }
            }
            if (ItemTypeEnum.LINK.value.equals(pushItem.getItemType())) {
                if (!itemTypes.contains(changeToItemTypeName(ItemTypeEnum.LINK.value))) {
                    itemTypes.add(changeToItemTypeName(ItemTypeEnum.LINK.value));
                }
            }
            if (ItemTypeEnum.APP.value.equals(pushItem.getItemType())) {
                if (!itemTypes.contains(changeToItemTypeName(ItemTypeEnum.APP.value))) {
                    itemTypes.add(changeToItemTypeName(ItemTypeEnum.APP.value));
                }
                if (pushItem.getAppId() != null) {
                    //应用
                    if (appDtoMap != null) {
                        AppDto appDto = appDtoMap.get(pushItem.getAppId());
                        if (appDto != null) {
                            pushItem.setApName(appDto.getTitle());
                            pushItem.setApTypeName(appDto.getTypeName());
                            pushItem.setApImage(appDto.getSquareImg());
                        }
                    }

                } else if (pushItem.getProductId() != null) {
                    //作品
                    if (productDtoMap != null) {
                        ProductDto productDto = productDtoMap.get(pushItem.getProductId());
                        if (productDto != null) {
                            pushItem.setApName(productDto.getProductName());
                            if (productDto.getProductTypeDto() != null) {
                                pushItem.setApTypeName(productDto.getProductTypeDto().getTypeName());
                            }
                            pushItem.setApImage(productDto.getCoverImg());
                        }
                    }
                }
            }
            if (ItemTypeEnum.IMAGE.value.equals(pushItem.getItemType())) {
                if (!itemTypes.contains(changeToItemTypeName(ItemTypeEnum.IMAGE.value))) {
                    itemTypes.add(changeToItemTypeName(ItemTypeEnum.IMAGE.value));
                }
            }
        }
        return itemTypes;
    }

    /**
     * 是否需要更新群发
     */
    private Boolean isNeedupdatepushSchedule(Push pushOld, Push push) {
        if (pushOld.getPushType().equals(push.getPushType())) {
            if (pushOld.getPushTime() != null && push.getPushTime() != null) {
                if (pushOld.getPushTime().equals(push.getPushTime())) {
                    return false;
                }
            } else if (pushOld.getPushTime() == null && push.getPushTime() == null) {
                return false;
            }
        }
        return true;
    }

    /**
     * 填充集合相关字段
     */
    @ParamLog("填充集合相关字段")
    private void fillPushDTOList(List<PushDTO> pushDTOS) {
        if (ListUtils.isEmpty(pushDTOS)) {
            return;
        }
        List<Long> pushIds = pushDTOS.stream().map(PushDTO::getId).collect(Collectors.toList());
        List<PushItem> pushItems = pushItemDao.getListByPushIds(pushIds);
        Map<Long, List<PushItem>> pushItemMap = new HashMap<>();
        List<Long> appIds = new ArrayList<>();
        List<Long> productIds = new ArrayList<>();
        Map<Long, AppDto> appDtoMap = new HashMap<>();
        Map<Long, ProductDto> productDtoMap = new HashMap<>();
        fillSomeMap(pushItems, pushItemMap, appIds, productIds);
        if (!ListUtils.isEmpty(appIds)) {
            appDtoMap = appConsr.mapByIds(appIds);
        }
        if (!ListUtils.isEmpty(productIds)) {
            productDtoMap = productConsr.getProBasesByIds(productIds);
        }
        for (PushDTO pushDTO : pushDTOS) {
            List<PushItem> pushItemsForOne = pushItemMap.get(pushDTO.getId());
            if (!ListUtils.isEmpty(pushItemsForOne)) {
                pushDTO.setPushItems(pushItemsForOne);
                pushDTO.setItemCount(pushItemsForOne.size());
                pushDTO.setItemTypes(fillAppItemAndGetItemTypes(pushItemsForOne, appDtoMap, productDtoMap));
            }
        }
    }

    /**
     * 处理一些map等
     *
     * @param pushItems
     * @param pushItemMap
     * @param appIds
     * @param productIds
     */
    private void fillSomeMap(List<PushItem> pushItems, Map<Long, List<PushItem>> pushItemMap, List<Long> appIds, List<Long> productIds) {
        if (pushItems == null || pushItemMap == null || appIds == null || productIds == null) {
            return;
        }
        for (PushItem pushItem : pushItems) {
            Long pushId = pushItem.getPushId();
            List<PushItem> pushItemsForOneOld = pushItemMap.get(pushId);
            if (pushItemsForOneOld != null) {
                pushItemsForOneOld.add(pushItem);
                pushItemMap.put(pushId, pushItemsForOneOld);
            } else {
                List<PushItem> pushItemsNew = new ArrayList<>();
                pushItemsNew.add(pushItem);
                pushItemMap.put(pushId, pushItemsNew);
            }
            if (ItemTypeEnum.APP.value.equals(pushItem.getItemType())) {
                if (pushItem.getAppId() != null) {
                    if (!appIds.contains(pushItem.getAppId())) {
                        appIds.add(pushItem.getAppId());
                    }
                } else if (pushItem.getProductId() != null) {
                    if (!productIds.contains(pushItem.getProductId())) {
                        productIds.add(pushItem.getProductId());
                    }
                }
            }
        }
    }

    /**
     * 转换
     *
     * @param value
     * @return
     */
    private String changeToItemTypeName(Integer value) {
        String name = "";
        if (ItemTypeEnum.TEXT.value.equals(value)) {
            name = "文本";
        }
        if (ItemTypeEnum.LINK.value.equals(value)) {
            name = "链接";
        }
        if (ItemTypeEnum.APP.value.equals(value)) {
            name = "应用";
        }
        if (ItemTypeEnum.IMAGE.value.equals(value)) {
            name = "图片";
        }
        return name;
    }

    /**
     * 每天发送
     */
    @ParamLog("每天发送")
    private void pushDay(Push push) {
        String cron;
        String pushTime=push.getPushTime();
        try {
            int hour = Integer.parseInt(pushTime.substring(0,2));
            int minute = Integer.parseInt(pushTime.substring(3,5));
            int second = Integer.parseInt(pushTime.substring(6,8));
            cron = second + " " + minute + " " + hour + " * * ?";
        } catch (Exception e) {
            throw new BookBizException(BookBizException.ERROR, "时间格式错误");
        }
        ScheduleJob scheduleJob = new ScheduleJob();
        scheduleJob.setJobGroup("book");
        scheduleJob.setJobName(PUSH_SCHEDULE_PRE + push.getId());
        scheduleJob.setCronExpression(cron);
        CallBackParam callBackParam = new CallBackParam();
        Map<String, Object> paramMap = new HashMap<>();
        //定时任务调用时用到的参数
        paramMap.put("pushId", push.getId());
        callBackParam.setParamMap(paramMap);
        callBackParam.setMethodName("sendGroupMessage");
        callBackParam.setBeanName("pushService");
        Map<String, Object> scheduleMap = new HashMap<>();
        scheduleMap.put("scheduleJob", scheduleJob);
        scheduleMap.put("callBackParam", callBackParam);
        try {
            scheduleService.addCronJob(scheduleMap);
        } catch (Exception e) {
            LOGGER.error("设置定时任务失败" + e.getMessage(), e);
            throw new BookBizException(BookBizException.ERROR, "定时任务设置失败");
        }
    }

    /**
     * 单次发送
     */
    @ParamLog("单次发送")
    private void pushOne(Push push) {
        Long second = (DateUtils.StringToDateTime(push.getPushTime()).getTime() - System.currentTimeMillis()) / 1000;
        ScheduleJob scheduleJob = new ScheduleJob();
        scheduleJob.setJobGroup("book");
        scheduleJob.setStartTimeFormat("ss");
        scheduleJob.setStartTime(second.intValue());
        scheduleJob.setRepeatCount(0);
        scheduleJob.setIntervalTime(0);
        scheduleJob.setIntervalTimeFormat("dd");
        scheduleJob.setJobName( PUSH_SCHEDULE_PRE + push.getId());
        CallBackParam callBackParam = new CallBackParam();
        Map<String, Object> paramMap = new HashMap<>();
        //定时任务调用时用到的参数
        paramMap.put("pushId", push.getId());
        callBackParam.setParamMap(paramMap);
        callBackParam.setMethodName("sendGroupMessage");
        callBackParam.setBeanName("pushService");
        Map<String, Object> scheduleMap = new HashMap<>();
        scheduleMap.put("scheduleJob", scheduleJob);
        scheduleMap.put("callBackParam", callBackParam);
        try {
            scheduleService.addSimpleJob(scheduleMap);
        } catch (Exception e) {
            LOGGER.error("设置定时任务失败" + e.getMessage(), e);
            throw new BookBizException(BookBizException.ERROR, "定时任务设置失败");
        }
    }


    /**
     * 填充消息项模型
     */
    @ParamLog("填充消息项模型")
    private void fillPushItemParam(Push push) {
        int i=0;
        for (PushItem pushItem : push.getPushItems()) {
            pushItem.setPushId(push.getId());
            pushItem.setSeqNum(++i);
            pushItem.setCreateUser(push.getCreateUser());
            pushItem.setUpdateUser(push.getCreateUser());
        }
    }

    /**
     * 设置群发关联表字段
     */
    @ParamLog("设置群发关联表字段")
    private void fillPushGroupParam(Push push) {
        for (PushGroup pushGroup : push.getPushGroups()) {
            pushGroup.setCreateUser(push.getCreateUser());
            pushGroup.setUpdateUser(push.getCreateUser());
            pushGroup.setPushId(push.getId());
        }
    }
}
