package com.pcloud.book.push.facade.impl;

import com.pcloud.book.base.exception.BookBizException;
import com.pcloud.book.push.biz.PushBiz;
import com.pcloud.book.push.entity.Push;
import com.pcloud.book.push.entity.PushGroup;
import com.pcloud.book.push.entity.PushItem;
import com.pcloud.book.push.facade.PushFacade;
import com.pcloud.common.dto.ResponseDto;
import com.pcloud.common.exceptions.BizException;
import com.pcloud.common.permission.PermissionException;
import com.pcloud.common.utils.SessionUtil;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * @Description
 * @Author ruansiyuan
 * @Date 2019/4/17 17:49
 **/
@RestController("pushFacade")
@RequestMapping("push")
public class PushFacadeImpl implements PushFacade {

    private static final Logger LOGGER = LoggerFactory.getLogger(PushFacadeImpl.class);

    @Autowired
    private PushBiz pushBiz;


    @ApiOperation("新增群发")
    @PostMapping("/createPush")
    @Override
    public ResponseDto<?> createPush(
            @RequestHeader("token") @ApiParam("token信息") String token,
            @RequestBody @ApiParam("群发模型") Push push
    ) throws BizException, PermissionException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        push.setCreateUser(partyId);
        push.setUpdateUser(partyId);
        pushBiz.createPush(push);
        return new ResponseDto<>();
    }

    @ApiOperation("更新群发")
    @PostMapping("/updatePush")
    @Override
    public ResponseDto<?> updatePush(
            @RequestHeader("token") @ApiParam("token信息") String token,
            @RequestBody @ApiParam("群发模型") Push push
    ) throws BizException, PermissionException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        push.setCreateUser(partyId);
        push.setUpdateUser(partyId);
        pushBiz.updatePush(push);
        return new ResponseDto<>();
    }

    @ApiOperation("删除群发")
    @GetMapping("/deletePush")
    @Override
    public ResponseDto<?> deletePush(
            @RequestHeader("token") @ApiParam("token信息") String token,
            @RequestParam("pushId") @ApiParam("群发id") Long pushId
    ) throws BizException, PermissionException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        pushBiz.deletePush(pushId,partyId);
        return new ResponseDto<>();
    }

    @ApiOperation("批量新增群发关联")
    @PostMapping("/createPushGroupBatch")
    @Override
    public ResponseDto<?> createPushGroupBatch(
            @RequestHeader("token") @ApiParam("token信息") String token,
            @RequestBody @ApiParam("群发关联模型集合") List<PushGroup> pushGroups
    ) throws BizException, PermissionException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        pushBiz.createPushGroupBatch(pushGroups,partyId);
        return new ResponseDto<>();
    }

    @ApiOperation("批量删除群发关联")
    @PostMapping("/deletePushGroupBatch")
    @Override
    public ResponseDto<?> deletePushGroupBatch(
            @RequestHeader("token") @ApiParam("token信息") String token,
            @RequestBody @ApiParam("群发关联id集合") List<Long> pushGroupIds
    ) throws BizException, PermissionException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        pushBiz.deletePushGroupBatch(pushGroupIds,partyId);
        return new ResponseDto<>();
    }

    @ApiOperation("新增群发项")
    @PostMapping("/createPushItem")
    @Override
    public ResponseDto<?> createPushItem(
            @RequestHeader("token") @ApiParam("token信息") String token,
            @RequestBody @ApiParam("群发项") PushItem pushItem
    ) throws BizException, PermissionException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        pushItem.setCreateUser(partyId);
        pushItem.setUpdateUser(partyId);
        pushBiz.createPushItem(pushItem);
        return new ResponseDto<>();
    }

    @ApiOperation("删除群发项")
    @GetMapping("/deletePushItem")
    @Override
    public ResponseDto<?> deletePushItem(
            @RequestHeader("token") @ApiParam("token信息") String token,
            @RequestParam("pushItemId") @ApiParam("群发项") Long pushItemId
    ) throws BizException, PermissionException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        pushBiz.deletePushItem(pushItemId,partyId);
        return new ResponseDto<>();
    }

    @ApiOperation("群发项排序")
    @PostMapping("/sortPushItems")
    @Override
    public ResponseDto<?> sortPushItems(
            @RequestHeader("token") @ApiParam("token信息") String token,
            @RequestBody @ApiParam("群发项id集合") List<Long> pushItemIds
    ) throws BizException, PermissionException {
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        pushBiz.sortPushItems(pushItemIds,partyId);
        return new ResponseDto<>();
    }

    @ApiOperation("获取定时群发列表")
    @GetMapping("/getTimingPushList")
    @Override
    public ResponseDto<?> getTimingPushList(
            @RequestHeader("token") @ApiParam("token信息") String token,
            @RequestParam("currentPage") @ApiParam("当前页") Integer currentPage,
            @RequestParam("numPerPage") @ApiParam("每页条数") Integer numPerPage
    ) throws BizException, PermissionException {
        if (currentPage == null || currentPage < 0) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "当前页不能为空且不能小于0！");
        }
        if (numPerPage == null || numPerPage <= 0) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "每页条数不能为空且必须大于0！");
        }
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        return new ResponseDto<>(pushBiz.getTimingPushList(partyId, currentPage, numPerPage));
    }

    @ApiOperation("更新群发消息状态为成功")
    @GetMapping("/updatePushStateSuccess")
    @Override
    public void updatePushStateSuccess(@RequestParam("pushGroupRecordId") @ApiParam("群发记录id") Integer pushGroupRecordId) throws BizException {
        LOGGER.info("更新群发消息状态为成功被调用+++++"+pushGroupRecordId);
        if (pushGroupRecordId == null) {
            return;
        }
        pushBiz.updatePushStateSuccess(pushGroupRecordId);
    }

    @ApiOperation("修改群发时间")
    @PostMapping("/updatePushTime")
    @Override
    public ResponseDto<?> updatePushTime(
            @RequestHeader("token") @ApiParam("token信息") String token,
            @RequestBody @ApiParam("群发模型") Push push
    ) throws BizException, PermissionException {
        if (push == null) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "群发不能为空！");
        }
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        push.setUpdateUser(partyId);
        pushBiz.updatePushTime(push);
        return new ResponseDto<>();
    }

    @ApiOperation("获取群发关联集合")
    @GetMapping("/getPushGroupList")
    @Override
    public ResponseDto<?> getPushGroupList(
            @RequestHeader("token") @ApiParam("token信息") String token,
            @RequestParam("pushId") @ApiParam("群发id") Long pushId,
            @RequestParam("currentPage") @ApiParam("当前页") Integer currentPage,
            @RequestParam("numPerPage") @ApiParam("每页条数") Integer numPerPage
    ) throws BizException, PermissionException {
        if (currentPage == null || currentPage < 0) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "当前页不能为空且不能小于0！");
        }
        if (numPerPage == null || numPerPage <= 0) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "每页条数不能为空且必须大于0！");
        }
        if (pushId==null){
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "群发id不能为空！");
        }
        SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        return new ResponseDto<>(pushBiz.getPushGroupList(pushId, currentPage, numPerPage));
    }

    @ApiOperation("获取群发记录列表")
    @GetMapping("/getPushRecordList")
    @Override
    public ResponseDto<?> getPushRecordList(
            @RequestHeader("token") @ApiParam("token信息") String token,
            @RequestParam("currentPage") @ApiParam("当前页") Integer currentPage,
            @RequestParam("numPerPage") @ApiParam("每页条数") Integer numPerPage
    ) throws BizException, PermissionException {
        if (currentPage == null || currentPage < 0) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "当前页不能为空且不能小于0！");
        }
        if (numPerPage == null || numPerPage <= 0) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "每页条数不能为空且必须大于0！");
        }
        Long partyId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        return new ResponseDto<>(pushBiz.getPushRecordList(partyId, currentPage, numPerPage));
    }

}
