package com.pcloud.book.advertising.biz.impl;

import java.math.BigDecimal;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import com.pcloud.appcenter.app.dto.AppTypeDto;
import com.pcloud.appcenter.app.service.AppTypeService;
import com.pcloud.book.advertising.biz.AdvertisingSpaceBiz;
import com.pcloud.book.advertising.dao.AdvertisingAdviserPermissionDao;
import com.pcloud.book.advertising.dao.AdvertisingAgentPermissionDao;
import com.pcloud.book.advertising.dao.AdvertisingClickRecordDao;
import com.pcloud.book.advertising.dao.AdvertisingDistributionBookDao;
import com.pcloud.book.advertising.dao.AdvertisingExposureRecordDao;
import com.pcloud.book.advertising.dao.AdvertisingIncomeDailyDao;
import com.pcloud.book.advertising.dao.AdvertisingMasterDao;
import com.pcloud.book.advertising.dao.AdvertisingSettlementMethodDao;
import com.pcloud.book.advertising.dao.AdvertisingSpaceDao;
import com.pcloud.book.advertising.dto.AdvertisingMasterDTO;
import com.pcloud.book.advertising.dto.AdvertisingSettlementMethodDTO;
import com.pcloud.book.advertising.dto.AdvertisingSpaceDTO;
import com.pcloud.book.advertising.dto.WechatGroupClickUserDTO;
import com.pcloud.book.advertising.entity.AdvertisingAdviserPermission;
import com.pcloud.book.advertising.entity.AdvertisingAgentPermission;
import com.pcloud.book.advertising.entity.AdvertisingClickRecord;
import com.pcloud.book.advertising.entity.AdvertisingDistributionBook;
import com.pcloud.book.advertising.entity.AdvertisingExposureRecord;
import com.pcloud.book.advertising.entity.AdvertisingMaster;
import com.pcloud.book.advertising.entity.AdvertisingSettlementMethod;
import com.pcloud.book.advertising.entity.AdvertisingSpace;
import com.pcloud.book.advertising.entity.DistributionWechatGroup;
import com.pcloud.book.advertising.enums.AdPositionDetailEnum;
import com.pcloud.book.advertising.enums.AdPositionEnum;
import com.pcloud.book.advertising.enums.AdPositionModeEnum;
import com.pcloud.book.advertising.enums.SettlementMethodEnum;
import com.pcloud.book.base.exception.BookBizException;
import com.pcloud.book.book.biz.BookBiz;
import com.pcloud.book.book.dao.BookDao;
import com.pcloud.book.book.dto.BookDto;
import com.pcloud.book.book.entity.BookAdviser;
import com.pcloud.book.book.set.BookSet;
import com.pcloud.book.consumer.app.AppConsr;
import com.pcloud.book.consumer.common.ExportConsr;
import com.pcloud.book.consumer.reader.ReaderConsr;
import com.pcloud.book.consumer.resource.ProductConsr;
import com.pcloud.book.consumer.user.AdviserConsr;
import com.pcloud.book.consumer.user.AgentConsr;
import com.pcloud.book.consumer.user.ChannelConsr;
import com.pcloud.book.consumer.wechatgroup.WechatGroupConsr;
import com.pcloud.book.group.dao.BookGroupClassifyDao;
import com.pcloud.book.group.dao.BookGroupDao;
import com.pcloud.book.group.dao.GroupQrcodeDao;
import com.pcloud.book.group.dto.AdGroupQrcodeDTO;
import com.pcloud.book.group.dto.BookGroupDTO;
import com.pcloud.book.group.dto.ClassifyDTO;
import com.pcloud.book.group.dto.GroupQrcodeInfoDTO;
import com.pcloud.book.group.entity.BookGroup;
import com.pcloud.book.group.entity.GroupQrcode;
import com.pcloud.book.util.common.CommonUtils;
import com.pcloud.channelcenter.qrcode.service.QrcodeSceneService;
import com.pcloud.common.core.aspect.ParamLog;
import com.pcloud.common.exceptions.BizException;
import com.pcloud.common.page.PageBean;
import com.pcloud.common.page.PageParam;
import com.pcloud.common.utils.DateNewUtils;
import com.pcloud.common.utils.DateUtils;
import com.pcloud.common.utils.ListUtils;
import com.pcloud.common.utils.ResponseHandleUtil;
import com.pcloud.common.utils.httpclient.UrlUtils;
import com.pcloud.common.utils.string.StringUtil;
import com.pcloud.readercenter.wechat.entity.WechatUser;
import com.pcloud.resourcecenter.product.service.ProductService;
import com.sdk.wxgroup.SendArticleMessageVO;
import com.sdk.wxgroup.SendPicMessageVO;
import com.sdk.wxgroup.SendTextMessageVO;
import com.sdk.wxgroup.WxGroupSDK;

/**
 * Description 广告位业务逻辑层接口实现类
 * @author PENG
 * @date 2019/2/28
 */
@Component("advertisingSpaceBiz")
public class AdvertisingSpaceBizImpl implements AdvertisingSpaceBiz {

    @Value("${wechat.group.link.prefix}")
    private String wechatGroupLinkPrefix;

    @Autowired
    private AdvertisingSpaceDao advertisingSpaceDao;
    @Autowired
    private AdviserConsr adviserConsr;
    @Autowired
    private AdvertisingDistributionBookDao advertisingDistributionBookDao;
    @Autowired
    private BookBiz bookBiz;
    @Autowired
    private AdvertisingAgentPermissionDao advertisingAgentPermissionDao;
    @Autowired
    private BookDao bookDao;
    @Autowired
    private BookSet bookSet;
    @Autowired
    private AdvertisingAdviserPermissionDao advertisingAdviserPermissionDao;
    @Autowired
    private QrcodeSceneService qrcodeSceneService;
    @Autowired
    private ChannelConsr channelConsr;
    @Autowired
    private AdvertisingExposureRecordDao advertisingExposureRecordDao;
    @Autowired
    private AdvertisingClickRecordDao advertisingClickRecordDao;
    @Autowired
    private ProductConsr productConsr;
    @Autowired
    private AppConsr appConsr;
    @Autowired
    private AdvertisingMasterDao advertisingMasterDao;
    @Autowired
    private AdvertisingSettlementMethodDao advertisingSettlementMethodDao;
    @Autowired
    private AdvertisingIncomeDailyDao advertisingIncomeDailyDao;
    @Autowired
    private BookGroupClassifyDao bookGroupClassifyDao;
    @Autowired
    private GroupQrcodeDao groupQrcodeDao;
    @Autowired
    private AgentConsr agentConsr;
    @Autowired
    private WechatGroupConsr wechatGroupConsr;
    @Autowired
    private ExportConsr exportConsr;
    @Autowired
    private ProductService productService;
    @Autowired
    private AppTypeService appTypeService;
    @Autowired
    private BookGroupDao bookGroupDao;
    @Autowired
    private ReaderConsr readerConsr;

    /**
     * 创建广告位
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @ParamLog("创建广告位")
    public Long create(AdvertisingSpace advertisingSpace) throws BizException {
        // 校验参数
        checkParamBeforeCreate(advertisingSpace);
        String settlementMethod = advertisingSpace.getSettlementMethod();
        advertisingSpace.setIsOpen(true);
        advertisingSpaceDao.insert(advertisingSpace);
        Long adId = advertisingSpace.getId();
        AdvertisingSettlementMethod method = new AdvertisingSettlementMethod();
        method.setMasterId(advertisingSpace.getMasterId());
        method.setAdId(adId);
        method.setSettlementMethod(settlementMethod);
        method.setPriceEachTime(advertisingSpace.getPriceEachTime());
        advertisingSettlementMethodDao.insert(method);
        return adId;
    }

    /**
     * 新增前校验参数
     */
    private void checkParamBeforeCreate(AdvertisingSpace advertisingSpace) {
        String adPosition = advertisingSpace.getAdPosition();
        if (null == advertisingSpace.getAdName() || null == adPosition || null == advertisingSpace.getAdPositionMode()
                || null == advertisingSpace.getMasterId() || null == advertisingSpace.getSettlementMethod()) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数有误！");
        }
        if (!AdPositionEnum.checkCodeExist(adPosition)) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "广告位置选择有误！");
        }
        if (!AdPositionModeEnum.checkCodeExist(adPosition, advertisingSpace.getAdPositionMode())) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "广告位置选择有误！");
        }
        if (AdPositionEnum.BOOK_GROUP_PAGE.equals(adPosition)
                && !AdPositionDetailEnum.checkCodeExist(adPosition, advertisingSpace.getAdPositionDetail())) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "广告位置选择有误！");
        }
        String settlementMethod = advertisingSpace.getSettlementMethod();
        if (AdPositionEnum.WECHAT_GROUP_MSG.equals(adPosition)
                && !SettlementMethodEnum.CPM.code.equals(settlementMethod)) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "微信群消息仅支持按曝光量结算！");
        }
        if (ArrayUtils.contains(SettlementMethodEnum.SETTLEMENT_METHOD_NEED_PRICE_GATHER, settlementMethod)
                && null == advertisingSpace.getPriceEachTime()) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "按点击量结算和按曝光量结算时每次价格不能为空！");
        }
    }

    /**
     * 修改广告位
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @ParamLog(value = "修改广告位", isAfterReturn = false)
    public void update(AdvertisingSpace advertisingSpace) throws BizException {
        // 校验参数
        checkParamBeforeCreate(advertisingSpace);
        if (null == advertisingSpace.getId()) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数有误！");
        }
        AdvertisingSpace space = advertisingSpaceDao.getById(advertisingSpace.getId());
        if (null == space) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "广告位不存在或已删除！");
        }
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("adId", advertisingSpace.getId());
        Integer bookNum = (Integer) advertisingDistributionBookDao.getBy(paramMap, "countByAdId");
        // 已投放
        if (null != bookNum && bookNum > 0) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "已投放书刊的广告位不能修改！");
        }
        advertisingSpaceDao.update(advertisingSpace);
        // 修改结算方式
        AdvertisingSettlementMethodDTO methodDTO = advertisingSettlementMethodDao.getByMasterIdAndAdId(space.getMasterId(), space.getId());
        AdvertisingSettlementMethod method = new AdvertisingSettlementMethod();
        method.setMasterId(advertisingSpace.getMasterId());
        method.setSettlementMethod(advertisingSpace.getSettlementMethod());
        method.setPriceEachTime(advertisingSpace.getPriceEachTime());
        if (null == methodDTO || null == methodDTO.getId()) {
            advertisingSettlementMethodDao.insert(method);
        } else {
            method.setId(methodDTO.getId());
            advertisingSettlementMethodDao.update(method);
        }
    }

    /**
     * 停用广告位
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @ParamLog(value = "停用广告位", isAfterReturn = false)
    public void stop(AdvertisingSpace advertisingSpace) throws BizException {
        if (null == advertisingSpace.getId()) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数有误！");
        }
        AdvertisingSpace space = advertisingSpaceDao.getById(advertisingSpace.getId());
        if (null == space) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "广告位不存在或已删除！");
        }
        if (!space.getIsOpen()) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "广告位已停用！");
        }
        advertisingSpace.setIsOpen(false);
        advertisingSpaceDao.update(advertisingSpace);
    }

    /**
     * 获取广告位信息
     */
    @Override
    @ParamLog(value = "获取广告位信息")
    public AdvertisingSpaceDTO get(Long adId) throws BizException {
        AdvertisingSpaceDTO spaceDTO = advertisingSpaceDao.getDTOById(adId);
        if (null != spaceDTO) {
            AdvertisingMaster master = advertisingMasterDao.getById(spaceDTO.getMasterId());
            if (null != master) {
                spaceDTO.setMasterName(master.getMasterName());
            }
            AdvertisingSettlementMethodDTO methodDTO = advertisingSettlementMethodDao.getByMasterIdAndAdId(spaceDTO.getMasterId(), spaceDTO.getId());
            if (null != methodDTO) {
                spaceDTO.setSettlementMethod(methodDTO.getSettlementMethod());
                spaceDTO.setSettlementMethodName(SettlementMethodEnum.getNameByCode(methodDTO.getSettlementMethod()));
                spaceDTO.setPriceEachTime(methodDTO.getPriceEachTime());
            }
            Map<String, Object> paramMap = new HashMap<>();
            paramMap.put("adId", adId);
            Integer bookNum = (Integer) advertisingDistributionBookDao.getBy(paramMap, "countByAdId");
            spaceDTO.setBookNum(bookNum);
            // 填充广告位置名称
            if (AdPositionEnum.WECHAT_GROUP_MSG.positionCode.equals(spaceDTO.getAdPosition())) {
                fillAdpositionName4WechatGroupAd(spaceDTO);
            } else {
                fillAdpositionName4CommonAd(spaceDTO);
            }
        }
        return spaceDTO;
    }

    /**
     * 投放书刊
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @ParamLog(value = "投放书刊", isAfterReturn = false)
    public void distributeBook(AdvertisingDistributionBook book) throws BizException {
        if (null == book.getAdId() || ListUtils.isEmpty(book.getBooks()) || null == book.getIsBookGroup()) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数有误！");
        }
        AdvertisingSpace advertisingSpace = advertisingSpaceDao.getById(book.getAdId());
        if (null == advertisingSpace || !advertisingSpace.getIsOpen()) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "广告位不存在或已关闭！");
        }
        AdvertisingSpace space = new AdvertisingSpace();
        space.setId(book.getAdId());
        advertisingSpaceDao.update(space);
        boolean paramError = book.getBooks().stream().anyMatch(bookAdviser -> null == bookAdviser.getBookId()
                || null == bookAdviser.getChannelId() || null == bookAdviser.getAdviserId());
        if (paramError) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数有误！");
        }
        // 校验书刊是否已投广告位
        Map<String, Object> paramMap = new HashMap<>();
        if (book.getIsBookGroup()) {
            // 社群书
            for (BookAdviser bookAdviser : book.getBooks()) {
                paramMap.put("adPositionDetail", advertisingSpace.getAdPositionDetail());
                paramMap.put("bookId", bookAdviser.getBookId());
                paramMap.put("channelId", bookAdviser.getChannelId());
                Integer existCount = (Integer) advertisingDistributionBookDao.getBy(paramMap, "checkBookGroupAdExist");
                if (null != existCount && existCount > 0) {
                    BookDto bookDto = bookBiz.getBaseById(bookAdviser.getBookId());
                    throw new BookBizException(BookBizException.PARAM_IS_ERROR, StringUtil.addBracket(bookDto.getBookName()) + "等社群书"
                            + AdPositionDetailEnum.getNameByCode(advertisingSpace.getAdPositionDetail()) + "已投放广告位！");
                }
            }
        } else {
            // 非社群书
            paramMap.put("adPositionDetail", advertisingSpace.getAdPositionDetail());
            paramMap.put("adPositionMode", advertisingSpace.getAdPositionMode());
            for (BookAdviser bookAdviser : book.getBooks()) {
                paramMap.put("bookId", bookAdviser.getBookId());
                paramMap.put("channelId", bookAdviser.getChannelId());
                Integer existCount = (Integer) advertisingDistributionBookDao.getBy(paramMap, "checkBookAdExist");
                if (null != existCount && existCount > 0) {
                    BookDto bookDto = bookBiz.getBaseById(bookAdviser.getBookId());
                    throw new BookBizException(BookBizException.PARAM_IS_ERROR, StringUtil.addBracket(bookDto.getBookName()) + "等书已投放该广告位！");
                }
            }
        }
        List<AdvertisingDistributionBook> list = new ArrayList<>();
        for (BookAdviser bookAdviser : book.getBooks()) {
            AdvertisingDistributionBook distributionBook = new AdvertisingDistributionBook();
            distributionBook.setAdId(book.getAdId());
            distributionBook.setBookId(bookAdviser.getBookId());
            distributionBook.setChannelId(bookAdviser.getChannelId());
            distributionBook.setAdviserId(bookAdviser.getAdviserId());
            Long agentId = adviserConsr.getAgentIdByAdviser(bookAdviser.getAdviserId());
            distributionBook.setAgentId(agentId);
            distributionBook.setIsBookGroup(book.getIsBookGroup());
            list.add(distributionBook);
        }
        advertisingDistributionBookDao.batchInsert(list);
    }

    /**
     * 投放微信群
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @ParamLog(value = "投放微信群", isAfterReturn = false)
    public void distributeWechatGroup(AdvertisingDistributionBook book) throws BizException {
        if (null == book.getAdId() || ListUtils.isEmpty(book.getGroups())) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数有误！");
        }
        AdvertisingSpace advertisingSpace = advertisingSpaceDao.getById(book.getAdId());
        if (null == advertisingSpace) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "广告位不存在！");
        }
        if (!AdPositionEnum.WECHAT_GROUP_MSG.positionCode.equals(advertisingSpace.getAdPosition())) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "该广告位不能投放到微信群！");
        }
        AdvertisingSpace space = new AdvertisingSpace();
        space.setId(book.getAdId());
        advertisingSpaceDao.update(space);
        boolean paramError = book.getGroups().stream().anyMatch(group -> null == group.getBookId()
                || null == group.getChannelId() || null == group.getAdviserId() || null == group.getClassifyId() || null == group.getQrcodeId());
        if (paramError) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数有误！");
        }
        // 校验广告位是否已投放微信群
        List<Long> qrcodeIds = book.getGroups().stream().map(DistributionWechatGroup::getQrcodeId).collect(Collectors.toList());
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("adId", book.getAdId());
        paramMap.put("qrcodeIds", qrcodeIds);
        List<Long> existIds = advertisingDistributionBookDao.checkQrcodeIdExist(paramMap);
        if (!ListUtils.isEmpty(existIds)) {
            Long qrcodeId = existIds.get(0);
            GroupQrcode groupQrcode = groupQrcodeDao.getById(qrcodeId);
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, StringUtil.addBracket(groupQrcode.getGroupName()) + "等" + existIds.size() + "个微信群已投放该广告位！");
        }
        List<AdvertisingDistributionBook> list = new ArrayList<>();
        Map<Long, Long> agentIdMap = new HashMap<>();
        for (DistributionWechatGroup group : book.getGroups()) {
            AdvertisingDistributionBook distributionBook = new AdvertisingDistributionBook();
            distributionBook.setAdId(book.getAdId());
            distributionBook.setBookId(group.getBookId());
            distributionBook.setChannelId(group.getChannelId());
            distributionBook.setAdviserId(group.getAdviserId());
            distributionBook.setClassifyId(group.getClassifyId());
            distributionBook.setQrcodeId(group.getQrcodeId());
            Long agentId;
            if (!agentIdMap.containsKey(group.getAdviserId())) {
                agentId = adviserConsr.getAgentIdByAdviser(group.getAdviserId());
                agentIdMap.put(group.getAdviserId(), agentId);
            } else {
                agentId = agentIdMap.get(group.getAdviserId());
            }
            distributionBook.setAgentId(agentId);
            distributionBook.setIsBookGroup(true);
            list.add(distributionBook);
        }
        advertisingDistributionBookDao.batchInsert(list);
        for (DistributionWechatGroup group : book.getGroups()) {
            // 获取群信息
            GroupQrcode groupQrcode = groupQrcodeDao.getById(group.getQrcodeId());
            String groupId = groupQrcode.getWeixinGroupId();
            ClassifyDTO classifyDTO = bookGroupClassifyDao.getById(group.getClassifyId());
            // 获取机器人微信号
            String altId = wechatGroupConsr.getRobotIdByGroupId(groupId);
            if (AdPositionModeEnum.TEXT_AND_LINK.modeCode.equals(advertisingSpace.getAdPositionMode())) {
                SendTextMessageVO sendTextMessageVO = new SendTextMessageVO();
                sendTextMessageVO.setGroupId(groupId);
                sendTextMessageVO.setAltId(altId);
                String adLink = "";
                if (!StringUtil.isEmpty(advertisingSpace.getAdLink())) {
                    if (advertisingSpace.getAdLink().contains("?")) {
                        adLink = advertisingSpace.getAdLink() + "&book_group_id=" + classifyDTO.getBookGroupId() + "&classify_id=" + group.getClassifyId()
                                + "&qrcode_id=" + group.getQrcodeId() + "&ad_id=" + advertisingSpace.getId();
                    } else {
                        adLink = advertisingSpace.getAdLink() + "?book_group_id=" + classifyDTO.getBookGroupId() + "&classify_id=" + group.getClassifyId()
                                + "&qrcode_id=" + group.getQrcodeId() + "&ad_id=" + advertisingSpace.getId();
                    }
                    adLink = UrlUtils.getShortUrl4Own(wechatGroupLinkPrefix + "/link?url=" + URLEncoder.encode(adLink));
                }
                sendTextMessageVO.setContent(advertisingSpace.getAdTitle() + "\n" + adLink);
                WxGroupSDK.sendTextMessage(sendTextMessageVO);
            } else if (AdPositionModeEnum.NEWS_SHARE.modeCode.equals(advertisingSpace.getAdPositionMode())) {
                SendArticleMessageVO sendArticleMessageVO = new SendArticleMessageVO();
                sendArticleMessageVO.setAltId(altId);
                sendArticleMessageVO.setTitle(advertisingSpace.getAdTitle());
                sendArticleMessageVO.setDescription(advertisingSpace.getAdSlogan());
                sendArticleMessageVO.setGroupId(groupId);
                String adLink = "";
                if (!StringUtil.isEmpty(advertisingSpace.getAdLink())) {
                    if (advertisingSpace.getAdLink().contains("?")) {
                        adLink = advertisingSpace.getAdLink() + "&book_group_id=" + classifyDTO.getBookGroupId() + "&classify_id=" + group.getClassifyId()
                                + "&qrcode_id=" + group.getQrcodeId() + "&ad_id=" + advertisingSpace.getId();
                    } else {
                        adLink = advertisingSpace.getAdLink() + "?book_group_id=" + classifyDTO.getBookGroupId() + "&classify_id=" + group.getClassifyId()
                                + "&qrcode_id=" + group.getQrcodeId() + "&ad_id=" + advertisingSpace.getId();
                    }
                }
                sendArticleMessageVO.setLinkUrl(adLink);
                sendArticleMessageVO.setPicUrl(advertisingSpace.getAdPic());
                WxGroupSDK.sendArticleMessage(sendArticleMessageVO);
            } else if (AdPositionModeEnum.JUST_PIC.modeCode.equals(advertisingSpace.getAdPositionMode())) {
                SendPicMessageVO sendPicMessageVO = new SendPicMessageVO();
                sendPicMessageVO.setAltId(altId);
                sendPicMessageVO.setGroupId(groupId);
                sendPicMessageVO.setPicUrl(advertisingSpace.getAdPic());
                WxGroupSDK.sendPicMessage(sendPicMessageVO);
            }
        }
        // 根据微信群人数埋点
        Map<Long, GroupQrcodeInfoDTO> map = groupQrcodeDao.listQrcodeInfoByIds(qrcodeIds);
        if (!MapUtils.isEmpty(map)) {
            List<AdvertisingExposureRecord> recordList = new ArrayList<>();
            for (DistributionWechatGroup group : book.getGroups()) {
                Long qrcodeId = group.getQrcodeId();
                if (map.containsKey(qrcodeId)) {
                    GroupQrcodeInfoDTO dto = map.get(qrcodeId);
                    Integer userNumber = dto.getUserNumber();
                    AdvertisingExposureRecord record = new AdvertisingExposureRecord();
                    record.setAdId(book.getAdId());
                    record.setBookId(group.getBookId());
                    record.setIsBookGroup(true);
                    record.setAdviserId(group.getAdviserId());
                    record.setAgentId(agentIdMap.get(group.getAdviserId()));
                    record.setChannelId(group.getChannelId());
                    record.setFromType("WECHAT_GROUP");
                    record.setFromId(qrcodeId);
                    record.setCount(userNumber.longValue());
                    recordList.add(record);
                }
            }
            if (!ListUtils.isEmpty(recordList)) {
                advertisingExposureRecordDao.insert(recordList);
            }
        }
    }

    /**
     * 获取出版社书刊权限
     */
    @Override
    @ParamLog("获取出版社书刊权限")
    public Boolean getAgentPermission(Long agentId) throws BizException {
        AdvertisingAgentPermission permission = advertisingAgentPermissionDao.getByAgentId(agentId);
        return null == permission ? true : permission.getIsOpen();
    }

    /**
     * 设置出版社书刊权限
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @ParamLog(value = "设置出版社书刊权限", isAfterReturn = false)
    public void setAgentPermission(AdvertisingAgentPermission agentPermission) throws BizException {
        if (null == agentPermission.getAgentId() || null == agentPermission.getIsOpen()) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数有误！");
        }
        AdvertisingAgentPermission permission = advertisingAgentPermissionDao.getByAgentId(agentPermission.getAgentId());
        if (null == permission) {
            advertisingAgentPermissionDao.insert(agentPermission);
        } else {
            agentPermission.setId(permission.getId());
            advertisingAgentPermissionDao.update(agentPermission);
        }
    }

    /**
     * 设置编辑书刊权限
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @ParamLog(value = "设置编辑书刊权限", isAfterReturn = false)
    public void setAdviserPermission(AdvertisingAdviserPermission adviserPermission) throws BizException {
        if (null == adviserPermission.getAdviserId() || null == adviserPermission.getIsOpen()
                || (!adviserPermission.getIsBatchOperation() && (null == adviserPermission.getBookId() || null == adviserPermission.getChannelId()))) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数有误！");
        }
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("adviserId", adviserPermission.getAdviserId());
        if (!adviserPermission.getIsBatchOperation()) {
            paramMap.put("bookId", adviserPermission.getBookId());
            paramMap.put("channelId", adviserPermission.getChannelId());
            AdvertisingAdviserPermission permission = (AdvertisingAdviserPermission) advertisingAdviserPermissionDao.getBy(paramMap, "getByAdviserId");
            if (null == permission) {
                advertisingAdviserPermissionDao.insert(adviserPermission);
            } else {
                adviserPermission.setId(permission.getId());
                advertisingAdviserPermissionDao.update(adviserPermission);
            }
        } else {
            // 删除编辑所有数据
            advertisingAdviserPermissionDao.deleteByAdviserId(adviserPermission.getAdviserId());
            List<BookDto> bookList = bookDao.listBookIdAndChannelId4Adviser(paramMap);
            if (!ListUtils.isEmpty(bookList)) {
                List<AdvertisingAdviserPermission> list = new ArrayList<>();
                for (BookDto bookDto : bookList) {
                    AdvertisingAdviserPermission permission = new AdvertisingAdviserPermission();
                    permission.setAdviserId(adviserPermission.getAdviserId());
                    permission.setIsOpen(adviserPermission.getIsOpen());
                    permission.setBookId(bookDto.getBookId());
                    permission.setChannelId(bookDto.getChannelId());
                    list.add(permission);
                }
                advertisingAdviserPermissionDao.batchInsert(list);
            }
        }
    }

    /**
     * 编辑端获取书刊列表
     */
    @Override
    public PageBean listBook4Adviser(Map<String, Object> paramMap, PageParam pageParam, Long adviserId) throws BizException {
        Long agentId = adviserConsr.getAgentIdByAdviser(adviserId);
        paramMap.put("agentId", agentId);
        PageBean pageBean = bookDao.listPage(pageParam, paramMap, "listBook4Adviser");
        if (pageBean == null || ListUtils.isEmpty(pageBean.getRecordList())) {
            return new PageBean(0, 0, new ArrayList<>());
        }
        // 填充渠道基础信息
        bookSet.setChannelInfoList(pageBean.getRecordList());
        return pageBean;
    }

    /**
     * 平台端获取某编辑书刊列表
     */
    @Override
    public PageBean listBook4Platform(Map<String, Object> paramMap, PageParam pageParam, Long adviserId) throws BizException {
        PageBean pageBean = bookDao.listPage(pageParam, paramMap, "listBook4Platform");
        if (pageBean == null || ListUtils.isEmpty(pageBean.getRecordList())) {
            return new PageBean(0, 0, new ArrayList<>());
        }
        return pageBean;
    }

    /**
     * 平台端获取社群书微信群列表
     */
    @Override
    public PageBean listGroupQrcode4Platform(Map<String, Object> paramMap, PageParam pageParam) throws BizException {
        PageBean pageBean = bookGroupClassifyDao.listPage(pageParam, paramMap, "listGroupQrcode4Platform");
        if (pageBean == null || ListUtils.isEmpty(pageBean.getRecordList())) {
            return new PageBean(0, 0, new ArrayList<>());
        }
        return pageBean;
    }

    /**
     * 获取广告位投放书刊列表
     */
    @Override
    public PageBean distributionBookList(Long adId, String name, PageParam pageParam) throws BizException {
        AdvertisingSpace space = advertisingSpaceDao.getById(adId);
        if (null == space) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "广告位不存在或已删除！");
        }
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("adId", adId);
        paramMap.put("name", StringUtil.isEmpty(name) ? null : name);
        PageBean pageBean = bookDao.listPage(pageParam, paramMap, "listBook4AdvertisingSpace");
        if (pageBean == null || ListUtils.isEmpty(pageBean.getRecordList())) {
            return new PageBean(0, 0, new ArrayList<>());
        }
        // 设置编辑名称
        bookSet.setAdviserName4BookDto(pageBean.getRecordList());
        // 设置出版名称
        bookSet.setAgentName4BookDto(pageBean.getRecordList());
        for (Object object : pageBean.getRecordList()) {
            BookDto bookDto = (BookDto) object;
            // 曝光量
            Long exposureNum = getExposureNumByAdId(adId, bookDto.getBookId(), bookDto.getAdviserId(), bookDto.getChannelId(), null);
            bookDto.setExposureNum(exposureNum);
            // 点击量
            Long clickNum = getClickNumByAdId(adId, bookDto.getBookId(), bookDto.getAdviserId(), bookDto.getChannelId(), null);
            bookDto.setClickNum(clickNum);
            BigDecimal clickRate = (null == bookDto.getExposureNum() || bookDto.getExposureNum().equals(0L)) ? BigDecimal.ZERO :
                    new BigDecimal(bookDto.getClickNum().doubleValue() / bookDto.getExposureNum()).setScale(2, BigDecimal.ROUND_HALF_UP);
            bookDto.setClickRate(clickRate);
            BigDecimal income = getIncomeByAdId(adId, bookDto.getBookId(), bookDto.getChannelId(), null);
            bookDto.setTotalIncome(income);
        }
        return pageBean;
    }

    /**
     * 获取广告位投放微信群列表
     */
    @Override
    public PageBean distributionWechatGroupList(Long adId, String name, PageParam pageParam) throws BizException {
        AdvertisingSpace space = advertisingSpaceDao.getById(adId);
        if (null == space) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "广告位不存在或已删除！");
        }
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("adId", adId);
        paramMap.put("name", StringUtil.isEmpty(name) ? null : name);
        PageBean pageBean = advertisingDistributionBookDao.listPage(pageParam, paramMap, "distributionWechatGroupList");
        if (pageBean == null || ListUtils.isEmpty(pageBean.getRecordList())) {
            return new PageBean(0, 0, new ArrayList<>());
        }
        List<Long> adviserIds = new ArrayList<>();
        List<Long> agentIds = new ArrayList<>();
        pageBean.getRecordList().forEach(obj -> {
            adviserIds.add(((AdGroupQrcodeDTO) obj).getAdviserId());
            agentIds.add(((AdGroupQrcodeDTO) obj).getAgentId());
        });
        Map<Long, String> adviserNameMap = adviserConsr.getNames(new ArrayList<>(new HashSet<>(adviserIds)));
        Map<Long, String> agentNameMap = agentConsr.getNames(new ArrayList<>(new HashSet<>(agentIds)));
        for (Object object : pageBean.getRecordList()) {
            AdGroupQrcodeDTO qrcodeDTO = (AdGroupQrcodeDTO) object;
            if (!MapUtils.isEmpty(adviserNameMap) && adviserNameMap.containsKey(qrcodeDTO.getAdviserId())) {
                qrcodeDTO.setAdviserName(adviserNameMap.get(qrcodeDTO.getAdviserId()));
            }
            if (!MapUtils.isEmpty(agentNameMap) && agentNameMap.containsKey(qrcodeDTO.getAgentId())) {
                qrcodeDTO.setAgentName(agentNameMap.get(qrcodeDTO.getAgentId()));
            }
            // 曝光量
            Long exposureNum = getExposureNumByGroupQrcodeId(qrcodeDTO.getGroupQrcodeId(), adId, null);
            qrcodeDTO.setExposureNum(exposureNum);
            // 点击量
            Long clickNum = getClickNumByGroupQrcodeId(qrcodeDTO.getGroupQrcodeId(), adId, null);
            qrcodeDTO.setClickNum(clickNum);
            BigDecimal clickRate = (null == qrcodeDTO.getExposureNum() || qrcodeDTO.getExposureNum().equals(0L)) ? BigDecimal.ZERO :
                    new BigDecimal(qrcodeDTO.getClickNum().doubleValue() / qrcodeDTO.getExposureNum()).setScale(2, BigDecimal.ROUND_HALF_UP);
            qrcodeDTO.setClickRate(clickRate);
            BigDecimal income = getIncomeByGroupQrcodeId(qrcodeDTO.getGroupQrcodeId(), adId, null);
            qrcodeDTO.setTotalIncome(income);
            // 点击读者量
            Long clickUserNum = getClickUserNumByGroupQrcodeId(qrcodeDTO.getGroupQrcodeId(), adId, null);
            qrcodeDTO.setClickUserNum(clickUserNum);
        }
        return pageBean;
    }

    /**
     * 获取广告位投放书刊列表
     */
    @Override
    public PageBean advertisingBookList(String name, PageParam pageParam) throws BizException {
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("name", StringUtil.isEmpty(name) ? null : name);
        PageBean pageBean = bookDao.listPage(pageParam, paramMap, "advertisingBookList");
        if (pageBean == null || ListUtils.isEmpty(pageBean.getRecordList())) {
            return new PageBean(0, 0, new ArrayList<>());
        }
        // 设置编辑名称
        bookSet.setAdviserName4BookDto(pageBean.getRecordList());
        // 设置出版名称
        bookSet.setAgentName4BookDto(pageBean.getRecordList());
        for (Object object : pageBean.getRecordList()) {
            BookDto bookDto = (BookDto) object;
            // 曝光量
            Long exposureNum = getExposureNumByBookId(bookDto.getBookId(), bookDto.getAdviserId(), bookDto.getChannelId());
            bookDto.setExposureNum(exposureNum);
            // 点击量
            Long clickNum = getClickNumByBookId(bookDto.getBookId(), bookDto.getAdviserId(), bookDto.getChannelId());
            bookDto.setClickNum(clickNum);
            BigDecimal clickRate = (null == bookDto.getExposureNum() || bookDto.getExposureNum().equals(0L)) ? BigDecimal.ZERO :
                    new BigDecimal(bookDto.getClickNum().doubleValue() / bookDto.getExposureNum()).setScale(2, BigDecimal.ROUND_HALF_UP);
            bookDto.setClickRate(clickRate);
            BigDecimal income = getIncomeByBookId(bookDto.getBookId(), bookDto.getChannelId());
            bookDto.setTotalIncome(income);
        }
        return pageBean;
    }

    /**
     * 获取广告位投放微信群列表
     */
    @Override
    public PageBean advertisingWechatGroupList(String name, PageParam pageParam) throws BizException {
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("name", StringUtil.isEmpty(name) ? null : name);
        PageBean pageBean = bookGroupClassifyDao.listPage(pageParam, paramMap, "advertisingWechatGroupList");
        if (pageBean == null || ListUtils.isEmpty(pageBean.getRecordList())) {
            return new PageBean(0, 0, new ArrayList<>());
        }
        for (Object object : pageBean.getRecordList()) {
            AdGroupQrcodeDTO qrcodeDTO = (AdGroupQrcodeDTO) object;
            // 曝光量
            Long exposureNum = getExposureNumByGroupQrcodeId(qrcodeDTO.getGroupQrcodeId(), null, null);
            qrcodeDTO.setExposureNum(exposureNum);
            // 点击量
            Long clickNum = getClickNumByGroupQrcodeId(qrcodeDTO.getGroupQrcodeId(), null, null);
            qrcodeDTO.setClickNum(clickNum);
            BigDecimal clickRate = (null == qrcodeDTO.getExposureNum() || qrcodeDTO.getExposureNum().equals(0L)) ? BigDecimal.ZERO :
                    new BigDecimal(qrcodeDTO.getClickNum().doubleValue() / qrcodeDTO.getExposureNum()).setScale(2, BigDecimal.ROUND_HALF_UP);
            qrcodeDTO.setClickRate(clickRate);
            BigDecimal income = getIncomeByGroupQrcodeId(qrcodeDTO.getGroupQrcodeId(), null, null);
            qrcodeDTO.setTotalIncome(income);
            // 点击读者量
            Long clickUserNum = getClickUserNumByGroupQrcodeId(qrcodeDTO.getGroupQrcodeId(), null, null);
            qrcodeDTO.setClickUserNum(clickUserNum);
        }
        return pageBean;
    }

    /**
     * 获取广告位投放微信群曝光量前几位
     */
    @Override
    public List<AdGroupQrcodeDTO> advertisingWechatGroupRankTop() throws BizException {
        List<AdGroupQrcodeDTO> list = advertisingExposureRecordDao.advertisingWechatGroupRankTop();
        return list;
    }

    /**
     * 获取书刊广告位曝光量
     */
    private Long getExposureNumByAdId(Long adId, Long bookId, Long adviserId, Long channelId, String statisMonth) {
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("adId", adId);
        paramMap.put("bookId", bookId);
        paramMap.put("adviserId", adviserId);
        paramMap.put("channelId", channelId);
        paramMap.put("statisMonth", statisMonth);
        Long exposureNum = (Long) advertisingExposureRecordDao.getBy(paramMap, "getExposureNum");
        return null == exposureNum ? 0L : exposureNum;
    }

    /**
     * 获取书刊广告位曝光量
     */
    private Long getExposureNumByBookId(Long bookId, Long adviserId, Long channelId) {
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("bookId", bookId);
        paramMap.put("adviserId", adviserId);
        paramMap.put("channelId", channelId);
        Long exposureNum = (Long) advertisingExposureRecordDao.getBy(paramMap, "getExposureNumByBookId");
        return null == exposureNum ? 0L : exposureNum;
    }

    /**
     * 获取书刊广告位曝光量
     */
    private Long getExposureNumByGroupQrcodeId(Long qrcodeId, Long adId, String statisMonth) {
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("qrcodeId", qrcodeId);
        paramMap.put("adId", adId);
        paramMap.put("statisMonth", statisMonth);
        Long exposureNum = (Long) advertisingExposureRecordDao.getBy(paramMap, "getExposureNumByGroupQrcodeId");
        return null == exposureNum ? 0L : exposureNum;
    }

    /**
     * 获取书刊广告位点击量
     */
    private Long getClickNumByAdId(Long adId, Long bookId, Long adviserId, Long channelId, String statisMonth) {
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("adId", adId);
        paramMap.put("bookId", bookId);
        paramMap.put("adviserId", adviserId);
        paramMap.put("channelId", channelId);
        paramMap.put("statisMonth", statisMonth);
        Long clickNum = (Long) advertisingClickRecordDao.getBy(paramMap, "getClickNum");
        return null == clickNum ? 0L : clickNum;
    }

    /**
     * 获取书刊广告位点击量
     */
    private Long getClickNumByBookId(Long bookId, Long adviserId, Long channelId) {
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("bookId", bookId);
        paramMap.put("adviserId", adviserId);
        paramMap.put("channelId", channelId);
        Long clickNum = (Long) advertisingClickRecordDao.getBy(paramMap, "getClickNumByBookId");
        return null == clickNum ? 0L : clickNum;
    }

    /**
     * 获取书刊广告位点击量
     */
    private Long getClickNumByGroupQrcodeId(Long qrcodeId, Long adId, String statisMonth) {
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("qrcodeId", qrcodeId);
        paramMap.put("adId", adId);
        paramMap.put("statisMonth", statisMonth);
        Long clickNum = (Long) advertisingClickRecordDao.getBy(paramMap, "getClickNumByGroupQrcodeId");
        return null == clickNum ? 0L : clickNum;
    }

    /**
     * 获取书刊广告位点击读者量
     */
    private Long getClickUserNumByGroupQrcodeId(Long qrcodeId, Long adId, String statisMonth) {
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("qrcodeId", qrcodeId);
        paramMap.put("adId", adId);
        paramMap.put("statisMonth", statisMonth);
        Long clickUserNum = (Long) advertisingClickRecordDao.getBy(paramMap, "getClickUserNumByGroupQrcodeId");
        return null == clickUserNum ? 0L : clickUserNum;
    }

    /**
     * 获取书刊广告位曝光量
     */
    private Long getExposureNumByAdId(Long adId) {
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("adId", adId);
        Long exposureNum = (Long) advertisingExposureRecordDao.getBy(paramMap, "getExposureNum");
        return null == exposureNum ? 0L : exposureNum;
    }

    /**
     * 获取书刊广告位曝光量
     */
    private Long getExposureNumByMasterId(Long masterId) {
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("masterId", masterId);
        Long exposureNum = (Long) advertisingExposureRecordDao.getBy(paramMap, "getExposureNumByMasterId");
        return null == exposureNum ? 0L : exposureNum;
    }

    /**
     * 获取书刊广告位点击量
     */
    private Long getClickNumByAdId(Long adId) {
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("adId", adId);
        Long clickNum = (Long) advertisingClickRecordDao.getBy(paramMap, "getClickNum");
        return null == clickNum ? 0L : clickNum;
    }

    /**
     * 获取书刊广告位点击量
     */
    private Long getClickNumByMasterId(Long masterId) {
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("masterId", masterId);
        Long clickNum = (Long) advertisingClickRecordDao.getBy(paramMap, "getClickNumByMasterId");
        return null == clickNum ? 0L : clickNum;
    }

    /**
     * 获取书刊广告位收益
     */
    private BigDecimal getIncomeByAdId(Long adId) {
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("adId", adId);
        BigDecimal income = (BigDecimal) advertisingIncomeDailyDao.getBy(paramMap, "getIncomeByAdId");
        return null == income ? BigDecimal.ZERO : income;
    }

    /**
     * 获取书刊广告位收益
     */
    private BigDecimal getIncomeByMasterId(Long masterId) {
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("masterId", masterId);
        BigDecimal income = (BigDecimal) advertisingIncomeDailyDao.getBy(paramMap, "getIncomeByMasterId");
        return null == income ? BigDecimal.ZERO : income;
    }

    /**
     * 获取书刊广告位收益
     */
    private BigDecimal getIncomeByAdId(Long adId, Long bookId, Long channelId, String statisMonth) {
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("adId", adId);
        paramMap.put("bookId", bookId);
        paramMap.put("channelId", channelId);
        paramMap.put("statisMonth", statisMonth);
        BigDecimal income = (BigDecimal) advertisingIncomeDailyDao.getBy(paramMap, "getIncomeByAdId");
        return null == income ? BigDecimal.ZERO : income;
    }

    /**
     * 获取书刊广告位点击量
     */
    private BigDecimal getIncomeByBookId(Long bookId, Long channelId) {
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("bookId", bookId);
        paramMap.put("channelId", channelId);
        BigDecimal income = (BigDecimal) advertisingIncomeDailyDao.getBy(paramMap, "getIncomeByBookId");
        return null == income ? BigDecimal.ZERO : income;
    }

    /**
     * 获取书刊广告位收益
     */
    private BigDecimal getIncomeByGroupQrcodeId(Long qrcodeId, Long adId, String statisMonth) {
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("qrcodeId", qrcodeId);
        paramMap.put("adId", adId);
        paramMap.put("statisMonth", statisMonth);
        BigDecimal income = (BigDecimal) advertisingIncomeDailyDao.getBy(paramMap, "getIncomeByGroupQrcodeId");
        return null == income ? BigDecimal.ZERO : income;
    }

    /**
     * 获取广告位列表
     */
    @Override
    public PageBean list(String name, PageParam pageParam, Boolean filterClose) throws BizException {
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("name", StringUtil.isEmpty(name) ? null : name);
        paramMap.put("filterClose", filterClose);
        PageBean pageBean = advertisingSpaceDao.listPage(pageParam, paramMap, "list");
        if (pageBean == null || ListUtils.isEmpty(pageBean.getRecordList())) {
            return new PageBean(0, 0, new ArrayList<>());
        }
        // 填充投放数据
        for (Object object : pageBean.getRecordList()) {
            AdvertisingSpaceDTO spaceDTO = (AdvertisingSpaceDTO) object;
            if (null == spaceDTO) {
                continue;
            }
            AdvertisingMaster master = advertisingMasterDao.getById(spaceDTO.getMasterId());
            if (null != master) {
                spaceDTO.setMasterName(master.getMasterName());
            }
            spaceDTO.setExposureNum(getExposureNumByAdId(spaceDTO.getId()));
            spaceDTO.setClickNum(getClickNumByAdId(spaceDTO.getId()));
            BigDecimal clickRate = (null == spaceDTO.getExposureNum() || spaceDTO.getExposureNum().equals(0L)) ? BigDecimal.ZERO :
                    new BigDecimal(spaceDTO.getClickNum().doubleValue() / spaceDTO.getExposureNum()).setScale(2, BigDecimal.ROUND_HALF_UP);
            spaceDTO.setClickRate(clickRate);
            BigDecimal income = getIncomeByAdId(spaceDTO.getId());
            spaceDTO.setTotalIncome(income);
            List<AdvertisingDistributionBook> list = advertisingDistributionBookDao.getByAdId(spaceDTO.getId());
            if (!ListUtils.isEmpty(list)) {
                spaceDTO.setBookNum(list.size());
                Long bookId = list.get(0).getBookId();
                BookDto bookDto = bookBiz.getBaseById(bookId);
                spaceDTO.setBookName(StringUtil.addBracket(bookDto.getBookName()));
            } else {
                spaceDTO.setBookNum(0);
            }
            // 填充广告位置名称
            if (AdPositionEnum.WECHAT_GROUP_MSG.positionCode.equals(spaceDTO.getAdPosition())) {
                fillAdpositionName4WechatGroupAd(spaceDTO);
            } else {
                fillAdpositionName4CommonAd(spaceDTO);
            }
        }
        return pageBean;
    }

    /**
     * 书刊广告位明细
     */
    @Override
    public PageBean advertisingDetail4Book(Long bookId, Long channelId, Long adviserId, String statisMonth, PageParam pageParam) throws BizException {
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("bookId", bookId);
        paramMap.put("channelId", channelId);
        paramMap.put("adviserId", adviserId);
        PageBean pageBean = advertisingSpaceDao.listPage(pageParam, paramMap, "advertisingDetail4Book");
        if (pageBean == null || ListUtils.isEmpty(pageBean.getRecordList())) {
            return new PageBean(0, 0, new ArrayList<>());
        }
        // 填充投放数据
        for (Object object : pageBean.getRecordList()) {
            AdvertisingSpaceDTO spaceDTO = (AdvertisingSpaceDTO) object;
            if (null == spaceDTO) {
                continue;
            }
            AdvertisingMaster master = advertisingMasterDao.getById(spaceDTO.getMasterId());
            if (null != master) {
                spaceDTO.setMasterName(master.getMasterName());
            }
            spaceDTO.setExposureNum(getExposureNumByAdId(spaceDTO.getId(), bookId, adviserId, channelId, statisMonth));
            spaceDTO.setClickNum(getClickNumByAdId(spaceDTO.getId(), bookId, adviserId, channelId, statisMonth));
            BigDecimal clickRate = (null == spaceDTO.getExposureNum() || spaceDTO.getExposureNum().equals(0L)) ? BigDecimal.ZERO :
                    new BigDecimal(spaceDTO.getClickNum().doubleValue() / spaceDTO.getExposureNum()).setScale(2, BigDecimal.ROUND_HALF_UP);
            spaceDTO.setClickRate(clickRate);
            BigDecimal income = getIncomeByAdId(spaceDTO.getId(), bookId, channelId, statisMonth);
            spaceDTO.setTotalIncome(income);
            // 填充广告位置名称
            fillAdpositionName4CommonAd(spaceDTO);
        }
        return pageBean;
    }

    /**
     * 导出书刊广告位明细
     */
    @Override
    public Map<String, Object> exportAdvertisingDetail4Book(Long bookId, Long channelId, Long adviserId, String statisMonth) throws BizException {
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("bookId", bookId);
        paramMap.put("channelId", channelId);
        paramMap.put("adviserId", adviserId);
        List<AdvertisingSpaceDTO> list = advertisingSpaceDao.advertisingDetail4Book(paramMap);
        if (ListUtils.isEmpty(list)) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "暂无数据导出！");
        }
        // 字段名
        String[] rowsName = {"序号", "广告位名称", "广告主", "广告位置", "投放时间", "曝光量", "点击量", "点击率", "累积收益"};
        List<Object[]> dataList = new ArrayList<>();
        Object[] objs;
        for (int i = 0; i < list.size(); i++) {
            AdvertisingSpaceDTO spaceDTO = list.get(i);
            objs = new Object[rowsName.length];
            objs[0] = i + 1;
            objs[1] = spaceDTO.getAdName();
            AdvertisingMaster master = advertisingMasterDao.getById(spaceDTO.getMasterId());
            objs[2] = null != master ? master.getMasterName() : "";
            fillAdpositionName4CommonAd(spaceDTO);
            objs[3] = spaceDTO.getAdPositionName();
            objs[4] = DateNewUtils.formatDate(spaceDTO.getDistributionTime());
            Long exposureNum = getExposureNumByAdId(spaceDTO.getId(), bookId, adviserId, channelId, statisMonth);
            objs[5] = exposureNum;
            Long clickNum = getClickNumByAdId(spaceDTO.getId(), bookId, adviserId, channelId, statisMonth);
            objs[6] = clickNum;
            BigDecimal clickRate = (null == exposureNum || exposureNum.equals(0L)) ? BigDecimal.ZERO :
                    new BigDecimal(clickNum.doubleValue() / exposureNum).setScale(2, BigDecimal.ROUND_HALF_UP);
            objs[7] = clickRate.multiply(new BigDecimal(100)).setScale(0, BigDecimal.ROUND_HALF_UP) + "%";
            BigDecimal income = getIncomeByAdId(spaceDTO.getId(), bookId, channelId, statisMonth);
            objs[8] = income + "元";
            dataList.add(objs);
        }
        BookDto bookDto = bookBiz.getBaseById(bookId);
        String[] dateStr = statisMonth.split("-");
        String fileName = StringUtil.addBracket(bookDto.getBookName()) + dateStr[0] + "年" + Integer.parseInt(dateStr[1]) + "月广告位明细";
        String fileUrl = exportConsr.exportExcel(fileName, rowsName, dataList);
        Map<String, Object> result = new HashMap<>();
        result.put("fileUrl", fileUrl);
        result.put("fileName", fileName);
        return result;
    }

    /**
     * 填充作品应用、社群码页面广告位名称
     */
    private void fillAdpositionName4CommonAd(AdvertisingSpaceDTO spaceDTO) {
        String adPositionName = "";
        String adPositionModeName = AdPositionModeEnum.getNameByCode(spaceDTO.getAdPositionMode());
        if (AdPositionEnum.BOOK_GROUP_PAGE.positionCode.equals(spaceDTO.getAdPosition())) {
            adPositionName = AdPositionDetailEnum.getNameByCode(spaceDTO.getAdPositionDetail()) + ">" + adPositionModeName;
        } else {
            int index = spaceDTO.getAdPositionDetail().indexOf("_");
            String type = spaceDTO.getAdPositionDetail().substring(0, index);
            String typeCode = spaceDTO.getAdPositionDetail().substring(index + 1, spaceDTO.getAdPositionDetail().length());
            if ("APP".equalsIgnoreCase(type)) {
                AppTypeDto appTypeDto = ResponseHandleUtil.parseResponse(appTypeService.getByTypeCode(typeCode), AppTypeDto.class);
                if (null != appTypeDto) {
                    adPositionName = appTypeDto.getTypeName() + ">" + adPositionModeName;
                } else {
                    adPositionName = adPositionModeName;
                }
            } else {
                String productTypeName = ResponseHandleUtil.parseResponse(productService.getProTypeName(typeCode), String.class);
                if (!StringUtil.isEmpty(productTypeName)) {
                    adPositionName = productTypeName + ">" + adPositionModeName;
                } else {
                    adPositionName = adPositionModeName;
                }
            }
        }
        spaceDTO.setAdPositionName(adPositionName);
    }

    /**
     * 填充作品应用、社群码页面广告位名称
     */
    private void fillAdpositionName4WechatGroupAd(AdvertisingSpaceDTO spaceDTO) {
        String adPositionName = AdPositionEnum.getNameByCode(spaceDTO.getAdPosition()) + ">" + AdPositionModeEnum.getNameByCode(spaceDTO.getAdPositionMode());
        spaceDTO.setAdPositionName(adPositionName);
    }

    /**
     * 微信群广告位明细
     */
    @Override
    public PageBean advertisingDetail4WechatGroup(Long qrcodeId, String statisMonth, PageParam pageParam) throws BizException {
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("qrcodeId", qrcodeId);
        PageBean pageBean = advertisingSpaceDao.listPage(pageParam, paramMap, "advertisingDetail4WechatGroup");
        if (pageBean == null || ListUtils.isEmpty(pageBean.getRecordList())) {
            return new PageBean(0, 0, new ArrayList<>());
        }
        // 填充投放数据
        for (Object object : pageBean.getRecordList()) {
            AdvertisingSpaceDTO spaceDTO = (AdvertisingSpaceDTO) object;
            if (null == spaceDTO) {
                continue;
            }
            AdvertisingMaster master = advertisingMasterDao.getById(spaceDTO.getMasterId());
            if (null != master) {
                spaceDTO.setMasterName(master.getMasterName());
            }
            spaceDTO.setExposureNum(getExposureNumByGroupQrcodeId(qrcodeId, spaceDTO.getId(), statisMonth));
            spaceDTO.setClickNum(getClickNumByGroupQrcodeId(qrcodeId, spaceDTO.getId(), statisMonth));
            BigDecimal clickRate = (null == spaceDTO.getExposureNum() || spaceDTO.getExposureNum().equals(0L)) ? BigDecimal.ZERO :
                    new BigDecimal(spaceDTO.getClickNum().doubleValue() / spaceDTO.getExposureNum()).setScale(2, BigDecimal.ROUND_HALF_UP);
            spaceDTO.setClickRate(clickRate);
            BigDecimal income = getIncomeByGroupQrcodeId(qrcodeId, spaceDTO.getId(), statisMonth);
            spaceDTO.setTotalIncome(income);
            // 点击读者量
            if (!AdPositionModeEnum.JUST_PIC.modeCode.equals(spaceDTO.getAdPositionMode()) && !StringUtil.isEmpty(spaceDTO.getAdLink())) {
                spaceDTO.setClickUserNum(getClickUserNumByGroupQrcodeId(qrcodeId, spaceDTO.getId(), statisMonth));
            } else {
                spaceDTO.setClickUserNum(0L);
            }
            // 填充广告位置名称
            spaceDTO.setAdPositionName(AdPositionModeEnum.getNameByCode(spaceDTO.getAdPositionMode()));
        }
        return pageBean;
    }

    /**
     * 导出微信群广告位明细
     */
    @Override
    public Map<String, Object> exportAdvertisingDetail4WechatGroup(Long qrcodeId, String statisMonth) throws BizException {
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("qrcodeId", qrcodeId);
        List<AdvertisingSpaceDTO> list = advertisingSpaceDao.advertisingDetail4WechatGroup(paramMap);
        if (ListUtils.isEmpty(list)) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "暂无数据导出！");
        }
        // 字段名
        String[] rowsName = {"序号", "广告位名称", "广告主", "广告位置", "投放时间", "曝光量", "点击量", "点击率", "累积收益"};
        List<Object[]> dataList = new ArrayList<>();
        Object[] objs;
        for (int i = 0; i < list.size(); i++) {
            AdvertisingSpaceDTO spaceDTO = list.get(i);
            objs = new Object[rowsName.length];
            objs[0] = i + 1;
            objs[1] = spaceDTO.getAdName();
            AdvertisingMaster master = advertisingMasterDao.getById(spaceDTO.getMasterId());
            objs[2] = null != master ? master.getMasterName() : "";
            objs[3] = AdPositionModeEnum.getNameByCode(spaceDTO.getAdPositionMode());
            objs[4] = DateNewUtils.formatDate(spaceDTO.getDistributionTime());
            Long exposureNum = getExposureNumByGroupQrcodeId(qrcodeId, spaceDTO.getId(), statisMonth);
            objs[5] = exposureNum;
            Long clickNum = getClickNumByGroupQrcodeId(qrcodeId, spaceDTO.getId(), statisMonth);
            objs[6] = clickNum;
            BigDecimal clickRate = (null == exposureNum || exposureNum.equals(0L)) ? BigDecimal.ZERO :
                    new BigDecimal(clickNum.doubleValue() / exposureNum).setScale(2, BigDecimal.ROUND_HALF_UP);
            objs[7] = clickRate.multiply(new BigDecimal(100)).setScale(0, BigDecimal.ROUND_HALF_UP) + "%";
            BigDecimal income = getIncomeByGroupQrcodeId(qrcodeId, spaceDTO.getId(), statisMonth);
            objs[8] = income + "元";
            dataList.add(objs);
        }
        GroupQrcode groupQrcode = groupQrcodeDao.getById(qrcodeId);
        String[] dateStr = statisMonth.split("-");
        String fileName = (null != groupQrcode ? groupQrcode.getGroupName() : "") + dateStr[0] + "年" + Integer.parseInt(dateStr[1]) + "月广告位明细";
        String fileUrl = exportConsr.exportExcel(fileName, rowsName, dataList);
        Map<String, Object> result = new HashMap<>();
        result.put("fileUrl", fileUrl);
        result.put("fileName", fileName);
        return result;
    }

    /**
     * 微信群广告位点击读者列表
     */
    @Override
    public PageBean clickUserList4AdvertisingWechatGroup(Long qrcodeId, Long adId, String statisMonth, PageParam pageParam) throws BizException {
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("qrcodeId", qrcodeId);
        paramMap.put("adId", adId);
        paramMap.put("statisMonth", statisMonth);
        PageBean pageBean = advertisingClickRecordDao.listPage(pageParam, paramMap, "clickUserList4AdvertisingWechatGroup");
        if (pageBean == null || ListUtils.isEmpty(pageBean.getRecordList())) {
            return new PageBean(0, 0, new ArrayList<>());
        }
        List<Long> wechatUserIds = new ArrayList<>();
        pageBean.getRecordList().forEach(obj -> {
            wechatUserIds.add(((WechatGroupClickUserDTO) obj).getWechatUserId());
        });
        Map<Long, WechatUser> wechatUserMap = readerConsr.getUserList(wechatUserIds);
        for (Object object : pageBean.getRecordList()) {
            WechatGroupClickUserDTO dto = (WechatGroupClickUserDTO) object;
            if (!MapUtils.isEmpty(wechatUserMap) && wechatUserMap.containsKey(dto.getWechatUserId())) {
                WechatUser wechatUser = wechatUserMap.get(dto.getWechatUserId());
                dto.setWechatUserName(wechatUser.getWechatUserNickname());
                dto.setWechatUserHeadurl(wechatUser.getWechatUserHeadurl());
                dto.setWechatUserSex(wechatUser.getWechatUserSex());
                dto.setWechatUserProvince(wechatUser.getWechatUserProvince());
                dto.setWechatUserCity(wechatUser.getWechatUserCity());
            }
        }
        return pageBean;
    }

    /**
     * 导出微信群广告位点击读者列表
     */
    @Override
    public Map<String, Object> exportClickUserList4AdvertisingWechatGroup(Long qrcodeId, Long adId, String statisMonth) throws BizException {
        AdvertisingSpace space = advertisingSpaceDao.getById(adId);
        if (null == space) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "广告位不存在或已删除！");
        }
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("qrcodeId", qrcodeId);
        paramMap.put("adId", adId);
        paramMap.put("statisMonth", statisMonth);
        List<WechatGroupClickUserDTO> list = advertisingClickRecordDao.clickUserList4AdvertisingWechatGroup(paramMap);
        if (ListUtils.isEmpty(list)) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "暂无数据导出！");
        }
        List<Long> wechatUserIds = list.stream().map(WechatGroupClickUserDTO::getWechatUserId).collect(Collectors.toList());
        Map<Long, WechatUser> wechatUserMap = readerConsr.getUserList(wechatUserIds);
        // 字段名
        String[] rowsName = {"序号", "读者信息", "读者标签", "点击时间"};
        List<Object[]> dataList = new ArrayList<>();
        Object[] objs;
        for (int i = 0; i < list.size(); i++) {
            WechatGroupClickUserDTO dto = list.get(i);
            objs = new Object[rowsName.length];
            objs[0] = i + 1;
            if (!MapUtils.isEmpty(wechatUserMap) && wechatUserMap.containsKey(dto.getWechatUserId())) {
                WechatUser wechatUser = wechatUserMap.get(dto.getWechatUserId());
                objs[1] = wechatUser.getWechatUserNickname();
                String sex = wechatUser.getWechatUserSex().equals(1) ? "男" : (wechatUser.getWechatUserSex().equals(2) ? "女" : "未知");
                objs[2] = sex + "，" + wechatUser.getWechatUserCity();
            } else {
                objs[1] = "";
                objs[2] = "";
            }
            objs[3] = DateNewUtils.formatDate(dto.getCreateTime());
            dataList.add(objs);
        }
        GroupQrcode groupQrcode = groupQrcodeDao.getById(qrcodeId);
        String[] dateStr = statisMonth.split("-");
        String fileName;
        if (StringUtil.isEmpty(statisMonth)) {
            fileName = (null != groupQrcode ? groupQrcode.getGroupName() : "") + dateStr[0] + "年" + Integer.parseInt(dateStr[1]) + "月" + space.getAdName() + "广告位点击读者明细";
        } else {
            fileName = space.getAdName() + "广告位" + (null != groupQrcode ? groupQrcode.getGroupName() : "") + "点击读者明细";
        }
        String fileUrl = exportConsr.exportExcel(fileName, rowsName, dataList);
        Map<String, Object> result = new HashMap<>();
        result.put("fileUrl", fileUrl);
        result.put("fileName", fileName);
        return result;
    }

    /**
     * 广告主广告位明细
     */
    @Override
    public PageBean advertisingDetail4Master(Long masterId, String statisMonth, PageParam pageParam) throws BizException {
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("masterId", masterId);
        PageBean pageBean = advertisingSpaceDao.listPage(pageParam, paramMap, "advertisingDetail4Master");
        if (pageBean == null || ListUtils.isEmpty(pageBean.getRecordList())) {
            return new PageBean(0, 0, new ArrayList<>());
        }
        // 填充投放数据
        for (Object object : pageBean.getRecordList()) {
            AdvertisingSpaceDTO spaceDTO = (AdvertisingSpaceDTO) object;
            if (null == spaceDTO) {
                continue;
            }
            Long exposureNum = 0L;
            Long clickNum = 0L;
            BigDecimal income = BigDecimal.ZERO;
            if (null != spaceDTO.getBookId()) {
                if (AdPositionEnum.WECHAT_GROUP_MSG.positionCode.equals(spaceDTO.getAdPosition())) {
                    exposureNum = getExposureNumByGroupQrcodeId(spaceDTO.getQrcodeId(), spaceDTO.getId(), statisMonth);
                    clickNum = getClickNumByGroupQrcodeId(spaceDTO.getQrcodeId(), spaceDTO.getId(), statisMonth);
                    income = getIncomeByGroupQrcodeId(spaceDTO.getQrcodeId(), spaceDTO.getId(), statisMonth);
                } else {
                    exposureNum = getExposureNumByAdId(spaceDTO.getId(), spaceDTO.getBookId(), spaceDTO.getAdviserId(), spaceDTO.getChannelId(), statisMonth);
                    clickNum = getClickNumByAdId(spaceDTO.getId(), spaceDTO.getBookId(), spaceDTO.getAdviserId(), spaceDTO.getChannelId(), statisMonth);
                    income = getIncomeByAdId(spaceDTO.getId(), spaceDTO.getBookId(), spaceDTO.getChannelId(), statisMonth);
                }
            }
            BigDecimal clickRate = (null == exposureNum || exposureNum.equals(0L)) ? BigDecimal.ZERO :
                    new BigDecimal(clickNum.doubleValue() / exposureNum).setScale(2, BigDecimal.ROUND_HALF_UP);
            spaceDTO.setExposureNum(exposureNum);
            spaceDTO.setClickNum(clickNum);
            spaceDTO.setClickRate(clickRate);
            spaceDTO.setTotalIncome(income);
            // 填充广告位置名称
            if (AdPositionEnum.WECHAT_GROUP_MSG.positionCode.equals(spaceDTO.getAdPosition())) {
                fillAdpositionName4WechatGroupAd(spaceDTO);
            } else {
                fillAdpositionName4CommonAd(spaceDTO);
            }
            // 填充投放渠道信息
            if (null != spaceDTO.getBookId()) {
                if (AdPositionEnum.WECHAT_GROUP_MSG.positionCode.equals(spaceDTO.getAdPosition())) {
                    GroupQrcode groupQrcode = groupQrcodeDao.getById(spaceDTO.getQrcodeId());
                    BookDto bookDto = bookBiz.getBaseById(spaceDTO.getBookId());
                    spaceDTO.setGroupQrcodeName(null != groupQrcode ? groupQrcode.getGroupName() : "");
                    spaceDTO.setBookName(StringUtil.addBracket(bookDto.getBookName()));
                    spaceDTO.setIsbn(bookDto.getIsbn());
                } else {
                    BookDto bookDto = bookBiz.getBaseById(spaceDTO.getBookId());
                    spaceDTO.setBookName(StringUtil.addBracket(bookDto.getBookName()));
                    spaceDTO.setIsbn(bookDto.getIsbn());
                    Map<Long, String> agentNameMap = agentConsr.getNames(Arrays.asList(spaceDTO.getAgentId()));
                    if (!MapUtils.isEmpty(agentNameMap) && agentNameMap.containsKey(spaceDTO.getAgentId())) {
                        spaceDTO.setAgentName(agentNameMap.get(spaceDTO.getAgentId()));
                    } else {
                        spaceDTO.setAgentName("");
                    }
                }
            }
        }
        return pageBean;
    }

    /**
     * 导出广告主广告位明细
     */
    @Override
    public Map<String, Object> exportAdvertisingDetail4Master(Long masterId, String statisMonth) throws BizException {
        AdvertisingMaster master = advertisingMasterDao.getById(masterId);
        if (null == master) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "广告主不存在！");
        }
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("masterId", masterId);
        List<AdvertisingSpaceDTO> list = advertisingSpaceDao.advertisingDetail4Master(paramMap);
        if (ListUtils.isEmpty(list)) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "暂无数据导出！");
        }
        // 字段名
        String[] rowsName = {"序号", "广告位名称", "对方ID", "广告位置", "投放渠道", "投放时间", "曝光量", "点击量", "点击率", "累积收益"};
        List<Object[]> dataList = new ArrayList<>();
        Object[] objs;
        for (int i = 0; i < list.size(); i++) {
            AdvertisingSpaceDTO spaceDTO = list.get(i);
            objs = new Object[rowsName.length];
            objs[0] = i + 1;
            objs[1] = spaceDTO.getAdName();
            objs[2] = spaceDTO.getSourceId();
            // 填充广告位置名称
            String distributionChannel = "";
            if (AdPositionEnum.WECHAT_GROUP_MSG.positionCode.equals(spaceDTO.getAdPosition())) {
                fillAdpositionName4WechatGroupAd(spaceDTO);
                if (null != spaceDTO.getBookId()) {
                    GroupQrcode groupQrcode = groupQrcodeDao.getById(spaceDTO.getQrcodeId());
                    BookDto bookDto = bookBiz.getBaseById(spaceDTO.getBookId());
                    distributionChannel = (null != groupQrcode ? groupQrcode.getGroupName() : "") + "\r\n社群书" + StringUtil.addBracket(bookDto.getBookName()) + "\r\n书号：" + bookDto.getIsbn();
                }
            } else {
                fillAdpositionName4CommonAd(spaceDTO);
                if (null != spaceDTO.getBookId()) {
                    BookDto bookDto = bookBiz.getBaseById(spaceDTO.getBookId());
                    Map<Long, String> agentNameMap = agentConsr.getNames(Arrays.asList(spaceDTO.getAgentId()));
                    String agentName = "";
                    if (!MapUtils.isEmpty(agentNameMap) && agentNameMap.containsKey(spaceDTO.getAgentId())) {
                        agentName = agentNameMap.get(spaceDTO.getAgentId());
                    }
                    distributionChannel = StringUtil.addBracket(bookDto.getBookName()) + "\r\n出版社：" + agentName + "\r\n书号：" + bookDto.getIsbn();
                }
            }
            objs[3] = spaceDTO.getAdPositionName();
            objs[4] = distributionChannel;
            objs[5] = DateNewUtils.formatDate(spaceDTO.getDistributionTime());
            Long exposureNum = 0L;
            Long clickNum = 0L;
            BigDecimal income = BigDecimal.ZERO;
            if (null != spaceDTO.getBookId()) {
                if (AdPositionEnum.WECHAT_GROUP_MSG.positionCode.equals(spaceDTO.getAdPosition())) {
                    exposureNum = getExposureNumByGroupQrcodeId(spaceDTO.getQrcodeId(), spaceDTO.getId(), statisMonth);
                    clickNum = getClickNumByGroupQrcodeId(spaceDTO.getQrcodeId(), spaceDTO.getId(), statisMonth);
                    income = getIncomeByGroupQrcodeId(spaceDTO.getQrcodeId(), spaceDTO.getId(), statisMonth);
                } else {
                    exposureNum = getExposureNumByAdId(spaceDTO.getId(), spaceDTO.getBookId(), spaceDTO.getAdviserId(), spaceDTO.getChannelId(), statisMonth);
                    clickNum = getClickNumByAdId(spaceDTO.getId(), spaceDTO.getBookId(), spaceDTO.getAdviserId(), spaceDTO.getChannelId(), statisMonth);
                    income = getIncomeByAdId(spaceDTO.getId(), spaceDTO.getBookId(), spaceDTO.getChannelId(), statisMonth);
                }
            }
            objs[6] = exposureNum;
            objs[7] = clickNum;
            BigDecimal clickRate = (null == exposureNum || exposureNum.equals(0L)) ? BigDecimal.ZERO :
                    new BigDecimal(clickNum.doubleValue() / exposureNum).setScale(2, BigDecimal.ROUND_HALF_UP);
            objs[8] = clickRate.multiply(new BigDecimal(100)).setScale(0, BigDecimal.ROUND_HALF_UP) + "%";
            objs[9] = income + "元";
            dataList.add(objs);
        }
        String[] dateStr = statisMonth.split("-");
        String fileName = master.getMasterName() + dateStr[0] + "年" + Integer.parseInt(dateStr[1]) + "月广告位明细";
        String fileUrl = exportConsr.exportExcel(fileName, rowsName, dataList);
        Map<String, Object> result = new HashMap<>();
        result.put("fileUrl", fileUrl);
        result.put("fileName", fileName);
        return result;
    }

    /**
     * 客户端获取书刊配置的广告位信息
     */
    @Override
    @ParamLog("客户端获取书刊配置的广告位信息")
    public List<AdvertisingSpaceDTO> get4Wechat(Long sceneId, Long adviserId, Long channelId, String adPositionDetail) throws BizException {
        Long agentId = adviserConsr.getAgentIdByAdviser(adviserId);
        Long bookId = ResponseHandleUtil.parseResponse(qrcodeSceneService.getBookId4SceneId(sceneId), Long.class);
        if (null == agentId || null == bookId) {
            return new ArrayList<>();
        }
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("bookId", bookId);
        paramMap.put("adviserId", adviserId);
        paramMap.put("channelId", channelId);
        paramMap.put("agentId", agentId);
        Boolean bookIsOpen = (Boolean) advertisingAdviserPermissionDao.getBy(paramMap, "checkBookOpen");
        if (null == bookIsOpen || !bookIsOpen) {
            return new ArrayList<>();
        }
        paramMap.put("adPositionDetail", adPositionDetail);
        List<AdvertisingSpaceDTO> list = advertisingSpaceDao.getDTOByBookId(paramMap);
        if (ListUtils.isEmpty(list)) {
            return new ArrayList<>();
        }
        if (!ListUtils.isEmpty(list)) {
            list.forEach(spaceDTO -> spaceDTO.setBookId(bookId));
        }
        return list;
    }

    /**
     * 获取社群书配置的广告位信息
     */
    @Override
    @ParamLog("获取社群书配置的广告位信息")
    public AdvertisingSpaceDTO getBookGroupAd4Wechat(Long bookGroupId, String adPositionDetail) throws BizException {
        BookGroupDTO bookGroupDTO = bookGroupDao.getDTOById(bookGroupId);
        if (null == bookGroupDTO) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "该社群书不存在或已删除！");
        }
        Long agentId = adviserConsr.getAgentIdByAdviser(bookGroupDTO.getCreateUser());
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("bookId", bookGroupDTO.getBookId());
        paramMap.put("adviserId", bookGroupDTO.getCreateUser());
        paramMap.put("channelId", bookGroupDTO.getChannelId());
        paramMap.put("agentId", agentId);
        Boolean bookIsOpen = (Boolean) advertisingAdviserPermissionDao.getBy(paramMap, "checkBookOpen");
        if (null == bookIsOpen || !bookIsOpen) {
            return new AdvertisingSpaceDTO();
        }
        paramMap.put("adPositionDetail", adPositionDetail);
        AdvertisingSpaceDTO spaceDTO = (AdvertisingSpaceDTO) advertisingSpaceDao.getBy(paramMap, "getDTOByBookId4BookGroup");
        if (null == spaceDTO) {
            return new AdvertisingSpaceDTO();
        }
        spaceDTO.setBookId(bookGroupDTO.getBookId());
        return spaceDTO;
    }

    /**
     * 曝光量埋点
     */
    @Override
    @ParamLog(value = "曝光量埋点", isAfterReturn = false)
    public void addExposureTrack(AdvertisingExposureRecord advertisingExposureRecord) throws BizException {
        if (StringUtil.isEmpty(advertisingExposureRecord.getFromType()) || null == advertisingExposureRecord.getFromId()
                || advertisingExposureRecord.getFromId() <= 0L) {
            return;
        }
        AdvertisingSpace space = advertisingSpaceDao.getById(advertisingExposureRecord.getAdId());
        if (null == space) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "广告位不存在或已删除！");
        }
        Long agentId = channelConsr.getParentId(advertisingExposureRecord.getChannelId());
        advertisingExposureRecord.setAgentId(agentId);
        if (AdPositionEnum.BOOK_GROUP_PAGE.positionCode.equals(space.getAdPosition())) {
            advertisingExposureRecord.setIsBookGroup(true);
        } else {
            advertisingExposureRecord.setIsBookGroup(false);
        }
        advertisingExposureRecord.setCount(1L);
        advertisingExposureRecordDao.insert(advertisingExposureRecord);
    }

    /**
     * 点击量埋点
     */
    @Override
    @ParamLog(value = "点击量埋点", isAfterReturn = false)
    public void addClickTrack(AdvertisingClickRecord advertisingClickRecord) throws BizException {
        if (StringUtil.isEmpty(advertisingClickRecord.getFromType()) || null == advertisingClickRecord.getFromId()
                || advertisingClickRecord.getFromId() <= 0L) {
            return;
        }
        if ("WECHAT_GROUP".equals(advertisingClickRecord.getFromType())) {
            BookGroup bookGroup = bookGroupDao.getById(advertisingClickRecord.getBookGroupId());
            if (null == bookGroup) {
                return;
            }
            advertisingClickRecord.setBookId(bookGroup.getBookId());
            advertisingClickRecord.setChannelId(bookGroup.getChannelId());
            advertisingClickRecord.setAdviserId(bookGroup.getCreateUser());
        }
        AdvertisingSpace space = advertisingSpaceDao.getById(advertisingClickRecord.getAdId());
        if (null == space) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "广告位不存在或已删除！");
        }
        Long agentId = channelConsr.getParentId(advertisingClickRecord.getChannelId());
        advertisingClickRecord.setAgentId(agentId);
        if (!AdPositionEnum.APP_OR_PRODUCT.positionCode.equals(space.getAdPosition())) {
            advertisingClickRecord.setIsBookGroup(true);
        } else {
            advertisingClickRecord.setIsBookGroup(false);
        }
        advertisingClickRecordDao.insert(advertisingClickRecord);
    }

    /**
     * 添加广告主
     */
    @Override
    @ParamLog(value = "添加广告主", isAfterReturn = false)
    @Transactional(rollbackFor = Exception.class)
    public void createMaster(AdvertisingMaster advertisingMaster) throws BizException {
        if (StringUtil.isEmpty(advertisingMaster.getMasterName()) || ListUtils.isEmpty(advertisingMaster.getSettlementMethodList())) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数有误！");
        }
        boolean paramError = advertisingMaster.getSettlementMethodList().stream().anyMatch(method ->
                StringUtil.isEmpty(method.getSettlementMethod()) || !ArrayUtils.contains(SettlementMethodEnum.SETTLEMENT_METHOD_GATHER, method.getSettlementMethod()));
        if (paramError) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数有误！");
        }
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("masterName", advertisingMaster.getMasterName());
        Boolean nameExist = (Boolean) advertisingMasterDao.getBy(paramMap, "checkMasterNameExist");
        if (null != nameExist && nameExist) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "广告主已存在！");
        }
        advertisingMasterDao.insert(advertisingMaster);
        Long masterId = advertisingMaster.getId();
        advertisingMaster.getSettlementMethodList().forEach(method -> method.setMasterId(masterId));
        advertisingSettlementMethodDao.insert(advertisingMaster.getSettlementMethodList());
    }

    /**
     * 修改广告主
     */
    @Override
    @ParamLog(value = "修改广告主", isAfterReturn = false)
    @Transactional(rollbackFor = Exception.class)
    public void updateMaster(AdvertisingMaster advertisingMaster) throws BizException {
        if (null == advertisingMaster.getId() || StringUtil.isEmpty(advertisingMaster.getMasterName()) || ListUtils.isEmpty(advertisingMaster.getSettlementMethodList())) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数有误！");
        }
        boolean paramError = advertisingMaster.getSettlementMethodList().stream().anyMatch(method ->
                StringUtil.isEmpty(method.getSettlementMethod()) || !ArrayUtils.contains(SettlementMethodEnum.SETTLEMENT_METHOD_GATHER, method.getSettlementMethod()));
        if (paramError) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数有误！");
        }
        Long masterId = advertisingMaster.getId();
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("masterName", advertisingMaster.getMasterName());
        paramMap.put("masterId", masterId);
        Boolean nameExist = (Boolean) advertisingMasterDao.getBy(paramMap, "checkMasterNameExist");
        if (null != nameExist && nameExist) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "广告主名称不能重复！");
        }
        advertisingMasterDao.update(advertisingMaster);
        // 删除广告主之前配置的结算方式
        advertisingSettlementMethodDao.deleteByMasterId(masterId);
        advertisingMaster.getSettlementMethodList().forEach(method -> method.setMasterId(masterId));
        advertisingSettlementMethodDao.insert(advertisingMaster.getSettlementMethodList());
    }

    /**
     * 获取广告主列表
     */
    @Override
    public PageBean getMasterList(PageParam pageParam, String masterName) throws BizException {
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("masterName", masterName);
        PageBean pageBean = advertisingMasterDao.listPage(pageParam, paramMap, "getMasterList");
        if (pageBean == null || ListUtils.isEmpty(pageBean.getRecordList())) {
            return new PageBean(0, 0, new ArrayList<>());
        }
        for (Object object : pageBean.getRecordList()) {
            AdvertisingMasterDTO masterDTO = (AdvertisingMasterDTO) object;
            List<AdvertisingSettlementMethodDTO> methodDTOList = advertisingSettlementMethodDao.getByMasterId(masterDTO.getId());
            if (!ListUtils.isEmpty(methodDTOList)) {
                methodDTOList.forEach(dto -> dto.setSettlementMethodName(SettlementMethodEnum.getNameByCode(dto.getSettlementMethod())));
            }
            masterDTO.setSettlementMethodList(methodDTOList);
            Long exposureNum = getExposureNumByMasterId(masterDTO.getId());
            masterDTO.setExposureNum(exposureNum);
            Long clickNum = getClickNumByMasterId(masterDTO.getId());
            masterDTO.setClickNum(clickNum);
            BigDecimal clickRate = (null == masterDTO.getExposureNum() || masterDTO.getExposureNum().equals(0L)) ? BigDecimal.ZERO :
                    new BigDecimal(masterDTO.getClickNum().doubleValue() / masterDTO.getExposureNum()).setScale(2, BigDecimal.ROUND_HALF_UP);
            masterDTO.setClickRate(clickRate);
            BigDecimal income = getIncomeByMasterId(masterDTO.getId());
            masterDTO.setTotalIncome(income);
        }
        return pageBean;
    }

    /**
     * 获取所有广告主
     */
    @Override
    public List<AdvertisingMasterDTO> getAllMaster() throws BizException {
        List<AdvertisingMasterDTO> list = advertisingMasterDao.getAllMaster();
        if (!ListUtils.isEmpty(list)) {
            for (AdvertisingMasterDTO masterDTO : list) {
                List<AdvertisingSettlementMethodDTO> methodDTOList = advertisingSettlementMethodDao.getByMasterId(masterDTO.getId());
                if (!ListUtils.isEmpty(methodDTOList)) {
                    methodDTOList.forEach(dto -> dto.setSettlementMethodName(SettlementMethodEnum.getNameByCode(dto.getSettlementMethod())));
                }
                masterDTO.setSettlementMethodList(methodDTOList);
            }
        }
        return list;
    }

    /**
     * 获取广告主信息
     */
    @Override
    public AdvertisingMasterDTO getMasterInfo(Long masterId) throws BizException {
        return advertisingMasterDao.getDTOById(masterId);
    }

    /**
     * 每日凌晨计算昨日广告位收益
     */
    @Override
    @ParamLog(value = "每日凌晨计算昨日广告位收益", isAfterReturn = false)
    public void calculateAdvertisingIncomeYesterday() throws BizException {
        new Thread() {
            @Override
            public void run() {
                Map<String, Object> paramMap = new HashMap<>();
                paramMap.put("createDay", DateNewUtils.getShortDateStr(DateNewUtils.addDay(new Date(), -1)));
                // 插入扫社群码页面和作品/应用广告位的昨日收益数据
                advertisingIncomeDailyDao.insertRecordWithoutWechatGroupMsgAd(paramMap);
                // 插入微信群消息广告位的昨日收益数据
                advertisingIncomeDailyDao.insertWechatGroupMsgAdRecord(paramMap);
            }
        }.start();
    }

    /**
     * 获取概览数据
     */
    @Override
    @ParamLog("获取概览数据")
    public Map<String, Object> overviewStatis() throws BizException {
        Map<String, Object> result = new HashMap<>();
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("statis", "all");
        Long advertisingSpaceNum = (Long) advertisingSpaceDao.getBy(paramMap, "getSpaceNum");
        result.put("advertisingSpaceNum", advertisingSpaceNum);
        Long distributionBookNum = (Long) advertisingDistributionBookDao.getBy(paramMap, "getDistributionBookNum");
        result.put("distributionBookNum", distributionBookNum);
        Long distributionWechatGroupNum = (Long) advertisingDistributionBookDao.getBy(paramMap, "getDistributionWechatGroupNum");
        result.put("distributionWechatGroupNum", distributionWechatGroupNum);
        Long exposureNum = (Long) advertisingExposureRecordDao.getBy(paramMap, "getTotalExposureNum");
        result.put("exposureNum", exposureNum);
        Long clickNum = (Long) advertisingClickRecordDao.getBy(paramMap, "getTotalClickNum");
        result.put("clickNum", clickNum);
        BigDecimal clickRate = (null == exposureNum || exposureNum.equals(0L)) ? BigDecimal.ZERO :
                new BigDecimal(clickNum.doubleValue() / exposureNum).setScale(2, BigDecimal.ROUND_HALF_UP);
        result.put("clickRate", clickRate);
        BigDecimal totalIncome = (BigDecimal) advertisingIncomeDailyDao.getBy(paramMap, "getTotalIncome");
        result.put("totalIncome", null == totalIncome ? BigDecimal.ZERO : totalIncome.setScale(2, BigDecimal.ROUND_HALF_UP));
        return result;
    }

    /**
     * 获取流量趋势
     */
    @Override
    public List<Object> getFlowStatisTrend(Integer day, String startDate, String endDate) throws BizException {
        if (null == day && (StringUtil.isEmpty(startDate) || StringUtil.isEmpty(endDate))) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数有误！");
        }
        Map<String, Object> paramMap = new HashMap<>();
        if (!StringUtil.isEmpty(startDate) && !StringUtil.isEmpty(endDate)) {
            paramMap.put("collectDates", CommonUtils.collectLocalDates(startDate, endDate));
            paramMap.put("startDate", startDate);
            paramMap.put("endDate", endDate);
        } else {
            Date startTime = DateUtils.addDay(new Date(), -day);
            Date endTime = DateUtils.addDay(new Date(), -1);
            paramMap.put("collectDates", CommonUtils.collectLocalDates(startTime, endTime));
            paramMap.put("startDate", DateNewUtils.getShortDateStr(startTime));
            paramMap.put("endDate", DateNewUtils.getShortDateStr(endTime));
        }
        return advertisingExposureRecordDao.listBy(paramMap, "getFlowStatisTrend");
    }

    /**
     * 获取收益趋势
     */
    @Override
    public List<Object> getIncomeStatisTrend(Integer day, String startDate, String endDate) throws BizException {
        if (null == day && (StringUtil.isEmpty(startDate) || StringUtil.isEmpty(endDate))) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数有误！");
        }
        Map<String, Object> paramMap = new HashMap<>();
        if (!StringUtil.isEmpty(startDate) && !StringUtil.isEmpty(endDate)) {
            paramMap.put("collectDates", CommonUtils.collectLocalDates(startDate, endDate));
        } else {
            Date startTime = DateUtils.addDay(new Date(), -day);
            Date endTime = DateUtils.addDay(new Date(), -1);
            paramMap.put("collectDates", CommonUtils.collectLocalDates(startTime, endTime));
        }
        return advertisingIncomeDailyDao.listBy(paramMap, "getIncomeStatisTrend");
    }

    /**
     * 获取微信群信息
     */
    @Override
    public AdGroupQrcodeDTO getGroupQrcodeInfo(Long qrcodeId) throws BizException {
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("qrcodeId", qrcodeId);
        AdGroupQrcodeDTO dto = (AdGroupQrcodeDTO) groupQrcodeDao.getBy(paramMap, "getGroupQrcodeInfo4Ad");
        if (null == dto) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "微信群不存在或已删除！");
        }
        Long adviserId = dto.getAdviserId();
        String adviserName = adviserConsr.getNameById(adviserId);
        dto.setAdviserName(adviserName);
        Long agentId = adviserConsr.getAgentIdByAdviser(adviserId);
        if (null != agentId) {
            Map<Long, String> agentNameMap = agentConsr.getNames(Arrays.asList(agentId));
            if (!MapUtils.isEmpty(agentNameMap) && agentNameMap.containsKey(agentId)) {
                dto.setAgentName(agentNameMap.get(agentId));
            }
        }
        return dto;
    }
}
