package com.pcloud.book.advertising.facade;

import org.codehaus.jackson.JsonParseException;
import org.springframework.cloud.netflix.feign.FeignClient;
import org.springframework.web.bind.annotation.CookieValue;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import com.pcloud.book.advertising.entity.AdvertisingAdviserPermission;
import com.pcloud.book.advertising.entity.AdvertisingAgentPermission;
import com.pcloud.book.advertising.entity.AdvertisingClickRecord;
import com.pcloud.book.advertising.entity.AdvertisingDistributionBook;
import com.pcloud.book.advertising.entity.AdvertisingExposureRecord;
import com.pcloud.book.advertising.entity.AdvertisingSpace;
import com.pcloud.common.dto.ResponseDto;
import com.pcloud.common.exceptions.BizException;
import com.pcloud.common.permission.PermissionException;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;

/**
 * Description 广告位接口层接口
 * @author PENG
 * @date 2019/2/28
 */
@FeignClient(value = "pcloud-service-book", qualifier = "advertisingSpaceFacadeCloud", path = "book/v1.0/advertising")
@Api(description = "广告位接口层接口")
public interface AdvertisingSpaceFacade {

    /**
     * 创建广告位
     * @param token            用户身份信息
     * @param advertisingSpace 广告位实体
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "创建广告位", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "token", value = "用户身份信息", required = false, dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "advertisingSpace", value = "广告位实体", required = false, dataType = "AdvertisingSpace", paramType = "body")
    })
    @RequestMapping(value = "create", method = RequestMethod.POST)
    ResponseDto<?> create(@RequestHeader("token") String token, @RequestBody AdvertisingSpace advertisingSpace)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 修改广告位
     * @param token            用户身份信息
     * @param advertisingSpace 广告位实体
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "修改广告位", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "token", value = "用户身份信息", required = false, dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "advertisingSpace", value = "广告位实体", required = false, dataType = "AdvertisingSpace", paramType = "body")
    })
    @RequestMapping(value = "update", method = RequestMethod.POST)
    ResponseDto<?> update(@RequestHeader("token") String token, @RequestBody AdvertisingSpace advertisingSpace)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 获取广告位信息
     * @param token 用户身份信息
     * @param adId  广告位ID
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "修改广告位", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "token", value = "用户身份信息", required = false, dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "adId", value = "广告位ID", required = false, dataType = "Long", paramType = "query")
    })
    @RequestMapping(value = "get", method = RequestMethod.GET)
    ResponseDto<?> get(@RequestHeader("token") String token, @RequestParam(value = "adId", required = false) Long adId)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 停用广告位
     * @param token            用户身份信息
     * @param advertisingSpace 广告位实体
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "停用广告位", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "token", value = "用户身份信息", required = false, dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "advertisingSpace", value = "广告位实体", required = false, dataType = "AdvertisingSpace", paramType = "body")
    })
    @RequestMapping(value = "stop", method = RequestMethod.POST)
    ResponseDto<?> stop(@RequestHeader("token") String token, @RequestBody AdvertisingSpace advertisingSpace)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 删除广告位
     * @param token            用户身份信息
     * @param advertisingSpace 广告位实体
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "删除广告位", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "token", value = "用户身份信息", required = false, dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "advertisingSpace", value = "广告位实体", required = false, dataType = "AdvertisingSpace", paramType = "body")
    })
    @RequestMapping(value = "delete", method = RequestMethod.POST)
    ResponseDto<?> delete(@RequestHeader("token") String token, @RequestBody AdvertisingSpace advertisingSpace)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 投放书刊
     * @param token                       用户身份信息
     * @param advertisingDistributionBook 广告位书刊实体
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "投放书刊", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "token", value = "用户身份信息", required = false, dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "advertisingDistributionBook", value = "广告位书刊实体", required = false, dataType = "AdvertisingDistributionBook", paramType = "body")
    })
    @RequestMapping(value = "distributeBook", method = RequestMethod.POST)
    ResponseDto<?> distributeBook(@RequestHeader("token") String token, @RequestBody AdvertisingDistributionBook advertisingDistributionBook)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 获取出版社书刊权限
     * @param token 用户身份信息
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "获取出版社书刊权限", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "token", value = "用户身份信息", required = false, dataType = "string", paramType = "header")
    })
    @RequestMapping(value = "getAgentPermission", method = RequestMethod.GET)
    ResponseDto<?> getAgentPermission(@RequestHeader("token") String token)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 设置出版社书刊权限
     * @param token                      用户身份信息
     * @param advertisingAgentPermission 广告位出版权限实体
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "设置出版社书刊权限", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "token", value = "用户身份信息", required = false, dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "advertisingAgentPermission", value = "广告位出版权限实体", required = false, dataType = "AdvertisingAgentPermission", paramType = "body")
    })
    @RequestMapping(value = "setAgentPermission", method = RequestMethod.POST)
    ResponseDto<?> setAgentPermission(@RequestHeader("token") String token, @RequestBody AdvertisingAgentPermission advertisingAgentPermission)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 设置编辑书刊权限
     * @param token                        用户身份信息
     * @param advertisingAdviserPermission 广告位编辑权限实体
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "设置编辑书刊权限", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "token", value = "用户身份信息", required = false, dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "advertisingAdviserPermission", value = "广告位编辑权限实体", required = false, dataType = "AdvertisingAdviserPermission", paramType = "body")
    })
    @RequestMapping(value = "setAdviserPermission", method = RequestMethod.POST)
    ResponseDto<?> setAdviserPermission(@RequestHeader("token") String token, @RequestBody AdvertisingAdviserPermission advertisingAdviserPermission)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 编辑端获取书刊列表
     * @param token       用户身份信息
     * @param name        书籍名称
     * @param currentPage 当前页
     * @param numPerPage  每页条数
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "编辑端获取书刊列表", httpMethod = "GET")
    @ApiImplicitParams({@ApiImplicitParam(name = "token", value = "token", dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "name", value = "书籍名称", dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "currentPage", value = "当前页", dataType = "int", paramType = "query"),
            @ApiImplicitParam(name = "numPerPage", value = "每页条数", dataType = "int", paramType = "query")
    })
    @RequestMapping(value = "listBook4Adviser", method = RequestMethod.GET)
    ResponseDto<?> listBook4Adviser(@RequestHeader("token") String token, @RequestParam(value = "name", required = false) String name,
            @RequestParam(value = "currentPage", required = false) Integer currentPage, @RequestParam(value = "numPerPage", required = false) Integer numPerPage)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 平台端获取某编辑书刊列表
     * @param token       用户身份信息
     * @param adviserId   编辑ID
     * @param name        书籍名称
     * @param currentPage 当前页
     * @param numPerPage  每页条数
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "平台端获取某编辑书刊列表", httpMethod = "GET")
    @ApiImplicitParams({@ApiImplicitParam(name = "token", value = "token", dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "adviserId", value = "编辑ID", dataType = "Long", paramType = "query"),
            @ApiImplicitParam(name = "name", value = "书籍名称", dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "currentPage", value = "当前页", dataType = "int", paramType = "query"),
            @ApiImplicitParam(name = "numPerPage", value = "每页条数", dataType = "int", paramType = "query")
    })
    @RequestMapping(value = "listBook4Platform", method = RequestMethod.GET)
    ResponseDto<?> listBook4Platform(@RequestHeader("token") String token, @RequestParam(value = "adviserId", required = false) Long adviserId,
            @RequestParam(value = "name", required = false) String name, @RequestParam(value = "currentPage", required = false) Integer currentPage,
            @RequestParam(value = "numPerPage", required = false) Integer numPerPage) throws PermissionException, BizException, JsonParseException;

    /**
     * 获取广告位列表
     * @param token       用户身份信息
     * @param name        广告位名称
     * @param currentPage 当前页
     * @param numPerPage  每页条数
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "获取广告位列表", httpMethod = "GET")
    @ApiImplicitParams({@ApiImplicitParam(name = "token", value = "token", dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "name", value = "广告位名称", dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "currentPage", value = "当前页", dataType = "int", paramType = "query"),
            @ApiImplicitParam(name = "numPerPage", value = "每页条数", dataType = "int", paramType = "query")
    })
    @RequestMapping(value = "list", method = RequestMethod.GET)
    ResponseDto<?> list(@RequestHeader("token") String token, @RequestParam(value = "name", required = false) String name,
            @RequestParam(value = "currentPage", required = false) Integer currentPage, @RequestParam(value = "numPerPage", required = false) Integer numPerPage)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 获取广告位投放书刊列表
     * @param token       用户身份信息
     * @param adId        广告位ID
     * @param name        书刊名称
     * @param currentPage 当前页
     * @param numPerPage  每页条数
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "获取广告位投放书刊列表", httpMethod = "GET")
    @ApiImplicitParams({@ApiImplicitParam(name = "token", value = "token", dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "adId", value = "广告位ID", dataType = "Long", paramType = "query"),
            @ApiImplicitParam(name = "name", value = "书刊名称", dataType = "string", paramType = "query"),
            @ApiImplicitParam(name = "currentPage", value = "当前页", dataType = "int", paramType = "query"),
            @ApiImplicitParam(name = "numPerPage", value = "每页条数", dataType = "int", paramType = "query")
    })
    @RequestMapping(value = "distributionBookList", method = RequestMethod.GET)
    ResponseDto<?> distributionBookList(@RequestHeader("token") String token, @RequestParam(value = "adId", required = false) Long adId,
            @RequestParam(value = "name", required = false) String name, @RequestParam(value = "currentPage", required = false) Integer currentPage,
            @RequestParam(value = "numPerPage", required = false) Integer numPerPage) throws PermissionException, BizException, JsonParseException;

    /**
     * 获取广告位投放作品/应用数据统计
     * @param token       用户身份信息
     * @param adId        广告位ID
     * @param currentPage 当前页
     * @param numPerPage  每页条数
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "获取广告位投放作品/应用数据统计", httpMethod = "GET")
    @ApiImplicitParams({@ApiImplicitParam(name = "token", value = "token", dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "adId", value = "广告位ID", dataType = "Long", paramType = "query"),
            @ApiImplicitParam(name = "currentPage", value = "当前页", dataType = "int", paramType = "query"),
            @ApiImplicitParam(name = "numPerPage", value = "每页条数", dataType = "int", paramType = "query")
    })
    @RequestMapping(value = "distributionAppStatics", method = RequestMethod.GET)
    ResponseDto<?> distributionAppStatics(@RequestHeader("token") String token, @RequestParam(value = "adId", required = false) Long adId,
            @RequestParam(value = "currentPage", required = false) Integer currentPage, @RequestParam(value = "numPerPage", required = false) Integer numPerPage)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 获取广告位投放时间统计
     * @param token       用户身份信息
     * @param adId        广告位ID
     * @param day        天数
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "获取广告位投放时间统计", httpMethod = "GET")
    @ApiImplicitParams({@ApiImplicitParam(name = "token", value = "token", dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "adId", value = "广告位ID", dataType = "Long", paramType = "query"),
            @ApiImplicitParam(name = "day", value = "天数", dataType = "Long", paramType = "query")
    })
    @RequestMapping(value = "distributionDateStatics", method = RequestMethod.GET)
    ResponseDto<?> distributionDateStatics(@RequestHeader("token") String token, @RequestParam(value = "adId", required = false) Long adId,
            @RequestParam(value = "day", required = false) Integer day) throws PermissionException, BizException, JsonParseException;

    /**
     * 获取书刊配置的广告位信息
     * @param userInfo 用户身份信息
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "获取书刊配置的广告位信息", httpMethod = "GET")
    @ApiImplicitParams({@ApiImplicitParam(name = "userInfo", value = "用户身份信息", dataType = "string", paramType = "header")
    })
    @RequestMapping(value = "get4Wechat", method = RequestMethod.GET)
    ResponseDto<?> get4Wechat(@CookieValue("userInfo") String userInfo)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 曝光量埋点
     * @param userInfo                  用户身份信息
     * @param advertisingExposureRecord 广告位曝光实体
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "曝光量埋点", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userInfo", value = "用户身份信息", required = false, dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "advertisingExposureRecord", value = "广告位曝光实体", required = false, dataType = "AdvertisingExposureRecord", paramType = "body")
    })
    @RequestMapping(value = "addExposureTrack", method = RequestMethod.POST)
    ResponseDto<?> addExposureTrack(@CookieValue("userInfo") String userInfo, @RequestBody AdvertisingExposureRecord advertisingExposureRecord)
            throws PermissionException, BizException, JsonParseException;

    /**
     * 点击量埋点
     * @param userInfo               用户身份信息
     * @param advertisingClickRecord 广告位点击实体
     * @return
     * @throws PermissionException
     * @throws BizException
     * @throws JsonParseException
     */
    @ApiOperation(value = "点击量埋点", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userInfo", value = "用户身份信息", required = false, dataType = "string", paramType = "header"),
            @ApiImplicitParam(name = "advertisingClickRecord", value = "广告位点击实体", required = false, dataType = "AdvertisingClickRecord", paramType = "body")
    })
    @RequestMapping(value = "addClickTrack", method = RequestMethod.POST)
    ResponseDto<?> addClickTrack(@CookieValue("userInfo") String userInfo, @RequestBody AdvertisingClickRecord advertisingClickRecord)
            throws PermissionException, BizException, JsonParseException;
}
