package com.pcloud.book.copyright.biz.impl;

import com.pcloud.book.base.enums.BookStatusEnum;
import com.pcloud.book.base.exception.BookBizException;
import com.pcloud.book.consumer.channel.QrcodeSceneConsr;
import com.pcloud.book.consumer.reader.ReaderConsr;
import com.pcloud.book.copyright.biz.BookAuthCodeBiz;
import com.pcloud.book.copyright.biz.BookAuthInfoBiz;
import com.pcloud.book.copyright.biz.BookAuthServeBiz;
import com.pcloud.book.copyright.biz.BookAuthUserBiz;
import com.pcloud.book.copyright.dao.BookAuthCodeDao;
import com.pcloud.book.copyright.entity.BookAuthCode;
import com.pcloud.book.copyright.entity.BookAuthUser;
import com.pcloud.book.copyright.vo.BookAuthInfoVO;
import com.pcloud.book.copyright.vo.CheckCodeParam;
import com.pcloud.book.copyright.vo.CheckIsAuthServeParam;
import com.pcloud.common.core.aspect.ParamLog;
import com.pcloud.common.utils.string.StringUtil;
import com.pcloud.readercenter.wechat.entity.WechatUser;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * @author lily
 * @date 2018/12/4 19:11
 */
@Component("bookAuthCodeBiz")
public class BookAuthCodeBizImpl implements BookAuthCodeBiz {

    @Autowired
    private BookAuthCodeDao bookAuthCodeDao;

    @Autowired
    private BookAuthUserBiz bookAuthUserBiz;

    @Autowired
    private BookAuthInfoBiz bookAuthInfoBiz;

    @Autowired
    private BookAuthServeBiz bookAuthServeBiz;
    @Autowired
    private QrcodeSceneConsr qrcodeSceneConsr;
    @Autowired
    private ReaderConsr readerConsr;
    @Override
    @ParamLog(value = "批量插入授权码",isBefore = false)
    public void insert(List<BookAuthCode> bookAuthCodes) {
        int insertLength = bookAuthCodes.size();
        int i = 0;
        while (insertLength > 5000) {
            bookAuthCodeDao.insert(bookAuthCodes.subList(i, i + 5000));
            i = i + 5000;
            insertLength = insertLength - 5000;
        }
        if (insertLength > 0) {
            bookAuthCodeDao.insert(bookAuthCodes.subList(i, i + insertLength));
        }
    }

    @Override
    public Integer getMaxBatchNum(Long bookId, Long channelId, Long adviserId) {
        Integer batchNum = bookAuthCodeDao.getMaxBatchNum(bookId, channelId, adviserId);
        return batchNum;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Integer checkCode(CheckCodeParam checkCodeParam, Long channelId, Long wechatUserId) {
        String code = checkCodeParam.getCode();
        if (StringUtils.isEmpty(code) || code.length() != 15) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "此码错误，请重新验证");
        }
        BookAuthInfoVO authBookInfo = bookAuthInfoBiz.getAuthBookInfo(checkCodeParam.getBookId(), channelId, checkCodeParam.getAdviserId(), checkCodeParam.getSceneId());
        if (authBookInfo == null || BookStatusEnum.NO_SET.value.equals(authBookInfo.getBookStatus())) {
            return BookStatusEnum.CodeUseTypeEnum.RIGHT.value;
        }
        Long bookId = authBookInfo.getBookId();
        Long adviserId = authBookInfo.getAdviserId();
        //校验用户是否已经授权过
        Boolean isHaveAuth = bookAuthUserBiz.checkIsHaveAuth(bookId, channelId, adviserId, wechatUserId);
        if (isHaveAuth) {
            return BookStatusEnum.CodeUseTypeEnum.RIGHT.value;
        }
        Boolean isHaveCode = bookAuthCodeDao.getIsHaveCode(bookId, channelId, adviserId, code);
        Integer count = bookAuthCodeDao.updateUseCount(bookId, channelId, adviserId, code, authBookInfo.getCodeUseCount());
        if (count < 1 && isHaveCode) {
            return BookStatusEnum.CodeUseTypeEnum.HAVE_USE.value;
        } else if (count < 1 && !isHaveCode) {
            return BookStatusEnum.CodeUseTypeEnum.NOT_RIGHT.value;
        } else {
            //新增一条校验成功记录
            addUserRecord(bookId, channelId, adviserId, wechatUserId);
            return BookStatusEnum.CodeUseTypeEnum.RIGHT.value;
        }
    }

    @Override
    @ParamLog("校验服务是否需要验证")
    public Boolean checkServe(CheckIsAuthServeParam checkIsAuthServeParam, Long channelId, Long wechatUserId) {
        Long serveId = checkIsAuthServeParam.getServeId();
        String serveType = checkIsAuthServeParam.getServeType();
        if(serveId == null || serveType == null){
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "请选择要进入的服务");
        }
        //校验图书是否开启保护
        BookAuthInfoVO authBookInfo = bookAuthInfoBiz.getAuthBookInfo(checkIsAuthServeParam.getBookId(), channelId, checkIsAuthServeParam.getAdviserId(), checkIsAuthServeParam.getSceneId());
        if(authBookInfo == null || BookStatusEnum.NO_SET.value.equals(authBookInfo.getBookStatus())){
            return false;
        }
        Long bookId = authBookInfo.getBookId();
        Long adviserId = authBookInfo.getAdviserId();
        List<Long> serveIds = new ArrayList<>();
        serveIds.add(serveId);
        Map<String, Boolean> isHaveServe = qrcodeSceneConsr.listIsInBook(bookId, channelId, adviserId, serveIds);
        if(isHaveServe == null) return false;
        //校验服务是否需要授权
        Boolean isNeedAuth = bookAuthServeBiz.checkIsNeedAuth(bookId, channelId, adviserId, serveId, serveType);

        if(!isNeedAuth){
            return false;
        }
        //校验用户是否已经授权过
        Boolean isHaveAuth = bookAuthUserBiz.checkIsHaveAuth(bookId, channelId, adviserId, wechatUserId);
        if(isHaveAuth){
            return false;
        }else{
            return true;
        }
    }

    @ParamLog("新增一条授权成功记录")
    private void addUserRecord(Long bookId, Long channelId, Long adviserId, Long wechatUserId) {
        BookAuthUser bookAuthUser = new BookAuthUser();
        bookAuthUser.setBookId(bookId);
        bookAuthUser.setChannelId(channelId);
        bookAuthUser.setAdviserId(adviserId);
        bookAuthUser.setWechatUserId(wechatUserId);
        bookAuthUser.setAuthCode(BookStatusEnum.AuthCodeTypeEnum.BY_CODE.value);
        WechatUser wechatUser = readerConsr.getWechatUser(wechatUserId);
        bookAuthUser.setProvince(wechatUser == null || StringUtil.isEmpty(wechatUser.getWechatUserProvince()) ? "未知" : wechatUser.getWechatUserProvince());
        bookAuthUser.setCity(wechatUser == null || StringUtil.isEmpty(wechatUser.getWechatUserCity()) ? "未知" : wechatUser.getWechatUserCity());
        bookAuthUserBiz.insert(bookAuthUser);
    }

}
