package com.pcloud.book.skill.biz.impl;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import com.pcloud.book.base.exception.BookBizException;
import com.pcloud.book.consumer.label.LabelConsr;
import com.pcloud.book.group.biz.GroupQrcodeBiz;
import com.pcloud.book.group.vo.GroupQrcodeBaseInfoVO;
import com.pcloud.book.reading.biz.ReadingActivityBiz;
import com.pcloud.book.skill.biz.PcloudGroupActivityBiz;
import com.pcloud.book.skill.dao.PcloudGroupActivityDao;
import com.pcloud.book.skill.dao.PcloudSkillDao;
import com.pcloud.book.skill.entity.PcloudGroupActivity;
import com.pcloud.book.skill.entity.PcloudSkill;
import com.pcloud.book.skill.enums.SkillTypeEnum;
import com.pcloud.book.skill.facade.request.SaveGroupActivityRequestVO;
import com.pcloud.book.skill.facade.request.UpdateGroupActivityRequestVO;
import com.pcloud.book.skill.facade.response.QueryGroupActivityResponseVO;
import com.pcloud.common.core.aspect.ParamLog;
import com.pcloud.common.page.PageBeanNew;
import com.pcloud.common.page.PageParam;
import com.pcloud.common.utils.string.StringUtilParent;
import com.sdk.wxgroup.SendGroupInviteVO;
import com.sdk.wxgroup.SendPicMessageVO;
import com.sdk.wxgroup.WxGroupSDK;

import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import javax.annotation.Resource;

import lombok.extern.slf4j.Slf4j;

@Slf4j
@Service
public class PcloudGroupActivityBizImpl implements PcloudGroupActivityBiz {

    @Resource
    private PcloudSkillDao pcloudSkillDao;
    @Resource
    private PcloudGroupActivityDao pcloudGroupActivityDao;
    @Resource
    private GroupQrcodeBiz groupQrcodeBiz;
    @Resource
    private ReadingActivityBiz readingActivityBiz;
    @Resource
    private LabelConsr labelConsr;

    @ParamLog("保存共读活动")
    @Override
    public void saveGroupActivity(SaveGroupActivityRequestVO vo) {
        vo.checkParam();
        PcloudGroupActivity entity = vo.getEntity();
        Date now = new Date();
        entity.setCreateUser(vo.getUserId());
        entity.setCreateTime(now);
        entity.setUpdateUser(vo.getUserId());
        entity.setUpdateTime(now);
        entity.setIsDelete(0);
        pcloudGroupActivityDao.insert(entity);

    }

    @Override
    public void updateGroupActivity(UpdateGroupActivityRequestVO vo) {
        vo.checkParam();
        PcloudGroupActivity activity = pcloudGroupActivityDao.getById(vo.getId());
        if (null == activity) {
            throw new BookBizException(BookBizException.ID_NOT_EXIST, "共读活动不存在");
        }
        PcloudGroupActivity entity = vo.getEntity();
        Date now = new Date();
        entity.setUpdateUser(vo.getUserId());
        entity.setUpdateTime(now);
        pcloudGroupActivityDao.updateByPrimaryKeySelective(entity);
    }

    @Override
    public void deleteGroupActivity(Long id, Long partyId) {
        if (Objects.isNull(id)) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "该共读活动不存在！");
        }
        PcloudSkill byRelId = pcloudSkillDao.getByRelId(id.intValue(), SkillTypeEnum.READING.value);
        if (Objects.nonNull(byRelId)) {
            throw new BookBizException(BookBizException.PARAM_IS_EXIST, "该共读活动已经关联技能，无法删除！");
        }
        pcloudGroupActivityDao.deleteById(id);
    }

    @Override
    @ParamLog("[processGroup]")
    public void processGroup(String wxId, String robotWxId, Long id) {
        PcloudGroupActivity byId = pcloudGroupActivityDao.getById(id);
        if (Objects.isNull(byId)) {
            log.error("[processGroup] error 共读不存在");
            return;
        }
        if (0 == byId.getJoinType()) {
            GroupQrcodeBaseInfoVO groupQrcode = groupQrcodeBiz.getBaseById(byId.getBookGroupQrcodeId());
            if (Objects.isNull(groupQrcode)) {
                log.error("[processGroup] error 共读对应社群不存在");
                return;
            }
            SendGroupInviteVO vo = new SendGroupInviteVO();
            vo.setWxId(wxId);
            vo.setAltId(robotWxId);
            vo.setIp(readingActivityBiz.getIP(robotWxId));
            vo.setWxGroupId(groupQrcode.getWeixinGroupId());
            WxGroupSDK.sendGroupInvite(vo);
            log.info("[processGroup] sendGroupInvite:{}", vo);
        }
        if (1 == byId.getJoinType()) {
            SendPicMessageVO vo = new SendPicMessageVO();
            vo.setPicUrl(byId.getGroupPic());
            vo.setWxGroupId(wxId);
            vo.setAltId(robotWxId);
            vo.setIp(readingActivityBiz.getIP(robotWxId));
            WxGroupSDK.sendPicMessage(vo);
        }
    }

    @Override
    public PageBeanNew<QueryGroupActivityResponseVO> queryGroupActivity(String name, Long proLabelId, Long purLabelId, Long depLabelId, Integer numPerPage, Integer currentPage) {
        PageParam pageParam = new PageParam(currentPage, numPerPage);
        Map<String, Object> paraMap = Maps.newHashMap();
        paraMap.put("name", name);
        paraMap.put("proLabelId", proLabelId);
        paraMap.put("purLabelId", purLabelId);
        paraMap.put("depLabelId", depLabelId);
        PageBeanNew<QueryGroupActivityResponseVO> queryGroupActivity = pcloudGroupActivityDao.listPageNew(pageParam, paraMap, "queryGroupActivity");
        if (null == queryGroupActivity || CollectionUtils.isEmpty(queryGroupActivity.getRecordList())) {
            return queryGroupActivity;
        }
        // 组装标签名称
        List<Long> labelId = Lists.newArrayList();
        for (QueryGroupActivityResponseVO responseVO : queryGroupActivity.getRecordList()) {
            labelId.add(responseVO.getProLabelId());
            labelId.add(responseVO.getPurLabelId());
            labelId.add(responseVO.getDepLabelId());
        }
        Map<Long, String> labelName = labelConsr.getLabelName(labelId);
        for (QueryGroupActivityResponseVO responseVO : queryGroupActivity.getRecordList()) {
            responseVO.setPurLabelName(labelName.get(responseVO.getPurLabelId()));
            responseVO.setProLabelName(labelName.get(responseVO.getProLabelId()));
            responseVO.setDepLabelName(labelName.get(responseVO.getDepLabelId()));
        }

        // 截取字符串
        for(QueryGroupActivityResponseVO vo : queryGroupActivity.getRecordList()){
            String filterStr = StringUtilParent.replaceHtml(vo.getDesc());
            String subStr = filterStr.length() > 60 ? filterStr.substring(0, 60) + "..." : filterStr;
            vo.setCutDesc(subStr);
        }

        return queryGroupActivity;
    }
}
