package com.pcloud.book.group.biz.impl;

import com.google.common.collect.Maps;

import com.pcloud.book.advertising.biz.AdvertisingSpaceBiz;
import com.pcloud.book.advertising.dto.QrcodeAdvertisingSpaceCountDTO;
import com.pcloud.book.base.exception.BookBizException;
import com.pcloud.book.consumer.label.LabelConsr;
import com.pcloud.book.consumer.reader.ReaderConsr;
import com.pcloud.book.consumer.user.AdviserConsr;
import com.pcloud.book.consumer.wechatgroup.WechatGroupConsr;
import com.pcloud.book.group.biz.BookGroupClassifyBiz;
import com.pcloud.book.group.biz.GroupAnnouncementBiz;
import com.pcloud.book.group.biz.GroupQrcodeBiz;
import com.pcloud.book.group.biz.WeixinQrcodeBiz;
import com.pcloud.book.group.dao.GroupQrcodeDao;
import com.pcloud.book.group.dto.ChangeGroupNameDTO;
import com.pcloud.book.group.dto.GroupAndUserNumberDTO;
import com.pcloud.book.group.dto.GroupNameAndMaxSeqDTO;
import com.pcloud.book.group.dto.GroupQrcodeDTO;
import com.pcloud.book.group.dto.GroupQrcodeInfo4Advertising;
import com.pcloud.book.group.dto.GroupQrcodeInfoDTO;
import com.pcloud.book.group.dto.PushAddUserMessageDTO;
import com.pcloud.book.group.dto.WeixinQrcodeDTO;
import com.pcloud.book.group.entity.GroupQrcode;
import com.pcloud.book.group.enums.QrcodeStatusEnum;
import com.pcloud.book.group.tools.SendWeixinRequestTools;
import com.pcloud.book.group.vo.ClassifyQrcodeVO;
import com.pcloud.book.group.vo.ClassifyVO;
import com.pcloud.book.group.vo.GroupQrcodeBaseInfoVO;
import com.pcloud.book.group.vo.ListGroupQrcodeResponseVO;
import com.pcloud.book.group.vo.ListQrcodeByClassifyParamVO;
import com.pcloud.book.group.vo.UpdateGroupQrcodeRequestVO;
import com.pcloud.book.keywords.dao.BookKeywordRecordDao;
import com.pcloud.book.keywords.dto.KeywordUserCountDTO;
import com.pcloud.book.mq.producer.BookMQProducer;
import com.pcloud.book.push.dao.PushGroupDao;
import com.pcloud.book.push.entity.PushGroup;
import com.pcloud.common.core.aspect.ParamLog;
import com.pcloud.common.page.PageBeanNew;
import com.pcloud.common.page.PageParam;
import com.pcloud.common.utils.ListUtils;
import com.pcloud.common.utils.cache.redis.JedisClusterUtils;
import com.pcloud.common.utils.string.StringUtil;
import com.pcloud.usercenter.party.adviser.dto.AdviserBaseInfoDto;
import com.sdk.wxgroup.WxGroupSDK;

import org.apache.commons.collections4.MapUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.stream.Collectors;

import lombok.extern.slf4j.Slf4j;

/**
 * @author lily
 * @date 2019/4/18 15:31
 */
@Slf4j
@Component("groupQrcodeBiz")
public class GroupQrcodeBizImpl implements GroupQrcodeBiz {
    private static final Logger LOGGER = LoggerFactory.getLogger(GroupQrcodeBizImpl.class);
    @Autowired
    private GroupQrcodeDao groupQrcodeDao;
    @Autowired
    private BookGroupClassifyBiz bookGroupClassifyBiz;
    @Autowired
    private WeixinQrcodeBiz weixinQrcodeBiz;
    @Autowired
    private WechatGroupConsr wechatGroupConsr;
    @Autowired
    private BookMQProducer bookMQProducer;
    @Autowired
    private BookKeywordRecordDao bookKeywordRecordDao;
    @Autowired
    private ReaderConsr readerConsr;
    @Autowired
    private GroupAnnouncementBiz groupAnnouncementBiz;
    @Autowired
    private PushGroupDao pushGroupDao;
    @Autowired
    private AdviserConsr adviserConsr;
    @Autowired
    private AdvertisingSpaceBiz advertisingSpaceBiz;
    @Autowired
    private LabelConsr labelConsr;

    private static final ExecutorService EXECUTOR_SERVICE = Executors.newFixedThreadPool(5);

    @Override
    @ParamLog("updateGroupQrcode")
    public void updateGroupQrcode(UpdateGroupQrcodeRequestVO vo) {
        this.groupQrcodeDao.updateGroupQrcode(vo.getGroupQrcodeId(), vo.getQrcodeUrl(), vo.getUserId());
    }

    @Override
    @ParamLog("listAllGroupQrcode")
    public PageBeanNew<ListGroupQrcodeResponseVO> listAllGroupQrcode(String wxGroupName, Integer currentPage, Integer numPerPage) {
        PageParam pageParam = new PageParam(currentPage, numPerPage);
        final Map<String, Object> map = Maps.newHashMap();
        map.put("groupName", StringUtil.isEmpty(wxGroupName) ? null : wxGroupName);
        final PageBeanNew<ListGroupQrcodeResponseVO> groupQrcode = this.groupQrcodeDao.listPageNew(pageParam, map, "listAllGroupQrcode");
        if (null == groupQrcode || CollectionUtils.isEmpty(groupQrcode.getRecordList())) {
            return groupQrcode;
        }
        final List<Long> adviserIds = groupQrcode.getRecordList().stream().map(ListGroupQrcodeResponseVO::getAdviserId).distinct().collect(Collectors.toList());
        final Map<Long, String> names = adviserConsr.getNames(adviserIds);
        for (ListGroupQrcodeResponseVO vo : groupQrcode.getRecordList()) {
            vo.setAdviserName(names.get(vo.getAdviserId()));
        }
        return groupQrcode;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    @ParamLog("新增群二维码")
    public Long insert(GroupQrcode groupQrcode) {
        groupQrcodeDao.insert(groupQrcode);
        return groupQrcode.getId();
    }

    @Override
    @ParamLog("根据分类删除二维码")
    public void deleteByClassifyId(Long classifyId) {
        groupQrcodeDao.deleteByClassifyId(classifyId);
    }

    @Override
    @ParamLog("获取群用户")
    public Map<Long, GroupAndUserNumberDTO> listGroupAndUserNumber(List<Long> classifyIds) {
        return groupQrcodeDao.listGroupAndUserNumber(classifyIds);
    }

    @Override
    @ParamLog("获取分类下的二维码")
    public PageBeanNew<ClassifyQrcodeVO> listQrcodeByClassify(ListQrcodeByClassifyParamVO listQrcodeByClassifyParamVO) {
        PageParam pageParam = new PageParam(listQrcodeByClassifyParamVO.getCurrentPage(), listQrcodeByClassifyParamVO.getNumPerPage());
        //获取分类基本信息
        ClassifyVO classify = bookGroupClassifyBiz.getClassify(listQrcodeByClassifyParamVO.getClassifyId());
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("classifyId", listQrcodeByClassifyParamVO.getClassifyId());
        paramMap.put("changeNumber", classify.getChangeNumber());
        //获取二维码列表
        PageBeanNew<ClassifyQrcodeVO> pageBeanNew = groupQrcodeDao.listPageNew(pageParam, paramMap, "listQrcodeByClassify");
        if (pageBeanNew == null || pageBeanNew.getRecordList() == null) {
            return new PageBeanNew<>(listQrcodeByClassifyParamVO.getCurrentPage(), listQrcodeByClassifyParamVO.getNumPerPage(), new ArrayList<>());
        }
        //设置退群率
        setQuitRate(pageBeanNew.getRecordList());
        //设置使用关键词次数
        setKeywordCount(pageBeanNew.getRecordList());
        return pageBeanNew;
    }

    @ParamLog(value = "设置关键词次数", isBefore = false)
    private void setKeywordCount(List<ClassifyQrcodeVO> classifyQrcodeVOS) {
        if (ListUtils.isEmpty(classifyQrcodeVOS)) {
            return;
        }
        List<Long> qrcodeIds = new ArrayList<>();
        for (ClassifyQrcodeVO classifyQrcodeVO : classifyQrcodeVOS) {
            qrcodeIds.add(classifyQrcodeVO.getWeixinQrcodeId());
        }
        if (ListUtils.isEmpty(qrcodeIds)) return;
        Map<Long, KeywordUserCountDTO> keywordMap = bookKeywordRecordDao.listKeywordUseCount(qrcodeIds);
        for (ClassifyQrcodeVO classifyQrcodeVO : classifyQrcodeVOS) {
            if (keywordMap == null || keywordMap.get(classifyQrcodeVO.getWeixinQrcodeId()) == null) {
                classifyQrcodeVO.setKeywordCount(0);
            } else {
                KeywordUserCountDTO keywordUserCountDTO = keywordMap.get(classifyQrcodeVO.getWeixinQrcodeId());
                classifyQrcodeVO.setKeywordCount(keywordUserCountDTO == null ? 0 : keywordUserCountDTO.getUseCount());
            }
          
        }
    }

    @ParamLog(value = "设置退群率", isBefore = false)
    private void setQuitRate(List<ClassifyQrcodeVO> classifyQrcodeVOS) {
        if (ListUtils.isEmpty(classifyQrcodeVOS)) {
            return;
        }
        List<String> weixinGroupIds = new ArrayList<>();
        for (ClassifyQrcodeVO classifyQrcodeVO : classifyQrcodeVOS) {
            weixinGroupIds.add(classifyQrcodeVO.getWeixinGroupId());
        }
        //获取退群率
        Map<String, BigDecimal> rateMap = wechatGroupConsr.mapOutGroupRate(weixinGroupIds);
        for (ClassifyQrcodeVO classifyQrcodeVO : classifyQrcodeVOS) {
            classifyQrcodeVO.setQuitRate(rateMap == null || rateMap.get(classifyQrcodeVO.getWeixinGroupId()) == null ? new BigDecimal(0) : rateMap.get(classifyQrcodeVO.getWeixinGroupId()));
        }
    }

    @Override
    @ParamLog("修改群名称")
    public void modifyWechatGroupName(String wechatGroupName, Long id, Long updateUser) {
        if (wechatGroupName == null) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "微信群名称不能为空");
        }
        ChangeGroupNameDTO changeGroupNameDTO = groupQrcodeDao.getUpdateGroupNameParam(id);
        groupQrcodeDao.modifyNameById(wechatGroupName, id, updateUser);
        if (changeGroupNameDTO != null) {
            //获取修改群名称小号
            String robotId = wechatGroupConsr.getRobotIdByGroupId(changeGroupNameDTO.getWxGroupId());
            SendWeixinRequestTools.changeGroupName(robotId, changeGroupNameDTO.getWxGroupId(), wechatGroupName);
        }
    }

    @Override
    @ParamLog("新增一个用户")
    public void addOneUser(String weixinGroupId, Integer memberCount, String nickName, String robotId) {
        // 收发一体
        String robotIdByGroupId = robotId;
        if(StringUtil.isBlank(robotId)) {
            log.info("[新增一个用户] : robotId is null weixinGroupId:{},memberCount:{},nickName:{},robotId:{}", weixinGroupId, memberCount, nickName, robotId);
            robotIdByGroupId = wechatGroupConsr.getRobotIdByGroupId(weixinGroupId);
        }
        if(robotIdByGroupId != null){
            Integer peopleCounts = WxGroupSDK.getPeopleCounts(weixinGroupId, robotIdByGroupId);
            if (peopleCounts != null) {
                memberCount = peopleCounts;
            }
        }
        //获取群基本信息
        GroupQrcodeDTO groupQrcodeDTO = groupQrcodeDao.getGroupQrcodeInfo(weixinGroupId);
        if (groupQrcodeDTO == null) {
            return;
        }
        //更新用户数
        groupQrcodeDao.updateUserNumber(weixinGroupId, memberCount);
        //如果人数超过限制，重新分配群
        if (memberCount >= groupQrcodeDTO.getChangeNumber() && QrcodeStatusEnum.ON_USE.value.equals(groupQrcodeDTO.getQrcodeState())) {
            //修改状态，重新分配一个群
            addWechatGroup(groupQrcodeDTO.getClassifyId(), groupQrcodeDTO.getId());
        }
        if (memberCount >= 100 && QrcodeStatusEnum.ON_USE.value.equals(groupQrcodeDTO.getUseState())) {
            //将二维码修改为已满群状态
            changeToOverNumber(groupQrcodeDTO.getWeixinQrcodeId(), groupQrcodeDTO.getId());
        }

        Map<String, Integer> nickNames = JedisClusterUtils.getJson("BOOK:WEIXINGROUP:GUIDEUSER" + weixinGroupId, Map.class);
        if (nickNames == null) {
            nickNames = new HashMap<>();
            nickNames.put(nickName, 0);
        } else {
            nickNames.put(nickName, 0);
        }
        LOGGER.info("新用户"+nickNames);
        JedisClusterUtils.setJson("BOOK:WEIXINGROUP:GUIDEUSER" + weixinGroupId, nickNames, 600);

        String value = JedisClusterUtils.getJson("BOOK:WEIXINGROUP:GUIDE" + weixinGroupId, String.class);

        if (StringUtil.isEmpty(value)) {
            JedisClusterUtils.setJson("BOOK:WEIXINGROUP:GUIDE" + weixinGroupId, weixinGroupId, 5);
        } else {
            return;
        }
        PushAddUserMessageDTO pushAddUserMessageDTO = new PushAddUserMessageDTO();
        pushAddUserMessageDTO.setWeixinGroupId(weixinGroupId);
        pushAddUserMessageDTO.setTime(System.currentTimeMillis());
        pushAddUserMessageDTO.setNickName(nickName);
        pushAddUserMessageDTO.setMemberCount(memberCount);
        pushAddUserMessageDTO.setRobotId(robotIdByGroupId);
        bookMQProducer.pushAddUserMessageTopic(pushAddUserMessageDTO);
    }


    @Override
    @ParamLog("获取群二维码相关信息")
    public Map<Long, GroupQrcodeInfoDTO> listQrcodeInfoByIds(List<Long> groupQrcodeIds) {
        return groupQrcodeDao.listQrcodeInfoByIds(groupQrcodeIds);
    }

    @Override
    public GroupQrcodeBaseInfoVO getBaseById(Long groupQrcodeId) {
        return groupQrcodeDao.getBaseById(groupQrcodeId);
    }

    @Override
    public void updateUserNumber(String weixinGroupId) {
        String robotIdByGroupId = wechatGroupConsr.getRobotIdByGroupId(weixinGroupId);
        if(robotIdByGroupId != null){
            Integer peopleCounts = WxGroupSDK.getPeopleCounts(weixinGroupId, robotIdByGroupId);
            if (peopleCounts != null) {
                //更新用户数
                groupQrcodeDao.updateUserNumber(weixinGroupId, peopleCounts);
            }
        }
    }

    @Override
    public Map<String, GroupQrcodeInfo4Advertising> listGroupQrcodeInfo4Advertising(List<String> wxGroupIds) {
        Map<String, GroupQrcodeInfo4Advertising> resultMap = new HashMap<>();
        List<GroupQrcodeInfo4Advertising> advertisingList = groupQrcodeDao.getGroupInfo4AdvertisingByWxGroupIds(wxGroupIds);
        if (ListUtils.isEmpty(advertisingList)) {
            return new HashMap<>();
        }
        List<Long> adviserIds = advertisingList.stream().filter(s -> s.getAdviserId() != null).map(GroupQrcodeInfo4Advertising::getAdviserId).distinct().collect(Collectors.toList());
        List<Long> qrcodeIds = advertisingList.stream().filter(s -> s.getGroupQrcodeId() != null).map(GroupQrcodeInfo4Advertising::getGroupQrcodeId).distinct().collect(Collectors.toList());
        List<Long> labelIds = new ArrayList<>();
        List<Long> proLabels = advertisingList.stream().filter(s -> s.getProLabelId() != null).map(GroupQrcodeInfo4Advertising::getProLabelId).distinct().collect(Collectors.toList());
        List<Long> depLabels = advertisingList.stream().filter(s -> s.getDepLabelId() != null).map(GroupQrcodeInfo4Advertising::getDepLabelId).distinct().collect(Collectors.toList());
        List<Long> purLabels = advertisingList.stream().filter(s -> s.getPurLabelId() != null).map(GroupQrcodeInfo4Advertising::getPurLabelId).distinct().collect(Collectors.toList());
        if (!ListUtils.isEmpty(proLabels)) {
            labelIds.addAll(proLabels);
        }
        if (!ListUtils.isEmpty(depLabels)) {
            labelIds.addAll(depLabels);
        }
        if (!ListUtils.isEmpty(purLabels)) {
            labelIds.addAll(purLabels);
        }
        //关闭广告位的出版社
        List<Long> closeAgentIds = advertisingSpaceBiz.getCloseAgentId();
        //出版社信息
        Map<Long, AdviserBaseInfoDto> baseInfoDtoMap = new HashMap<>();
        Map<Long, String> adviserNameMap = new HashMap<>();
        //获取广告位数量
        Map<Long, QrcodeAdvertisingSpaceCountDTO> advertisingSpaceNumMap = new HashMap<>();
        //标签
        Map<Long, String> labelMap = new HashMap<>();
        if (!ListUtils.isEmpty(adviserIds)) {
            baseInfoDtoMap = adviserConsr.getAdviserId2AdviserInfoDtoMap(adviserIds);
            adviserNameMap = adviserConsr.getNames(adviserIds);
        }
        if (!ListUtils.isEmpty(qrcodeIds)) {
            advertisingSpaceNumMap = advertisingSpaceBiz.mapAdvertisingSpaceNum(qrcodeIds);
        }

        if (!ListUtils.isEmpty(labelIds)) {
            labelMap = labelConsr.getLabelName(labelIds);
        }
        for (GroupQrcodeInfo4Advertising advertising : advertisingList) {
            if (!MapUtils.isEmpty(baseInfoDtoMap) && null != advertising.getAdviserId()) {
                AdviserBaseInfoDto adviserBaseInfoDto = baseInfoDtoMap.get(advertising.getAdviserId());
                if (null != adviserBaseInfoDto) {
                    advertising.setAgentId(adviserBaseInfoDto.getAgentId());
                    advertising.setAgentName(adviserBaseInfoDto.getAgentName());
                }
            }
            if (!MapUtils.isEmpty(adviserNameMap) && null != advertising.getAdviserId()) {
                advertising.setAdviserName(adviserNameMap.get(advertising.getAdviserId()));
            }
            if (!ListUtils.isEmpty(closeAgentIds) && null != advertising.getAgentId()) {
                if (closeAgentIds.contains(advertising.getAgentId())) {
                    advertising.setOpen(false);
                }
            }
            if (!MapUtils.isEmpty(advertisingSpaceNumMap) && null != advertising.getGroupQrcodeId()) {
                QrcodeAdvertisingSpaceCountDTO dto = advertisingSpaceNumMap.get(advertising.getGroupQrcodeId());
                if (null != dto) {
                    advertising.setAdvertisingSpaceNum(dto.getAdvertisingSpaceNum());
                }
            }
            if (!MapUtils.isEmpty(labelMap)) {
                if (null != advertising.getProLabelId() && labelMap.containsKey(advertising.getProLabelId())) {
                    advertising.setProLabelName(labelMap.get(advertising.getProLabelId()));
                }
                if (null != advertising.getDepLabelId() && labelMap.containsKey(advertising.getDepLabelId())) {
                    advertising.setDepLabelName(labelMap.get(advertising.getDepLabelId()));
                }
                if (null != advertising.getPurLabelId() && labelMap.containsKey(advertising.getPurLabelId())) {
                    advertising.setPurLabelName(labelMap.get(advertising.getPurLabelId()));
                }
            }
            resultMap.put(advertising.getWxGroupId(), advertising);
        }
        return resultMap;
    }

    /**
     * 重新分配一个群
     */
    @Transactional(rollbackFor = Exception.class)
    public void addWechatGroup(Long classifyId, Long groupQrcodeId) {
        WeixinQrcodeDTO oneQrcode = weixinQrcodeBiz.getOneQrcode();
        if (oneQrcode != null) {
            //将二维码改为已满群状态
            Integer number = groupQrcodeDao.changeQrcodeState(groupQrcodeId, QrcodeStatusEnum.OVER_NUMBER.value);
            if (number > 0){
                //获取标签对应基本信息
                GroupNameAndMaxSeqDTO groupNameAndMaxSeqDTO = bookGroupClassifyBiz.getGroupNameAndMaxSeq(classifyId);
                if (groupNameAndMaxSeqDTO != null) {
                    Integer maxSeq = groupNameAndMaxSeqDTO.getMaxSeq() + 1;
                    String groupName = groupNameAndMaxSeqDTO.getGroupQrcodeName() + groupNameAndMaxSeqDTO.getClassify() + maxSeq + "群";
                    GroupQrcode groupQrcode = new GroupQrcode();
                    groupQrcode.setWeixinGroupId(oneQrcode.getWeixinGroupId());
                    groupQrcode.setQrcodeUrl(oneQrcode.getQrcodeUrl());
                    groupQrcode.setCreateUser(groupNameAndMaxSeqDTO.getCreateUser());
                    groupQrcode.setWeixinQrcodeId(oneQrcode.getId());
                    groupQrcode.setGroupSeq(maxSeq);
                    groupQrcode.setQrcodeHeadUrl(readerConsr.getNineHeadUrl());
                    groupQrcode.setUserNumber(oneQrcode.getUserNumber());
                    groupQrcode.setGroupName(groupName);
                    groupQrcode.setClassifyId(classifyId);
                    this.insert(groupQrcode);
                    EXECUTOR_SERVICE.execute(()->{
                        //获取该社群书下是否有群公告
                        ClassifyVO classifyVO = bookGroupClassifyBiz.getClassify(classifyId);
                        groupAnnouncementBiz.setAnForGroup(classifyVO.getBookGroupId(), classifyId, oneQrcode.getWeixinGroupId());
                        //补充该分类下的群发关联
                        List<PushGroup> pushGroups = pushGroupDao.getListByClassifyId(classifyId);
                        if (!ListUtils.isEmpty(pushGroups)){
                            PushGroup pushGroup = pushGroups.get(0);
                            pushGroup.setBookGroupQrcodeId(groupQrcode.getId());
                            pushGroupDao.insert(pushGroup);
                        }
                    });
                    //String robotId = wechatGroupConsr.getRobotIdByGroupId(oneQrcode.getWeixinGroupId());
                    SendWeixinRequestTools.changeGroupName(oneQrcode.getRobotWxId(), oneQrcode.getWeixinGroupId(), groupName);
                }
            }
        }
    }

    @ParamLog("将二维码改为已满群")
    @Transactional(rollbackFor = Exception.class)
    public void changeToOverNumber(Long weixinQrcodeId, Long id) {
        groupQrcodeDao.updateUseState(id, QrcodeStatusEnum.OVER_NUMBER.value);
        weixinQrcodeBiz.updateUseState(weixinQrcodeId, QrcodeStatusEnum.OVER_NUMBER.value);
    }

    @Override
    public List<Long> listQrcodeIdsByLabelQuery(String query, Long proLabelId, Long depLabelId, Long purLabelId) {
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("query", StringUtil.isEmpty(query) ? null : query);
        paramMap.put("proLabelId",proLabelId);
        paramMap.put("depLabelId",depLabelId);
        paramMap.put("purLabelId",purLabelId);
        List<Long> list = groupQrcodeDao.listQrcodeIdsByLabelQuery(paramMap);
        if (ListUtils.isEmpty(list)){
            list = new ArrayList<>();
        }
        return list;
    }

    @Override
    public List<Long> filterDeleteId(List<String> wxGroupIdList) {
        List<Long> qrcodeIds = groupQrcodeDao.filterDeleteQrcodeId(wxGroupIdList);
        if (ListUtils.isEmpty(qrcodeIds)){
            return new ArrayList<>();
        }else {
            return qrcodeIds;
        }
    }

}
