/**
 *
 */
package com.pcloud.common.utils.cache.redis;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Maps;
import com.pcloud.common.utils.ListUtils;
import com.pcloud.common.utils.string.StringUtil;

import org.springframework.util.CollectionUtils;
import redis.clients.jedis.JedisCluster;

/**
 * @描述:redis集群API
 * @作者:songx
 * @创建时间:2017年5月11日,下午4:15:40
 * @版本:1.0
 */
@Component("jedisClusterUtils")
public class JedisClusterUtils {

    /**
     *
     */
    private static Logger LOGGER = LoggerFactory.getLogger(JedisClusterUtils.class);

    /**
     * 默认过期时间(2小时)
     */
    private static final Integer REDIS_SESSION_TIMEOUT = 2 * 60 * 60;

    private static JedisCluster jedisCluster;

    /**
     * 保存一个字符串到redis中,长期有效
     *
     * @param key   键
     * @param value 值
     * @return
     */
    public static boolean set(String key, String value) {
        try {
            jedisCluster.set(key, value);
        } catch (Exception e) {
            LOGGER.warn("jedis缓存保存失败:" + e.getMessage(), e);
            return false;
        }
        return true;
    }

    /**
     * 根据前缀匹配模糊删除
     *
     * @param fuzzyKey 模糊前缀
     * @return 删除的数量
     */
    public static int fuzzyDel(String fuzzyKey) {
        if (StringUtils.isBlank(fuzzyKey)) {
            return 0;
        }
        Set<String> keys = jedisCluster.hkeys(fuzzyKey);
        if (CollectionUtils.isEmpty(keys)) {
            return 0;
        }
        int num = 0;
        for (String key : keys) {
            if (del(key)) {
                num++;
            }
        }
        return num;
    }

    /**
     * 保存一个字符串到redis中,长期有效
     *
     * @param key   键
     * @param value 值
     * @return
     */
    public static boolean getSet(String key, String value) {
        try {
            jedisCluster.getSet(key, value);
        } catch (Exception e) {
            LOGGER.warn("jedis缓存保存失败:" + e.getMessage(), e);
            return false;
        }
        return true;
    }

    public static String getSetR(String key,String value){
        String set;
        try {
            set = jedisCluster.getSet(key, value);
        } catch (Exception e) {
            LOGGER.warn("jedis缓存保存失败:" + e.getMessage(), e);
            return null;
        }
        return set;
    }

    /**
     * 保存一个字符串到redis中并指定过期时间
     *
     * @param key     键
     * @param value   值
     * @param seconds 有效时间，为null使用默认时间30分钟
     * @return 旧值
     */
    public static String getSet(String key, String value, Integer seconds) {
        String set;
        try {
            set = jedisCluster.getSet(key, value);
            expire(key, seconds);
        } catch (Exception e) {
            LOGGER.warn("jedis getSet缓存保存失败:" + e.getMessage(), e);
            return null;
        }
        return set;
    }

    /**
     * 保存一个字符串到redis中并指定过期时间
     *
     * @param key     键
     * @param value   值
     * @param seconds 有效时间，为null使用默认时间30分钟
     * @return
     */
    public static boolean set(String key, String value, Integer seconds) {
        try {
            jedisCluster.set(key, value);
            expire(key, seconds);
        } catch (Exception e) {
            LOGGER.warn("jedis缓存保存失败:" + e.getMessage(), e);
            return false;
        }
        return true;
    }

    /**
     * 保存一个字符串到redis中，长期有效
     *
     * @param objecKey   键
     * @param objecValue 缓存对象
     * @return
     */
    public static boolean setObject(Object key, Object value) {
        try {
            jedisCluster.set(SerializeUtils.serialize(key), SerializeUtils.serialize(value));
        } catch (Exception e) {
            LOGGER.warn("jedis缓存保存失败:" + e.getMessage(), e);
            return false;
        }
        return true;
    }

    /**
     * 保存一个字符串到redis中并指定过期时间
     *
     * @param objecKey   键
     * @param objecValue 缓存对象
     * @param seconds    时间,如果为null，则使用默认时间30分钟
     * @return
     */
    public static boolean setObject(Object key, Object value, Integer seconds) {
        if (key == null || value == null) {
            return false;
        }
        try {
            jedisCluster.set(SerializeUtils.serialize(key), SerializeUtils.serialize(value));
            expireObject(key, seconds);
        } catch (Exception e) {
            LOGGER.warn("jedis缓存保存失败:" + e.getMessage(), e);
            return false;
        }
        return true;
    }

    /**
     * 当键在redis中不存在时才设置
     *
     * @param key
     * @param value
     * @return
     */
    public static boolean setnx(String key, String value) {
        long result = 0;
        try {
            result = jedisCluster.setnx(key, value);
        } catch (Exception e) {
            LOGGER.warn("jedis Setn缓存保存失败:" + e.getMessage(), e);
            return false;
        }
        return result == 0 ? false : true;
    }

    /**
     * 追加一个字符串到指定的key
     *
     * @param key
     * @param value
     * @return
     */
    public static boolean append(String key, String value) {
        try {
            jedisCluster.append(key, value);
        } catch (Exception e) {
            LOGGER.warn("jedis缓存保存失败:" + e.getMessage(), e);
            return false;
        }
        return true;
    }

    /**
     * 根据缓存键获取Redis缓存中的值.<br/>
     *
     * @param key 键
     * @return
     */
    public static String get(String key) {
        try {
            return jedisCluster.get(key);
        } catch (Exception e) {
            LOGGER.warn("jedis获取缓存失败:" + e.getMessage(), e);
            return null;
        }
    }

    /**
     * 根据缓存键获取Redis缓存中的对象.<br/>
     *
     * @param objecKey 键
     * @return
     */
    public static Object getObject(Object key) {
        try {
            byte[] bytes = jedisCluster.get(SerializeUtils.serialize(key));
            return bytes == null ? null : SerializeUtils.unSerialize(bytes);
        } catch (Exception e) {
            LOGGER.warn("jedis获取缓存失败:" + e.getMessage(), e);
        }
        return null;
    }

    /**
     * 根据缓存键清除Redis缓存中的对象.
     *
     * @param key
     * @return
     */
    public static boolean delObject(Object key) {
        try {
            jedisCluster.del(SerializeUtils.serialize(key));
        } catch (Exception e) {
            LOGGER.warn("jedis删除缓存失败:" + e.getMessage(), e);
            return false;
        }
        return true;
    }

    /**
     * 根据缓存键批量清除Redis缓存中的对象.
     *
     * @param keys
     * @return
     */
    public static boolean delObject(Object... keys) {
        try {
            jedisCluster.del(SerializeUtils.serialize(keys));
        } catch (Exception e) {
            LOGGER.warn("jedis批量删除缓存失败:" + e.getMessage(), e);
            return false;
        }
        return true;
    }

    /**
     * 根据缓存键删除Redis缓存中的值.
     *
     * @param key
     * @return
     */
    public static boolean del(String key) {
        try {
            jedisCluster.del(key);
        } catch (Exception e) {
            LOGGER.warn("jedis删除缓存失败:" + e.getMessage(), e);
            return false;
        }
        return true;
    }

    /**
     * 根据缓存键批量清除Redis缓存中的值.
     *
     * @param keys
     * @return
     */
    public static boolean del(String... keys) {
        try {
            jedisCluster.del(keys);
        } catch (Exception e) {
            LOGGER.warn("jedis批量删除缓存失败:" + e.getMessage(), e);
            return false;
        }
        return true;
    }

    /**
     * 保存一个Map对象到redis中,长期有效
     *
     * @param key  键
     * @param hash Map组合的字符串
     * @return
     */
    public static boolean hmset(String key, Map<String, String> hash) {
        try {
            jedisCluster.hmset(key, hash);
        } catch (Exception e) {
            LOGGER.warn("jedis缓存保存失败:" + e.getMessage(), e);
            return false;
        }
        return true;
    }

    /**
     * 保存一个Map对象到redis中并指定过期时间
     *
     * @param key     键
     * @param hash    Map组合的字符串
     * @param seconds 时间
     * @return
     */
    public static boolean hmset(String key, Map<String, String> hash, Integer seconds) {
        try {
            jedisCluster.hmset(key, hash);
            expire(key, seconds);
        } catch (Exception e) {
            LOGGER.warn("jedis缓存保存失败:" + e.getMessage(), e);
            return false;
        }
        return true;
    }

    /**
     * 添加一个内容到指定key的hash中
     *
     * @param key
     * @param field
     * @param value
     * @return
     */
    public static Boolean hset(String key, String field, String value) {
        try {
            jedisCluster.hset(key, field, value);
        } catch (Exception e) {
            LOGGER.warn("jedis添加内容到指定key的hash失败:" + e.getMessage(), e);
            return false;
        }
        return true;
    }

    /**
     * 添加一个内容到指定key的hash中,并转换成json格式
     *
     * @param key
     * @param field
     * @param value
     * @return
     */
    public static Boolean hset2Json(String key, String field, Object value) {
        try {
            jedisCluster.hset(key, field, JSONObject.toJSONString(value));
        } catch (Exception e) {
            LOGGER.warn("jedis添加内容到指定key的hash失败:" + e.getMessage(), e);
            return false;
        }
        return true;
    }

    /**
     * 添加一个Fieid内容到指定key的hash中,如果Fieid存在就不添加，反之添加
     *
     * @param key
     * @param field
     * @param value
     * @return
     */
    public static boolean hsetnx(String key, String field, String value) {
        try {
            jedisCluster.hsetnx(key, field, value);
        } catch (Exception e) {
            LOGGER.warn("jedis缓存保存失败:" + e.getMessage(), e);
            return false;
        }
        return true;
    }

    /**
     * 添加一个Fieid内容到指定key的hash中,如果Fieid存在就不添加，反之添加
     *
     * @param key
     * @param field
     * @param value
     * @return
     */
    public static boolean hsetnx2Json(String key, String field, Object value) {
        try {
            jedisCluster.hsetnx(key, field, JSONObject.toJSONString(value));
        } catch (Exception e) {
            LOGGER.warn("jedis缓存保存失败:" + e.getMessage(), e);
            return false;
        }
        return true;
    }

    /**
     * 指定Map中file的值自增
     *
     * @param key
     * @param field
     * @param value
     * @return
     */
    public static Long hincrBy(String key, String field, Long value) {
        try {
            return jedisCluster.hincrBy(key, field, value);
        } catch (Exception e) {
            LOGGER.warn("jedis缓存保存失败:" + e.getMessage(), e);
            return null;
        }
    }

    /**
     * 添加一个内容到指定key的hash中
     *
     * @param key
     * @param field
     * @param value
     * @return
     */
    @Deprecated
    public static Boolean hsetObject(Object key, Object field, Object value) {
        try {
            jedisCluster.hset(SerializeUtils.serialize(key), SerializeUtils.serialize(field),
                    SerializeUtils.serialize(value));
        } catch (Exception e) {
            LOGGER.warn("jedis添加内容到指定key的hash失败:" + e.getMessage(), e);
            return false;
        }
        return true;
    }

    /**
     * 根据缓存键获取Redis缓存中的Map.<br/>
     *
     * @param key
     * @param fields 具体的map键
     * @return
     */
    public static Map<String, String> hgetAll(String key) {
        try {
            return jedisCluster.hgetAll(key);
        } catch (Exception e) {
            LOGGER.warn("jedis获取缓存失败:" + e.getMessage(), e);
            return null;
        }
    }

    /**
     * 根据缓存键获取Redis缓存中的Map.<br/>
     * 可以指定Map具体的键值获取精确数据
     *
     * @param key
     * @param fields 具体的map键
     * @return
     */
    public static Map<String, String> hmget(String key, String... fields) {
        Map<String, String> resultMap = Maps.newHashMap();
        try {
            // 获取完整的map集合
            if (fields == null || fields.length == 0) {
                return jedisCluster.hgetAll(key);
            } else {
                List<String> resultList = jedisCluster.hmget(key, fields);
                if (!ListUtils.isEmpty(resultList)) {
                    for (int i = 0; i < resultList.size(); i++) {
                        resultMap.put(fields[i], resultList.get(i));
                    }
                }
            }
        } catch (Exception e) {
            LOGGER.warn("jedis获取缓存失败:" + e.getMessage(), e);
            return null;
        }
        return resultMap;
    }

    /**
     * 从指定hash中取值
     *
     * @param key
     * @param field
     * @return
     */
    public static String hget(String key, String field) {
        try {
            return jedisCluster.hget(key, field);
        } catch (Exception e) {
            LOGGER.warn("jedis从指定hash中取值失败:" + e.getMessage(), e);
            return null;
        }
    }

    /**
     * 从指定hash中取值,并将取到的json字符串转换成指定的对象
     *
     * @param key
     * @param field
     * @param clazz
     * @return
     */
    public static <T> T hgetJson2Class(String key, String field, Class<T> clazz) {
        try {
            return JSONObject.parseObject(jedisCluster.hget(key, field), clazz);
        } catch (Exception e) {
            LOGGER.warn("jedis从指定hash中取值失败:" + e.getMessage(), e);
            return null;
        }
    }

    /**
     * 从指定hash中取值,并将取到的json字符串转换成指定的对象集合
     *
     * @param key
     * @param field
     * @param clazz
     * @return
     */
    public static <T> List<T> hgetJson2List(String key, String field, Class<T> clazz) {
        try {
            return JSONArray.parseArray(jedisCluster.hget(key, field), clazz);
        } catch (Exception e) {
            LOGGER.warn("jedis从指定hash中取值失败:" + e.getMessage(), e);
            return null;
        }
    }

    /**
     * 从指定hash中取值
     *
     * @param key
     * @param field
     * @return
     */
    @Deprecated
    public static Object hgetObject(Object key, Object field) {
        try {
            byte[] bytes = jedisCluster.hget(SerializeUtils.serialize(key), SerializeUtils.serialize(field));
            return bytes == null ? null : SerializeUtils.unSerialize(bytes);
        } catch (Exception e) {
            LOGGER.warn("jedis从指定hash中取对象失败:" + e.getMessage(), e);
        }
        return null;
    }

    /**
     * 根据缓存键删除map中的指定键值
     *
     * @param key   缓存键
     * @param field map键值
     * @return 删除成功返回1，失败返回0
     */
    public static Boolean hdel(String key, String... fields) {
        long result = 0L;
        try {
            result = jedisCluster.hdel(key, fields);
        } catch (Exception e) {
            LOGGER.warn("jedis删除map中的指定键值失败:" + e.getMessage(), e);
        }
        return result > 0 ? true : false;
    }

    /**
     * 从hash中删除指定filed的对象
     *
     * @param key
     * @param field
     * @return
     */
    @Deprecated
    public static Boolean hdelObject(Object key, Object... field) {
        long result = 0L;
        try {
            result = jedisCluster.hdel(SerializeUtils.serialize(key), SerializeUtils.serialize(field));
        } catch (Exception e) {
            LOGGER.warn("jedis从hash中删除指定filed的对象失败:" + e.getMessage(), e);
        }
        return result > 0 ? true : false;
    }

    /**
     * 获取指定key中存放的hash值个数
     *
     * @param key
     * @return
     */
    public static Long hlen(String key) {
        try {
            return jedisCluster.hlen(key);
        } catch (Exception e) {
            LOGGER.warn("jedis获取指定key中存放的hash值个数失败:" + e.getMessage(), e);
        }
        return null;
    }

    /**
     * 判断指定key中是否存在fileId键
     *
     * @param key
     * @return
     */
    public static Boolean hexists(String key, String fileId) {
        try {
            return jedisCluster.hexists(key, fileId);
        } catch (Exception e) {
            LOGGER.warn("jedis判断指定key中是否存在fileId键失败:" + e.getMessage(), e);
        }
        return null;
    }

    /**
     * 获取指定key中存放的hash值个数
     *
     * @param key
     * @return
     */
    public static Long hlenObject(Object key) {
        try {
            return jedisCluster.hlen(SerializeUtils.serialize(key));
        } catch (Exception e) {
            LOGGER.warn("jedis获取指定key中存放的hash值个数失败:" + e.getMessage(), e);
        }
        return null;
    }

    /**
     * 获取指定key中map对象中的所有key
     *
     * @param key
     * @return
     */
    public static Set<String> hkeys(String key) {
        try {
            return jedisCluster.hkeys(key);
        } catch (Exception e) {
            LOGGER.warn("jedis获取指定key中map对象中的所有key失败:" + e.getMessage(), e);
        }
        return null;
    }

    /**
     * 添加内容到指定key的list中,先进后出
     *
     * @param key
     * @param field
     * @param value
     * @return
     */
    public static Boolean lpush(String key, String value) {
        Long result = 0L;
        try {
            result = jedisCluster.lpush(key, value);
        } catch (Exception e) {
            LOGGER.warn("jedis添加内容到指定key的list中失败:" + e.getMessage(), e);
        }
        return result == 0 ? false : true;
    }

    /**
     * 添加内容到指定key的list中,先进后出
     *
     * @param key
     * @param obj
     * @return
     */
    public static Boolean lpushJson(String key, Object obj) {
        Long result = 0L;
        try {
            result = jedisCluster.lpush(key, JSONObject.toJSONString(obj));
        } catch (Exception e) {
            LOGGER.warn("jedis添加内容到指定key的list中失败:" + e.getMessage(), e);
        }
        return result == 0 ? false : true;
    }

    /**
     * 添加内容到指定key的list中,先进后出
     *
     * @param key
     * @param value
     * @return
     */
    public static Boolean lpushObject(Object key, Object value) {
        Long result = 0L;
        try {
            result = jedisCluster.lpush(SerializeUtils.serialize(key), SerializeUtils.serialize(value));
        } catch (Exception e) {
            LOGGER.warn("jedis添加内容到指定key的list中失败:" + e.getMessage(), e);
        }
        return result == 0 ? false : true;
    }

    /**
     * 添加内容到指定key的list中,先进先出
     *
     * @param key
     * @param field
     * @param value
     * @return
     */
    public static Boolean rpush(String key, String value) {
        Long result = 0L;
        try {
            result = jedisCluster.rpush(key, value);
        } catch (Exception e) {
            LOGGER.warn("jedis添加内容到指定key的list中失败:" + e.getMessage(), e);
        }
        return result == 0 ? false : true;
    }

    /**
     * 添加内容到指定key的list中,先进先出
     *
     * @param key
     * @param obj
     * @return
     */
    public static Boolean rpushJson(String key, Object obj) {
        Long result = 0L;
        try {
            result = jedisCluster.rpush(key, JSONObject.toJSONString(obj));
        } catch (Exception e) {
            LOGGER.warn("jedis添加内容到指定key的list中失败:" + e.getMessage(), e);
        }
        return result == 0 ? false : true;
    }

    /**
     * 添加内容到指定key的list中,先进先出
     *
     * @param key
     * @param field
     * @param value
     * @return
     */
    public static Boolean rpushObject(Object key, Object value) {
        Long result = 0L;
        try {
            result = jedisCluster.rpush(SerializeUtils.serialize(key), SerializeUtils.serialize(value));
        } catch (Exception e) {
            LOGGER.warn("jedis添加内容到指定key的list中失败:" + e.getMessage(), e);
        }
        return result == 0 ? false : true;
    }

    /**
     * 获取指定key的list集合
     *
     * @param key
     * @param start list中的起始位置，从0开始
     * @param len   长度，-1获取所有
     * @return
     */
    public static List<String> lrange(String key, long start, long len) {
        try {
            return jedisCluster.lrange(key, start, len);
        } catch (Exception e) {
            LOGGER.warn("jedis获取指定key的list集合失败:" + e.getMessage(), e);
        }
        return null;
    }

    /**
     * 获取指定key的list长度
     *
     * @param key
     * @return
     */
    public static Long llen(String key) {
        try {
            return jedisCluster.llen(key);
        } catch (Exception e) {
            LOGGER.warn("jedis获取指定key的list长度失败:" + e.getMessage(), e);
        }
        return null;
    }

    /**
     * 获取指定key的list长度
     *
     * @param key
     * @return
     */
    public static Long llenObject(Object key) {
        try {
            return jedisCluster.llen(SerializeUtils.serialize(key));
        } catch (Exception e) {
            LOGGER.warn("jedis获取指定key的list长度失败:" + e.getMessage(), e);
        }
        return null;
    }

    /**
     * 列表出栈（左端）
     *
     * @param key
     * @return
     */
    public static String lpop(String key) {
        try {
            return jedisCluster.lpop(key);
        } catch (Exception e) {
            LOGGER.warn("jedis列表出栈（左端）失败:" + e.getMessage(), e);
        }
        return null;
    }

    /**
     * 列表出栈（左端）
     *
     * @param key
     * @return
     */
    public static List<String> lpop(String key, long count) {
        try {
            List<String> list = new ArrayList<String>();
            for (int i = 0; i < count; i++) {
                String vel = jedisCluster.lpop(key);
                if (vel != null) {
                    list.add(vel);
                }
            }
            return list;
        } catch (Exception e) {
            LOGGER.warn("jedis列表出栈（左端）失败:" + e.getMessage(), e);
        }
        return null;
    }

    /**
     * 列表出栈（左端）
     *
     * @param key
     * @return
     */
    public static <T> T lpopJson(String key, Class<T> clazz) {
        try {
            String vel = jedisCluster.lpop(key);
            if (!StringUtil.isEmpty(vel)) {
                return JSONObject.parseObject(vel, clazz);
            }
        } catch (Exception e) {
            LOGGER.warn("jedis列表出栈（左端）失败:" + e.getMessage(), e);
        }
        return null;
    }

    /**
     * 列表出栈（左端）
     *
     * @param key
     * @return
     */
    public static <T> List<T> lpopJson(String key, long count, Class<T> clazz) {
        try {
            List<T> list = new ArrayList<T>();
            for (int i = 0; i < count; i++) {
                String vel = jedisCluster.lpop(key);
                if (!StringUtil.isEmpty(vel)) {
                    list.add(JSONObject.parseObject(vel, clazz));
                }
            }
            return list;
        } catch (Exception e) {
            LOGGER.warn("jedis列表出栈（左端）失败:" + e.getMessage(), e);
        }
        return null;
    }

    /**
     * 列表出栈（右端）
     *
     * @param key
     * @return
     */
    public static String rpop(String key) {
        try {
            return jedisCluster.rpop(key);
        } catch (Exception e) {
            LOGGER.warn("jedis列表出栈（右端）失败:" + e.getMessage(), e);
        }
        return null;
    }

    /**
     * 列表出栈（右端）
     *
     * @param key
     * @return
     */
    public static List<String> rpop(String key, long count) {
        try {
            List<String> list = new ArrayList<String>();
            for (int i = 0; i < count; i++) {
                String vel = jedisCluster.rpop(key);
                if (vel != null) {
                    list.add(vel);
                }
            }
            return list;
        } catch (Exception e) {
            LOGGER.warn("jedis列表出栈（右端）失败:" + e.getMessage(), e);
        }
        return null;
    }

    /**
     * 列表出栈（右端）
     *
     * @param key
     * @return
     */
    public static <T> T rpopJson(String key, Class<T> clazz) {
        try {
            String vel = jedisCluster.rpop(key);
            if (!StringUtil.isEmpty(vel)) {
                return JSONObject.parseObject(vel, clazz);
            }
        } catch (Exception e) {
            LOGGER.warn("jedis列表出栈（右端）失败:" + e.getMessage(), e);
        }
        return null;
    }

    /**
     * 列表出栈（右端）
     *
     * @param key
     * @return
     */
    public static <T> List<T> rpopJson(String key, long count, Class<T> clazz) {
        try {
            List<T> list = new ArrayList<T>();
            for (int i = 0; i < count; i++) {
                String vel = jedisCluster.rpop(key);
                if (!StringUtil.isEmpty(vel)) {
                    list.add(JSONObject.parseObject(vel, clazz));
                }
            }
            return list;
        } catch (Exception e) {
            LOGGER.warn("jedis列表出栈（右端）失败:" + e.getMessage(), e);
        }
        return null;
    }

    /**
     * 删除指定区间之外的元素
     *
     * @param key
     * @return
     */
    public static String ltrim(String key, long start, long len) {
        try {
            return jedisCluster.ltrim(key, start, len);
        } catch (Exception e) {
            LOGGER.warn("jedis删除指定区间之外的元素失败:" + e.getMessage(), e);
        }
        return null;
    }

    /**
     * 有序序列(Sort-Set)集合：添加数据
     *
     * @param key
     * @return
     */
    public static boolean zset(String key, double score, String member) {
        try {
            jedisCluster.zadd(key, score, member);
        } catch (Exception e) {
            LOGGER.error("jedis有序序列集合：添加数据失败:" + e.getMessage(), e);
            return false;
        }
        return true;
    }

    /**
     * 获取指定key的Sort-Set集合长度
     *
     * @return
     */
    public static Long zcard(String key) {
        try {
            return jedisCluster.zcard(key);
        } catch (Exception e) {
            LOGGER.error("jedis有序序列集合长度失败:" + e.getMessage(), e);
        }
        return null;
    }

    /**
     * 获取指定key的Sort-Set集合中member元素所在的位置排行（从 0 开始）
     *
     * @return
     */
    public static Long zrank(String key, String member) {
        try {
            return jedisCluster.zrank(key, member);
        } catch (Exception e) {
            LOGGER.error("jedis获取有序序列集合中MEMBER位置失败:" + e.getMessage(), e);
        }
        return null;
    }

    /**
     * 获取指定key的Sort-Set集合的元素(按照score从小到大排序)
     *
     * @return
     */
    public static Set<String> zrange(String key, long start, long end) {
        try {
            return jedisCluster.zrange(key, start, end);
        } catch (Exception e) {
            LOGGER.error("jedis有序序列集合的元素失败:" + e.getMessage(), e);
        }
        return null;
    }

    /**
     * 获取指定key的Sort-Set集合的元素(按照score从大到小排序)
     *
     * @return
     */
    public static Set<String> zrevrange(String key, long start, long end) {
        try {
            return jedisCluster.zrevrange(key, start, end);
        } catch (Exception e) {
            LOGGER.error("jedis有序序列集合的元素失败:" + e.getMessage(), e);
        }
        return null;
    }

    /**
     * 删除指定key的Sort-Set集合的元素
     *
     * @return
     */
    public static boolean zrem(String key, long start, long end) {
        try {
            jedisCluster.zremrangeByRank(key, start, end);
        } catch (Exception e) {
            LOGGER.error("jedis删除有序序列集合的元素失败:" + e.getMessage(), e);
            return false;
        }
        return true;
    }

    /**
     * 根据clazz和key获取对应 的类型
     *
     * @param key
     * @param clazz
     * @return
     */
    public static <T> T getClass(Object key, Class<T> clazz) {
        try {
            Object object = getObject(key);
            return object == null ? null : clazz.cast(object);
        } catch (Exception e) {
            LOGGER.warn("jedis根据clazz和key获取对应 的类型:" + e.getMessage(), e);
        }
        return null;
    }

    /**
     * 根据clazz和key获取对应的list
     *
     * @param key
     * @param clazz
     * @return
     */
    @SuppressWarnings("unchecked")
    public static <T> List<T> getList(Object key, Class<T> clazz) {
        try {
            Object object = getObject(key);
            return object == null ? null : (List<T>) object;
        } catch (Exception e) {
            LOGGER.warn("jedis根据clazz和key获取对应的list:" + e.getMessage(), e);
        }
        return null;
    }

    /**
     * 根据clazz和key获取对应的map
     *
     * @param key
     * @param clazz
     * @return
     */
    @SuppressWarnings("unchecked")
    public static <K, V> Map<K, V> getMap(Object key, Class<K> kClazz, Class<V> vClazz) {
        try {
            Object object = getObject(key);
            return object == null ? null : (Map<K, V>) object;
        } catch (Exception e) {
            LOGGER.warn("jedis根据clazz和key获取对应的map:" + e.getMessage(), e);
        }
        return null;
    }

    /**
     * 根据key，及clazz获取对象.使用json进行转换
     *
     * @param key
     * @param clazz
     * @return
     */
    public static <T> T getJson(String key, Class<T> clazz) {
        String jsonValue = get(key);
        if (StringUtil.isEmpty(jsonValue)) {
            return null;
        }
        try {
            return JSONObject.parseObject(jsonValue, clazz);
        } catch (Exception e) {
            LOGGER.warn("jedis根据key获取clazz对象,JSON转换异常:" + e.getMessage(), e);
        }
        return null;
    }

    /**
     * 根据key，及clazz获取对象集合.使用json进行转换
     *
     * @param key
     * @param clazz
     * @return
     */
    public static <T> List<T> getJsonList(String key, Class<T> clazz) {
        String jsonValue = get(key);
        if (StringUtil.isEmpty(jsonValue)) {
            return null;
        }
        try {
            return JSONArray.parseArray(jsonValue, clazz);
        } catch (Exception e) {
            LOGGER.warn("jedis根据key获取clazz对象,JSON转换异常:" + e.getMessage(), e);
        }
        return null;
    }

    /**
     * 缓存对象.使用json进行转换
     *
     * @param key
     * @param obj
     */
    public static Boolean setJson(String key, Object obj) {
        try {
            return set(key, JSONObject.toJSONString(obj));
        } catch (Exception e) {
            LOGGER.warn("jedis根据key缓存对象,JSON转换异常:" + e.getMessage(), e);
        }
        return false;
    }

    /**
     * 缓存对象.使用json进行转换,带有效期
     *
     * @param key
     * @param obj
     * @param seconds 有效期(秒)
     * @return
     */
    public static Boolean setJson(String key, Object obj, Integer seconds) {
        try {
            return set(key, JSONObject.toJSONString(obj), seconds);
        } catch (Exception e) {
            LOGGER.warn("jedis根据key缓存对象,JSON转换异常:" + e.getMessage(), e);
        }
        return false;
    }

    /**
     * 缓存对象集合.使用json进行转换
     *
     * @param key
     * @param obj
     * @return
     */
    public static Boolean setJsonList(String key, Object obj) {
        try {
            return set(key, JSONArray.toJSONString(obj));
        } catch (Exception e) {
            LOGGER.warn("jedis根据key缓存对象,JSON转换异常:" + e.getMessage(), e);
        }
        return false;
    }

    /**
     * 缓存对象集合.使用json进行转换
     *
     * @param key
     * @param obj
     * @return
     */
    public static Boolean setJsonList(String key, Object obj, Integer seconds) {
        try {
            return set(key, JSONArray.toJSONString(obj), seconds);
        } catch (Exception e) {
            LOGGER.warn("jedis根据key缓存对象,JSON转换异常:" + e.getMessage(), e);
        }
        return false;
    }

    /**
     * 设置超时时间
     *
     * @param key
     * @param seconds
     * @return
     */
    public static boolean expire(String key, Integer seconds) {
        try {
            jedisCluster.expire(key, seconds == null || seconds < 1 ? REDIS_SESSION_TIMEOUT : seconds);
        } catch (Exception e) {
            LOGGER.warn("jedis设置超时时间失败:" + e.getMessage(), e);
            return false;
        }
        return true;
    }

    /**
     * 设置超时时间
     *
     * @param key
     * @param seconds 超时时间（单位为秒）
     * @return
     */
    public static boolean expireObject(Object key, Integer seconds) {
        try {
            jedisCluster.expire(SerializeUtils.serialize(key),
                    seconds == null || seconds < 1 ? REDIS_SESSION_TIMEOUT : seconds);
        } catch (Exception e) {
            LOGGER.warn("jedis设置超时时间失败:" + e.getMessage(), e);
            return false;
        }
        return true;
    }

    /**
     * 查看键还有多长时间过期(秒)
     *
     * @param key
     * @return
     */
    public static Long ttl(String key) {
        try {
            return jedisCluster.ttl(key);
        } catch (Exception e) {
            LOGGER.warn("jedis查看过期时间失败:" + e.getMessage(), e);
        }
        return null;
    }

    /**
     * 查看键还有多长时间过期(秒)
     *
     * @param key
     * @return
     */
    public static Long ttlObject(Object key) {
        try {
            return jedisCluster.ttl(SerializeUtils.serialize(key));
        } catch (Exception e) {
            LOGGER.warn("jedis查看过期时间失败:" + e.getMessage(), e);
        }
        return null;
    }

    /**
     * 判断一个key是否存在
     *
     * @param key
     * @return
     */
    public static boolean exists(String key) {
        try {
            return jedisCluster.exists(key);
        } catch (Exception e) {
            LOGGER.warn("jedis判断key是否存在失败:" + e.getMessage(), e);
        }
        return false;
    }

    /**
     * 判断一个key是否存在
     *
     * @param key
     * @return
     */
    public static boolean existsObject(Object key) {
        try {
            return jedisCluster.exists(SerializeUtils.serialize(key));
        } catch (Exception e) {
            LOGGER.warn("jedis判断key是否存在失败:" + e.getMessage(), e);
        }
        return false;
    }

    /**
     * 缓存值自增1
     *
     * @param key
     * @return
     */
    public static Long incr(String key) {
        Long result = null;
        try {
            result = jedisCluster.incr(key);
        } catch (Exception e) {
            LOGGER.warn("jedis缓存值自增1失败:" + e.getMessage(), e);
        }
        return result;
    }

    /**
     * 缓存值自增1
     *
     * @param key
     * @return
     */
    public static Long incrObject(Object key) {
        Long result = null;
        try {
            result = jedisCluster.incr(SerializeUtils.serialize(key));
        } catch (Exception e) {
            LOGGER.warn("jedis缓存值自增1失败:" + e.getMessage(), e);
        }
        return result;
    }

    /**
     * 缓存值自减1
     *
     * @param key
     * @return
     */
    public static Long decr(String key) {
        Long result = null;
        try {
            result = jedisCluster.decr(key);
        } catch (Exception e) {
            LOGGER.warn("jedis缓存值自减1失败:" + e.getMessage(), e);
        }
        return result;
    }

    /**
     * 缓存值自减1
     *
     * @param key
     * @return
     */
    public static Long decrObject(Object key) {
        Long result = null;
        try {
            result = jedisCluster.decr(SerializeUtils.serialize(key));
        } catch (Exception e) {
            LOGGER.warn("jedis缓存值自减1失败:" + e.getMessage(), e);
        }
        return result;
    }

    @Autowired
    public void setJedisCluster(JedisCluster jedisCluster) {
        JedisClusterUtils.jedisCluster = jedisCluster;
    }

}
