package com.pcloud.common.core.aspect;

import com.pcloud.common.utils.DateUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.RandomUtils;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.AfterReturning;
import org.aspectj.lang.annotation.Pointcut;
import org.springframework.util.ResourceUtils;
import redis.clients.util.JedisClusterCRC16;
import redis.clients.util.SafeEncoder;

import java.io.*;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * 添加监控，查看
 */
//@Component
//@Aspect
@Slf4j
public class JedisClusterAspect {
    //需要统计的方法调用次数
    private Map<String, ConcurrentHashMap<String, AtomicInteger>> solfDistributedMap = new ConcurrentHashMap<>();
    private Map<String, ConcurrentHashMap<String, AtomicInteger>> tempMap = new ConcurrentHashMap<>();
    private LFU<String, String> cache = new LFU<>(10000);

    public JedisClusterAspect() {
        //初始化redis操作
        new PringLogThread().start();
    }

    @Pointcut("execution(* redis.clients.jedis.JedisCluster.*(..))")
    public void pointcut() {

    }

    class PringLogThread extends Thread {
        @Override
        public void run() {
            String path = getResourceBasePath() + File.separator + "redislog" + File.separator;
            File file = new File(path);
            if (!file.exists()) {
                file.mkdirs();
            }
            while (true) {
                BufferedWriter out = null;
                try {
                    Thread.sleep(60*60 * 1000);
                    tempMap.putAll(solfDistributedMap);
                    solfDistributedMap.clear();
                    String fileName = "hotkey" + DateUtils.getYmdHmsTime() + ".log";
                    out = new BufferedWriter(new FileWriter(path + fileName));
                    for (Map.Entry<String, ConcurrentHashMap<String, AtomicInteger>> entry : tempMap.entrySet()) {
                        for (Map.Entry<String, AtomicInteger> childEntry : entry.getValue().entrySet()) {
                            out.write("key前缀："+entry.getKey() + "，节点：" + childEntry.getKey());
                            out.write("，调用次数：" + childEntry.getValue());
                            out.newLine();
                        }
                    }
                    tempMap.clear();
                    Map<String, LFU<String, String>.HitRate> hotMap = cache.getHotKeyMap();
                    for (Map.Entry<String, LFU<String, String>.HitRate> entry : hotMap.entrySet()) {
                        LFU<String, String>.HitRate hitRate = entry.getValue();
                        out.write("key：" + hitRate.getKey() + "，槽位：" + hitRate.getV() + "，次数：" + hitRate.getHitCount());
                        out.newLine();
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                } finally {
                    try {
                        out.close();
                    } catch (IOException e) {
                    }
                }
            }
        }
    }

    @AfterReturning(value = "pointcut()", returning = "result")
    public void afterReturning(JoinPoint joinPoint, Object result) {
        String methodName = joinPoint.getSignature().getName();
        Object[] obj = joinPoint.getArgs();
        String key = obj[0].toString();
        setSoltDistributed(key);
        cache.put(key, getNodeName(key));
        int randomInt = RandomUtils.nextInt(1,101);
        //100里面1个数，小于等于10的概率就是10%
        if(randomInt<10){
            log.warn("采样hotkey methodName {} key {}",methodName,key);
        }
        if(key.startsWith("readercenter")){
            log.warn("统计增长比较快的key methodName {} key {}",methodName,key);
        }
    }
    private String getNodeName(String key){
        int slot = JedisClusterCRC16.getSlot(SafeEncoder.encode(key));
        String nodeName = "";
        if (slot >= 0 && slot <= 4095) {
            nodeName = "firstNode";
        } else if (slot >= 4096 && slot <= 8191) {
            nodeName = "secondNode";
        } else if (slot >= 8192 && slot <= 12287) {
            nodeName = "thirdNode";
        } else if (slot > 12288 && slot <= 16383) {
            nodeName = "fourthNode";
        }
        return nodeName;
    }
    /**
     * 计算key的分布
     *
     * @param key
     */
    private void setSoltDistributed(String key) {
        String[] keys = key.split(":");
        if (keys.length < 2) {
            return;
        }
        String keyPrefix = key.substring(0, key.lastIndexOf(":"));
        String nodeName = getNodeName(key);
        ConcurrentHashMap<String, AtomicInteger> map = solfDistributedMap.get(keyPrefix);
        if (map == null) {
            map = new ConcurrentHashMap<String, AtomicInteger>();
            solfDistributedMap.put(keyPrefix,map);
        }
        AtomicInteger count = map.get(nodeName);
        if (count == null) {
            count = new AtomicInteger(1);
        } else {
            count.getAndIncrement();
        }
        map.put(nodeName, count);
    }

    private static String getResourceBasePath() {
        // 获取跟目录
        File path = null;
        try {
            path = new File(ResourceUtils.getURL("classpath:").getPath());
        } catch (FileNotFoundException e) {
            // nothing to do
        }
        if (path == null || !path.exists()) {
            path = new File("");
        }

        String pathStr = path.getAbsolutePath();
        // 如果是在eclipse中运行，则和target同级目录,如果是jar部署到服务器，则默认和jar包同级
        pathStr = pathStr.replace("\\target\\classes", "");

        return pathStr;
    }

    public void addMissCount(ConcurrentHashMap<String, AtomicInteger> map, String key) {
        AtomicInteger missCount = map.get(key);
        if (missCount == null) {
            missCount = new AtomicInteger(1);
        } else {
            missCount.getAndIncrement();
        }
        map.put(key, missCount);
    }

}
