package com.pcloud.common.core.zipkin;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.cloud.sleuth.Sampler;
import org.springframework.cloud.sleuth.Span;

import java.util.BitSet;
import java.util.Map;
import java.util.Random;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.ReentrantReadWriteLock;

/**
 * 自定义采样率Sampler
 */
@Slf4j
public class PercentageLocalSampler implements Sampler {

    //采样率最小值
    private static final Float MIN_PERCENTAGE = 0.0f;
    //采样率最大值
    private static final Float MAX_PERCENTAGE = 1.0f;
    private Map<String, BitSet> sampleDecisionsMap;
    //本地采样器
    private final SamplerLocalProperties configuration;

    private Float globalPercentage;
    private static final String all = "all";
    //替换访问地址前缀正则表达式
    private static final String regex = "^(http://www\\.|http://|www\\.|http:)";
    //uri访问次数
    private final Map<String, AtomicInteger> concurrentSampleCount;
    //刷新本地采样器和读取采样器配置使用读写锁，该场景读大于写，后期可优化为类似eureka注册表多级缓存
    ReentrantReadWriteLock readWriteLock = new ReentrantReadWriteLock();
    //读锁
    ReentrantReadWriteLock.ReadLock readLock = readWriteLock.readLock();
    //写锁
    ReentrantReadWriteLock.WriteLock writeLock = readWriteLock.writeLock();

    public PercentageLocalSampler(SamplerLocalProperties configuration) {
        this.configuration = configuration;
        sampleDecisionsMap = buildRandomBit();
        concurrentSampleCount = new ConcurrentHashMap<>();
        concurrentSampleCount.put(all, new AtomicInteger(0));
    }

    public Map<String, AtomicInteger> getConcurrentSampleCount() {
        return this.concurrentSampleCount;
    }

    @Override
    public boolean isSampled(Span currentSpan) {
        try {
            readLock.lock();
            if (currentSpan == null) {
                return false;
            }
            // 获取span中的请求uri
            String uri = currentSpan.getName();
            uri = uri.replaceFirst(regex, "");
            AtomicInteger count = this.concurrentSampleCount.get(all);
            // 获取全局的bitSet
            BitSet bitSet = this.sampleDecisionsMap.get(all);
            // 获取全局的采样率
            float percentage = this.configuration.getPercentage();
            for (UriSampleProperties sampleProperties : configuration.getUriSamples()) {
                // 正则匹配
                if (uri.matches(sampleProperties.getUriRegex())) {
                    //匹配上了自定义采样率的正则
                    // 多个线程会有并发问题，加个局部锁
                    synchronized (this) {
                        // 判断当前uri是否在map中
                        if (!concurrentSampleCount.containsKey(uri)) {
                            concurrentSampleCount.put(uri, new AtomicInteger(0));
                        }
                    }
                    // 获取当前URI对应的访问次数
                    count = concurrentSampleCount.get(uri);
                    // 获取当前URI对应的bitSet
                    bitSet = sampleDecisionsMap.get(sampleProperties.getUriRegex());
                    // 获取当前URI对应的采样率
                    percentage = sampleProperties.getUriPercentage();
                    break;
                }
            }
            log.warn("replace uri {} , percentage {}", uri, percentage);
            // 如果采样率是0 ，直接返回false
            if (percentage == MIN_PERCENTAGE) {
                return false;
            } else if (percentage == MAX_PERCENTAGE) { // 如果采样率是1 ，那么直接返回true
                return true;
            }
            synchronized (this) {
                // 访问次数加1
                final int i = count.getAndIncrement();
                // 判断当前的访问 次数是否在 bitSet中，存在则返回true
                boolean result = bitSet.get(i);
                // 等于99的时候，重新设置为0
                if (i == 99) {
                    count.set(0);
                }
                return result;
            }
        } finally {
            readLock.unlock();
        }
    }

    private static BitSet randomBitSet(int size, int cardinality, Random rnd) {
        BitSet result = new BitSet(size);
        int[] chosen = new int[cardinality];
        int i;
        for (i = 0; i < cardinality; ++i) {
            chosen[i] = i;
            result.set(i);
        }
        for (; i < size; ++i) {
            int j = rnd.nextInt(i + 1);
            if (j < cardinality) {
                result.clear(chosen[j]);
                result.set(i);
                chosen[j] = i;
            }
        }
        return result;
    }

    private Map<String, BitSet> buildRandomBit() {
        Map<String, BitSet> map = new ConcurrentHashMap<>();
        int size = 100;
        // 设置全局的采样率
        int outOf100 = (int) (configuration.getPercentage() * size);
        map.put(all, randomBitSet(size, outOf100, new Random()));
        if (CollectionUtils.isNotEmpty(configuration.getUriSamples())) {
            for (UriSampleProperties sampleProperties : configuration.getUriSamples()) {
                // 设置个性化的采样率
                map.put(sampleProperties.getUriRegex(), randomBitSet(size,
                        (int) (sampleProperties.getUriPercentage() * size), new Random()));
            }
        }
        return map;
    }

    public void initPercentage(SamplerLocalProperties samplerRemoteProperties) {
        try {
            writeLock.lock();
            configuration.setPercentage(samplerRemoteProperties.getPercentage());
            configuration.setUriSamples(samplerRemoteProperties.getUriSamples());
            sampleDecisionsMap = buildRandomBit();
        } finally {
            writeLock.unlock();
        }
    }

    public SamplerLocalProperties getConfiguration() {
        return configuration;
    }

}
