package com.pcloud.book.advertising.biz;

import java.util.List;
import java.util.Map;
import com.pcloud.book.advertising.dto.AdvertisingSpaceDTO;
import com.pcloud.book.advertising.entity.AdvertisingAdviserPermission;
import com.pcloud.book.advertising.entity.AdvertisingAgentPermission;
import com.pcloud.book.advertising.entity.AdvertisingClickRecord;
import com.pcloud.book.advertising.entity.AdvertisingDistributionBook;
import com.pcloud.book.advertising.entity.AdvertisingExposureRecord;
import com.pcloud.book.advertising.entity.AdvertisingSpace;
import com.pcloud.common.exceptions.BizException;
import com.pcloud.common.page.PageBean;
import com.pcloud.common.page.PageParam;

/**
 * Description 广告位业务逻辑层接口
 * @author PENG
 * @date 2019/2/28
 */
public interface AdvertisingSpaceBiz {

    /**
     * 创建广告位
     * @param advertisingSpace 广告位实体
     * @return
     * @throws BizException
     */
    Long create(AdvertisingSpace advertisingSpace) throws BizException;

    /**
     * 修改广告位
     * @param advertisingSpace 广告位实体
     * @return
     * @throws BizException
     */
    void update(AdvertisingSpace advertisingSpace) throws BizException;

    /**
     * 停用广告位
     * @param advertisingSpace 广告位实体
     * @throws BizException
     */
    void stop(AdvertisingSpace advertisingSpace) throws BizException;

    /**
     * 删除广告位
     * @param advertisingSpace 广告位实体
     * @throws BizException
     */
    void delete(AdvertisingSpace advertisingSpace) throws BizException;

    /**
     * 获取广告位信息
     * @param adId 广告位ID
     * @return
     * @throws BizException
     */
    AdvertisingSpaceDTO get(Long adId) throws BizException;

    /**
     * 投放书刊
     * @param book 广告位书刊实体
     * @return
     * @throws BizException
     */
    void distributeBook(AdvertisingDistributionBook book) throws BizException;

    /**
     * 获取出版社书刊权限
     * @param agentId 出版社ID
     * @return
     * @throws BizException
     */
    Boolean getAgentPermission(Long agentId) throws BizException;

    /**
     * 设置出版社书刊权限
     * @param agentPermission 广告位出版权限实体
     * @throws BizException
     */
    void setAgentPermission(AdvertisingAgentPermission agentPermission) throws BizException;

    /**
     * 设置编辑书刊权限
     * @param adviserPermission 广告位编辑权限实体
     * @throws BizException
     */
    void setAdviserPermission(AdvertisingAdviserPermission adviserPermission) throws BizException;

    /**
     * 编辑端获取书刊列表
     * @param paramMap  参数Map
     * @param pageParam 分页参数
     * @param adviserId 编辑ID
     * @return
     * @throws BizException
     */
    PageBean listBook4Adviser(Map<String, Object> paramMap, PageParam pageParam, Long adviserId) throws BizException;

    /**
     * 平台端获取某编辑书刊列表
     * @param paramMap  参数Map
     * @param pageParam 分页参数
     * @param adviserId 编辑ID
     * @return
     * @throws BizException
     */
    PageBean listBook4Platform(Map<String, Object> paramMap, PageParam pageParam, Long adviserId) throws BizException;

    /**
     * 获取广告位投放书刊列表
     * @param adId      广告位ID
     * @param name      书刊名称
     * @param pageParam 分页参数
     * @return
     * @throws BizException
     */
    PageBean distributionBookList(Long adId, String name, PageParam pageParam) throws BizException;

    /**
     * 获取广告位投放作品/应用数据统计
     * @param adId 广告位ID
     * @param pageParam 分页参数
     * @return
     * @throws BizException
     */
    PageBean distributionAppStatics(Long adId, PageParam pageParam) throws BizException;

    /**
     * 获取广告位投放时间统计
     * @param adId 广告位ID
     * @param day 天数
     * @return
     * @throws BizException
     */
    List<Object> distributionDateStatics(Long adId, Integer day) throws BizException;

    /**
     * 获取广告位列表
     * @param name      名称
     * @param pageParam 分页参数
     * @return
     * @throws BizException
     */
    PageBean list(String name, PageParam pageParam) throws BizException;

    /**
     * 客户端获取书刊配置的广告位信息
     * @param sceneId   二维码ID
     * @param adviserId 编辑ID
     * @param channelId 运营ID
     * @return
     * @throws BizException
     */
    AdvertisingSpaceDTO get4Wechat(Long sceneId, Long adviserId, Long channelId) throws BizException;

    /**
     * 曝光量埋点
     * @param advertisingExposureRecord 广告位曝光实体
     * @throws BizException
     */
    void addExposureTrack(AdvertisingExposureRecord advertisingExposureRecord) throws BizException;

    /**
     * 点击量埋点
     * @param advertisingClickRecord 广告位点击实体
     * @throws BizException
     */
    void addClickTrack(AdvertisingClickRecord advertisingClickRecord) throws BizException;
}
