package com.pcloud.book.advertising.biz.impl;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import com.pcloud.appcenter.app.dto.AppDto;
import com.pcloud.book.advertising.biz.AdvertisingSpaceBiz;
import com.pcloud.book.advertising.dao.AdvertisingAdviserPermissionDao;
import com.pcloud.book.advertising.dao.AdvertisingAgentPermissionDao;
import com.pcloud.book.advertising.dao.AdvertisingClickRecordDao;
import com.pcloud.book.advertising.dao.AdvertisingDistributionBookDao;
import com.pcloud.book.advertising.dao.AdvertisingExposureRecordDao;
import com.pcloud.book.advertising.dao.AdvertisingSpaceDao;
import com.pcloud.book.advertising.dto.AdvertisingSpaceDTO;
import com.pcloud.book.advertising.dto.DistributionAppDTO;
import com.pcloud.book.advertising.entity.AdvertisingAdviserPermission;
import com.pcloud.book.advertising.entity.AdvertisingAgentPermission;
import com.pcloud.book.advertising.entity.AdvertisingClickRecord;
import com.pcloud.book.advertising.entity.AdvertisingDistributionBook;
import com.pcloud.book.advertising.entity.AdvertisingExposureRecord;
import com.pcloud.book.advertising.entity.AdvertisingSpace;
import com.pcloud.book.advertising.enums.AdPositionEnum;
import com.pcloud.book.advertising.enums.AdSourceEnum;
import com.pcloud.book.advertising.enums.DistributionTypeEnum;
import com.pcloud.book.base.exception.BookBizException;
import com.pcloud.book.book.biz.BookBiz;
import com.pcloud.book.book.dao.BookDao;
import com.pcloud.book.book.dto.BookDto;
import com.pcloud.book.book.entity.BookAdviser;
import com.pcloud.book.book.set.BookSet;
import com.pcloud.book.consumer.app.AppConsr;
import com.pcloud.book.consumer.resource.ProductConsr;
import com.pcloud.book.consumer.user.AdviserConsr;
import com.pcloud.book.consumer.user.ChannelConsr;
import com.pcloud.book.util.common.CommonUtils;
import com.pcloud.channelcenter.qrcode.service.QrcodeSceneService;
import com.pcloud.common.core.aspect.ParamLog;
import com.pcloud.common.exceptions.BizException;
import com.pcloud.common.page.PageBean;
import com.pcloud.common.page.PageParam;
import com.pcloud.common.utils.DateUtils;
import com.pcloud.common.utils.ListUtils;
import com.pcloud.common.utils.ResponseHandleUtil;
import com.pcloud.common.utils.string.StringUtil;
import com.pcloud.resourcecenter.product.dto.Product4BookDTO;

/**
 * Description 广告位业务逻辑层接口实现类
 * @author PENG
 * @date 2019/2/28
 */
@Component("advertisingSpaceBiz")
public class AdvertisingSpaceBizImpl implements AdvertisingSpaceBiz {

    @Autowired
    private AdvertisingSpaceDao advertisingSpaceDao;
    @Autowired
    private AdviserConsr adviserConsr;
    @Autowired
    private AdvertisingDistributionBookDao advertisingDistributionBookDao;
    @Autowired
    private BookBiz bookBiz;
    @Autowired
    private AdvertisingAgentPermissionDao advertisingAgentPermissionDao;
    @Autowired
    private BookDao bookDao;
    @Autowired
    private BookSet bookSet;
    @Autowired
    private AdvertisingAdviserPermissionDao advertisingAdviserPermissionDao;
    @Autowired
    private QrcodeSceneService qrcodeSceneService;
    @Autowired
    private ChannelConsr channelConsr;
    @Autowired
    private AdvertisingExposureRecordDao advertisingExposureRecordDao;
    @Autowired
    private AdvertisingClickRecordDao advertisingClickRecordDao;
    @Autowired
    private ProductConsr productConsr;
    @Autowired
    private AppConsr appConsr;

    /**
     * 创建广告位
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @ParamLog("创建广告位")
    public Long create(AdvertisingSpace advertisingSpace) throws BizException {
        if (null == advertisingSpace.getAdName() || null == advertisingSpace.getAdPosition()
                || null == advertisingSpace.getAdSource() || null == advertisingSpace.getAdLink()) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数有误！");
        }
        if (!AdPositionEnum.checkCodeExist(advertisingSpace.getAdPosition())) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "广告位置选择有误！");
        }
        if (!ArrayUtils.contains(AdSourceEnum.AD_SOURCE_GATHER, advertisingSpace.getAdSource())) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "广告来源选择有误！");
        }
        advertisingSpace.setDistributionType(DistributionTypeEnum.SIGN_BOOK.value);
        advertisingSpace.setIsOpen(true);
        advertisingSpaceDao.insert(advertisingSpace);
        return advertisingSpace.getId();
    }

    /**
     * 修改广告位
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @ParamLog(value = "修改广告位", isAfterReturn = false)
    public void update(AdvertisingSpace advertisingSpace) throws BizException {
        if (null == advertisingSpace.getId() || null == advertisingSpace.getAdName() || null == advertisingSpace.getAdPosition()
                || null == advertisingSpace.getAdSource() || null == advertisingSpace.getAdLink()) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数有误！");
        }
        if (!AdPositionEnum.checkCodeExist(advertisingSpace.getAdPosition())) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "广告位置选择有误！");
        }
        if (!ArrayUtils.contains(AdSourceEnum.AD_SOURCE_GATHER, advertisingSpace.getAdSource())) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "广告来源选择有误！");
        }
        AdvertisingSpace space = advertisingSpaceDao.getById(advertisingSpace.getId());
        if (null == space) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "广告位不存在或已删除！");
        }
        advertisingSpaceDao.update(advertisingSpace);
    }

    /**
     * 停用广告位
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @ParamLog(value = "停用广告位", isAfterReturn = false)
    public void stop(AdvertisingSpace advertisingSpace) throws BizException {
        if (null == advertisingSpace.getId()) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数有误！");
        }
        AdvertisingSpace space = advertisingSpaceDao.getById(advertisingSpace.getId());
        if (null == space) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "广告位不存在或已删除！");
        }
        if (!space.getIsOpen()) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "广告位已停用！");
        }
        advertisingSpace.setIsOpen(false);
        advertisingSpaceDao.update(advertisingSpace);
    }

    /**
     * 删除广告位
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @ParamLog(value = "删除广告位", isAfterReturn = false)
    public void delete(AdvertisingSpace advertisingSpace) throws BizException {
        if (null == advertisingSpace.getId()) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数有误！");
        }
        AdvertisingSpace space = advertisingSpaceDao.getById(advertisingSpace.getId());
        if (null == space) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "广告位不存在或已删除！");
        }
        if (DistributionTypeEnum.NO_SIGN_BOOK.value.equals(space.getDistributionType())) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "广告位不能删除！");
        } else {
            Map<String, Object> paramMap = new HashMap<>();
            paramMap.put("adId", advertisingSpace.getId());
            Boolean hasBook = (Boolean) advertisingDistributionBookDao.getBy(paramMap, "checkAdSpaceHasBook");
            if (null != hasBook && hasBook) {
                throw new BookBizException(BookBizException.PARAM_IS_ERROR, "广告位不能删除！");
            }
        }
        advertisingSpaceDao.deleteById(advertisingSpace.getId());
    }

    /**
     * 获取广告位信息
     */
    @Override
    @ParamLog(value = "获取广告位信息")
    public AdvertisingSpaceDTO get(Long adId) throws BizException {
        return advertisingSpaceDao.getDTOById(adId);
    }

    /**
     * 投放书刊
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @ParamLog(value = "投放书刊", isAfterReturn = false)
    public void distributeBook(AdvertisingDistributionBook book) throws BizException {
        if (null == book.getAdId() || null == book.getDistributionType()
                || !ArrayUtils.contains(DistributionTypeEnum.DISTRIBUTION_TYPE_GATHER, book.getDistributionType())
                || (DistributionTypeEnum.SIGN_BOOK.value.equals(book.getDistributionType()) && (ListUtils.isEmpty(book.getBooks()) || null == book.getAdviserId()))) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数有误！");
        }
        if (DistributionTypeEnum.NO_SIGN_BOOK.value.equals(book.getDistributionType())) {
            Map<String, Object> paramMap = new HashMap<>();
            paramMap.put("adId", book.getAdId());
            AdvertisingSpaceDTO spaceDTO = (AdvertisingSpaceDTO) advertisingSpaceDao.getBy(paramMap, "getNoSignBookAd");
            if (null != spaceDTO) {
                throw new BookBizException(BookBizException.PARAM_IS_ERROR, "只能创建一个不指定书刊的广告位！");
            }
        }
        AdvertisingSpace space = new AdvertisingSpace();
        space.setId(book.getAdId());
        space.setDistributionType(book.getDistributionType());
        advertisingSpaceDao.update(space);
        if (DistributionTypeEnum.SIGN_BOOK.value.equals(book.getDistributionType())) {
            boolean paramError = book.getBooks().stream().anyMatch(bookAdviser -> null == bookAdviser.getBookId() || null == bookAdviser.getChannelId());
            if (paramError) {
                throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数有误！");
            }
            List<Long> bookIds = book.getBooks().stream().map(BookAdviser::getBookId).collect(Collectors.toList());
            List<Long> existIds = advertisingDistributionBookDao.checkBookIdExist(bookIds);
            if (!ListUtils.isEmpty(existIds)) {
                Long bookId = existIds.get(0);
                BookDto bookDto = bookBiz.getBaseById(bookId);
                throw new BookBizException(BookBizException.PARAM_IS_ERROR, StringUtil.addBracket(bookDto.getBookName()) + "等" + existIds.size() + "本书已投放广告位！");
            }
            List<AdvertisingDistributionBook> list = new ArrayList<>();
            Long agentId = adviserConsr.getAgentIdByAdviser(book.getAdviserId());
            for (BookAdviser bookAdviser : book.getBooks()) {
                AdvertisingDistributionBook distributionBook = new AdvertisingDistributionBook();
                distributionBook.setAdId(book.getAdId());
                distributionBook.setBookId(bookAdviser.getBookId());
                distributionBook.setChannelId(bookAdviser.getChannelId());
                distributionBook.setAdviserId(book.getAdviserId());
                distributionBook.setAgentId(agentId);
                list.add(distributionBook);
            }
            advertisingDistributionBookDao.batchInsert(list);
        }
    }

    /**
     * 获取出版社书刊权限
     */
    @Override
    @ParamLog("获取出版社书刊权限")
    public Boolean getAgentPermission(Long agentId) throws BizException {
        AdvertisingAgentPermission permission = advertisingAgentPermissionDao.getByAgentId(agentId);
        return null == permission ? true : permission.getIsOpen();
    }

    /**
     * 设置出版社书刊权限
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @ParamLog(value = "设置出版社书刊权限", isAfterReturn = false)
    public void setAgentPermission(AdvertisingAgentPermission agentPermission) throws BizException {
        if (null == agentPermission.getAgentId() || null == agentPermission.getIsOpen()) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数有误！");
        }
        AdvertisingAgentPermission permission = advertisingAgentPermissionDao.getByAgentId(agentPermission.getAgentId());
        if (null == permission) {
            advertisingAgentPermissionDao.insert(agentPermission);
        } else {
            agentPermission.setId(permission.getId());
            advertisingAgentPermissionDao.update(agentPermission);
        }
    }

    /**
     * 设置编辑书刊权限
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @ParamLog(value = "设置编辑书刊权限", isAfterReturn = false)
    public void setAdviserPermission(AdvertisingAdviserPermission adviserPermission) throws BizException {
        if (null == adviserPermission.getAdviserId() || null == adviserPermission.getIsOpen()
                || (!adviserPermission.getIsBatchOperation() && (null == adviserPermission.getBookId() || null == adviserPermission.getChannelId()))) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数有误！");
        }
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("adviserId", adviserPermission.getAdviserId());
        if (!adviserPermission.getIsBatchOperation()) {
            paramMap.put("bookId", adviserPermission.getBookId());
            paramMap.put("channelId", adviserPermission.getChannelId());
            AdvertisingAdviserPermission permission = (AdvertisingAdviserPermission) advertisingAdviserPermissionDao.getBy(paramMap, "getByAdviserId");
            if (null == permission) {
                advertisingAdviserPermissionDao.insert(adviserPermission);
            } else {
                adviserPermission.setId(permission.getId());
                advertisingAdviserPermissionDao.update(adviserPermission);
            }
        } else {
            // 删除编辑所有数据
            advertisingAdviserPermissionDao.deleteByAdviserId(adviserPermission.getAdviserId());
            List<BookDto> bookList = bookDao.listBookIdAndChannelId4Adviser(paramMap);
            if (!ListUtils.isEmpty(bookList)) {
                List<AdvertisingAdviserPermission> list = new ArrayList<>();
                for (BookDto bookDto : bookList) {
                    AdvertisingAdviserPermission permission = new AdvertisingAdviserPermission();
                    permission.setAdviserId(adviserPermission.getAdviserId());
                    permission.setIsOpen(adviserPermission.getIsOpen());
                    permission.setBookId(bookDto.getBookId());
                    permission.setChannelId(bookDto.getChannelId());
                    list.add(permission);
                }
                advertisingAdviserPermissionDao.batchInsert(list);
            }
        }
    }

    /**
     * 编辑端获取书刊列表
     */
    @Override
    public PageBean listBook4Adviser(Map<String, Object> paramMap, PageParam pageParam, Long adviserId) throws BizException {
        Long agentId = adviserConsr.getAgentIdByAdviser(adviserId);
        paramMap.put("agentId", agentId);
        PageBean pageBean = bookDao.listPage(pageParam, paramMap, "listBook4Adviser");
        if (pageBean == null || ListUtils.isEmpty(pageBean.getRecordList())) {
            return new PageBean(0, 0, new ArrayList<>());
        }
        // 填充渠道基础信息
        bookSet.setChannelInfoList(pageBean.getRecordList());
        return pageBean;
    }

    /**
     * 平台端获取某编辑书刊列表
     */
    @Override
    public PageBean listBook4Platform(Map<String, Object> paramMap, PageParam pageParam, Long adviserId) throws BizException {
        Long agentId = adviserConsr.getAgentIdByAdviser(adviserId);
        paramMap.put("agentId", agentId);
        PageBean pageBean = bookDao.listPage(pageParam, paramMap, "listBook4Platform");
        if (pageBean == null || ListUtils.isEmpty(pageBean.getRecordList())) {
            return new PageBean(0, 0, new ArrayList<>());
        }
        return pageBean;
    }

    /**
     * 获取广告位投放书刊列表
     */
    @Override
    public PageBean distributionBookList(Long adId, String name, PageParam pageParam) throws BizException {
        AdvertisingSpace space = advertisingSpaceDao.getById(adId);
        if (null == space) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "广告位不存在或已删除！");
        }
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("adId", adId);
        paramMap.put("name", StringUtil.isEmpty(name) ? null : name);
        PageBean pageBean;
        if (DistributionTypeEnum.SIGN_BOOK.value.equals(space.getDistributionType())) {
            pageBean = bookDao.listPage(pageParam, paramMap, "listBook4AdvertisingSpace");
        } else {
            pageBean = bookDao.listPage(pageParam, paramMap, "listBook4AdvertisingSpaceNoSignBook");
        }
        if (pageBean == null || ListUtils.isEmpty(pageBean.getRecordList())) {
            return new PageBean(0, 0, new ArrayList<>());
        }
        // 设置编辑名称
        bookSet.setAdviserName4BookDto(pageBean.getRecordList());
        // 设置出版名称
        bookSet.setAgentName4BookDto(pageBean.getRecordList());
        // 填充渠道公众号信息
        bookSet.setChannelOfficialAccountInfoList(pageBean.getRecordList());
        for (Object object : pageBean.getRecordList()) {
            BookDto bookDto = (BookDto) object;
            if (!DistributionTypeEnum.SIGN_BOOK.value.equals(space.getDistributionType())) {
                bookDto.setDistributionTime(space.getCreateTime());
            }
            // 曝光量
            Long exposureNum = getExposureNumByBookId(adId, bookDto.getBookId(), bookDto.getAdviserId(), bookDto.getChannelId());
            bookDto.setExposureNum(exposureNum);
            // 点击量
            Long clickNum = getClickNumByBookId(adId, bookDto.getBookId(), bookDto.getAdviserId(), bookDto.getChannelId());
            bookDto.setClickNum(clickNum);
        }
        return pageBean;
    }

    /**
     * 获取广告位投放作品/应用数据统计
     */
    @Override
    public PageBean distributionAppStatics(Long adId, PageParam pageParam) throws BizException {
        AdvertisingSpace space = advertisingSpaceDao.getById(adId);
        if (null == space) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "广告位不存在或已删除！");
        }
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("adId", adId);
        PageBean pageBean = advertisingExposureRecordDao.listPage(pageParam, paramMap, "distributionAppStatics");
        if (pageBean == null || ListUtils.isEmpty(pageBean.getRecordList())) {
            return new PageBean(0, 0, new ArrayList<>());
        }
        List<Long> productIds = new ArrayList<>();
        List<Long> appIds = new ArrayList<>();
        for (Object object : pageBean.getRecordList()) {
            DistributionAppDTO dto = (DistributionAppDTO) object;
            if ("PRODUCT".equals(dto.getFromType())) {
                productIds.add(dto.getFromId());
            } else if ("APP".equals(dto.getFromType())) {
                appIds.add(dto.getFromId());
            }
        }
        Map<Long, Product4BookDTO> productMap = productConsr.getProBasesWithSceneByIds(productIds);
        Map<Long, AppDto> appMap = appConsr.getBaseByIds(appIds);
        for (Object object : pageBean.getRecordList()) {
            DistributionAppDTO dto = (DistributionAppDTO) object;
            if ("PRODUCT".equals(dto.getFromType()) && !MapUtils.isEmpty(productMap) && productMap.containsKey(dto.getFromId())) {
                Product4BookDTO product4BookDTO = productMap.get(dto.getFromId());
                dto.setTitle(product4BookDTO.getProductName());
                dto.setSceneCode(product4BookDTO.getProductSceneCode());
                dto.setSceneName(product4BookDTO.getProductSceneName());
                dto.setCreateUser(product4BookDTO.getMerchantId());
                dto.setCreateUserName(product4BookDTO.getMerchantName());
                dto.setCreateRole(product4BookDTO.getIsSuperMerchant() ? "超级作者" : "普通作者");
            } else if ("APP".equals(dto.getFromType()) && !MapUtils.isEmpty(appMap) && appMap.containsKey(dto.getFromId())) {
                AppDto appDto = appMap.get(dto.getFromId());
                dto.setTitle(appDto.getTitle());
                dto.setSceneCode(appDto.getSceneCode());
                dto.setSceneName(appDto.getSceneName());
                dto.setCreateUser(appDto.getCreatedUser());
                dto.setCreateUserName(appDto.getAdviserName());
                dto.setCreateRole("编辑");
            }
        }
        return pageBean;
    }

    /**
     * 获取广告位投放时间统计
     */
    @Override
    @ParamLog("获取广告位投放时间统计")
    public List<Object> distributionDateStatics(Long adId, Integer day) throws BizException {
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("adId", adId);
        Date startTime = DateUtils.addDay(new Date(), -day);
        Date endTime = DateUtils.addDay(new Date(), -1);
        paramMap.put("collectDates", CommonUtils.collectLocalDates(startTime, endTime));
        return advertisingExposureRecordDao.listBy(paramMap, "distributionDateStatics");
    }

    /**
     * 获取书刊广告位曝光量
     */
    private Long getExposureNumByBookId(Long adId, Long bookId, Long adviserId, Long channelId) {
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("adId", adId);
        paramMap.put("bookId", bookId);
        paramMap.put("adviserId", adviserId);
        paramMap.put("channelId", channelId);
        Long exposureNum = (Long) advertisingExposureRecordDao.getBy(paramMap, "getExposureNum");
        return null == exposureNum ? 0L : exposureNum;
    }

    /**
     * 获取书刊广告位点击量
     */
    private Long getClickNumByBookId(Long adId, Long bookId, Long adviserId, Long channelId) {
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("adId", adId);
        paramMap.put("bookId", bookId);
        paramMap.put("adviserId", adviserId);
        paramMap.put("channelId", channelId);
        Long clickNum = (Long) advertisingClickRecordDao.getBy(paramMap, "getClickNum");
        return null == clickNum ? 0L : clickNum;
    }

    /**
     * 获取书刊广告位曝光量
     */
    private Long getExposureNumByAdId(Long adId) {
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("adId", adId);
        Long exposureNum = (Long) advertisingExposureRecordDao.getBy(paramMap, "getExposureNum");
        return null == exposureNum ? 0L : exposureNum;
    }

    /**
     * 获取书刊广告位点击量
     */
    private Long getClickNumByAdId(Long adId) {
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("adId", adId);
        Long clickNum = (Long) advertisingClickRecordDao.getBy(paramMap, "getClickNum");
        return null == clickNum ? 0L : clickNum;
    }

    /**
     * 获取广告位列表
     */
    @Override
    @ParamLog("获取广告位列表")
    public PageBean list(String name, PageParam pageParam) throws BizException {
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("name", StringUtil.isEmpty(name) ? null : name);
        PageBean pageBean = advertisingSpaceDao.listPage(pageParam, paramMap, "list");
        if (pageBean == null || ListUtils.isEmpty(pageBean.getRecordList())) {
            return new PageBean(0, 0, new ArrayList<>());
        }
        // 填充投放书籍
        for (Object object : pageBean.getRecordList()) {
            AdvertisingSpaceDTO spaceDTO = (AdvertisingSpaceDTO) object;
            if (null == spaceDTO) {
                continue;
            }
            spaceDTO.setExposureNum(getExposureNumByAdId(spaceDTO.getId()));
            spaceDTO.setClickNum(getClickNumByAdId(spaceDTO.getId()));
            if (DistributionTypeEnum.SIGN_BOOK.value.equals(spaceDTO.getDistributionType())) {
                List<AdvertisingDistributionBook> list = advertisingDistributionBookDao.getByAdId(spaceDTO.getId());
                if (!ListUtils.isEmpty(list)) {
                    spaceDTO.setBookNum(list.size());
                    Long bookId = list.get(0).getBookId();
                    BookDto bookDto = bookBiz.getBaseById(bookId);
                    spaceDTO.setBookName(StringUtil.addBracket(bookDto.getBookName()));
                } else {
                    spaceDTO.setBookNum(0);
                }
            }
        }
        return pageBean;
    }

    /**
     * 客户端获取书刊配置的广告位信息
     */
    @Override
    @ParamLog("客户端获取书刊配置的广告位信息")
    public AdvertisingSpaceDTO get4Wechat(Long sceneId, Long adviserId, Long channelId) throws BizException {
        Long agentId = adviserConsr.getAgentIdByAdviser(adviserId);
        Long bookId = ResponseHandleUtil.parseResponse(qrcodeSceneService.getBookId4SceneId(sceneId), Long.class);
        if (null == agentId || null == bookId) {
            return new AdvertisingSpaceDTO();
        }
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("bookId", bookId);
        paramMap.put("adviserId", adviserId);
        paramMap.put("channelId", channelId);
        paramMap.put("agentId", agentId);
        Boolean bookIsOpen = (Boolean) advertisingAdviserPermissionDao.getBy(paramMap, "checkBookOpen");
        if (null == bookIsOpen || !bookIsOpen) {
            return new AdvertisingSpaceDTO();
        }
        AdvertisingSpaceDTO spaceDTO = (AdvertisingSpaceDTO) advertisingSpaceDao.getBy(paramMap, "getDTOByBookId");
        if (null == spaceDTO) {
            spaceDTO = (AdvertisingSpaceDTO) advertisingSpaceDao.getBy(paramMap, "getNoSignBookAd");
        }
        if (null != spaceDTO) {
            spaceDTO.setBookId(bookId);
        }
        return spaceDTO;
    }

    /**
     * 曝光量埋点
     */
    @Override
    @ParamLog(value = "曝光量埋点", isAfterReturn = false)
    public void addExposureTrack(AdvertisingExposureRecord advertisingExposureRecord) throws BizException {
        if (StringUtil.isEmpty(advertisingExposureRecord.getFromType()) || null == advertisingExposureRecord.getFromId()
                || advertisingExposureRecord.getFromId() <= 0L) {
            return;
        }
        Long agentId = channelConsr.getParentId(advertisingExposureRecord.getChannelId());
        advertisingExposureRecord.setAgentId(agentId);
        advertisingExposureRecordDao.insert(advertisingExposureRecord);
    }

    /**
     * 点击量埋点
     */
    @Override
    @ParamLog(value = "点击量埋点", isAfterReturn = false)
    public void addClickTrack(AdvertisingClickRecord advertisingClickRecord) throws BizException {
        if (StringUtil.isEmpty(advertisingClickRecord.getFromType()) || null == advertisingClickRecord.getFromId()
                || advertisingClickRecord.getFromId() <= 0L) {
            return;
        }
        Long agentId = channelConsr.getParentId(advertisingClickRecord.getChannelId());
        advertisingClickRecord.setAgentId(agentId);
        advertisingClickRecordDao.insert(advertisingClickRecord);
    }
}
