package com.pcloud.book.advertising.facade.impl;

import java.util.HashMap;
import java.util.Map;
import org.codehaus.jackson.JsonParseException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.CookieValue;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import com.pcloud.book.advertising.biz.AdvertisingSpaceBiz;
import com.pcloud.book.advertising.dto.AdvertisingSpaceDTO;
import com.pcloud.book.advertising.entity.AdvertisingAdviserPermission;
import com.pcloud.book.advertising.entity.AdvertisingAgentPermission;
import com.pcloud.book.advertising.entity.AdvertisingClickRecord;
import com.pcloud.book.advertising.entity.AdvertisingDistributionBook;
import com.pcloud.book.advertising.entity.AdvertisingExposureRecord;
import com.pcloud.book.advertising.entity.AdvertisingSpace;
import com.pcloud.book.advertising.facade.AdvertisingSpaceFacade;
import com.pcloud.book.base.exception.BookBizException;
import com.pcloud.common.dto.ResponseDto;
import com.pcloud.common.exceptions.BizException;
import com.pcloud.common.page.PageParam;
import com.pcloud.common.permission.PermissionException;
import com.pcloud.common.utils.SessionUtil;
import com.pcloud.common.utils.cookie.Cookie;
import com.pcloud.common.utils.string.StringUtil;

/**
 * Description 广告位接口层接口实现类
 * @author PENG
 * @date 2019/2/28
 */
@RestController("advertisingSpaceFacade")
@RequestMapping("/advertising")
public class AdvertisingSpaceFacadeImpl implements AdvertisingSpaceFacade {

    @Autowired
    private AdvertisingSpaceBiz advertisingSpaceBiz;

    /**
     * 创建广告位
     */
    @Override
    @RequestMapping(value = "create", method = RequestMethod.POST)
    public ResponseDto<?> create(@RequestHeader("token") String token, @RequestBody AdvertisingSpace advertisingSpace)
            throws PermissionException, BizException, JsonParseException {
        if (null == advertisingSpace) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        SessionUtil.getToken4Redis(token);
        Map<String, Object> result = new HashMap<>();
        result.put("adId", advertisingSpaceBiz.create(advertisingSpace));
        return new ResponseDto<>(result);
    }

    /**
     * 修改广告位
     */
    @Override
    @RequestMapping(value = "update", method = RequestMethod.POST)
    public ResponseDto<?> update(@RequestHeader("token") String token, @RequestBody AdvertisingSpace advertisingSpace)
            throws PermissionException, BizException, JsonParseException {
        if (null == advertisingSpace) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        SessionUtil.getToken4Redis(token);
        advertisingSpaceBiz.update(advertisingSpace);
        return new ResponseDto<>();
    }

    /**
     * 获取广告位信息
     */
    @Override
    @RequestMapping(value = "get", method = RequestMethod.GET)
    public ResponseDto<?> get(@RequestHeader("token") String token, @RequestParam(value = "adId", required = false) Long adId)
            throws PermissionException, BizException, JsonParseException {
        if (null == adId) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        SessionUtil.getToken4Redis(token);
        return new ResponseDto<>(advertisingSpaceBiz.get(adId));
    }

    /**
     * 停用广告位
     */
    @Override
    @RequestMapping(value = "stop", method = RequestMethod.POST)
    public ResponseDto<?> stop(@RequestHeader("token") String token, @RequestBody AdvertisingSpace advertisingSpace)
            throws PermissionException, BizException, JsonParseException {
        if (null == advertisingSpace) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        SessionUtil.getToken4Redis(token);
        advertisingSpaceBiz.stop(advertisingSpace);
        return new ResponseDto<>();
    }

    /**
     * 删除广告位
     */
    @Override
    @RequestMapping(value = "delete", method = RequestMethod.POST)
    public ResponseDto<?> delete(@RequestHeader("token") String token, @RequestBody AdvertisingSpace advertisingSpace)
            throws PermissionException, BizException, JsonParseException {
        if (null == advertisingSpace) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        SessionUtil.getToken4Redis(token);
        advertisingSpaceBiz.delete(advertisingSpace);
        return new ResponseDto<>();
    }

    /**
     * 投放书刊
     */
    @Override
    @RequestMapping(value = "distributeBook", method = RequestMethod.POST)
    public ResponseDto<?> distributeBook(@RequestHeader("token") String token, @RequestBody AdvertisingDistributionBook advertisingDistributionBook)
            throws PermissionException, BizException, JsonParseException {
        if (null == advertisingDistributionBook) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        advertisingSpaceBiz.distributeBook(advertisingDistributionBook);
        return new ResponseDto<>();
    }

    /**
     * 获取出版社书刊权限
     */
    @Override
    @RequestMapping(value = "getAgentPermission", method = RequestMethod.GET)
    public ResponseDto<?> getAgentPermission(@RequestHeader("token") String token)
            throws PermissionException, BizException, JsonParseException {
        Long agentId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        Map<String, Object> result = new HashMap<>();
        result.put("isOpen", advertisingSpaceBiz.getAgentPermission(agentId));
        return new ResponseDto<>(result);
    }

    /**
     * 设置出版社书刊权限
     */
    @Override
    @RequestMapping(value = "setAgentPermission", method = RequestMethod.POST)
    public ResponseDto<?> setAgentPermission(@RequestHeader("token") String token, @RequestBody AdvertisingAgentPermission advertisingAgentPermission)
            throws PermissionException, BizException, JsonParseException {
        if (null == advertisingAgentPermission) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        Long agentId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        advertisingAgentPermission.setAgentId(agentId);
        advertisingSpaceBiz.setAgentPermission(advertisingAgentPermission);
        return new ResponseDto<>();
    }

    /**
     * 设置编辑书刊权限
     */
    @Override
    @RequestMapping(value = "setAdviserPermission", method = RequestMethod.POST)
    public ResponseDto<?> setAdviserPermission(@RequestHeader("token") String token, @RequestBody AdvertisingAdviserPermission advertisingAdviserPermission)
            throws PermissionException, BizException, JsonParseException {
        if (null == advertisingAdviserPermission) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        advertisingAdviserPermission.setAdviserId(adviserId);
        if (null == advertisingAdviserPermission.getIsBatchOperation()) {
            advertisingAdviserPermission.setIsBatchOperation(false);
        }
        advertisingSpaceBiz.setAdviserPermission(advertisingAdviserPermission);
        return new ResponseDto<>();
    }

    /**
     * 编辑端获取书刊列表
     */
    @Override
    @RequestMapping(value = "listBook4Adviser", method = RequestMethod.GET)
    public ResponseDto<?> listBook4Adviser(@RequestHeader("token") String token, @RequestParam(value = "name", required = false) String name,
            @RequestParam(value = "currentPage", required = false) Integer currentPage, @RequestParam(value = "numPerPage", required = false) Integer numPerPage)
            throws PermissionException, BizException, JsonParseException {
        Long adviserId = (Long) SessionUtil.getVlaue(token, SessionUtil.PARTY_ID);
        if (currentPage == null || numPerPage == null || currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("adviserId", adviserId);
        paramMap.put("name", StringUtil.isEmpty(name) ? null : name);
        return new ResponseDto<>(advertisingSpaceBiz.listBook4Adviser(paramMap, new PageParam(currentPage, numPerPage), adviserId));
    }

    /**
     * 平台端获取某编辑书刊列表
     */
    @Override
    @RequestMapping(value = "listBook4Platform", method = RequestMethod.GET)
    public ResponseDto<?> listBook4Platform(@RequestHeader("token") String token, @RequestParam(value = "adviserId", required = false) Long adviserId,
            @RequestParam(value = "name", required = false) String name, @RequestParam(value = "currentPage", required = false) Integer currentPage,
            @RequestParam(value = "numPerPage", required = false) Integer numPerPage) throws PermissionException, BizException, JsonParseException {
        if (currentPage == null || numPerPage == null || currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        if (null == adviserId) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        SessionUtil.getToken4Redis(token);
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("adviserId", adviserId);
        paramMap.put("name", StringUtil.isEmpty(name) ? null : name);
        return new ResponseDto<>(advertisingSpaceBiz.listBook4Platform(paramMap, new PageParam(currentPage, numPerPage), adviserId));
    }

    /**
     * 获取广告位列表
     */
    @Override
    @RequestMapping(value = "list", method = RequestMethod.GET)
    public ResponseDto<?> list(@RequestHeader("token") String token, @RequestParam(value = "name", required = false) String name,
            @RequestParam(value = "currentPage", required = false) Integer currentPage, @RequestParam(value = "numPerPage", required = false) Integer numPerPage)
            throws PermissionException, BizException, JsonParseException {
        if (currentPage == null || numPerPage == null || currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        return new ResponseDto<>(advertisingSpaceBiz.list(name, new PageParam(currentPage, numPerPage)));
    }

    /**
     * 获取广告位投放书刊列表
     */
    @Override
    @RequestMapping(value = "distributionBookList", method = RequestMethod.GET)
    public ResponseDto<?> distributionBookList(@RequestHeader("token") String token, @RequestParam(value = "adId", required = false) Long adId,
            @RequestParam(value = "name", required = false) String name, @RequestParam(value = "currentPage", required = false) Integer currentPage,
            @RequestParam(value = "numPerPage", required = false) Integer numPerPage) throws PermissionException, BizException, JsonParseException {
        if (currentPage == null || numPerPage == null || currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        if (null == adId) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        SessionUtil.getToken4Redis(token);
        return new ResponseDto<>(advertisingSpaceBiz.distributionBookList(adId, name, new PageParam(currentPage, numPerPage)));
    }

    /**
     * 获取广告位投放作品/应用数据统计
     */
    @Override
    @RequestMapping(value = "distributionAppStatics", method = RequestMethod.GET)
    public ResponseDto<?> distributionAppStatics(@RequestHeader("token") String token, @RequestParam(value = "adId", required = false) Long adId,
            @RequestParam(value = "currentPage", required = false) Integer currentPage, @RequestParam(value = "numPerPage", required = false) Integer numPerPage)
            throws PermissionException, BizException, JsonParseException {
        if (currentPage == null || numPerPage == null || currentPage < 0 || numPerPage < 0) {
            throw BookBizException.PAGE_PARAM_DELETION;
        }
        if (null == adId) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        SessionUtil.getToken4Redis(token);
        return new ResponseDto<>(advertisingSpaceBiz.distributionAppStatics(adId, new PageParam(currentPage, numPerPage)));
    }

    /**
     * 获取广告位投放时间统计
     */
    @Override
    @RequestMapping(value = "distributionDateStatics", method = RequestMethod.GET)
    public ResponseDto<?> distributionDateStatics(@RequestHeader("token") String token, @RequestParam(value = "adId", required = false) Long adId,
            @RequestParam(value = "day", required = false) Integer day) throws PermissionException, BizException, JsonParseException {
        if (null == adId) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        SessionUtil.getToken4Redis(token);
        if (null == day) {
            day = 7;
        }
        return new ResponseDto<>(advertisingSpaceBiz.distributionDateStatics(adId, day));
    }

    /**
     * 获取书刊配置的广告位信息
     */
    @Override
    @RequestMapping(value = "get4Wechat", method = RequestMethod.GET)
    public ResponseDto<?> get4Wechat(@CookieValue("userInfo") String userInfo)
            throws PermissionException, BizException, JsonParseException {
        Long sceneId = Cookie.getId(userInfo, Cookie._SCENE_ID);
        Long adviserId = Cookie.getId(userInfo, Cookie._ADVISER_ID);
        Long channelId = Cookie.getId(userInfo, Cookie._CHANNEL_ID);
        if (null == sceneId || null == adviserId || null == channelId) {
            return new ResponseDto<>(new AdvertisingSpaceDTO());
        }
        return new ResponseDto<>(advertisingSpaceBiz.get4Wechat(sceneId, adviserId, channelId));
    }

    /**
     * 曝光量埋点
     */
    @Override
    @RequestMapping(value = "addExposureTrack", method = RequestMethod.POST)
    public ResponseDto<?> addExposureTrack(@CookieValue("userInfo") String userInfo, @RequestBody AdvertisingExposureRecord advertisingExposureRecord)
            throws PermissionException, BizException, JsonParseException {
        if (null == advertisingExposureRecord || null == advertisingExposureRecord.getAdId() || null == advertisingExposureRecord.getBookId()) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        Long sceneId = Cookie.getId(userInfo, Cookie._SCENE_ID);
        Long adviserId = Cookie.getId(userInfo, Cookie._ADVISER_ID);
        Long channelId = Cookie.getId(userInfo, Cookie._CHANNEL_ID);
        Long officialAccountsId = Cookie.getId(userInfo, Cookie._OFFICIAL_ACCOUNTS_ID);
        Long wechatUserId = Cookie.getId(userInfo, Cookie._WECHAT_USER_ID);
        advertisingExposureRecord.setSceneId(sceneId);
        advertisingExposureRecord.setAdviserId(adviserId);
        advertisingExposureRecord.setChannelId(channelId);
        advertisingExposureRecord.setOfficialAccountId(officialAccountsId);
        advertisingExposureRecord.setWechatUserId(wechatUserId);
        advertisingSpaceBiz.addExposureTrack(advertisingExposureRecord);
        return new ResponseDto<>();
    }

    /**
     * 点击量埋点
     */
    @Override
    @RequestMapping(value = "addClickTrack", method = RequestMethod.POST)
    public ResponseDto<?> addClickTrack(@CookieValue("userInfo") String userInfo, @RequestBody AdvertisingClickRecord advertisingClickRecord)
            throws PermissionException, BizException, JsonParseException {
        if (null == advertisingClickRecord || null == advertisingClickRecord.getAdId() || null == advertisingClickRecord.getBookId()) {
            throw new BookBizException(BookBizException.PARAM_IS_NULL, "参数有误！");
        }
        Long sceneId = Cookie.getId(userInfo, Cookie._SCENE_ID);
        Long adviserId = Cookie.getId(userInfo, Cookie._ADVISER_ID);
        Long channelId = Cookie.getId(userInfo, Cookie._CHANNEL_ID);
        Long officialAccountsId = Cookie.getId(userInfo, Cookie._OFFICIAL_ACCOUNTS_ID);
        Long wechatUserId = Cookie.getId(userInfo, Cookie._WECHAT_USER_ID);
        advertisingClickRecord.setSceneId(sceneId);
        advertisingClickRecord.setAdviserId(adviserId);
        advertisingClickRecord.setChannelId(channelId);
        advertisingClickRecord.setOfficialAccountId(officialAccountsId);
        advertisingClickRecord.setWechatUserId(wechatUserId);
        advertisingSpaceBiz.addClickTrack(advertisingClickRecord);
        return new ResponseDto<>();
    }
}
