package com.pcloud.book.push.check;

import com.pcloud.book.base.exception.BookBizException;
import com.pcloud.book.push.entity.Push;
import com.pcloud.book.push.entity.PushGroup;
import com.pcloud.book.push.entity.PushItem;
import com.pcloud.book.push.enums.ItemTypeEnum;
import com.pcloud.book.push.enums.PushTypeEnum;
import com.pcloud.common.core.aspect.ParamLog;
import com.pcloud.common.utils.DateUtils;
import com.pcloud.common.utils.ListUtils;
import com.pcloud.common.utils.string.StringUtil;
import org.springframework.stereotype.Component;

import java.util.Date;
import java.util.List;

/**
 * @Description
 * @Author ruansiyuan
 * @Date 2019/4/18 15:48
 **/
@Component("pushCheck")
public class PushCheck {

    /**
     * 创建群发参数校验
     *
     * @param push
     */
    @ParamLog("创建群发参数校验")
    public void createPushParamCheck(Push push) {
        if (push == null) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "参数为空！");
        }
        if (push.getPushType() == null) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "群发类型不能为空！");
        }
        checkPushTime(push);
        if (ListUtils.isEmpty(push.getPushGroups())) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "发送群不能为空！");
        }
        if (ListUtils.isEmpty(push.getPushItems())) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "发送消息项不能为空！");
        }
        if (push.getPushItems().size() > 4) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "群发消息项不能超过4条！");
        }
        for (PushGroup pushGroup : push.getPushGroups()) {
            if (pushGroup == null) {
                throw new BookBizException(BookBizException.PARAM_IS_ERROR, "发送群不能为空！");
            }
            if (pushGroup.getBookId() == null) {
                throw new BookBizException(BookBizException.PARAM_IS_ERROR, "发送群书id不能为空！");
            }
            if (pushGroup.getClassifyId() == null) {
                throw new BookBizException(BookBizException.PARAM_IS_ERROR, "发送群分类id不能为空！");
            }
            if (pushGroup.getBookGroupQrcodeId()==null) {
                throw new BookBizException(BookBizException.PARAM_IS_ERROR, "发送群二维码id不能为空！");
            }
        }
        for (PushItem pushItem : push.getPushItems()) {
            if (pushItem == null) {
                throw new BookBizException(BookBizException.PARAM_IS_ERROR, "发送消息项不能为空！");
            }
            if (pushItem.getItemType() == null) {
                throw new BookBizException(BookBizException.PARAM_IS_ERROR, "发送消息项内容类型不能为空！");
            }
            if (ItemTypeEnum.TEXT.value.equals(pushItem.getItemType()) && StringUtil.isEmpty(pushItem.getTextContent())) {
                throw new BookBizException(BookBizException.PARAM_IS_ERROR, "发送消息项内容类型为文本时，文本不能为空！");
            }
            if (ItemTypeEnum.LINK.value.equals(pushItem.getItemType()) && StringUtil.isEmpty(pushItem.getLinkUrl())) {
                throw new BookBizException(BookBizException.PARAM_IS_ERROR, "发送消息项内容类型为链接时，链接不能为空！");
            }
            if (ItemTypeEnum.APP.value.equals(pushItem.getItemType()) && (pushItem.getAppId() == null && pushItem.getProductId() == null)) {
                throw new BookBizException(BookBizException.PARAM_IS_ERROR, "发送消息项内容类型为应用或作品时，应用id或作品id不能为空！");
            }
            if (ItemTypeEnum.APP.value.equals(pushItem.getItemType()) && pushItem.getAppId() != null && StringUtil.isEmpty(pushItem.getAppUrl())) {
                throw new BookBizException(BookBizException.PARAM_IS_ERROR, "发送消息项内容类型为应用时，应用链接不能为空！");
            }
            if (ItemTypeEnum.APP.value.equals(pushItem.getItemType()) && pushItem.getProductId() != null && StringUtil.isEmpty(pushItem.getProductUrl())) {
                throw new BookBizException(BookBizException.PARAM_IS_ERROR, "发送消息项内容类型为作品时，作品链接不能为空！");
            }
            if (ItemTypeEnum.IMAGE.value.equals(pushItem.getItemType()) && StringUtil.isEmpty(pushItem.getImageUrl())) {
                throw new BookBizException(BookBizException.PARAM_IS_ERROR, "发送消息项内容类型为图片时，图片不能为空！");
            }
        }

    }

    /**
     * 校验时间
     *
     * @param push
     */
    public void checkPushTime(Push push) {
        if (push == null) {
            return;
        }
        if (PushTypeEnum.DAY.value.equals(push.getPushType()) || PushTypeEnum.ONE.value.equals(push.getPushType())) {
            if (StringUtil.isEmpty(push.getPushTime())) {
                throw new BookBizException(BookBizException.PARAM_IS_ERROR, "设置为定时发送时发送时间不能为空！");
            }
            if (PushTypeEnum.ONE.value.equals(push.getPushType())) {
                Date pushOneDate = DateUtils.StringToDateTime(push.getPushTime());
                if (pushOneDate == null) {
                    throw new BookBizException(BookBizException.PARAM_IS_ERROR, "时间格式错误，应为2019-01-01 12:00:00！");
                }
                if (pushOneDate.getTime() - System.currentTimeMillis() <= 0) {
                    throw new BookBizException(BookBizException.PARAM_IS_ERROR, "发送时间不能早于当前时间！");
                }
            }
            if (PushTypeEnum.DAY.value.equals(push.getPushType())) {
                if (!push.getPushTime().matches("([0-1]?[0-9]|2[0-3]):([0-5][0-9]):([0-5][0-9])$")) {
                    throw new BookBizException(BookBizException.PARAM_IS_ERROR, "时间格式不对，应为HH:mm:ss！");
                }
            }
        }
    }

    /**
     * 更新群发参数校验
     * @param push
     */
    public void updatePushParamCheck(Push push) {
        createPushParamCheck(push);
        if (push.getId()==null){
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "群发id不能为空！");
        }
    }

    /**
     * 批量新增群发关联参数校验
     * @param pushGroups
     */
    public void createPushGroupBatchParamCheck(List<PushGroup> pushGroups) {
        if (ListUtils.isEmpty(pushGroups)){
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "群发关联为空！");
        }
        for (PushGroup pushGroup:pushGroups){
            if (pushGroup==null){
                throw new BookBizException(BookBizException.PARAM_IS_ERROR, "群发关联为空！");
            }
            if (pushGroup.getPushId()==null){
                throw new BookBizException(BookBizException.PARAM_IS_ERROR, "群发id不能为空！");
            }
            if (pushGroup.getBookId() == null) {
                throw new BookBizException(BookBizException.PARAM_IS_ERROR, "发送群书id不能为空！");
            }
            if (pushGroup.getClassifyId() == null) {
                throw new BookBizException(BookBizException.PARAM_IS_ERROR, "发送群分类id不能为空！");
            }
            if (pushGroup.getBookGroupQrcodeId()==null) {
                throw new BookBizException(BookBizException.PARAM_IS_ERROR, "发送群二维码id不能为空！");
            }
        }
    }

    /**
     * 新增群发消息项校验
     * @param pushItem
     */
    public void createPushItemParamCheck(PushItem pushItem) {
        if (pushItem == null) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "发送消息项不能为空！");
        }
        if (pushItem.getPushId() == null) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "群发id不能为空！");
        }
        if (pushItem.getItemType() == null) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "发送消息项内容类型不能为空！");
        }
        if (ItemTypeEnum.TEXT.value.equals(pushItem.getItemType()) && StringUtil.isEmpty(pushItem.getTextContent())) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "发送消息项内容类型为文本时，文本不能为空！");
        }
        if (ItemTypeEnum.LINK.value.equals(pushItem.getItemType()) && StringUtil.isEmpty(pushItem.getLinkUrl())) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "发送消息项内容类型为链接时，链接不能为空！");
        }
        if (ItemTypeEnum.APP.value.equals(pushItem.getItemType()) && (pushItem.getAppId() == null && pushItem.getProductId() == null)) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "发送消息项内容类型为应用时，应用id或作品id不能为空！");
        }
        if (ItemTypeEnum.IMAGE.value.equals(pushItem.getItemType()) && StringUtil.isEmpty(pushItem.getImageUrl())) {
            throw new BookBizException(BookBizException.PARAM_IS_ERROR, "发送消息项内容类型为图片时，图片不能为空！");
        }
    }
}
