/**
 *
 */
package com.pcloud.common.utils;

import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.time.temporal.TemporalAdjusters;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.pcloud.common.utils.string.StringUtil;

/**
 * @描述：jdk8中新增的日期处理类,更安全、更精确也更明确
 * @作者：songx
 * @创建时间：2017年7月20日,上午9:11:37 @版本：1.0
 */
public class LocalDateUtils {
    
    private static final DateTimeFormatter TIME = DateTimeFormatter.ofPattern("HHmmss");
    private static final DateTimeFormatter SHORT_MILLISECOND = DateTimeFormatter.ofPattern("yyyyMMddHHmmssSSS");
    private static final DateTimeFormatter SHORT_DATETIME = DateTimeFormatter.ofPattern("yyyyMMddHHmmss");
    private static final DateTimeFormatter DATETIME = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
    private static final DateTimeFormatter SHORT_DATE = DateTimeFormatter.ofPattern("yyyyMMdd");
    private static final DateTimeFormatter DATE = DateTimeFormatter.ofPattern("yyyy-MM-dd");
    
    /**
     * 获取当前的日期字符串(yyyy-MM-dd)
     *
     * @return
     */
    public static String getDateNow() {
        return LocalDate.now().format(DATE);
    }
    
    /**
     * 获取当前的日期字符串(yyyyMMdd)
     *
     * @return
     */
    public static String getShortDateNow() {
        return LocalDate.now().format(SHORT_DATE);
    }
    
    /**
     * 获取当前的时间字符串(HHmmss)
     *
     * @return
     */
    public static String getShortTimeNow() {
        return LocalDateTime.now().format(TIME);
    }
    
    /**
     * 获取当前的日期字符串(yyyy-MM-dd HH:mm:ss)
     *
     * @return
     */
    public static String getDateTimeNow() {
        return LocalDateTime.now().format(DATETIME);
    }
    
    /**
     * 获取当前的日期字符串(yyyyMMddHHmmss)
     *
     * @return
     */
    public static String getShortDateTimeNow() {
        return LocalDateTime.now().format(SHORT_DATETIME);
    }
    
    /**
     * 获取当前的日期字符串(yyyyMMddHHmmssSSS)
     *
     * @return
     */
    public static String getYmdhmss() {
        return LocalDateTime.now().format(SHORT_MILLISECOND);
    }
    
    /**
     * 时间戳转换成字符串格式的时间
     *
     * @param time
     * @return
     */
    public static String convertToString(long time) {
        return LocalDateTime.ofInstant(Instant.ofEpochMilli(time), ZoneId.systemDefault()).toString().replace("T", " ");
    }
    
    /**
     * 转换成Stirng类型的时间字符串(yyyy-MM-dd HH:mm:ss)
     *
     * @param localDateTime
     * @return
     */
    public static String convertToString(LocalDateTime localDateTime) {
        if (localDateTime == null) {
            return null;
        }
        return localDateTime.format(DATETIME);
    }
    
    /**
     * 获取当前日前多少天以前的日期
     *
     * @param days
     * @return
     */
    public static LocalDate getMinusDays(long days) {
        LocalDate localDate = LocalDate.now();
        return localDate.minusDays(days);
    }
    
    /**
     * 获取指定日前多少天以前的日期
     *
     * @param date
     * @param days
     * @return
     */
    public static LocalDate getMinusDays(LocalDate date, long days) {
        if (date == null) {
            throw new IllegalArgumentException("date is null");
        }
        return date.minusDays(days);
    }
    
    /**
     * 获取当前日前多少小时以前的日期
     *
     * @param hours
     * @return
     */
    public static LocalDateTime getMinusHours(long hours) {
        LocalDateTime localDateTime = LocalDateTime.now();
        return localDateTime.minusHours(hours);
    }
    
    /**
     * 获取当前日前多少分钟以前的日期
     *
     * @param minutes
     * @return
     */
    public static LocalDateTime getMinusMinutes(long minutes) {
        LocalDateTime localDateTime = LocalDateTime.now();
        return localDateTime.minusMinutes(minutes);
    }
    
    /**
     * 获取指定日前多少天以后的日期
     *
     * @param date
     * @param days
     * @return
     */
    public static LocalDateTime getPlusDays(Date date, long days) {
        LocalDateTime localDateTime = convertDateTime(date);
        return localDateTime.plusDays(days);
    }
    
    /**
     * String转换为localDate
     *
     * @param date
     * @return
     */
    public static LocalDate convertDate(String date) {
        if (StringUtil.isEmpty(date)) {
            return null;
        }
        return LocalDate.parse(date);
    }
    
    /**
     * Date转换为localDate
     *
     * @param date
     * @return
     */
    public static LocalDate convertDate(Date date) {
        LocalDateTime localDateTime = convertDateTime(date);
        if (localDateTime == null) {
            return null;
        }
        return localDateTime.toLocalDate();
    }
    
    /**
     * Date转换为localDateTime
     *
     * @param date
     * @return
     */
    public static LocalDateTime convertDateTime(Date date) {
        if (date == null) {
            return null;
        }
        Instant instant = date.toInstant();
        return LocalDateTime.ofInstant(instant, ZoneId.systemDefault());
    }
    
    /**
     * string转换为localDateTime
     *
     * @param dateTime
     * @return
     */
    public static LocalDateTime convertDateTime(String dateTime) {
        if (StringUtil.isEmpty(dateTime)) {
            return null;
        }
        if (!dateTime.contains("T")) {
            dateTime = dateTime.replace(" ", "T");
        }
        return LocalDateTime.parse(dateTime);
    }
    
    /**
     * 判断当前日期是否在指定日前之前,含当前日期
     *
     * @param date
     * @return
     */
    public static boolean isBefore(Date date) {
        if (date == null) {
            return false;
        }
        LocalDateTime localDateTime = convertDateTime(date);
        return LocalDateTime.now().isBefore(localDateTime);
    }
    
    /**
     * 判断当前日期是否在指定日前之后,含当前日期
     *
     * @param date
     * @return
     */
    public static boolean isAfter(Date date) {
        if (date == null) {
            return false;
        }
        LocalDateTime localDateTime = convertDateTime(date);
        return LocalDateTime.now().isAfter(localDateTime);
    }
    
    /**
     * 判断当前日期是否在指定日前之后,含当前日期
     *
     * @param localDateTime
     * @return
     */
    public static boolean isAfter(LocalDateTime localDateTime) {
        if (localDateTime == null) {
            return false;
        }
        return LocalDateTime.now().isAfter(localDateTime);
    }
    
    /**
     * 判断第一个日期是否在第二个日前之后
     *
     * @param firstDateTime
     * @param secondDateTime
     * @return
     */
    public static boolean isAfter(LocalDateTime firstDateTime, LocalDateTime secondDateTime) {
        if (firstDateTime == null || secondDateTime == null) {
            return false;
        }
        return firstDateTime.isAfter(secondDateTime);
    }
    
    /**
     * 判断第一个日期是否在第二个日前之后
     *
     * @param firstDateTime
     * @param secondDateTime
     * @return
     */
    public static boolean isAfter(String firstDateTime, String secondDateTime) {
        if (firstDateTime == null || secondDateTime == null) {
            return false;
        }
        return convertDateTime(firstDateTime).isAfter(convertDateTime(secondDateTime));
    }
    
    /**
     * 判断第一个日期是否在第二个日前之后
     *
     * @param firstDateTime
     * @param secondDateTime
     * @return
     */
    public static boolean isAfter(Date firstDateTime, Date secondDateTime) {
        if (firstDateTime == null || secondDateTime == null) {
            return false;
        }
        return convertDateTime(firstDateTime).isAfter(convertDateTime(secondDateTime));
    }
    
    /**
     * 判断当前日期是否在一定的日期之内,包含前后的两天，例：2017-07-15->[2017-07-01, 2017-07-15]
     *
     * @param startDate
     * @param endDate
     * @return
     */
    public static boolean isInIncludeDate(LocalDate startDate, LocalDate endDate) {
        if (startDate == null || endDate == null) {
            return false;
        }
        LocalDate nowTime = LocalDate.now();
        if (nowTime.isAfter(startDate.plusDays(-1)) && nowTime.isBefore(endDate.plusDays(1))) {
            return true;
        } else {
            return false;
        }
    }
    
    /**
     * 判断当前日期是否在一定的日期之内,包含前后的两天，例：2017-07-15->[2017-07-01, 2017-07-15]
     *
     * @param startDate
     * @param endDate
     * @return
     */
    public static boolean isInIncludeDateString(String startDate, String endDate) {
        return isInIncludeDate(LocalDate.parse(startDate), LocalDate.parse(endDate));
    }
    
    /**
     * 判断当前日期是否在一定的日期之内,不包含前后的两天，例：2017-07-15->(2017-07-01, 2017-07-15)
     *
     * @param startDate
     * @param endDate
     * @return
     */
    public static boolean isInBetweenDate(LocalDate startDate, LocalDate endDate) {
        if (startDate == null || endDate == null) {
            return false;
        }
        LocalDate nowTime = LocalDate.now();
        if (nowTime.isAfter(startDate) && nowTime.isBefore(endDate)) {
            return true;
        } else {
            return false;
        }
    }
    
    /**
     * 判断当前日期是否在一定的日期之内,不包含前后的两天，例：2017-07-15->(2017-07-01, 2017-07-15)
     *
     * @param startDate
     * @param endDate
     * @return
     */
    public static boolean isInBetweenDateString(String startDate, String endDate) {
        return isInBetweenDate(LocalDate.parse(startDate), LocalDate.parse(endDate));
    }
    
    /**
     * 判断当前日期是否在一定的日期之内， 例：2017-07-15 12:12:12->(2017-07-01 12:12:12, 2017-07-15
     * 12:12:12)
     *
     * @param startDateTime
     * @param endDateTime
     * @return
     */
    public static boolean isInBetweenDateTime(LocalDateTime startDateTime, LocalDateTime endDateTime) {
        if (startDateTime == null || endDateTime == null) {
            return false;
        }
        LocalDateTime nowTime = LocalDateTime.now();
        if (nowTime.isAfter(startDateTime) && nowTime.isBefore(endDateTime)) {
            return true;
        } else {
            return false;
        }
    }
    
    /**
     * 判断当前日期是否在一定的日期之内， 例：2017-07-15 12:12:12->(2017-07-01 12:12:12, 2017-07-15
     * 12:12:12)
     *
     * @param startDateTime
     * @param endDateTime
     * @return
     */
    public static boolean isInBetweenDateTime(Date startDateTime, Date endDateTime) {
        return isInBetweenDateTime(convertDateTime(startDateTime), convertDateTime(endDateTime));
    }
    
    /**
     * 判断当前日期是否在一定的日期之内， 例：2017-07-15 12:12:12->(2017-07-01 12:12:12, 2017-07-15
     * 12:12:12)
     *
     * @param startDateTime
     * @param endDateTime
     * @return
     */
    public static boolean isInBetweenDateTimeString(String startDateTime, String endDateTime) {
        return isInBetweenDateTime(LocalDateTime.parse(startDateTime), LocalDateTime.parse(endDateTime));
    }
    
    /**
     * 获取两个时间差，天，时，分
     *
     * @param stratDateTime
     * @param endDateTime
     * @return
     */
    public static long[] getDatePoor(LocalDateTime stratDateTime, LocalDateTime endDateTime) {
        if (stratDateTime == null || stratDateTime == null) {
            return null;
        }
        Duration duration = Duration.between(stratDateTime, endDateTime);
        // 计算差多少天
        long day = duration.toDays();
        // 计算差多少小时
        long hour = duration.toHours() - day * 24;
        // 计算差多少分钟
        long min = duration.toMinutes() - duration.toHours() * 60;
        return new long[]{day, hour, min};
    }
    
    /**
     * 获取指定时间与当前日期相差多少小时
     *
     * @param date
     * @return
     */
    public static long getHourPoor(Date date) {
        if (date == null) {
            return -1;
        }
        return getHourPoor(convertDateTime(date));
    }
    
    /**
     * 获取指定时间与当前日期相差多少小时
     *
     * @param dateTime
     * @return
     */
    public static long getHourPoor(LocalDateTime dateTime) {
        if (dateTime == null) {
            return -1;
        }
        Duration duration = Duration.between(dateTime, LocalDateTime.now());
        return duration.toHours();
    }
    
    /**
     * 获取指定时间与当前日期相差多少天
     *
     * @param date
     * @return
     */
    public static long getDayPoor(Date date) {
        if (date == null) {
            return -1;
        }
        return getDayPoor(convertDateTime(date));
    }
    
    /**
     * 获取指定时间与当前日期相差多少天
     *
     * @param dateTime
     * @return
     */
    public static long getDayPoor(LocalDateTime dateTime) {
        if (dateTime == null) {
            throw new IllegalArgumentException("dateTime is null");
        }
        Duration duration = Duration.between(dateTime, LocalDateTime.now());
        return duration.toDays();
    }
    
    /**
     * 获取指定时间相差多少天
     *
     * @param startDate 格式：2017-01-01
     * @param endDate   格式：2017-02-01
     * @return
     */
    public static long getDayPoor(String startDate, String endDate) {
        if (StringUtil.isEmpty(startDate) || StringUtil.isEmpty(endDate)) {
            throw new IllegalArgumentException("startDate or endDate is null");
        }
        return getDayPoor(convertDate(startDate), convertDate(endDate));
    }
    
    /**
     * 获取指定时间相差多少天
     *
     * @param startDate
     * @param endDate
     * @return
     */
    public static long getDayPoor(LocalDate startDate, LocalDate endDate) {
        if (startDate == null || endDate == null) {
            throw new IllegalArgumentException("startDate or endDate is null");
        }
        return startDate.until(endDate, ChronoUnit.DAYS);
    }
    
    /**
     * 获取月份
     *
     * @param dateTime 时间格式：2019-01-01 00:00:00
     * @return
     */
    public static int getMonth(String dateTime) {
        if (StringUtil.isEmpty(dateTime)) {
            return 0;
        }
        return convertDateTime(dateTime).getMonthValue();
    }
    
    /**
     * 获取当月第一天
     *
     * @param date
     * @return
     */
    public static LocalDate firstDay(LocalDate date) {
        if (date == null) {
            throw new IllegalArgumentException("date is null");
        }
        return date.with(TemporalAdjusters.firstDayOfMonth());
    }
    
    /**
     * 获取当月最后一天
     *
     * @param date
     * @return
     */
    public static LocalDate lastDay(LocalDate date) {
        if (date == null) {
            throw new IllegalArgumentException("date is null");
        }
        return date.with(TemporalAdjusters.lastDayOfMonth());
    }
    
    /**
     * 获取指定日期和天数以前的日期时间集合(不含指定日期)
     *
     * @param date
     * @return
     */
    public static List<LocalDate> getBeforeDays(LocalDate date, long days) {
        if (date == null) {
            throw new IllegalArgumentException("date is null");
        }
        List<LocalDate> localDates = new ArrayList<>();
        for (long i = days; i > 0; i--) {
            localDates.add(getMinusDays(date, i));
        }
        return localDates;
    }
    
    /**
     * 获取指定日期之间的日期集合(含指定日期)
     *
     * @param startdate
     * @param endDate
     * @return
     */
    public static List<LocalDate> getDates(LocalDate startdate, LocalDate endDate) {
        if (startdate == null || endDate == null) {
            throw new IllegalArgumentException("date is null");
        }
        return Stream.iterate(startdate, date -> date.plusDays(1))
                .limit(ChronoUnit.DAYS.between(startdate, endDate) + 1).collect(Collectors.toList());
    }
    
}
