package com.pcloud.common.utils.word.dto;

import java.util.List;

import org.apache.poi.xwpf.usermodel.ParagraphAlignment;
import org.apache.poi.xwpf.usermodel.TextAlignment;

import com.pcloud.common.utils.word.enums.ParagraphType;
import lombok.Getter;
import lombok.Setter;

/**
 * word中的段落
 *
 * @author：songx
 * @date：2019/5/27,19:00
 */
@Getter
@Setter
public class ParagraphDTO {

    /**
     * 段落类型
     */
    private ParagraphType paragraphType;

    /**
     * 是否页眉
     */
    private boolean header;

    /**
     * 是否页脚
     */
    private boolean footer;

    /**
     * 段落前间距，与前一个段落之间的间距
     */
    private int spacingBefore;

    /**
     * 段落后间距，与后一个段落之间的间距
     */
    private int spacingAfter;

    /**
     * 段落纵向对齐
     */
    private ParagraphAlignment paragraphAlignment;

    /**
     * 段落水平对齐
     */
    private TextAlignment textAlignment;

    /**
     * 段落后空多少行
     */
    private int afterEmpty;

    /**
     * 段落中的块,与run只能同时存在一个,runs优先级高
     */
    private List<RunDTO> runs;

    /**
     * 段落中的块,与runs只能同时存在一个,runs优先级高
     */
    private RunDTO run;

    public static ParagraphDTO newInstance() {
        return new ParagraphDTO();
    }

    /**
     * 段落
     *
     * @param paragraphType
     * @param run
     * @return
     */
    public static ParagraphDTO newParagraph(ParagraphType paragraphType, RunDTO run) {
        return newParagraph(paragraphType, null, null, run, 0);
    }

    /**
     * 段落
     *
     * @param paragraphType
     * @param run
     * @return
     */
    public static ParagraphDTO newParagraph(ParagraphType paragraphType, RunDTO run, int afterEmpty) {
        return newParagraph(paragraphType, null, null, run, afterEmpty);
    }

    /**
     * 段落
     *
     * @param paragraphType
     * @param run
     * @return
     */
    public static ParagraphDTO newParagraph(ParagraphType paragraphType, TextAlignment textAlignment, RunDTO run,
        int afterEmpty) {
        return newParagraph(paragraphType, null, textAlignment, run, afterEmpty);
    }

    /**
     * 段落
     *
     * @param paragraphType
     * @param run
     * @return
     */
    public static ParagraphDTO newParagraph(ParagraphType paragraphType, ParagraphAlignment paragraphAlignment,
        RunDTO run, int afterEmpty) {
        return newParagraph(paragraphType, paragraphAlignment, null, run, afterEmpty);
    }

    /**
     * 段落
     *
     * @param paragraphType
     * @param run
     * @return
     */
    public static ParagraphDTO newParagraph(ParagraphType paragraphType, ParagraphAlignment paragraphAlignment,
        TextAlignment textAlignment, RunDTO run, int afterEmpty) {
        ParagraphDTO paragraphDTO = ParagraphDTO.newInstance();
        paragraphDTO.setRun(run);
        paragraphDTO.setParagraphType(paragraphType);
        paragraphDTO.setParagraphAlignment(paragraphAlignment);
        paragraphDTO.setTextAlignment(textAlignment);
        paragraphDTO.setAfterEmpty(afterEmpty);
        return paragraphDTO;
    }

    /**
     * 段落
     *
     * @param paragraphType
     * @param runs
     * @return
     */
    public static ParagraphDTO newParagraph(ParagraphType paragraphType, List<RunDTO> runs) {
        ParagraphDTO paragraphDTO = ParagraphDTO.newInstance();
        paragraphDTO.setRuns(runs);
        paragraphDTO.setParagraphType(paragraphType);
        return paragraphDTO;
    }

    /**
     * 段落
     *
     * @param paragraphType
     * @param runs
     * @return
     */
    public static ParagraphDTO newParagraph(ParagraphType paragraphType, List<RunDTO> runs, int afterEmpty) {
        ParagraphDTO paragraphDTO = ParagraphDTO.newInstance();
        paragraphDTO.setRuns(runs);
        paragraphDTO.setParagraphType(paragraphType);
        paragraphDTO.setAfterEmpty(afterEmpty);
        return paragraphDTO;
    }

    /**
     * 段落
     *
     * @param paragraphType
     * @param runs
     * @return
     */
    public static ParagraphDTO newParagraph(ParagraphType paragraphType, ParagraphAlignment paragraphAlignment,
        List<RunDTO> runs, int afterEmpty) {
        ParagraphDTO paragraphDTO = ParagraphDTO.newInstance();
        paragraphDTO.setRuns(runs);
        paragraphDTO.setParagraphType(paragraphType);
        paragraphDTO.setParagraphAlignment(paragraphAlignment);
        paragraphDTO.setAfterEmpty(afterEmpty);
        return paragraphDTO;
    }

    /**
     * 页眉
     *
     * @param paragraphType
     * @param run
     * @return
     */
    public static ParagraphDTO newHeader(ParagraphType paragraphType, RunDTO run) {
        ParagraphDTO paragraphDTO = ParagraphDTO.newInstance();
        paragraphDTO.setHeader(true);
        paragraphDTO.setRun(run);
        paragraphDTO.setParagraphType(paragraphType);
        return paragraphDTO;
    }

    /**
     * 页眉
     *
     * @param paragraphType
     * @param runs
     * @return
     */
    public static ParagraphDTO newHeader(ParagraphType paragraphType, List<RunDTO> runs) {
        ParagraphDTO paragraphDTO = ParagraphDTO.newInstance();
        paragraphDTO.setHeader(true);
        paragraphDTO.setRuns(runs);
        paragraphDTO.setParagraphType(paragraphType);
        return paragraphDTO;
    }


    /**
     * 页脚
     *
     * @param paragraphType
     * @param run
     * @return
     */
    public static ParagraphDTO newFooter(ParagraphType paragraphType, ParagraphAlignment alignment, RunDTO run) {
        ParagraphDTO paragraphDTO = ParagraphDTO.newInstance();
        paragraphDTO.setFooter(true);
        paragraphDTO.setRun(run);
        paragraphDTO.setParagraphType(paragraphType);
        paragraphDTO.setParagraphAlignment(alignment);
        return paragraphDTO;
    }

    /**
     * 空行
     *
     * @return
     */
    public static ParagraphDTO newEmpty() {
        ParagraphDTO paragraphDTO = ParagraphDTO.newInstance();
        paragraphDTO.setRun(RunTextDTO.newEmpty());
        paragraphDTO.setParagraphType(ParagraphType.text);
        return paragraphDTO;
    }

}
